/*
 *  szecore_proc.c: ipv6 hw router linux driver, proc interface
 *  Copyright (c) 2004-2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <generated/utsrelease.h>
#include <linux/module.h>

#include <linux/init.h>
#include <linux/io.h>
#include <linux/proc_fs.h>
#include <linux/poll.h>
#include <linux/vmalloc.h>
#include <linux/fs.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/pci.h>
#include <linux/uaccess.h>

#include "szedata.h"
#include "szedatak.h"

static DEFINE_MUTEX(info_mutex);

struct szedata_info_private_data {
	struct szedata_info_buffer *rbuffer;
	struct szedata_info_buffer *wbuffer;
	struct szedata_info_entry *entry;
	void *file_private_data;
};

static int szedata_info_version_init(void);
static int szedata_info_version_done(void);
static void szedata_info_read_stats(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer);
static void szedata_info_read_debug(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer);
static void szedata_info_read_pktpages(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer);

/**
 * szedata_iprintf - printf on the procfs buffer
 * @buffer: the procfs buffer
 * @fmt: the printf format
 *
 * Outputs the string on the procfs buffer just like printf().
 *
 * Returns the size of output string.
 */
int szedata_iprintf(struct szedata_info_buffer *buffer, char *fmt, ...)
{
	va_list args;
	int res;
	char sbuffer[512];

	if (buffer->stop || buffer->error)
		return 0;
	va_start(args, fmt);
	res = vscnprintf(sbuffer, sizeof(sbuffer), fmt, args);
	va_end(args);
	if (buffer->size + res >= buffer->len) {
		buffer->stop = 1;
		return 0;
	}
	strcpy(buffer->curr, sbuffer);
	buffer->curr += res;
	buffer->size += res;
	return res;
}
EXPORT_SYMBOL(szedata_iprintf);

static struct proc_dir_entry *szedata_proc_root;

static void szedata_remove_proc_entry(struct proc_dir_entry *parent,
				     struct proc_dir_entry *de)
{
	if (de)
		remove_proc_entry(de->name, parent);
}

static ssize_t szedata_info_entry_read(struct file *file, char __user *buffer,
				   size_t count, loff_t *offset)
{
	struct szedata_info_private_data *data;
	struct szedata_info_entry *entry;
	struct szedata_info_buffer *buf;
	size_t size = 0;
	loff_t pos;

	data = file->private_data;
	pos = *offset;
	if (pos < 0 || (long) pos != pos || (ssize_t) count < 0)
		return -EIO;
	if ((unsigned long) pos + (unsigned long) count < (unsigned long) pos)
		return -EIO;
	entry = data->entry;
	switch (entry->content) {
	case SZEDATA_INFO_CONTENT_TEXT:
		buf = data->rbuffer;
		if (buf == NULL)
			return -EIO;
		if (pos >= buf->size)
			return 0;
		size = buf->size - pos;
		size = min(count, size);
		if (copy_to_user(buffer, buf->buffer + pos, size))
			return -EFAULT;
		break;
	}
	if ((ssize_t) size > 0)
		*offset = pos + size;
	return size;
}
static ssize_t szedata_info_entry_write(struct file *file,
		const char __user *buffer, size_t count, loff_t *offset)
{
	struct szedata_info_private_data *data;
	struct szedata_info_entry *entry;
	struct szedata_info_buffer *buf;
	size_t size = 0;
	off_t pos;

	data = file->private_data;
	pos = *offset;
	if (pos < 0 || (long) pos != pos || (ssize_t) count < 0)
		return -EIO;
	if ((unsigned long) pos + (unsigned long) count < (unsigned long) pos)
		return -EIO;
	entry = data->entry;
	switch (entry->content) {
	case SZEDATA_INFO_CONTENT_TEXT:
		buf = data->wbuffer;
		if (buf == NULL)
			return -EIO;
		if (pos >= (long)buf->len)
			return -ENOMEM;
		size = buf->len - pos;
		size = min(count, size);
		if (copy_from_user(buf->buffer + pos, buffer, size))
			return -EFAULT;
		if ((long)buf->size < pos + size)
			buf->size = pos + size;
		break;
	}
	if ((ssize_t) size > 0)
		*offset = pos + size;
	return size;
}

static int szedata_info_entry_open(struct inode *inode, struct file *file)
{
	struct szedata_info_entry *entry;
	struct szedata_info_private_data *data;
	struct szedata_info_buffer *buffer;
	struct proc_dir_entry *p;
	int mode, err;

	mutex_lock(&info_mutex);
	p = PDE(inode);
	entry = p == NULL ? NULL : (struct szedata_info_entry *)p->data;
	if (entry == NULL) {
		mutex_unlock(&info_mutex);
		return -ENODEV;
	}
	if (!try_module_get(entry->module)) {
		err = -EFAULT;
		goto __error1;
	}
	mode = file->f_flags & O_ACCMODE;
	if (mode == O_RDONLY || mode == O_RDWR) {
		if (entry->content == SZEDATA_INFO_CONTENT_TEXT &&
				!entry->c.text.read_size) {
			err = -ENODEV;
			goto __error;
		}
	}
	if (mode == O_WRONLY || mode == O_RDWR) {
		if (printk_ratelimit())
			printk(KERN_INFO "%s: writing not implemented\n",
					__func__);
		err = -ENODEV;
		goto __error;
	}
	data = kmalloc(sizeof(struct szedata_info_private_data), GFP_KERNEL);
	if (data == NULL) {
		err = -ENOMEM;
		goto __error;
	}
	memset(data, 0, sizeof(struct szedata_info_private_data));
	data->entry = entry;
	switch (entry->content) {
	case SZEDATA_INFO_CONTENT_TEXT:
		if (mode == O_RDONLY || mode == O_RDWR) {
			buffer = kmalloc(sizeof(struct szedata_info_buffer),
					GFP_KERNEL);
			if (buffer == NULL) {
				kfree(data);
				err = -ENOMEM;
				goto __error;
			}
			memset(buffer, 0, sizeof(struct szedata_info_buffer));
			buffer->len = (entry->c.text.read_size +
				      (PAGE_SIZE - 1)) & ~(PAGE_SIZE - 1);
			buffer->buffer = vmalloc(buffer->len);
			if (buffer->buffer == NULL) {
				kfree(buffer);
				kfree(data);
				err = -ENOMEM;
				goto __error;
			}
			buffer->curr = buffer->buffer;
			data->rbuffer = buffer;
		}
		break;
	}
	file->private_data = data;
	mutex_unlock(&info_mutex);
	if (entry->content == SZEDATA_INFO_CONTENT_TEXT &&
	    (mode == O_RDONLY || mode == O_RDWR)) {
		if (entry->c.text.read) {
			mutex_lock(&entry->access);
			entry->c.text.read(entry, data->rbuffer);
			mutex_unlock(&entry->access);
		}
	}
	return 0;

__error:
	module_put(entry->module);
__error1:
	mutex_unlock(&info_mutex);
	return err;
}

static int szedata_info_entry_release(struct inode *inode, struct file *file)
{
	struct szedata_info_entry *entry;
	struct szedata_info_private_data *data;
	int mode;

	mode = file->f_flags & O_ACCMODE;
	data = file->private_data;
	entry = data->entry;
	switch (entry->content) {
	case SZEDATA_INFO_CONTENT_TEXT:
		if (mode == O_RDONLY || mode == O_RDWR) {
			vfree(data->rbuffer->buffer);
			kfree(data->rbuffer);
		}
		break;
	}
	module_put(entry->module);
	kfree(data);
	return 0;
}

static struct file_operations szedata_info_entry_operations = {
	.owner =	THIS_MODULE,
	.read =		szedata_info_entry_read,
	.write =	szedata_info_entry_write,
	.open =		szedata_info_entry_open,
	.release =	szedata_info_entry_release,
};

/**
 * szedata_create_proc_entry - create a procfs entry
 * @name: the name of the proc file
 * @mode: the file permission bits, S_Ixxx
 * @parent: the parent proc-directory entry
 *
 * Creates a new proc file entry with the given name and permission
 * on the given directory.
 *
 * Returns the pointer of new instance or NULL on failure.
 */
static struct proc_dir_entry *szedata_create_proc_entry(const char *name,
			mode_t mode, struct proc_dir_entry *parent)
{
	struct proc_dir_entry *p;
	p = create_proc_entry(name, mode, parent);
#ifdef CONFIG_PROC_ENTRY_HAS_OWNER
	if (p)
		p->owner = THIS_MODULE;
#endif
	return p;
}

int __init szedata_info_init(void)
{
	struct proc_dir_entry *p;

	p = szedata_create_proc_entry("driver/szedata",
			S_IFDIR | S_IRUGO | S_IXUGO, NULL);
	if (p == NULL)
		return -ENOMEM;
	szedata_proc_root = p;
	szedata_info_version_init();
	return 0;
}

int szedata_info_done(void)
{
	szedata_info_version_done();
	szedata_remove_proc_entry(szedata_proc_root->parent, szedata_proc_root);
	return 0;
}

/**
 * szedata_info_device_create - create a szedata device proc file
 * @device: device to create info for
 *
 * called from szedata.c
 */
int szedata_info_device_create(struct szedata_device *device)
{
	char str[8];
	struct szedata_info_entry *entry;

	sprintf(str, "device%i", device->device);
	entry = szedata_info_create_module_entry(device->owner, str, NULL);
	if (entry == NULL)
		return -ENOMEM;
	entry->mode = S_IFDIR | S_IRUGO | S_IXUGO;
	if (szedata_info_register(entry) < 0) {
		szedata_info_free_entry(entry);
		return -ENOMEM;
	}
	device->proc_root = entry;
	entry = szedata_info_create_module_entry(device->owner, "stats",
			device->proc_root);
	if (entry) {
		szedata_info_set_text_ops(entry, device,
				device->max_apps * 1024,
				szedata_info_read_stats);
		if (szedata_info_register(entry) < 0) {
			szedata_info_free_entry(entry);
			entry = NULL;
		}
	}
	device->proc_stats = entry;
	entry = szedata_info_create_module_entry(device->owner, "debug",
			device->proc_root);
	if (entry) {
		szedata_info_set_text_ops(entry, device, 1024,
				szedata_info_read_debug);
		if (szedata_info_register(entry) < 0) {
			szedata_info_free_entry(entry);
			entry = NULL;
		}
	}
	device->proc_debug = entry;
	entry = szedata_info_create_module_entry(device->owner, "pktpages",
			device->proc_root);
	if (entry) {
		szedata_info_set_text_ops(entry, device, 128 * 1024,
				szedata_info_read_pktpages);
		if (szedata_info_register(entry) < 0) {
			szedata_info_free_entry(entry);
			entry = NULL;
		}
	}
	device->proc_pktpages = entry;
	return 0;
}

/**
 * szedata_info_device_free - de-register the device proc file
 * @device: device to unregister info for
 *
 * called from init.c
 */
int szedata_info_device_free(struct szedata_device *device)
{
	if (device->proc_pktpages) {
		szedata_info_unregister(device->proc_pktpages);
		device->proc_pktpages = NULL;
	}
	if (device->proc_debug) {
		szedata_info_unregister(device->proc_debug);
		device->proc_debug = NULL;
	}
	if (device->proc_stats) {
		szedata_info_unregister(device->proc_stats);
		device->proc_stats = NULL;
	}
	if (device->proc_root) {
		szedata_info_unregister(device->proc_root);
		device->proc_root = NULL;
	}
	return 0;
}

/**
 * szedata_info_create_entry - create an info entry
 * @name: the proc file name
 *
 * Creates an info entry with the given file name and initializes as
 * the default state.
 *
 * Usually called from other functions such as
 * szedata_info_create_device_entry().
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
static struct szedata_info_entry *szedata_info_create_entry(const char *name)
{
	struct szedata_info_entry *entry;
	entry = kzalloc(sizeof(struct szedata_info_entry), GFP_KERNEL);
	if (entry == NULL)
		return NULL;
	entry->name = kmalloc(strlen(name) + 1, GFP_KERNEL);
	if (entry->name == NULL) {
		kfree(entry);
		return NULL;
	}
	strcpy((char *)entry->name, name);
	entry->mode = S_IFREG | S_IRUGO;
	entry->content = SZEDATA_INFO_CONTENT_TEXT;
	mutex_init(&entry->access);
	return entry;
}
EXPORT_SYMBOL(szedata_info_create_entry);

/**
 * szedata_info_create_module_entry - create an info entry for the given module
 * @module: the module pointer
 * @name: the file name
 * @parent: the parent directory
 *
 * Creates a new info entry and assigns it to the given module.
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
struct szedata_info_entry *szedata_info_create_module_entry(
		struct module *module, const char *name,
		struct szedata_info_entry *parent)
{
	struct szedata_info_entry *entry = szedata_info_create_entry(name);
	if (entry) {
		entry->module = module;
		entry->parent = parent;
	}
	return entry;
}
EXPORT_SYMBOL(szedata_info_create_module_entry);

/**
 * szedata_info_create_device_entry - create an info entry for the given device
 * @device: the device instance
 * @name: the file name
 * @parent: the parent directory
 *
 * Creates a new info entry and assigns it to the given device.
 *
 * Returns the pointer of the new instance, or NULL on failure.
 */
struct szedata_info_entry *szedata_info_create_device_entry(
		struct szedata_device *device, const char *name,
		struct szedata_info_entry *parent)
{
	struct szedata_info_entry *entry = szedata_info_create_entry(name);
	if (entry) {
		entry->module = device->owner;
		entry->device = device;
		entry->parent = parent;
	}
	return entry;
}

/**
 * szedata_info_free_entry - release the info entry
 * @entry: the info entry
 *
 * Releases the info entry.  Don't call this after registered.
 */
void szedata_info_free_entry(struct szedata_info_entry *entry)
{
	if (entry == NULL)
		return;
	if (entry->name)
		kfree((char *)entry->name);
	if (entry->private_free)
		entry->private_free(entry);
	kfree(entry);
}
EXPORT_SYMBOL(szedata_info_free_entry);

/**
 * szedata_info_register - register the info entry
 * @entry: the info entry
 *
 * Registers the proc info entry.
 *
 * Returns zero if successful, or a negative error code on failure.
 */
int szedata_info_register(struct szedata_info_entry *entry)
{
	struct proc_dir_entry *root, *p = NULL;

	root = entry->parent == NULL ? szedata_proc_root : entry->parent->p;
	mutex_lock(&info_mutex);
	p = szedata_create_proc_entry(entry->name, entry->mode, root);
	if (!p) {
		mutex_unlock(&info_mutex);
		return -ENOMEM;
	}
#ifdef CONFIG_PROC_ENTRY_HAS_OWNER
	p->owner = entry->module;
#endif
	if (!S_ISDIR(entry->mode))
		p->proc_fops = &szedata_info_entry_operations;
	p->size = entry->size;
	p->data = entry;
	entry->p = p;
	mutex_unlock(&info_mutex);
	return 0;
}
EXPORT_SYMBOL(szedata_info_register);

/**
 * szedata_info_unregister - de-register the info entry
 * @entry: the info entry
 *
 * De-registers the info entry and releases the instance.
 *
 * Returns zero if successful, or a negative error code on failure.
 */
int szedata_info_unregister(struct szedata_info_entry *entry)
{
	struct proc_dir_entry *root;

	root = entry->parent == NULL ? szedata_proc_root : entry->parent->p;
	mutex_lock(&info_mutex);
	szedata_remove_proc_entry(root, entry->p);
	mutex_unlock(&info_mutex);
	szedata_info_free_entry(entry);
	return 0;
}
EXPORT_SYMBOL(szedata_info_unregister);

static struct szedata_info_entry *szedata_info_version_entry;

static void szedata_info_version_read(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer)
{
	static const char kernel_version[] = UTS_RELEASE;

	szedata_iprintf(buffer,
		    "SZEData Driver Version " CONFIG_SZEDATA_VERSION ".\n"
		    "Compiled on " __DATE__ " " __TIME__ " for kernel %s"
#ifdef CONFIG_SMP
		    " (SMP)"
#endif
#ifdef MODVERSIONS
		    " with versioned symbols"
#endif
		    ".\n", kernel_version);
}

static int __init szedata_info_version_init(void)
{
	struct szedata_info_entry *entry;

	entry = szedata_info_create_module_entry(THIS_MODULE, "version", NULL);
	if (entry == NULL)
		return -ENOMEM;
	entry->c.text.read_size = 256;
	entry->c.text.read = szedata_info_version_read;
	if (szedata_info_register(entry) < 0) {
		szedata_info_free_entry(entry);
		return -ENOMEM;
	}
	szedata_info_version_entry = entry;
	return 0;
}

static int szedata_info_version_done(void)
{
	if (szedata_info_version_entry)
		szedata_info_unregister(szedata_info_version_entry);
	return 0;
}

static void szedata_info_read_stats(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer)
{
	struct szedata_device *device = entry->private_data;
	unsigned int i;
	struct szedata_instance *instance;

	spin_lock_irq(&device->lock);
	szedata_iprintf(buffer, "General:\n");
	szedata_iprintf(buffer, "  Alloc blocks:\t\t%u\n",
			device->alloc_blocks);
	szedata_iprintf(buffer, "  Alloc bsize:\t\t%u\n",
			device->alloc_block_size);
	szedata_iprintf(buffer, "  Alloc mmap:\t\t%u\n",
			device->alloc_mmap_pages);
	szedata_iprintf(buffer, "  Alloc failed:\t\t%u\n",
			device->alloc_failed);
	szedata_iprintf(buffer, "  Alloc over:\t\t%u\n", device->alloc_over);
	szedata_iprintf(buffer, "  Active apps:\t\t%08x\n",
			device->active_apps);
	szedata_iprintf(buffer, "  Running apps:\t\t%08x\n", device->running);
	for (i = 0; i < device->max_apps; i++) {
		instance = &device->appinst[i];
		szedata_iprintf(buffer, "Application #%i:\n", i);
		szedata_iprintf(buffer, "  Subs. interfaces:\t%08x\n",
				instance->active_ports);
		szedata_iprintf(buffer, "  Waiting packets:\t%u\n",
				instance->apps_count);
		szedata_iprintf(buffer, "  Waiting peak pkts:\t%u\n",
				instance->apps_peak_count);
		szedata_iprintf(buffer, "  Poll threshold:\t%u\n",
				instance->poll_threshold);
		szedata_iprintf(buffer, "  Total locks:\t\t%llu\n",
				instance->apps_slocked);
		szedata_iprintf(buffer, "  Total unlocks:\t%llu\n",
				instance->apps_sunlocked);
	}
	spin_unlock_irq(&device->lock);
}

#define do_stats(block, what, locked, apps) do { \
	what++; \
	locked += block->locked_count; \
	apps += block->app_count; \
} while (0)

static void szedata_info_read_debug(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer)
{
	struct szedata_device *device = entry->private_data;
	struct szedata_block *block;
	unsigned int lused = 0, pused = 0, dused = 0, pfree = 0;
	unsigned int locked = 0, apps = 0;

	spin_lock_irq(&device->lock);
	list_for_each_entry(block, &device->locked_pool, list)
		do_stats(block, lused, locked, apps);
	list_for_each_entry(block, &device->used_pool, list)
		do_stats(block, pused, locked, apps);
	list_for_each_entry(block, &device->driver_pool, list)
		do_stats(block, dused, locked, apps);
	list_for_each_entry(block, &device->free_pool, list)
		do_stats(block, pfree, locked, apps);
	szedata_iprintf(buffer, "General:\n");
	szedata_iprintf(buffer, "  Locked pool:\t\t%u\n", lused);
	szedata_iprintf(buffer, "  Used pool:\t\t%u\n", pused);
	szedata_iprintf(buffer, "  Driver pool:\t\t%u\n", dused);
	szedata_iprintf(buffer, "  Free pool:\t\t%u\n", pfree);
	szedata_iprintf(buffer, "  Total locks:\t\t%u\n", locked);
	szedata_iprintf(buffer, "  Total waiting:\t%u\n", apps);
	spin_unlock_irq(&device->lock);
}

static void szedata_info_read_pktpages(struct szedata_info_entry *entry,
		struct szedata_info_buffer *buffer)
{
	struct szedata_device *device = entry->private_data;
	int i, last;
	struct page *page;
	void *vaddr;

	last = device->alloc_mmap_pages / PAGE_SIZE;
	for (i = 0; i < last; i++) {
		vaddr = device->mmap_page_table[i];
		page = virt_to_page(vaddr);
		szedata_iprintf(buffer, "%p->%p:%lx:%i\n", vaddr, page,
				page->flags, page_count(page));
	}
}
