/*
 * szedata 2 hw module
 *
 * Copyright (c) 2008 CESNET
 * Copyright (c) 2008 Jiri Slaby <jirislaby@gmail.com>
 *
 * Licensed under GPLv2
 */

#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/irq.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mutex.h>
#include <linux/pci.h>
#include <linux/timer.h>

#include "combo6k.h"
#include "combo6x.h"
#include "szedata2k.h"

#include "sze2c6x.h"

#include "kocompat.h"

#define SZET_BLOCK_COUNT	(64*1024*1024/SZET_BLOCK_SIZE)
#define SZET_BLOCK_SIZE		PAGE_SIZE

struct szedata2_hw {
	struct szedata2 *sd;
	struct combo6 *combo;
	struct szedata2_block *desc;
	unsigned int interfaces[2];
	unsigned int desc_pgs;
};

static unsigned long block_size = SZET_BLOCK_SIZE;
static unsigned int block_count = 0;
static unsigned int block_count_rx[32] = {
	[0 ... 31] = SZET_BLOCK_COUNT,
};
static unsigned int block_count_tx[32] = {
	[0 ... 31] = SZET_BLOCK_COUNT,
};
static unsigned int *const block_counts[2] = { block_count_rx, block_count_tx };
static int allocate_on_modprobe = 1;
static unsigned int poll_divisor = 8;

static inline unsigned int szedata2_hw_space_to_dir(unsigned int space)
{
	return space == SZE2_MMIO_RX ? SZE2_HW_DIR_RX : SZE2_HW_DIR_TX;
}

static struct szedata2_block *szedata2_hw_get_desc(struct szedata2_hw *sdhw,
		unsigned int space, unsigned int area)
{
	struct szedata2_block *desc = sdhw->desc;
	unsigned int a, page = 0, rx = space == SZE2_MMIO_RX;

	for (a = 0; a < (rx ? area : sdhw->interfaces[0]); a++)
		page += DIV_ROUND_UP(block_count_rx[a], PAGE_SIZE / 8 - 1);
	if (!rx)
		for (a = 0; a < area; a++)
			page += DIV_ROUND_UP(block_count_tx[a],
					PAGE_SIZE / 8 - 1);

	BUG_ON(page >= sdhw->desc_pgs);

	return desc + page;
}

static int szedata2_hw_alloc_descs(struct szedata2_hw *sdhw)
{
	unsigned int a, b;
	int ret;

	if (WARN_ON(sdhw->desc_pgs))
		return -EINVAL;

	for (a = 0; a < 2; a++)
		for (b = 0; b < sdhw->interfaces[a]; b++) {
			/*
			 * each entry is 8 bytes, we have PAGE_SIZE bytes minus
			 * one entry (pointer to the next block) per page
			 */
			sdhw->desc_pgs += DIV_ROUND_UP(block_counts[a][b],
					PAGE_SIZE / 8 - 1);
		}

	sdhw->desc = szedata2_alloc_dma(&sdhw->combo->pci->dev,
			sdhw->desc_pgs, PAGE_SIZE);
	if (IS_ERR(sdhw->desc)) {
		printk(KERN_ERR "szedata2: can't alloc pci space for "
				"descriptors\n");
		ret = PTR_ERR(sdhw->desc);
		goto err;
	}

	return 0;
err:
	sdhw->desc_pgs = 0;
	return ret;
}

static void szedata2_hw_free_descs(struct szedata2_hw *sdhw)
{
	if (!sdhw->desc_pgs)
		return;

	szedata2_free_dma(&sdhw->combo->pci->dev, sdhw->desc,
			sdhw->desc_pgs, PAGE_SIZE);
	sdhw->desc_pgs = 0;
}

static void szedata2_hw_fill_descs(struct szedata2_hw *sdhw,
		unsigned int space, unsigned int area)
{
	struct szedata2_block *dsc;
	__le64 *uninitialized_var(v);
	dma_addr_t fst_phys;
	unsigned int a, pos;

	if (!block_counts[space][area])
		return;

	/* fill descs */
	dsc = szedata2_hw_get_desc(sdhw, space, area);
	fst_phys = dsc->phys;
	pos = 0;
	while (1) {
		v = dsc->virt;
		for (a = 0; a < PAGE_SIZE / 8 - 1; a++) {
			dma_addr_t p;
			szedata2_get_phys(sdhw->sd, space, area, &p,
					pos * block_size, 1);
			*v++ = cpu_to_le64(p);
			if (++pos >= block_counts[space][area])
				goto finish;
		}

		dsc++;
		*v = cpu_to_le64(dsc->phys | SZE2_HW_DESCS_PTR);
	}
finish:
	*v = cpu_to_le64(fst_phys | SZE2_HW_DESCS_PTR);
}

static int szedata2_hw_alloc_dma(struct szedata2_hw *sdhw)
{
	unsigned int a;
	int ret;

	ret = szedata2_alloc_dmaspace(sdhw->sd, SZE2_MMIO_RX,
			sdhw->interfaces[0], block_count_rx, block_size,
			poll_divisor);
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace RX failed\n");
		goto err;
	}

	ret = szedata2_alloc_dmaspace(sdhw->sd, SZE2_MMIO_TX,
			sdhw->interfaces[1], block_count_tx, block_size,
			poll_divisor);
	if (ret) {
		printk(KERN_ERR "szedata2_alloc_dmaspace TX failed\n");
		goto err_rx;
	}

	ret = szedata2_hw_alloc_descs(sdhw);
	if (ret)
		goto err_tx;

	for (a = 0; a < sdhw->interfaces[0]; a++)
		szedata2_hw_fill_descs(sdhw, SZE2_MMIO_RX, a);
	for (a = 0; a < sdhw->interfaces[1]; a++)
		szedata2_hw_fill_descs(sdhw, SZE2_MMIO_TX, a);

	return 0;
err_tx:
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_TX);
err_rx:
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_RX);
err:
	return ret;
}

static void szedata2_hw_free_dma(struct szedata2_hw *sdhw)
{
	szedata2_hw_free_descs(sdhw);
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_RX);
	szedata2_free_dmaspace(sdhw->sd, SZE2_MMIO_TX);
}

static int szedata2_hw_open(struct szedata2 *sd)
{
	struct szedata2_hw *sdhw = sd->private;
	int ret;

	if (!allocate_on_modprobe) {
		ret = szedata2_hw_alloc_dma(sdhw);
		if (ret)
			return ret;
	}

	return 0;
}

static void szedata2_hw_close(struct szedata2 *sd)
{
	if (!allocate_on_modprobe)
		szedata2_hw_free_dma(sd->private);
}

static int szedata2_hw_start(struct szedata2 *sd, unsigned int space,
		unsigned int area)
{
	struct szedata2_hw *sdhw = sd->private;
	struct szedata2_block *dsc = szedata2_hw_get_desc(sdhw, space, area);
	struct combo6 *combo6 = sdhw->combo;
	unsigned int counter = 0;

	printk(KERN_DEBUG "starting %cX:iface%u\n", space ? 'T' : 'R', area);

	space = szedata2_hw_space_to_dir(space);

	/* set sizes */
	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_BUFSIZE),
			block_size * block_counts[space][area] - 1);
	/* set initial pointer to dsc[0] */
	combo6_fpga_writel(combo6, SZE2_HW_DESCS(space, area),
			(dsc->phys & 0xffffffffU) | SZE2_HW_DESCS_PTR);
	combo6_fpga_writel(combo6, SZE2_HW_DESCS(space, area) + 4,
			(u64)dsc->phys >> 32);

	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_STARTPTR),
			0);

	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_ENDPTR), 0);

	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_CTL),
			SZE2_HW_CTL_START);

	/* intr init */
	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_IRQ), 0);

	while (1) {
		u32 stat = combo6_fpga_readl(combo6, SZE2_HW_DMA(space, area,
					SZE2_HW_STAT));
		if (stat == SZE2_HW_STAT_IDLE || stat == SZE2_HW_STAT_RUNNING)
			break;
		if (counter++ > 100) {
			dev_warn(&combo6->pci->dev, "card didn't become ready "
					"in 100 us (no idle or running after "
					"start), stat=%.8x!\n", stat);
			break;
		}
		udelay(1);
	}

	combo6_br_writel(combo6, PCR_INTR_SET, SZE2_HW_INTR_RX |
			SZE2_HW_INTR_TX);

	return 0;
}

static void szedata2_hw_stop(struct szedata2 *sd, unsigned int space,
		unsigned int area)
{
	struct szedata2_hw *sdhw = sd->private;
	struct combo6 *combo6 = sdhw->combo;
	unsigned int counter = 0;

	printk(KERN_DEBUG "stopping %cX:iface%u\n", space ? 'T' : 'R', area);

	combo6_br_writel(combo6, PCR_INTR_MASK, 0);
	combo6_br_readl(combo6, PCR_INTR_MASK); /* posting */

	space = szedata2_hw_space_to_dir(space);

	if (space == SZE2_HW_DIR_TX)
		while (1) {
			u32 s, e;
			s = combo6_fpga_readl(combo6, SZE2_HW_DMA(space, area,
						SZE2_HW_STARTPTR));
			e = combo6_fpga_readl(combo6, SZE2_HW_DMA(space, area,
						SZE2_HW_ENDPTR));
			if (s == e)
				break;
			if (combo_fatal_signal_pending(current)) {
				dev_warn(&combo6->pci->dev, "ptrs not equal, "
						"killed by user\n");
				break;
			}
			msleep(10);
		}

	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_CTL),
			SZE2_HW_CTL_STOP);
	while (1) {
		u32 stat = combo6_fpga_readl(combo6, SZE2_HW_DMA(space, area,
					SZE2_HW_STAT));
		if (stat == SZE2_HW_STAT_IDLE)
			break;
		if (counter++ > 100) {
			dev_warn(&combo6->pci->dev, "card didn't stop in 1 "
					"whole second (no idle after stop), "
					"stat=%.8x!\n", stat);
			break;
		}
		msleep(10);
	}
}

static unsigned long szedata2_hw_get_ptr(struct szedata2 *sd,
		unsigned int space, unsigned int area, unsigned int which)
{
	struct szedata2_hw *sdhw = sd->private;
	struct combo6 *combo6 = sdhw->combo;

	space = szedata2_hw_space_to_dir(space);

	/* nonzero `which' means tail
	 * RX: START is tail changed by us, END is head changed by HW
	 * TX: START is tail changed by HW, END is head changed by us
	 */
	return combo6_fpga_readl(combo6, SZE2_HW_DMA(space, area,
				which ? SZE2_HW_STARTPTR : SZE2_HW_ENDPTR));
}

static void szedata2_hw_set_ptr(struct szedata2 *sd, unsigned int space,
		unsigned int area, unsigned long ptr)
{
	struct szedata2_hw *sdhw = sd->private;
	struct combo6 *combo6 = sdhw->combo;

	space = szedata2_hw_space_to_dir(space);

	/* we alter either RX tail or TX head */
	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area,
			space == SZE2_HW_DIR_RX ? SZE2_HW_STARTPTR :
			SZE2_HW_ENDPTR), ptr);
}

static void szedata2_hw_set_intr(struct szedata2 *sd, unsigned int space,
		unsigned int area, unsigned long value)
{
	struct szedata2_hw *sdhw = sd->private;
	struct combo6 *combo6 = sdhw->combo;
	u32 irq = SZE2_HW_IRQ_PTR(value) | SZE2_HW_IRQ_PTRE;

	if (space == SZE2_MMIO_RX)
		irq |= SZE2_HW_IRQ_TIMEOUTE;

	space = szedata2_hw_space_to_dir(space);

	combo6_fpga_writel(combo6, SZE2_HW_DMA(space, area, SZE2_HW_IRQ), irq);
}

static void szedata2_hw_set_timeout(struct szedata2 *sd, unsigned int space,
		unsigned int area, unsigned long value)
{
	struct szedata2_hw *sdhw = sd->private;
	struct combo6 *combo = sdhw->combo;

	space = szedata2_hw_space_to_dir(space);

	combo6_fpga_writel(combo, SZE2_HW_DMA(space, area, SZE2_HW_TIMEOUT),
			SZE2_HW_TIMEOUT_NS(value));
}

static void szedata2_hw_destroy(struct szedata2 *sd)
{
	struct szedata2_hw *sdhw = sd->private;

	szedata2_hw_free_descs(sdhw);
}

/*
 * INIT
 */

static int szedata2_hw_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	struct szedata2_hw *sdhw;
	struct szedata2 *sd;
	unsigned int ifaces[2];
	int ret;

	ifaces[0] = ifaces[1] = interfaces;

	BUG_ON(ifaces[0] >= ARRAY_SIZE(block_count_rx));
	BUG_ON(ifaces[1] >= ARRAY_SIZE(block_count_tx));

	sd = szedata2_alloc(sizeof(struct szedata2_hw), &combo->pci->dev);
	if (IS_ERR(sd)) {
		printk(KERN_ERR "szedata2_alloc failed\n");
		ret = PTR_ERR(sd);
		goto err_dest;
	}
	sd->owner = THIS_MODULE;
	sd->open = szedata2_hw_open;
	sd->close = szedata2_hw_close;
	sd->start = szedata2_hw_start;
	sd->stop = szedata2_hw_stop;
	sd->get_ptr = szedata2_hw_get_ptr;
	sd->set_ptr = szedata2_hw_set_ptr;
	sd->set_intr = szedata2_hw_set_intr;
	sd->set_timeout = szedata2_hw_set_timeout;
	sd->destroy = szedata2_hw_destroy;

	sdhw = sd->private;
	sdhw->sd = sd;
	memcpy(sdhw->interfaces, ifaces, sizeof(ifaces));
	sdhw->combo = combo;

	ret = szedata2_set_timeout(sd, 10000, 1000000);
	if (ret)
		goto err_dest;

	ret = szedata2_hw_alloc_dma(sdhw);
	if (ret)
		goto err_dest;

	ret = szedata2_register(sd);
	if (ret) {
		printk(KERN_ERR "szedata2_register failed\n");
		goto err_dest;
	}

	if (szedata2_eth_add(sd, ifaces, NULL))
		printk(KERN_ERR "szedata2_eth_init failed\n");

	combo->private_data = sd;

	return 0;
err_dest:
	szedata2_destroy(sd);
	return ret;
}

static int szedata2_hw_detach(struct combo6 *combo)
{
	struct szedata2 *sd = combo->private_data;

	szedata2_eth_remove(sd);
	szedata2_destroy(sd);

	return 0;
}

static irqreturn_t szedata2_hw_interrupt(struct combo6 *combo,
		unsigned int mask)
{
	struct szedata2 *sd = combo->private_data;
	struct szedata2_hw *sdhw = sd->private;
	u32 rx = 0, tx = 0;

	/* ACKs */
	if (sdhw->interfaces[0] != 0)
		rx = combo6_fpga_readl(combo, SZE2_HW_RX_IRQ);
	if (sdhw->interfaces[1] != 0)
		tx = combo6_fpga_readl(combo, SZE2_HW_TX_IRQ);

	if (!rx && !tx)
		return IRQ_NONE;

	szedata2_intr(sd, mask, rx, tx);

	return IRQ_HANDLED;
}

static const struct combo_device_id szedata2x_ids[] = {
	{
		.id_lsw	= 0x41f10100,
		.id_hsw = 0x41f103ff,
		.id_text = "NIFIC_C6X",
	}, {
		.id_lsw	= 0x41c10300,
		.id_hsw = 0x41c103ff,
		.id_text = "NIC_NetCOPE",
	}, {
		.id_lsw	= 0xf1010300,
		.id_hsw = 0xf10104ff,
		.id_text = "Flexible_FlowMon",
	}, {
	}
};

static struct combo_driver szedata2x_ops = {
	.drv = {
		.owner  = THIS_MODULE,
		.name	= "szedata2",
	},
	.dhw	= DHW_COMBO6X,
	.id_table = szedata2x_ids,
	.attach	= szedata2_hw_attach,
	.detach	= szedata2_hw_detach,
	.interrupt = szedata2_hw_interrupt,
};

static void copy_block_count(unsigned int dir, unsigned int area)
{
	if (block_count)
		block_counts[dir][area] = block_count;

	if (block_counts[dir][area] &&
			!is_power_of_2(block_counts[dir][area])) {
		block_counts[dir][area] =
			roundup_pow_of_two(block_counts[dir][area]);
		printk(KERN_INFO "szedata2: block_count for %cx[%u] is not "
				"power of 2, rounding up to %u\n",
				dir ? 't' : 'r', area, block_counts[dir][area]);
	}
}

int szedata2_hw_init(void)
{
	unsigned int a, b, sum_blocks = 0;

	if (block_size != PAGE_ALIGN(block_size)) {
		block_size = PAGE_ALIGN(block_size);
		printk(KERN_INFO "szedata2: block_size is not multiple of "
				"page size, rounding up to %lu\n",
				block_size);
	}

	if (poll_divisor < 1 || poll_divisor > block_size) {
		poll_divisor = clamp_t(unsigned int, poll_divisor,
				1, block_size);
		printk(KERN_INFO "szedata2: poll_divisor is not between 1 and "
				"block_size, clamped to %u\n", poll_divisor);
	}

	BUILD_BUG_ON(ARRAY_SIZE(block_count_rx) != ARRAY_SIZE(block_count_tx));

	for (a = 0; a < 2; a++)
		for (b = 0; b < ARRAY_SIZE(block_count_rx); b++) {
			copy_block_count(a, b);
			sum_blocks += block_counts[a][b];
		}

	if (!sum_blocks) {
		printk(KERN_ERR "%s: overall zero block count?\n", __func__);
		return -EINVAL;
	}

	combo6x_module_ref();

	return combo_register_driver(&szedata2x_ops);
}

void szedata2_hw_exit(void)
{
	combo_unregister_driver(&szedata2x_ops);
}

module_init(szedata2_hw_init);
module_exit(szedata2_hw_exit);

MODULE_LICENSE("GPL v2");

module_param(block_size, ulong, S_IRUGO);
MODULE_PARM_DESC(block_size, "block size [PAGE_SIZE]");
module_param(block_count, uint, S_IRUGO);
MODULE_PARM_DESC(block_count, "block count (overrides entries in both rx and "
		"tx arrays)");
module_param_array(block_count_rx, uint, NULL, S_IRUGO);
MODULE_PARM_DESC(block_count_rx, "block count array for rx "
		"[64*1024*1024/PAGE_SIZE]");
module_param_array(block_count_tx, uint, NULL, S_IRUGO);
MODULE_PARM_DESC(block_count_tx, "block count array for tx "
		"[64*1024*1024/PAGE_SIZE]");
module_param(allocate_on_modprobe, bool, 0400);
MODULE_PARM_DESC(allocate_on_modprobe, "allocate ring bufs on modprobe [yes]");
module_param(poll_divisor, uint, 0400);
MODULE_PARM_DESC(poll_divisor, "block_size*block_count/poll_divisor data wake "
		"up applications [8]");
