/*
 *  c6pcreth.c: ipv6 hw router driver, universal DMA driver for PCR bridge
 *  Copyright (c) 2006 CESNET
 *  Author(s): Jaroslav Kysela <perex@perex.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#include <linux/module.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/sched.h>

#include "combo6k.h"
#include "combo6x.h"
#include "combo6pcr.h"
#include "combo6pcrdma.h"

#define TX_TIMEOUT  (4*HZ)

#if 0
#define C6ETH_DEBUG
#endif
#if 0
#define C6ETH_RX_DEBUG
#endif
#if 0
#define C6ETH_TX_DEBUG
#endif

#define C6ETH_PKTSIZE	1536		/* maximal packet size (jumbo???) */

#define C6ETH_RX_BUFFERS	256		/* enqueued buffers per interface */
#define C6ETH_TX_BUFFERS	256		/* enqueued buffers per interface */

#define C6ETH_RX_FRAGS		(C6ETH_RX_BUFFERS * PCRDMA_NSEGS)
#define C6ETH_TX_FRAGS		(C6ETH_TX_BUFFERS * PCRDMA_NSEGS)
#define C6ETH_TX_BLOCK		(PCRDMA_NPORTS * PCRDMA_NSEGS)

#define C6ETH_TX_PCI_PHYS(priv, pos) \
	    ((priv)->sgdesc_phys + \
	     (C6ETH_RX_BUFFERS + (pos / PCRDMA_NSEGS)) * sizeof(struct pcrdma_dma))

#define C6ETH_DMA_RX_FLAGS	DMA_FLAGS_INTR_MASK
#define C6ETH_DMA_TX_FLAGS	0

#define PCRDMA_LIMIT_BLOCKS	0 /* unneeded here */

static char *dev_template = "c6eth";

struct c6eth_skbdma {
	struct sk_buff *skb;
	skb_frag_t *frag;
	struct pcrdma_seg *seg;	/* first DMA segment for sk_buff */
	struct pcrdma_dma *dma;
	dma_addr_t dma_addr;	/* address on PCI bus */
};

struct c6eth_private {
	struct combo6 *combo6;
	int interfaces;
	struct net_device *dev[4];
	unsigned long cseg_size;
	struct pcrdma_cseg *cseg_virt;
	dma_addr_t cseg_phys;
	struct pcrdma_config *config_virt;
	dma_addr_t config_phys;
	unsigned long sgdesc_size;
	char *sgdesc_virt;
	dma_addr_t sgdesc_phys;
	struct c6eth_skbdma rx_skbdma[C6ETH_RX_FRAGS];
	struct c6eth_skbdma tx_skbdma[C6ETH_TX_FRAGS];
	unsigned int rx_pos;			/* to rx_skbdma */
	unsigned int tx_pos[PCRDMA_NPORTS];	/* to tx_skbdma */
	unsigned int tx_pkt[PCRDMA_NPORTS];	/* actual queued packets */
	unsigned int tx_run_head[2];		/* first segment */
	unsigned int tx_run_tail[2];		/* last segment */
	unsigned int tx_run_fill[2];		/* filled segments */
	unsigned int tx_cur_head;		/* first segment */
	unsigned int tx_cur_tail;		/* last segment */
	unsigned int tx_cur_fill;		/* filled segments */
	unsigned int tx_run, tx_runf;
	unsigned int tx_run_count;
	unsigned int tx_queue;
	u32 hh_size;
	u32 pkt_skip;
	u32 tx_hw_packets;
	u32 enable;
	u32 pcrver;
	struct mutex open_mutex;
	struct combo6_info_entry *proc_regs;
};

struct c6eth_netdev {
	unsigned int index;
	struct net_device *dev;
	struct combo6 *combo6;
	struct combo6_driver *driver;
	struct net_device_stats stats;
};

MODULE_AUTHOR("CESNET; Jaroslav Kysela <perex@perex.cz>");
MODULE_DESCRIPTION("Combo6 IPV6 HW Router Linux Driver - Universal DMA Interface for PCR based code");
MODULE_LICENSE("GPL");

static int rx_copybreak = 256;
module_param(rx_copybreak, int, 0644);
MODULE_PARM_DESC(rx_copybreak, "Copy breakpoint for copy-only-tiny-frames");

static int netdev_open(struct net_device *dev);
static int netdev_close(struct net_device *dev);
static struct net_device_stats *get_stats(struct net_device *dev);
static void set_rx_mode(struct net_device *dev);
static int netdev_ioctl(struct net_device *dev, struct ifreq *rq, int cmd);
static void tx_timeout(struct net_device *dev);
static int change_mtu(struct net_device *dev, int new_mtu);
static int c6eth_tx(struct sk_buff *skb, struct net_device *dev);
static void check_rx(struct combo6 *combo);
static void check_tx(struct combo6 *combo);
static void c6eth_tx_start(struct c6eth_private *xpriv);

/*
 * interrupt handler
 */
static irqreturn_t c6eth_interrupt(struct combo6 *combo, unsigned int xmask)
{
	if (xmask & PCRDMA_INTR_RX_MASK)
		check_rx(combo);
	if (xmask & PCRDMA_INTR_TX_MASK)
		check_tx(combo);
	if (xmask & PCRDMA_INTR_ERR_MASK) {
		//printk(KERN_ERR "GPR(1) 0x%x\n", combo6_br_readl(driver->combo, PCR_GPR(1)));
		//printk(KERN_ERR "DEBUG 0x%x\n", combo6_br_readl(driver->combo, PCRDMA_DEBUG));
		printk(KERN_ERR "combo6 pcidma error IRQ!\n");
	}
	return IRQ_HANDLED;
}

/*
 * packet I/O part
 */

static void my_alloc_skb(struct combo6 *combo6,
			struct c6eth_skbdma *sd)
{
	sd->skb = dev_alloc_skb(C6ETH_PKTSIZE + 2);
	if (sd->skb == NULL) {
		sd->seg->seg_stlen = 0;
		return;
	}
	skb_reserve(sd->skb, 2);
	sd->seg->seg_addr = cpu_to_le32(
			      pci_map_single(combo6->pci, sd->skb->data,
					     C6ETH_PKTSIZE + 2,
					     PCI_DMA_FROMDEVICE));
	sd->seg->seg_stlen = cpu_to_le32(C6ETH_PKTSIZE);
}

static void check_rx(struct combo6 *combo6)
{
	struct c6eth_private *xpriv = combo6->private_data;
	struct c6eth_netdev *priv;
	struct net_device *netdev;
	struct c6eth_skbdma *sd;
	struct pcrdma_dma *dma = NULL;
	dma_addr_t dma_addr = 0;	/* prevent warning */
	u32 seg_stlen, seg_len, port;

	while (1) {
		sd = &xpriv->rx_skbdma[xpriv->rx_pos];
		if (sd->dma != dma) {
			if (dma != NULL) {
				/* clear DONE flag */
				dma->dma_flags = cpu_to_le32(C6ETH_DMA_RX_FLAGS);
				pci_dma_sync_single_for_device(combo6->pci,
							       dma_addr,
							       sizeof(struct pcrdma_dma),
							       PCI_DMA_TODEVICE);
			}
			dma = sd->dma;
			dma_addr = sd->dma_addr;
			pci_dma_sync_single_for_cpu(combo6->pci,
						    sd->dma_addr,
						    sizeof(struct pcrdma_dma),
						    PCI_DMA_FROMDEVICE);
		}
		if ((sd->dma->dma_flags & cpu_to_le32(DMA_FLAGS_DONE_MASK)) == 0)
			break;
		if (sd->skb == NULL) {
			my_alloc_skb(combo6, sd);
			goto __rx_skip;
		}
#ifdef C6ETH_RX_DEBUG
		printk("RX DMA (%i) finished, flags = 0x%x, seg_stlen = 0x%x\n", xpriv->rx_pos, sd->dma->dma_flags, sd->seg->seg_stlen);
#endif
		seg_stlen = le32_to_cpu(sd->seg->seg_stlen);
		seg_len = SEG_STLEN_LENGTH(seg_stlen);
		if (seg_stlen & (3 << SEG_STLEN_PORT_SHIFT)) {
			port = (seg_stlen & (2 << SEG_STLEN_PORT_SHIFT)) ? 1 : 0;
		} else {
			port = (seg_stlen & (8 << SEG_STLEN_PORT_SHIFT)) ? 3 : 2;
		}
		if ((xpriv->enable & CTRL_ENABLE(port)) == 0) {
		      __rx_skip:
			sd->seg->seg_stlen = cpu_to_le32(C6ETH_PKTSIZE);
			goto __rx_next;
		}
		netdev = xpriv->dev[port];
		priv = netdev_priv(netdev);
		if (seg_stlen & SEG_STLEN_ERR_MASK)
			priv->stats.rx_errors++;
		if (seg_stlen & SEG_STLEN_CRCERR_MASK)
			priv->stats.rx_crc_errors++;
		if (seg_stlen & SEG_STLEN_OVER_MASK)
			priv->stats.rx_over_errors++;
		if ((seg_stlen & SEG_STLEN_FOP_MASK) == 0 ||
		    (seg_stlen & SEG_STLEN_EOP_MASK) == 0 ||
		    (seg_stlen & SEG_STLEN_PORT_MASK) == 0 ||
		    seg_len == 0) {
			goto __rx_skip;
		}
		pci_dma_sync_single_for_cpu(combo6->pci, le32_to_cpu(sd->seg->seg_addr),
					    seg_len, PCI_DMA_FROMDEVICE);
		if (seg_len <= rx_copybreak) {
			struct sk_buff *new_skb;

			new_skb = dev_alloc_skb(seg_len + 2);
			if (new_skb) {
				new_skb->dev = netdev;
				new_skb->ip_summed = sd->skb->ip_summed;
				skb_reserve(new_skb, 2);
				memcpy(new_skb->data, sd->skb->data, seg_len);
				skb_put(new_skb, seg_len);
				new_skb->protocol = eth_type_trans(new_skb, netdev);
				netif_rx(new_skb);
				sd->seg->seg_stlen = cpu_to_le32(C6ETH_PKTSIZE);
			} else {
				goto __std_rx;
			}
		} else {
		      __std_rx:
			pci_unmap_single(combo6->pci, le32_to_cpu(sd->seg->seg_addr),
					 C6ETH_PKTSIZE, PCI_DMA_FROMDEVICE);
			sd->skb->dev = netdev;
			skb_put(sd->skb, seg_len);
			sd->skb->protocol = eth_type_trans(sd->skb, netdev);
			netif_rx(sd->skb);
			my_alloc_skb(combo6, sd);
		}
		priv->stats.rx_packets++;
		priv->stats.rx_bytes += seg_len;
	      __rx_next:
		xpriv->rx_pos++;
		xpriv->rx_pos %= C6ETH_RX_FRAGS;
	}
}

static void check_tx(struct combo6 *combo6)
{
	struct c6eth_private *xpriv = combo6->private_data;
	struct c6eth_netdev *priv;
	struct c6eth_skbdma *sd;
	struct pcrdma_dma *dma;
	unsigned int tmp;
	u32 seg_stlen, seg_len, port;

      __again:
	if (xpriv->tx_run_count == 0)
		return;
	dma = NULL;
	sd = NULL;
	while (xpriv->tx_run_fill[xpriv->tx_run] > 0) {
		sd = &xpriv->tx_skbdma[xpriv->tx_run_head[xpriv->tx_run]];
		if (sd->dma != dma) {
			if (dma != NULL) {
				/* clear DONE flag */
				dma->dma_flags = cpu_to_le32(C6ETH_DMA_TX_FLAGS);
				pci_dma_sync_single_for_device(combo6->pci,
							       sd->dma_addr,
							       sizeof(struct pcrdma_dma),
							       PCI_DMA_TODEVICE);
			}
			dma = sd->dma;
			pci_dma_sync_single_for_cpu(combo6->pci,
						    sd->dma_addr,
						    sizeof(struct pcrdma_dma),
						    PCI_DMA_FROMDEVICE);
		}
		if ((sd->dma->dma_flags & cpu_to_le32(DMA_FLAGS_DONE_MASK)) == 0) {
			dma = NULL;
			break;
		}
		if (sd->skb == NULL && sd->frag == NULL) {
			tmp = PCRDMA_NSEGS - (xpriv->tx_run_head[xpriv->tx_run] % PCRDMA_NSEGS);
			xpriv->tx_run_head[xpriv->tx_run] += tmp;
			xpriv->tx_run_head[xpriv->tx_run] %= C6ETH_TX_FRAGS;
			xpriv->tx_run_fill[xpriv->tx_run] -= tmp;
			continue;
		}
#ifdef C6ETH_TX_DEBUG
		printk("TX DMA (%i) finished, flags = 0x%x, seg_stlen = 0x%x\n", xpriv->tx_run_head[xpriv->tx_run], sd->dma->dma_flags, sd->seg->seg_stlen);
#endif
		seg_stlen = le32_to_cpu(sd->seg->seg_stlen);
		seg_len = SEG_STLEN_LENGTH(seg_stlen);
		if (seg_stlen & (3 << SEG_STLEN_PORT_SHIFT)) {
			port = (seg_stlen & (2 << SEG_STLEN_PORT_SHIFT)) ? 1 : 0;
		} else {
			port = (seg_stlen & (8 << SEG_STLEN_PORT_SHIFT)) ? 3 : 2;
		}
		priv = netdev_priv(xpriv->dev[port]);
		if (sd->skb) {
			pci_unmap_single(combo6->pci,
					 le32_to_cpu(sd->seg->seg_addr),
					 seg_len,
					 PCI_DMA_TODEVICE);
			dev_kfree_skb_irq(sd->skb);
			sd->skb = NULL;
			priv->stats.tx_packets++;
			priv->stats.tx_bytes += seg_len;
		}
		if (sd->frag) {
			pci_unmap_single(combo6->pci,
					 le32_to_cpu(sd->seg->seg_addr),
					 seg_len,
					 PCI_DMA_TODEVICE);
			sd->frag = NULL;
			priv->stats.tx_bytes += seg_len;
		}
		sd->seg->seg_stlen = 0;
		xpriv->tx_run_head[xpriv->tx_run]++;
		xpriv->tx_run_head[xpriv->tx_run] %= C6ETH_TX_FRAGS;
		xpriv->tx_run_fill[xpriv->tx_run]--;
	}
	if (dma != NULL) {
		/* clear DONE flag */
		dma->dma_flags = cpu_to_le32(C6ETH_DMA_TX_FLAGS);
		pci_dma_sync_single_for_device(combo6->pci,
					       sd->dma_addr,
					       sizeof(struct pcrdma_dma),
					       PCI_DMA_TODEVICE);
	}
	if (xpriv->tx_run_fill[xpriv->tx_run] == 0) {
#ifdef C6ETH_TX_DEBUG
		printk("TX done (0x%x) - restarting (queue = 0x%x)\n", combo6_br_readl(xpriv->combo6, PCRDMA_CTRL), xpriv->tx_queue);
#endif
		xpriv->enable &= ~(xpriv->tx_run ? CTRL_ENABLE_TX1 : CTRL_ENABLE_TX0);
		xpriv->tx_run ^= 1;
		xpriv->tx_run_count--;
		c6eth_tx_start(xpriv);
		if (xpriv->tx_run_count > 0)
			goto __again;
	}
}

static void c6eth_tx_start_hw(struct c6eth_private *xpriv)
{
	u32 set_enable = 0, clr_enable = 0;
	int changed = 0;

	if (xpriv->tx_queue == 0)
		return;
	if (xpriv->tx_run_count == 0)
		return;
	if (xpriv->enable & CTRL_ENABLE_TX)
		return;
#ifdef C6ETH_TX_DEBUG
	printk("START HW: tx_queue = 0x%x\n", xpriv->tx_queue);
#endif
	if ((xpriv->tx_queue & 1) && !(xpriv->enable & CTRL_ENABLE_TX0)) {
		combo6_br_writel(xpriv->combo6, PCRDMA_TX0HEAD,
			 xpriv->tx_skbdma[xpriv->tx_run_head[0]].dma_addr);
		xpriv->enable |= CTRL_ENABLE_TX0;
		set_enable |= CTRL_ENABLE_TX0;
		xpriv->tx_queue &= ~1;
#ifdef C6ETH_TX_DEBUG
		printk("start TX0 - %i [0x%x, 0x%x, 0x%x, 0x%x]\n", xpriv->tx_run_head[0], xpriv->tx_skbdma[xpriv->tx_run_head[0]].dma_addr, xpriv->tx_skbdma[xpriv->tx_run_head[0]].dma->dma_flags, xpriv->tx_skbdma[xpriv->tx_run_head[0]].dma->dma_next, xpriv->tx_skbdma[xpriv->tx_run_head[0]].seg->seg_addr);
#endif
		changed = 1;
	}
	if ((xpriv->tx_queue & 2) && !(xpriv->enable & CTRL_ENABLE_TX1)) {
		combo6_br_writel(xpriv->combo6, PCRDMA_TX1HEAD,
			 xpriv->tx_skbdma[xpriv->tx_run_head[1]].dma_addr);
		xpriv->enable |= CTRL_ENABLE_TX1;
		set_enable |= CTRL_ENABLE_TX1;
		xpriv->tx_queue &= ~2;
#ifdef C6ETH_TX_DEBUG
		printk("start TX1 - %i [0x%x, 0x%x, 0x%x, 0x%x]\n", xpriv->tx_run_head[1], xpriv->tx_skbdma[xpriv->tx_run_head[1]].dma_addr, xpriv->tx_skbdma[xpriv->tx_run_head[1]].dma->dma_flags, xpriv->tx_skbdma[xpriv->tx_run_head[1]].dma->dma_next, xpriv->tx_skbdma[xpriv->tx_run_head[1]].seg->seg_addr);
#endif
		changed = 1;
	}
	if (changed) {
		if ((xpriv->enable & CTRL_ENABLE_TX) == CTRL_ENABLE_TX0) {
			set_enable |= CTRL_SET_TX0;
			clr_enable |= CTRL_SET_TX1;
		} else if ((xpriv->enable & CTRL_ENABLE_TX) == CTRL_ENABLE_TX1) {
			set_enable |= CTRL_SET_TX1;
			clr_enable |= CTRL_SET_TX0;
		}
		combo6_br_writel(xpriv->combo6, PCRDMA_CLR_CTRL, clr_enable);
		combo6_br_writel(xpriv->combo6, PCRDMA_SET_CTRL, set_enable);
	}
}

static void c6eth_tx_start(struct c6eth_private *xpriv)
{
	unsigned int port, pos;
	unsigned int head, tail, fill;
	struct c6eth_skbdma *sd, *psd;

	if (xpriv->tx_queue)
		c6eth_tx_start_hw(xpriv);
	if (xpriv->tx_run_count == 2)
		return;
	port = combo6_br_readl(xpriv->combo6, PCRDMA_CTRL);
	if ((port & CTRL_ENABLE_TX) == CTRL_ENABLE_TX) {
		printk(KERN_ERR "combo6x(%i): TX start - stat 0x%x broken\n", xpriv->combo6->index, port);
		return;
	}
	for (port = 0; port < PCRDMA_NPORTS; port++) {
		if ((xpriv->tx_pos[port] % PCRDMA_NSEGS) == 0 &&
		    xpriv->tx_pos[port] >= xpriv->tx_cur_tail - (C6ETH_TX_BLOCK - 1) &&
		    xpriv->tx_pos[port] < xpriv->tx_cur_tail) {
			continue;
		}
		break;
	}
	if (port == PCRDMA_NPORTS) {
		if (xpriv->tx_cur_fill == C6ETH_TX_BLOCK)
			return;
		xpriv->tx_cur_fill -= C6ETH_TX_BLOCK;
		xpriv->tx_cur_tail -= C6ETH_TX_BLOCK;
		xpriv->tx_cur_tail %= C6ETH_TX_FRAGS;
	}
	if (xpriv->tx_cur_head == xpriv->tx_cur_tail ||
	    xpriv->tx_cur_fill < C6ETH_TX_BLOCK) {
		printk(KERN_ERR "TX mismatch!!!\n");
		return;
	}
#ifdef C6ETH_TX_DEBUG
	printk("start_flags [0x%p, %i] = 0x%x, end_flags [0x%p, %i] = 0x%x\n",
		  xpriv->tx_skbdma[xpriv->tx_cur_head].dma,
		  xpriv->tx_cur_head,
		  xpriv->tx_skbdma[xpriv->tx_cur_head].dma->dma_flags,
		  xpriv->tx_skbdma[xpriv->tx_cur_tail].dma,
		  xpriv->tx_cur_tail,
		  xpriv->tx_skbdma[xpriv->tx_cur_tail].dma->dma_flags);
#endif
	head = xpriv->tx_cur_head;
	tail = xpriv->tx_cur_tail;
	fill = xpriv->tx_cur_fill;
	xpriv->tx_cur_head = (xpriv->tx_cur_tail + 1) % C6ETH_TX_FRAGS;
	xpriv->tx_cur_tail = (xpriv->tx_cur_head + C6ETH_TX_BLOCK - 1) %
				C6ETH_TX_FRAGS;
	xpriv->tx_cur_fill = C6ETH_TX_BLOCK;
	for (port = 0; port < PCRDMA_NPORTS; port++) {
		xpriv->tx_pos[port] = xpriv->tx_cur_head + port * PCRDMA_NSEGS;
		xpriv->tx_pkt[port] = 0;
	}

#if 1
#if 0
	printk("== head = %i, tail = %i, fill = %i ==\n", head, tail, fill);
	for (pos = head; fill > 0; fill -= PCRDMA_NSEGS, pos += PCRDMA_NSEGS) {
		pos %= C6ETH_TX_FRAGS;
		sd = &xpriv->tx_skbdma[pos];
		printk("pos: %i, flags = 0x%x, next = 0x%x, skb = %p, frag = %p\n", pos, le32_to_cpu(sd->dma->dma_flags), sd->dma->dma_next, sd->skb, sd->frag);
	}
#endif
	/* optimization, skip all unused blocks */
	psd = &xpriv->tx_skbdma[head];
	pos = head;
	tail = head + PCRDMA_NSEGS - 1;
	for (port = fill; port > 0; port -= PCRDMA_NSEGS) {
		sd = &xpriv->tx_skbdma[pos];
		sd->dma->dma_flags = cpu_to_le32(C6ETH_DMA_TX_FLAGS);
		if (sd->skb == NULL && sd->frag == NULL) {
			if (head == pos) {
				head = (pos + PCRDMA_NSEGS) % C6ETH_TX_FRAGS;
				psd = sd;
			} else {
				sd->dma->dma_flags |= cpu_to_le32(DMA_FLAGS_DONE_MASK);
			}
			goto __next;
		}
		psd->dma->dma_next = cpu_to_le32(C6ETH_TX_PCI_PHYS(xpriv, pos));
		psd = sd;
		tail = (pos + PCRDMA_NSEGS - 1) % C6ETH_TX_FRAGS;
	      __next:
		pos += PCRDMA_NSEGS;
		pos %= C6ETH_TX_FRAGS;
	}
	if (head < tail)
		fill = (tail + 1) - head;
	else
		fill = (C6ETH_TX_FRAGS - head) + (tail + 1);
#if 0
	for (pos = head; fill > 0; fill -= PCRDMA_NSEGS, pos += PCRDMA_NSEGS) {
		pos %= C6ETH_TX_FRAGS;
		sd = &xpriv->tx_skbdma[pos];
		printk("Xpos: %i, flags = 0x%x, next = 0x%x, skb = %p\n", pos, le32_to_cpu(sd->dma->dma_flags), sd->dma->dma_next, sd->skb);
	}
	fill = ((tail + 1) - head) % C6ETH_TX_FRAGS;
	printk("head = %i, tail = %i, fill = %i\n", head, tail, fill);
#endif
#endif
	xpriv->tx_run_head[xpriv->tx_runf] = head;
	xpriv->tx_run_tail[xpriv->tx_runf] = tail;
	xpriv->tx_run_fill[xpriv->tx_runf] = fill;

	xpriv->tx_skbdma[tail].dma->dma_flags |=
				cpu_to_le32(DMA_FLAGS_INTR_MASK |
					    DMA_FLAGS_LAST_MASK);

	port = fill;
	if (head + port > C6ETH_TX_FRAGS)
		port = C6ETH_TX_FRAGS - head;
	pci_dma_sync_single_for_device(xpriv->combo6->pci,
				       xpriv->tx_skbdma[head].dma_addr,
				       port * sizeof(struct pcrdma_dma),
				       PCI_DMA_TODEVICE);
	port = fill - port;
	if (port != 0)
		pci_dma_sync_single_for_device(xpriv->combo6->pci,
				       xpriv->tx_skbdma[0].dma_addr,
				       port * sizeof(struct pcrdma_dma),
				       PCI_DMA_TODEVICE);

	xpriv->tx_queue |= 1 << xpriv->tx_runf;
	xpriv->tx_runf ^= 1;
	xpriv->tx_run_count++;
	c6eth_tx_start_hw(xpriv);
}

static int c6eth_tx(struct sk_buff *skb, struct net_device *dev)
{
	struct c6eth_netdev *priv = netdev_priv(dev);
	struct c6eth_private *xpriv = priv->combo6->private_data;
	struct c6eth_skbdma *sd;
	u32 nr_frags = skb_shinfo(skb)->nr_frags + 1, remaining;
	unsigned int port = priv->index, i;
	struct sk_buff *new_skb;

	if (xpriv->tx_hw_packets == 0) { /* I'm sorry, TX is not supported */
		dev_kfree_skb(skb);
		return 0;
	}
#if 0
	if (skb->len != 0x62) {
		dev_kfree_skb(skb);
		printk("discarding TX skb with len 0x%x\n", skb->len);
		return 0;
	}
#endif
	if (nr_frags > PCRDMA_NSEGS) {
#ifndef CONFIG_OLD_SKB_LINEARIZE
		if (skb_linearize(skb) < 0) {
#else
		if (skb_linearize(skb, GFP_ATOMIC) < 0) {
#endif
			printk(KERN_ERR "combo6: unable to handle fragmented skb(%i) in TX\n", nr_frags);
			dev_kfree_skb(skb);
			return 0;
		}
		nr_frags = 1;
	}
	if (skb->len < ETH_ZLEN) {
		new_skb = dev_alloc_skb(ETH_ZLEN);
#ifndef CONFIG_OLD_SKB_LINEARIZE
		if (skb_linearize(skb) < 0 || new_skb == NULL) {
#else
		if (skb_linearize(skb, GFP_ATOMIC) < 0 || new_skb == NULL) {
#endif
			printk(KERN_ERR "combo6: unable to handle ETH_ZLEN in TX\n");
			if (new_skb)
				dev_kfree_skb(new_skb);
			dev_kfree_skb(skb);
			return 0;
		}
		new_skb->dev = dev;
		memcpy(new_skb->data, skb->data, skb->len);
		memset(new_skb->data + skb->len, 0, ETH_ZLEN - skb->len);
		skb_put(new_skb, ETH_ZLEN);
		dev_kfree_skb(skb);
		skb = new_skb;
	}
	/* note the filling of SG entries is a bit complicated because */
	/* we must split packets for ports and the hardware has limited */
	/* number of TX packets and we do want to have similar priority */
	/* for all ports */
	spin_lock_irq(&priv->combo6->reg_lock);
	if (xpriv->tx_cur_fill + xpriv->tx_run_fill[0] +
	    xpriv->tx_run_fill[1] > C6ETH_TX_FRAGS) {
		/* oops: no more free buffers */
		c6eth_tx_start(xpriv);
		spin_unlock_irq(&priv->combo6->reg_lock);
		netif_stop_queue(dev);
		return -EAGAIN;
	}
	remaining = PCRDMA_NSEGS - (xpriv->tx_pos[port] % PCRDMA_NSEGS);
	if (remaining < nr_frags || xpriv->tx_pkt[port] >= xpriv->tx_hw_packets) {
		xpriv->tx_pos[port] += remaining + C6ETH_TX_BLOCK - PCRDMA_NSEGS;
		xpriv->tx_pos[port] %= C6ETH_TX_FRAGS;
		xpriv->tx_pkt[port] = 0;
		xpriv->tx_cur_tail += C6ETH_TX_BLOCK;
		xpriv->tx_cur_tail %= C6ETH_TX_FRAGS;
		xpriv->tx_cur_fill += C6ETH_TX_BLOCK;
		if (xpriv->tx_cur_fill + xpriv->tx_run_fill[0] +
		    xpriv->tx_run_fill[1] > C6ETH_TX_FRAGS) {
			/* oops: no more free buffers */
			c6eth_tx_start(xpriv);
			spin_unlock_irq(&priv->combo6->reg_lock);
			netif_stop_queue(dev);
			return -EAGAIN;
		}
	}
#if 0
	printk("TX: len = 0x%x [0x%x], nr_frags = %i, port = %i, tx_run_count = %i\n", skb->len - skb->data_len, (u32)ETH_ZLEN, nr_frags, port, xpriv->tx_run_count);
#endif
	sd = &xpriv->tx_skbdma[xpriv->tx_pos[port]];
	sd->skb = skb;		/* only for first fragment */
	sd->frag = NULL;
	sd->seg->seg_stlen = cpu_to_le32(
				((1 << port) << SEG_STLEN_PORT_SHIFT) |
				(skb->len - skb->data_len) |
				SEG_STLEN_FOP_MASK |
				(nr_frags == 1 ? SEG_STLEN_EOP_MASK : 0));
	sd->seg->seg_addr = cpu_to_le32(
			      pci_map_single(xpriv->combo6->pci,
					     skb->data, skb->len - skb->data_len,
					     PCI_DMA_TODEVICE));
	for (i = 1; i < nr_frags; i++) {
		skb_frag_t *frag = &skb_shinfo(skb)->frags[i-1];
		void *addr = (void *) page_address(frag->page + frag->page_offset);

		sd = &xpriv->tx_skbdma[++xpriv->tx_pos[port]];
		sd->skb = NULL;
		sd->frag = frag;
		sd->seg->seg_addr = cpu_to_le32(
				      pci_map_single(xpriv->combo6->pci,
						     addr, frag->size,
						     PCI_DMA_TODEVICE));
		sd->seg->seg_stlen = cpu_to_le32(
				    ((1 << port) << SEG_STLEN_PORT_SHIFT) |
				    frag->size |
				    (i == nr_frags - 1 ? SEG_STLEN_EOP_MASK : 0));
	}
	xpriv->tx_pkt[port]++;
	if ((++xpriv->tx_pos[port] % PCRDMA_NSEGS) == 0) {
		xpriv->tx_pos[port] += C6ETH_TX_BLOCK - PCRDMA_NSEGS;
		xpriv->tx_pos[port] %= C6ETH_TX_FRAGS;
		xpriv->tx_pkt[port] = 0;
		xpriv->tx_cur_tail += C6ETH_TX_BLOCK;
		xpriv->tx_cur_tail %= C6ETH_TX_FRAGS;
		xpriv->tx_cur_fill += C6ETH_TX_BLOCK;
	}
	c6eth_tx_start(xpriv);
	spin_unlock_irq(&priv->combo6->reg_lock);
	return 0;
}

static int c6eth_do_config(struct c6eth_private *priv)
{
	struct pcrdma_cseg *cseg = priv->cseg_virt;
	u32 timeout = 50;

	combo6_br_writel(priv->combo6, PCRDMA_CTRLHEAD, priv->cseg_phys);

	cseg->cseg_addr = priv->config_phys;
	cseg->cseg_lbaddr = cpu_to_le32(0xffffff00); /* configuration */
	cseg->cseg_len = cpu_to_le32(sizeof(struct pcrdma_config));
	cseg->cseg_next = 0;
	cseg->cseg_flags = cpu_to_le32(CSEG_FLAGS_LAST_MASK);

	pci_dma_sync_single_for_device(priv->combo6->pci,
				       priv->cseg_phys,
				       sizeof(struct pcrdma_cseg),
				       PCI_DMA_TODEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, priv->enable |= CTRL_ENABLE_CTRL);
	combo6_br_readl(priv->combo6, PCRDMA_CTRL);

	while (1) {
		schedule_timeout_uninterruptible(1);
		pci_dma_sync_single_for_cpu(priv->combo6->pci,
					    priv->cseg_phys,
					    sizeof(struct pcrdma_cseg),
					    PCI_DMA_FROMDEVICE);
		if (cseg->cseg_flags & CSEG_FLAGS_DONE_MASK)
			break;
		if (--timeout == 0)
			break;
	}

	pci_dma_sync_single_for_cpu(priv->combo6->pci,
				    priv->config_phys,
				    sizeof(struct pcrdma_config),
				    PCI_DMA_FROMDEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, priv->enable &= ~CTRL_ENABLE_CTRL);

	if (timeout == 0) {
		printk(KERN_ERR "c6pcreth: unable to read DMA PPC configuration\n");
		return -EIO;
	}


	priv->pcrver = le32_to_cpu(priv->config_virt->version);
	if (priv->pcrver < PCRDMA_VERSION(0, 2) ||
	    priv->pcrver > PCRDMA_VERSION(0, 3)) {
		printk(KERN_ERR "c6pcreth: DMA PPC version does not match (0x%x)\n", le32_to_cpu(priv->config_virt->version));
		return -EIO;
	}

	priv->hh_size = le32_to_cpu(priv->config_virt->hh_size);
	priv->pkt_skip = le32_to_cpu(priv->config_virt->pkt_skip);
	priv->tx_hw_packets = le32_to_cpu(priv->config_virt->tx_hw_packets);
	if ((priv->tx_hw_packets != 0 && priv->tx_hw_packets < 4) || priv->tx_hw_packets > 64) {
		printk(KERN_ERR "c6pcreth: TX HW packet count is wrong (%u)?\n", priv->tx_hw_packets);
		return -EIO;
	}

#if 0
	printk("timeout = 0x%x, phys = 0x%x\n", timeout, priv->config_phys);
	printk("config.pkt_threshold = 0x%x, time_threshold = 0x%x, res[0] = 0x%x, res[3] = 0x%x\n", priv->config_virt->pkt_intr_threshold, priv->config_virt->time_intr_threshold, priv->config_virt->reserved[0], priv->config_virt->reserved[3]);
	for (i = 0; i < 8; i++)
		printk("GPR(%i) = 0x%x\n", i, combo6_br_readl(priv->combo6, PCR_GPR(i)));
#endif

	if (priv->hh_size > 0 && priv->pkt_skip < priv->hh_size) {
		priv->pkt_skip = priv->hh_size;
		priv->config_virt->pkt_skip = cpu_to_le32(priv->hh_size);
	} else {
		return 0;
	}

	cseg->cseg_flags = cpu_to_le32(CSEG_FLAGS_LAST_MASK|CSEG_FLAGS_WRITE_MASK);

	combo6_br_writel(priv->combo6, PCRDMA_CTRLHEAD, priv->cseg_phys);

	pci_dma_sync_single_for_device(priv->combo6->pci,
				       priv->cseg_phys,
				       sizeof(struct pcrdma_cseg),
				       PCI_DMA_TODEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, priv->enable |= CTRL_ENABLE_CTRL);
	combo6_br_readl(priv->combo6, PCRDMA_CTRL);

	while (1) {
		schedule_timeout_uninterruptible(1);
		pci_dma_sync_single_for_cpu(priv->combo6->pci,
					    priv->cseg_phys,
					    sizeof(struct pcrdma_cseg),
					    PCI_DMA_FROMDEVICE);
		if (cseg->cseg_flags & CSEG_FLAGS_DONE_MASK)
			break;
		if (--timeout == 0)
			break;
	}

	pci_dma_sync_single_for_cpu(priv->combo6->pci,
				    priv->config_phys,
				    sizeof(struct pcrdma_config),
				    PCI_DMA_FROMDEVICE);

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, priv->enable &= ~CTRL_ENABLE_CTRL);

	if (timeout == 0) {
		printk(KERN_ERR "c6pcreth: unable to write DMA PPC configuration\n");
		return -EIO;
	}

	priv->pkt_skip = le32_to_cpu(priv->config_virt->pkt_skip);

	return 0;
}

static void init_vars(struct c6eth_private *priv)
{
	unsigned int port;

	priv->rx_pos = 0;
	for (port = 0; port < PCRDMA_NPORTS; port++) {
		priv->tx_pos[port] = port * PCRDMA_NSEGS;
		priv->tx_pkt[port] = 0;
	}
	priv->tx_cur_head = priv->tx_run_head[0] = priv->tx_run_head[1] = 0;
	priv->tx_cur_tail = priv->tx_run_tail[0] = priv->tx_run_tail[1] = C6ETH_TX_BLOCK - 1;
	priv->tx_cur_fill = priv->tx_cur_tail + 1;
	priv->tx_run_fill[0] = priv->tx_run_fill[1] = 0;
	priv->tx_run = priv->tx_runf = 0;
	priv->tx_run_count = 0;
	priv->tx_queue = 0;
}

static void pcr_dma_finish(struct c6eth_private *priv)
{
	u32 ctrl;

	if (priv->pcrver < PCRDMA_VERSION(0, 3)) {
		combo6x_dma_finish(priv->combo6);
		return;
	}

	while (1) {
		ctrl = combo6_br_readl(priv->combo6, PCRDMA_CTRL);
		if ((ctrl & (CTRL_STATUS_RX_RUNNING|CTRL_STATUS_TX_RUNNING)) == 0)
			break;
		udelay(1);
	}
}

static void free_segments(struct c6eth_private *priv)
{
	struct c6eth_skbdma *sd;
	unsigned pos, i, j;

	pcr_dma_finish(priv);

	pos = 0;
	for (i = 0; i < C6ETH_RX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->rx_skbdma[pos];
			if (sd->skb) {
				sd->seg->seg_stlen = cpu_to_le32(C6ETH_PKTSIZE);
			} else {
				sd->seg->seg_stlen = 0;
			}
			pos++;
		}
		sd->dma->dma_flags &= ~cpu_to_le32(DMA_FLAGS_DONE_MASK);
	}
	pos = 0;
	for (i = 0; i < C6ETH_TX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->tx_skbdma[pos];
			if (sd->skb) {
				pci_unmap_single(priv->combo6->pci,
						 sd->seg->seg_addr,
						 C6ETH_PKTSIZE + 2,
						 PCI_DMA_TODEVICE);
				dev_kfree_skb(sd->skb);
			}
			sd->skb = NULL;
			sd->seg->seg_addr = 0;
			sd->seg->seg_stlen = 0;
			pos++;
		}
		sd->dma->dma_flags &= ~cpu_to_le32(DMA_FLAGS_DONE_MASK);
	}

	init_vars(priv);
}

static void c6eth_regs_read(struct combo6_info_entry *entry, struct combo6_info_buffer *buffer)
{
	struct combo6 *combo = entry->private_data;
	struct c6eth_private *priv = combo->private_data;
	struct c6eth_skbdma *sd;
	unsigned int i;

	combo6_iprintf(buffer, "PCRDMA_CTRL:\t%08x\n", combo6_br_readl(combo, PCRDMA_CTRL));
	combo6_iprintf(buffer, "PCRDMA_GPR1:\t%08x\n", combo6_br_readl(combo, PCR_GPR(1)));
	combo6_iprintf(buffer, "PCRDMA_RXHEAD:\t%08x [%08x]\n", combo6_br_readl(combo, PCRDMA_RXHEAD), (u32)priv->rx_skbdma[0].dma_addr);
	combo6_iprintf(buffer, "PCRDMA_TX0HEAD:\t%08x [%08x]\n", combo6_br_readl(combo, PCRDMA_TX0HEAD), (u32)priv->tx_skbdma[0].dma_addr);
	combo6_iprintf(buffer, "PCRDMA_TX1HEAD:\t%08x\n", combo6_br_readl(combo, PCRDMA_TX1HEAD));
	combo6_iprintf(buffer, "PCRDMA_CTRLHEAD:%08x\n", combo6_br_readl(combo, PCRDMA_CTRLHEAD));
	combo6_iprintf(buffer, "PCRDMA_DEBUG:\t%08x\n", combo6_br_readl(combo, PCRDMA_DEBUG));
	combo6_iprintf(buffer, "PCRDMA_STATE:\t%08x\n", combo6_br_readl(combo, PCRDMA_STATE));
	combo6_iprintf(buffer, "\n");
	combo6_iprintf(buffer, "PCR_INTR_STAT:\t%08x\n", combo6_br_readl(combo, PCR_INTR_STAT));
	combo6_iprintf(buffer, "PCR_DMA_ASRC:\t%08x\n", combo6_br_readl(combo, PCR_DMA_ADDR_SRC));
	combo6_iprintf(buffer, "PCR_DMA_ADST:\t%08x\n", combo6_br_readl(combo, PCR_DMA_ADDR_DST));
	combo6_iprintf(buffer, "PCR_DMA_COUNT:\t%08x\n", combo6_br_readl(combo, PCR_DMA_COUNT));
	combo6_iprintf(buffer, "PCR_DMA_CMD:\t%08x\n", combo6_br_readl(combo, PCR_DMA_CMD));
	combo6_iprintf(buffer, "\n");
	combo6_iprintf(buffer, "enable:\t\t%08x\n", priv->enable);
	combo6_iprintf(buffer, "tx_hw_packets:\t%08x\n", priv->tx_hw_packets);
	combo6_iprintf(buffer, "hh_size:\t%08x\n", priv->hh_size);
	combo6_iprintf(buffer, "pkt_skip:\t%08x\n", priv->pkt_skip);
	spin_lock_irq(&combo->reg_lock);
	sd = &priv->rx_skbdma[priv->rx_pos];
	combo6_iprintf(buffer, "rx_pos:\t\t%i [%08x] [%08x]\n", priv->rx_pos, sd->seg->seg_stlen, sd->dma->dma_flags);
	for (i = 0; i < PCRDMA_NPORTS; i++) {
		combo6_iprintf(buffer, "tx_pos[%i]:\t%i\n", i, priv->tx_pos[i]);
		combo6_iprintf(buffer, "tx_pkt[%i]:\t%i\n", i, priv->tx_pkt[i]);
	}
	combo6_iprintf(buffer, "tx_cur_head:\t%i\n", priv->tx_cur_head);
	combo6_iprintf(buffer, "tx_cur_tail:\t%i\n", priv->tx_cur_tail);
	combo6_iprintf(buffer, "tx_cur_fill:\t%i\n", priv->tx_cur_fill);
	combo6_iprintf(buffer, "tx_run_head:\t%i : %i\n", priv->tx_run_head[0], priv->tx_run_head[1]);
	combo6_iprintf(buffer, "tx_run_tail:\t%i : %i\n", priv->tx_run_tail[0], priv->tx_run_tail[1]);
	combo6_iprintf(buffer, "tx_run_fill:\t%i : %i\n", priv->tx_run_fill[0], priv->tx_run_fill[1]);
	combo6_iprintf(buffer, "tx_run/f:\t%i : %i\n", priv->tx_run, priv->tx_runf);
	combo6_iprintf(buffer, "tx_run_count:\t%i\n", priv->tx_run_count);
	combo6_iprintf(buffer, "tx_queue:\t%i\n", priv->tx_queue);
	spin_unlock_irq(&combo->reg_lock);
#if 0
	combo6_iprintf(buffer, "\n");
	for (i = 0; i < 4; i++) {
		combo6_iprintf(buffer, "rx_bufstat[%i]:\t%08x [%i]\n", i, combo6_fpga_readl(combo, 0x26088 + i * 0x8000), combo6_fpga_readl(combo, 0x26094 + i * 0x8000));
		combo6_iprintf(buffer, "tx_bufstat[%i]:\t%08x [%i]\n", i, combo6_fpga_readl(combo, 0x2608c + i * 0x8000), combo6_fpga_readl(combo, 0x26098 + i * 0x8000));
	}
#endif
}

/*
 * initialization and detection part
 */
static int c6eth_dev_init(struct combo6 *combo, int idx, struct net_device **rdev)
{
	static const struct net_device_ops ops = {
		.ndo_open = &netdev_open,
		.ndo_stop = &netdev_close,
		.ndo_start_xmit = &c6eth_tx,
		.ndo_get_stats = &get_stats,
		.ndo_set_multicast_list = &set_rx_mode,
		.ndo_do_ioctl = &netdev_ioctl,
		.ndo_tx_timeout = &tx_timeout,
		.ndo_change_mtu = &change_mtu,
	};
	struct net_device *dev;
	struct c6eth_netdev *priv;
	int err = 0;

	dev = alloc_etherdev(sizeof(*priv));
	if (strstr(dev_template, "%d"))
		strcpy(dev->name, dev_template);
	else
		sprintf(dev->name, "%s%d%d", dev_template,
				combo->index, idx);
	dev->destructor = free_netdev;
	priv = netdev_priv(dev);
	priv->index = idx;
	priv->dev = dev;
	priv->combo6 = combo;
	/* HW addr */
	dev->dev_addr[0] = 0x00;
	dev->dev_addr[1] = 0x11;
	dev->dev_addr[2] = 0x17;
	dev->dev_addr[3] = 0xff;
	dev->dev_addr[4] = 0xff;
	dev->dev_addr[5] = (combo->index << 2) | idx;
	/* The chip-specific entries in the driver structure. */
#ifdef CONFIG_HAVE_NET_DEVICE_OPS
	dev->netdev_ops = &ops;
#else
	combo6_copy_ndev_ops(dev, &ops);
#endif
	dev->watchdog_timeo = TX_TIMEOUT;
	err = register_netdev(dev);
	if (err < 0) {
		printk(KERN_ERR "failed to create network driver %s\n", dev->name);
		free_netdev(dev);
		return err;
	}
	*rdev = dev;
	return 0;
}

static void c6eth_dev_done(struct net_device *dev)
{
	if (dev != NULL)
		unregister_netdev(dev);
}

static int c6eth_detach(struct combo6 *combo)
{
	struct c6eth_private *priv = combo->private_data;
	struct c6eth_skbdma *sd;
	int idx;
	unsigned pos, i, j;

	spin_lock_irq(&priv->combo6->reg_lock);

	if (priv->enable & CTRL_ENABLE_MASK) {
		spin_unlock_irq(&priv->combo6->reg_lock);
		return -EBUSY;
	}

	combo6_br_writel(priv->combo6, PCRDMA_CTRL, 0);
	combo6_br_writel(priv->combo6, PCRDMA_CTRL, 0);

	spin_unlock_irq(&priv->combo6->reg_lock);

	pcr_dma_finish(priv);
	combo6x_stop_pcippc(priv->combo6);

	pos = 0;
	for (i = 0; i < C6ETH_RX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->rx_skbdma[pos];
			if (sd->skb) {
				pci_unmap_single(combo->pci,
						 le32_to_cpu(sd->seg->seg_addr),
						 C6ETH_PKTSIZE + 2,
						 PCI_DMA_FROMDEVICE);
				dev_kfree_skb(sd->skb);
			}
			pos++;
		}
	}
	pos = 0;
	for (i = 0; i < C6ETH_TX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->tx_skbdma[pos + PCRDMA_NSEGS - 1 - j];
			if (sd->skb) {
				pci_unmap_single(combo->pci,
						 le32_to_cpu(sd->seg->seg_addr),
						 SEG_STLEN_LENGTH(le32_to_cpu(sd->seg->seg_stlen)),
						 PCI_DMA_TODEVICE);
				dev_kfree_skb(sd->skb);
			}
			if (sd->frag) {
				pci_unmap_single(combo->pci,
						 le32_to_cpu(sd->seg->seg_addr),
						 SEG_STLEN_LENGTH(le32_to_cpu(sd->seg->seg_stlen)),
						 PCI_DMA_TODEVICE);
			}
			pos++;
		}
	}

	if (priv->sgdesc_virt != NULL)
		dma_free_coherent(&combo->pci->dev, priv->sgdesc_size,
				priv->sgdesc_virt, priv->sgdesc_phys);
	if (priv->cseg_virt != NULL)
		dma_free_coherent(&combo->pci->dev, priv->cseg_size,
				priv->cseg_virt, priv->cseg_phys);
	for (idx = 0; idx < priv->interfaces; idx++)
		c6eth_dev_done(priv->dev[idx]);
	if (priv->proc_regs)
		combo6_info_unregister(priv->proc_regs);
	vfree(priv);
	combo->private_data = NULL;
	return 0;
}

static int c6eth_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	int idx, err;
	struct c6eth_private *priv;
	struct pcrdma_dma *dma;
	struct pcrdma_seg *seg;
	struct c6eth_skbdma *sd;
	dma_addr_t dma_addr, dma_faddr;
	unsigned int pos, i, j;
	struct combo6_info_entry *entry;
	u32 brver = combo->pcibr_version;

	if (interfaces < 1 || interfaces > PCRDMA_NPORTS)
		return -EIO;

	if (PCR_ID_MAJOR(brver) < 4 ||
	    (PCR_ID_MAJOR(brver) == 4 &&
	     PCR_ID_MINOR(brver) < 1)) {
		printk(KERN_ERR "c6pcreth: Old version of PCI bridge, aborting\n");
		return -EIO;
	}

	/* disable interrupts */
	combo6_br_writel(combo, PCR_INTR_MASK, 0);
	/* check if PPC has good firmware (program) */
	if (combo6_br_readl(combo, PCRDMA_DEBUG) != 0x4f6f4b6b) {
		udelay(500);
		if (combo6_br_readl(combo, PCRDMA_DEBUG) != 0x4f6f4b6b) {
			printk(KERN_ERR "c6pcreth: DMA PPC firmware signature is wrong: 0x%x\n",
					combo6_br_readl(combo, PCRDMA_DEBUG));
			return -EIO;
		}
	}
	/* disable PPC program */
	combo6_br_writel(combo, PCRDMA_CTRL, 0);
	priv = vmalloc(sizeof(*priv));
	if (priv == NULL)
		return -ENOMEM;
	memset(priv, 0, sizeof(*priv));

	mutex_init(&priv->open_mutex);
	priv->combo6 = combo;
	priv->interfaces = interfaces;
	for (idx = 0; idx < interfaces; idx++) {
		err = c6eth_dev_init(combo, idx, &priv->dev[idx]);
		if (err < 0)
			goto __error;
	}

	priv->cseg_size = sizeof(struct pcrdma_cseg) + sizeof(struct pcrdma_config);
	priv->cseg_virt = dma_alloc_coherent(&combo->pci->dev,
			priv->cseg_size, &priv->cseg_phys, GFP_KERNEL);
	if (priv->cseg_virt == NULL) {
		printk(KERN_ERR "unable to allocate memory for cseg SG list for ethernet devices combo%d\n",
				combo->index);
		err = -ENOMEM;
		goto __error;
	}
	priv->config_virt = (struct pcrdma_config *)(priv->cseg_virt + 1);
	priv->config_phys = priv->cseg_phys + sizeof(struct pcrdma_cseg);
	priv->sgdesc_size = sizeof(struct pcrdma_dma) *
				(C6ETH_RX_BUFFERS + C6ETH_TX_BUFFERS);
	priv->sgdesc_virt = dma_alloc_coherent(&combo->pci->dev,
			priv->sgdesc_size, &priv->sgdesc_phys, GFP_KERNEL);
	if (priv->sgdesc_virt == NULL) {
		printk(KERN_ERR "unable to allocate memory for SG list for ethernet devices combo%d\n",
				combo->index);
		err = -ENOMEM;
		goto __error;
	}
	dma = (struct pcrdma_dma *)priv->sgdesc_virt;
	dma_addr = dma_faddr = priv->sgdesc_phys;
	combo6_br_writel(priv->combo6, PCRDMA_RXHEAD, dma_faddr);
	pos = 0;
	for (i = 0; i < C6ETH_RX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->rx_skbdma[pos];
			sd->seg = seg = &dma->dma_segs[j];
			sd->dma = dma;
			sd->dma_addr = dma_addr;
			my_alloc_skb(combo, sd);
			pos++;
		}
		dma->dma_next = cpu_to_le32(dma_faddr +
				((i + 1) % C6ETH_RX_BUFFERS)
				  * sizeof(struct pcrdma_dma));
		dma->dma_flags = cpu_to_le32(C6ETH_DMA_RX_FLAGS);
		dma++;
		dma_addr += sizeof(struct pcrdma_dma);
	}
	pci_dma_sync_single_for_device(priv->combo6->pci,
				       dma_faddr,
				       sizeof(struct pcrdma_dma) * C6ETH_RX_BUFFERS,
				       PCI_DMA_TODEVICE);
	dma_faddr = dma_addr;
	combo6_br_writel(priv->combo6, PCRDMA_TX0HEAD, dma_faddr);
	combo6_br_writel(priv->combo6, PCRDMA_TX1HEAD, dma_faddr);
	pos = 0;
	for (i = 0; i < C6ETH_TX_BUFFERS; i++) {
		for (j = 0; j < PCRDMA_NSEGS; j++) {
			sd = &priv->tx_skbdma[pos];
			sd->seg = seg = &dma->dma_segs[j];
			sd->dma = dma;
			sd->dma_addr = dma_addr;
			seg->seg_stlen = 0;
			pos++;
		}
		dma->dma_next = cpu_to_le32(dma_faddr +
				((i + 1) % C6ETH_TX_BUFFERS)
				  * sizeof(struct pcrdma_dma));
		dma->dma_flags = cpu_to_le32(C6ETH_DMA_TX_FLAGS);
		dma++;
		dma_addr += sizeof(struct pcrdma_dma);
	}
	pci_dma_sync_single_for_device(priv->combo6->pci,
				       dma_faddr,
				       sizeof(struct pcrdma_dma) * C6ETH_TX_BUFFERS,
				       PCI_DMA_TODEVICE);
	init_vars(priv);
	if ((err = c6eth_do_config(priv)) < 0)
		goto __error;
	/* enable interrupts */
	combo6_br_writel(priv->combo6, PCR_INTR_SET, PCRDMA_INTR_RX_MASK | PCRDMA_INTR_TX_MASK | PCRDMA_INTR_ERR_MASK);
	combo->private_data = priv;

	entry = combo6_info_create_module_entry(THIS_MODULE, "c6pcrdma-regs", priv->combo6->proc_root);
	if (entry) {
		combo6_info_set_text_ops(entry, combo, 4096, c6eth_regs_read);
		if (combo6_info_register(entry) < 0) {
			combo6_info_free_entry(entry);
			entry = NULL;
		}
	}
	priv->proc_regs = entry;
	return 0;

      __error:
	combo->private_data = NULL;
	c6eth_detach(combo);
	return err;
}

/*
 * linux network layer
 */

static int netdev_open(struct net_device *dev)
{
	struct c6eth_netdev *priv = netdev_priv(dev);
	struct c6eth_private *xpriv = priv->combo6->private_data;
	u32 set_enable = 0;

	mutex_lock(&xpriv->open_mutex);
	spin_lock_irq(&priv->combo6->reg_lock);
	if ((xpriv->enable & CTRL_ENABLE_RX) == 0) {
		set_enable |= CTRL_ENABLE_RX;
		xpriv->enable |= CTRL_ENABLE_RX;
		combo6_br_writel(priv->combo6, PCRDMA_RXHEAD, xpriv->rx_skbdma[0].dma_addr);
		xpriv->rx_pos = 0;
	}
	set_enable |= CTRL_ENABLE(priv->index);
	xpriv->enable |= CTRL_ENABLE(priv->index);
	combo6_br_writel(priv->combo6, PCRDMA_SET_CTRL, set_enable);
	c6eth_tx_start_hw(xpriv);
	spin_unlock_irq(&priv->combo6->reg_lock);
	mutex_unlock(&xpriv->open_mutex);

	netif_start_queue(dev);

	return 0;

}

static int netdev_close(struct net_device *dev)
{
	struct c6eth_netdev *priv = netdev_priv(dev);
	struct c6eth_private *xpriv = priv->combo6->private_data;
	u32 clr_enable = 0;

	netif_stop_queue(dev);

	mutex_lock(&xpriv->open_mutex);
	spin_lock_irq(&priv->combo6->reg_lock);
	xpriv->enable &= ~CTRL_ENABLE(priv->index);
	clr_enable |= CTRL_ENABLE(priv->index);
	if ((xpriv->enable & CTRL_ENABLE_MASK) == 0) {
		xpriv->enable &= ~(CTRL_ENABLE_RX | CTRL_ENABLE_TX);
		clr_enable |= CTRL_ENABLE_RX | CTRL_ENABLE_TX;
		xpriv->tx_queue = 0;
	}
	combo6_br_writel(priv->combo6, PCRDMA_CLR_CTRL, clr_enable);
	c6eth_tx_start_hw(xpriv);
	spin_unlock_irq(&priv->combo6->reg_lock);
	if ((xpriv->enable & CTRL_ENABLE_MASK) == 0)
		free_segments(xpriv);
	mutex_unlock(&xpriv->open_mutex);

	return 0;
}

static struct net_device_stats *get_stats(struct net_device *dev)
{
	struct c6eth_netdev *priv = netdev_priv(dev);

	/* FIXME: ok, read stats from registers here */
	return &priv->stats;
}

static void set_rx_mode(struct net_device *dev)
{
}

static int netdev_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	if (!netif_running(dev))
		return -EINVAL;
#if 0
	if (cmd == SIOCETHTOOL)
		return netdev_ethtool_ioctl((void *) rq->ifr_data);
#endif
	return -ENXIO;
}

static void tx_timeout(struct net_device *dev)
{
	struct c6eth_netdev *priv = netdev_priv(dev);

	spin_lock_irq(&priv->combo6->reg_lock);
	check_tx(priv->combo6);
	spin_unlock_irq(&priv->combo6->reg_lock);
	if (netif_queue_stopped(dev))
		netif_wake_queue(dev);
}

static int change_mtu(struct net_device *dev, int new_mtu)
{
	if ((new_mtu < 68) || (new_mtu > 8191))
		return -EINVAL;
	if (netif_running(dev))
		return -EBUSY;
	dev->mtu = new_mtu;
	return 0;
}

/*
 * real initialization
 */

#define pcrdma_attach c6eth_attach
#define pcrdma_detach c6eth_detach
#define pcrdma_interrupt c6eth_interrupt
#include "pcrdma_ops.c"

static int __init c6eth_init(void)
{
	combo6x_module_ref();

	return combo_register_driver(&pcrdma_ops);
}

static void __exit c6eth_exit(void)
{
	combo_unregister_driver(&pcrdma_ops);
}

module_init(c6eth_init)
module_exit(c6eth_exit)
