#include <linux/fs.h>
#include <linux/list.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/spinlock.h>
#include <linux/uaccess.h>

#include <asm/atomic.h>

#include "core.h"
#include "combo6.h"

struct comboctl_message_wrap {
	struct list_head list;
	struct comboctl_message msg;
};

static atomic_t comboctl_openers = ATOMIC_INIT(0);
static LIST_HEAD(messages);
static DEFINE_SPINLOCK(messages_lock);
static DECLARE_WAIT_QUEUE_HEAD(messages_wait);

/**
 * combo_queue_message - queue a message for /dev/comboctl
 *
 * @card: index of the card which is the source of the message
 * @message: message to send
 *
 * When the source is not known, -1 shall be passed as @card.
 * It is safe to call this function in the interrupt context.
 */
int combo_queue_message(int card, int message)
{
	struct comboctl_message_wrap *msg;
	unsigned long flags;

	msg = kmalloc(sizeof(*msg), GFP_ATOMIC);
	if (!msg)
		return -ENOMEM;

	msg->msg.card = card;
	msg->msg.message = message;

	spin_lock_irqsave(&messages_lock, flags);
	if (atomic_read(&comboctl_openers))
		list_add_tail(&msg->list, &messages);
	else
		kfree(msg);
	spin_unlock_irqrestore(&messages_lock, flags);

	wake_up(&messages_wait);

	return 0;
}
EXPORT_SYMBOL_GPL(combo_queue_message);

static int comboctl_open(struct inode *inode, struct file *file)
{
	nonseekable_open(inode, file);
	if (atomic_inc_return(&comboctl_openers) > 1) {
		atomic_dec(&comboctl_openers);
		return -EBUSY;
	}
	return 0;
}

static struct comboctl_message_wrap *dequeue_msg(void)
{
	struct comboctl_message_wrap *msg = NULL;

	spin_lock_irq(&messages_lock);
	if (!list_empty(&messages)) {
		msg = list_first_entry(&messages, struct comboctl_message_wrap,
				list);
		list_del(&msg->list);
	}
	spin_unlock_irq(&messages_lock);

	return msg;
}

static ssize_t comboctl_read(struct file *file, char __user *buf, size_t len,
		loff_t *offp)
{
	struct comboctl_message_wrap *msg;
	ssize_t ret;

	msg = dequeue_msg();
	if (!msg) {
		if (file->f_flags & O_NONBLOCK)
			return 0;

		ret = wait_event_interruptible(messages_wait,
				(msg = dequeue_msg()));
		if (ret)
			return ret;
	}

	ret = sizeof(msg->msg);
	if (copy_to_user(buf, &msg->msg, ret))
		ret = -EFAULT;

	kfree(msg);

	return ret;
}

static long comboctl_ioctl(struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	void __user *argp = (void __user *)arg;

	switch (cmd) {
	case COMBOCTL_IOC_VERSION:
		return put_user(1, (u8 __user *)argp);
	}
	return -ENXIO;
}

static int comboctl_release(struct inode *inode, struct file *file)
{
	spin_lock_irq(&messages_lock);
	while (!list_empty(&messages)) {
		struct comboctl_message_wrap *msg;
		msg = list_first_entry(&messages, struct comboctl_message_wrap, list);
		list_del(&msg->list);
		kfree(msg);
	}
	atomic_dec(&comboctl_openers);
	spin_unlock_irq(&messages_lock);

	return 0;
}

static struct file_operations comboctl_fops = {
	.owner = THIS_MODULE,
	.open = comboctl_open,
	.release = comboctl_release,
	.read = comboctl_read,
	.unlocked_ioctl = comboctl_ioctl,
};

static struct miscdevice combo_misc = {
	.minor = MISC_DYNAMIC_MINOR,
	.name = "comboctl",
	.fops = &comboctl_fops,
};

int comboctl_init(void)
{
	return misc_register(&combo_misc);
}

void comboctl_exit(void)
{
	misc_deregister(&combo_misc);
}
