/**
 *  spacev2.c: driver module, used to test DMA transfers with comboV2 cards
 *  Copyright (c) 2004 CESNET
 *  Author(s): Patrik Beck <xbeckp00@stud.fit.vutbr.cz>
 *             Petr Kastovsky <kastovsky@liberouter.org>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#define DEBUG 	1

#include <linux/delay.h>
#include <linux/module.h>
#include <asm/io.h>
#include <asm/uaccess.h>
#include <linux/signal.h>
#include <linux/miscdevice.h>
#include <linux/interrupt.h>
#include <linux/init.h>
#include <linux/pci.h>
#include <linux/slab.h>
#include <linux/mm.h>
#include <linux/cdev.h>

#define PDEBUG(fmt, ...)	pr_debug("spacev2: " fmt, ##__VA_ARGS__)
#define PERR(fmt, ...)	printk(KERN_ERR "spacev2: " fmt, ##__VA_ARGS__)

#include "combo6.h"
#include "combo6k.h"
#include "combov2.h"
#include "space.h"
#include "cv2eth.h"


MODULE_AUTHOR("Patrik Beck <beck@liberouter.org>");
MODULE_AUTHOR("Petr Kastovsky <kastovsky@liberouter.org>");
MODULE_DESCRIPTION("Test space allocation");
MODULE_LICENSE("GPL");

///////////////////////////////////////////////////////////////////////////////
// comboV2 module parameters
///////////////////////////////////////////////////////////////////////////////
static int enable[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 };      /* Enable this card */
static int invalidmemsize[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Invalid memory region, fix to 64MB */
static int bootprotect[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 1 }; /* Check design boot */
static int bootdelay[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Boot delay in ms before fpga access */
static int usemode[COMBO6_CARDS] = { [0 ... (COMBO6_CARDS-1)] = 0 }; /* Use mode (see README) */

module_param_array(enable, int, NULL, 0);
MODULE_PARM_DESC(enable, "Enable ComboV2 card.");
module_param_array(invalidmemsize, int, NULL, 0);
MODULE_PARM_DESC(invalidmemsize, "Invalid PCI memory area size - fix to 64MB.");
module_param_array(bootprotect, int, NULL, 0);
MODULE_PARM_DESC(bootprotect, "Enable(default)=1/Disable=0 boot protection checks.");
module_param_array(bootdelay, int, NULL, 0);
MODULE_PARM_DESC(bootdelay, "Boot delay in ms before FPGA access.");
module_param_array(usemode, int, NULL, 0);
MODULE_PARM_DESC(usemode, "Use mode (for debugging only). See README.");

///////////////////////////////////////////////////////////////////////////////
// spaceV2 module parameters
///////////////////////////////////////////////////////////////////////////////

// parameters initialization
static int sp_alloc_pages = 1;
static int max_bufsize = 4096;

module_param_named(alloc_pages, sp_alloc_pages, int, S_IRUGO);
MODULE_PARM_DESC(alloc_pages, "Number of allocated pages.");
module_param(max_bufsize, int, S_IRUGO);
MODULE_PARM_DESC(max_bufsize, "Maximal size of buffer for burst io.");

//static struct combo6_driver *device;
//static struct cdev cdev;
//static dma_addr_t phys_addr;
//static void *virt_addr;

struct spacev2_private {
	struct combo6 *combo;
	dma_addr_t phys_addr;
	void *virt_addr;
};

static struct spacev2_private *spg_priv;

// Function declarations
static irqreturn_t spacev2_interrupt(struct combo6 *combo, unsigned int mask);
static int spacev2_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces);
static int spacev2_detach(struct combo6 *combo);
static int cdev_spacev2_open(struct inode *inode, struct file *filp);
static int cdev_spacev2_release(struct inode *inode, struct file *filp);
static long cdev_spacev2_ioctl(struct file *filp, unsigned int cmd,
		unsigned long arg);
static int __devinit spacev2_probe(struct pci_dev *pci, const struct pci_device_id *id);
static void __devexit spacev2_remove(struct pci_dev *pci);


static struct pci_device_id spacev2_ids[] = {
	{ .vendor 	 = PCI_VENDOR_ID_CESNET, .device = 0x6d05,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xFF00 }, /* ML555 */
	{ .vendor 	 = PCI_VENDOR_ID_CESNET, .device = 0xc032,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xFF00 }, /* ComboV2 */
	{ .vendor 	 = PCI_VENDOR_ID_CESNET, .device = 0xc132,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xFF00 }, /* ComboV2 */
	{ .vendor 	 = PCI_VENDOR_ID_CESNET, .device = 0xc232,
	  .subvendor = PCI_VENDOR_ID_CESNET, .subdevice = 0xFF00 }, /* ComboV2 */
	{ 0, }

};
MODULE_DEVICE_TABLE(pci, spacev2_ids);

static struct pci_driver driver = {
	.name 		= "spacev2drv",
	.id_table 	= spacev2_ids,
	.probe 		= spacev2_probe,
	.remove 	= __devexit_p(spacev2_remove),
};

/**
 * Operation definitions
 */
static const struct combo_device_id space_ids[] = {
	{
		.id_lsw 	= 0x00000000, 
		.id_hsw 	= 0xFFFFFFFF,
		.id_text 	= "",
	}, {
	}
};

static struct combo_driver space_ops = {
	.drv = {
		.owner  	= THIS_MODULE,
		.name		= "spacev2",
	},
	.dhw		= DHW_COMBOV2,
	.id_table	= space_ids,
	.attach		= spacev2_attach,
	.detach		= spacev2_detach,
	.interrupt 	= spacev2_interrupt,
};

/**
 * Char device file operations
 */
static struct file_operations space_fops = {
	.owner 		= THIS_MODULE,
	.open 		= cdev_spacev2_open,
	.release 	= cdev_spacev2_release,
	.unlocked_ioctl = cdev_spacev2_ioctl,
};	

static struct miscdevice spacev2_dev;

/**
 * interrupt handler
 */
static irqreturn_t spacev2_interrupt(struct combo6 *combo, unsigned int mask)
{

	/*
	u32 irq_status;
	irq_status = combo6_fpga_readl(combo, PAC_HW_TX_IRQ);
	if (irq_status != 0) {
		if (printk_ratelimit())
			PDEBUG("TxIRQ status %u\n", irq_status);
		pac_tx_irq(combo);
	}*/
	struct siginfo info;
	struct task_struct *task_list;
	memset(&info, 0, sizeof(struct siginfo));
	info.si_signo = 10;
	info.si_code = SI_QUEUE;
	info.si_int = 1234;
	for_each_process(task_list)
	{
		send_sig_info(10, &info, task_list);//send signal to user land 
	}
	PDEBUG(".. interrupt\n");
	return IRQ_HANDLED;
}


/**
 *  initialization and detection part
 */
static int spacev2_attach(struct combo6 *combo,
		const struct combo_device_id *id, int interfaces)
{
	struct spacev2_private *priv;
	int ret = 0;

	PDEBUG(".. attach\n");

	pci_dev_get(combo->pci);
	priv = vmalloc(sizeof(struct spacev2_private));
	if (priv == NULL)
		return -ENOMEM;

	priv->combo = combo;
	priv->virt_addr = pci_alloc_consistent(combo->pci,
		   	PAGE_SIZE*sp_alloc_pages, &priv->phys_addr);
	if (priv->virt_addr == NULL) {
		PERR("Failed to allocate pci pages (%d)!\n", ret);
		ret = -ENOMEM;
		goto err1;
	}
	PDEBUG(".. allocated pci pages\n");

	spacev2_dev.minor = MISC_DYNAMIC_MINOR;
	spacev2_dev.name = "space";
	spacev2_dev.fops = &space_fops;
	ret = misc_register(&spacev2_dev);
	if (ret != 0) {
		PERR("Failed to register misc device (%d)!\n", ret);
		goto err2;
	}

	PDEBUG(".. registered misc device %d\n", spacev2_dev.minor);

	combo->private_data = priv;
	spg_priv = priv;

	PDEBUG(".. attach end\n");
	return 0;

err2:
	pci_free_consistent(combo->pci, PAGE_SIZE*sp_alloc_pages,
		   	priv->virt_addr, priv->phys_addr);

err1:
	vfree(priv);
	pci_dev_put(combo->pci);

	return ret;
}

static int spacev2_detach(struct combo6 *combo) {

	int ret = 0;
	struct spacev2_private *priv = combo->private_data;

	PDEBUG(".. detach\n");
	ret = misc_deregister(&spacev2_dev);
	if (ret != 0)
		PERR("Failed to unregister misc device (%d)!\n", ret);

	pci_free_consistent(combo->pci, PAGE_SIZE*sp_alloc_pages,
		   	priv->virt_addr, priv->phys_addr);

	vfree(priv);
	pci_dev_put(combo->pci);
	PDEBUG(".. detach end\n");

	return ret;
}


///////////////////////////////////////////////////////////////////////////////
// char device file functions 
///////////////////////////////////////////////////////////////////////////////
static int cdev_spacev2_open(struct inode *inode, struct file *filp) {
	PDEBUG(".. cdev open\n");
	return 0;
}

static int cdev_spacev2_release(struct inode *inode, struct file *filp) {
	PDEBUG(".. cdev release\n");
	return 0;
}

static long cdev_spacev2_ioctl(struct file *filp, unsigned int cmd,
		unsigned long arg)
{
	static DEFINE_MUTEX(ioctl_lock);
	t_transaction t;
	t_biotransaction t_bio;
//	t_pactransaction t_pac;
	void* p;

	dma_addr_t phys_addr = spg_priv->phys_addr;
	void *virt_addr = spg_priv->virt_addr;
	struct combo6 *combo = spg_priv->combo;
	struct tx_info tx;
	char *buf;
	int ret = 0;
	
	PDEBUG("space ioctl ...\n");
	
	//detecting wrong ioctl
	if (_IOC_TYPE(cmd) != SPACE_MAGIC || _IOC_NR(cmd) > SPACE_IOC_MAX) {
		PDEBUG("\twrong IOCTL command\n");
		return -ENOTTY;
	}
   
	PDEBUG("arg ok\n");

	mutex_lock(&ioctl_lock);
	switch (cmd) {
	case SPACE_IOC_GETPHYSADDR:
		ret = (int)phys_addr;
		break;
	case SPACE_IOC_WRITE:
		if(copy_from_user(&t, (void*)arg, sizeof(t_transaction))) {
			ret = -EFAULT;
			break;
		}
		if (copy_from_user(virt_addr + t.pos, t.data, t.len))
			ret = -EFAULT;
		break;

	case SPACE_IOC_READ:
		if(copy_from_user(&t, (void*)arg, sizeof(t_transaction))) {
			ret = -EFAULT;
			break;
		}
		if(copy_to_user(t.data, virt_addr + t.pos, t.len))
			ret = -EFAULT;
		break;	

	case SPACE_IOC_BIOWRITE:
		if(copy_from_user(&t_bio, (void*)arg, sizeof(t_biotransaction))) {
			ret = -EFAULT;
			break;
		}
		p = combo->lmem_virt + (unsigned long)t_bio.addr;
		PDEBUG("Burst write to %p len %zX\n", p, t_bio.len);
		if (t_bio.len > max_bufsize) {
			printk(KERN_ERR "Wrong data length.\n");
			ret = -EINVAL;
			break;
		}
		buf = kmalloc(t_bio.len, GFP_KERNEL);
		if (!buf) {
			ret = -ENOMEM;
			break;
		}
		if (copy_from_user(buf, t_bio.data, t_bio.len)) {
			ret = -EFAULT;
			break;
		}
		memcpy_toio(p, buf, t_bio.len);
		break;

	case SPACE_IOC_BIOREAD:
		if(copy_from_user(&t_bio, (void*)arg, sizeof(t_biotransaction))) {
			ret = -EFAULT;
			break;
		}
		p = combo->lmem_virt + (unsigned long)t_bio.addr;
		PDEBUG("Burst read from %p len %zX\n", p, t_bio.len);
		if (t_bio.len > max_bufsize) {
			printk(KERN_ERR "Wrong data length.\n");
			ret = -EINVAL;
			break;
		}
		buf = kmalloc(t_bio.len, GFP_KERNEL);
		if (!buf) {
			ret = -ENOMEM;
			break;
		}
		memcpy_fromio(buf, p, t_bio.len);
		if (copy_to_user(t_bio.data, buf, t_bio.len))
			ret = -EFAULT;
		break;

	case SPACE_IOC_PACINIT: /* init channel */
		tx.pkt_size = 64;
		tx.ifc_index = 0;
		pac_init_tx(combo, &tx);
		break;
	case SPACE_IOC_PACSTR: /* start channel */
		pac_start_tx(combo);
		break;
	case SPACE_IOC_PACSTP: /* stop channel */
		pac_stop_tx(combo);
		break;
	default:
		printk(KERN_ERR "Bad IOCTL command.\n");
		ret = -ENOTTY;
		break;
	}
	mutex_unlock(&ioctl_lock);
	return ret;
}
///////////////////////////////////////////////////////////////////////////////
// new driver loading part
///////////////////////////////////////////////////////////////////////////////
static int __devinit spacev2_probe(struct pci_dev *pci,
					const struct pci_device_id *id)
{
	struct combov2_init init = {};
	static int dev;

	PDEBUG(".. probe\n");
	if (!enable[dev]) {
		dev++;
		return -ENOENT;
	}
	init.invalidmemsize = invalidmemsize[dev];
	init.bootprotect = bootprotect[dev];
	init.bootdelay = bootdelay[dev];
	init.usemode = usemode[dev]; // | 0x2;

	return combov2_probe(pci, id, &init, THIS_MODULE);
}

static void __devexit spacev2_remove(struct pci_dev *pci)
{
	PDEBUG(".. remove\n");
	combov2_remove(pci);
	PDEBUG(".. remove end\n");
}


///////////////////////////////////////////////////////////////////////////////
// initialization and end function
///////////////////////////////////////////////////////////////////////////////
static int __init spacev2_init(void) {

	int ret;

	ret = combo_register_driver(&space_ops);
	if (ret < 0) {
		printk(KERN_ERR "spacev2: can't register combov2 device\n");
		return ret;
	}

	PDEBUG(".. module init\n");
	ret = pci_register_driver(&driver);
	if (ret) {
		PERR("Failed to register pci driver (%d)!\n", ret);
		combo_unregister_driver(&space_ops);
	}

	PDEBUG(".. module init end\n");
	return ret;

}

static void __exit spacev2_exit(void) {


	PDEBUG(".. module exit\n");
	pci_unregister_driver(&driver);
	PDEBUG(".. module exit end\n");
	combo_unregister_driver(&space_ops);
}

module_init(spacev2_init)
module_exit(spacev2_exit)

#ifndef MODULE

/* format is: spacev2=enable */

static int __init spacev2_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	(void)(get_option(&str,&enable[nr_dev]) == 2);
	nr_dev++;
	return 1;
}

__setup("spacev2=", spacev2_setup);

#endif
