-- dma_up_int.vhd: DMA UP interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture dma_up_int_arch of dma_up_int is

   -- Types declaration -------------------------------------------------------

   type state_type is (S_IDLE, S_WRITE_TRANSACTION, S_WAIT_FOR_TAG); 

   -- Signals declaration -----------------------------------------------------

   -- FSM signals
   signal present_state : state_type;
   signal next_state    : state_type;


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                                 FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process(present_state, DMA_UP_SRC_RDY, DMA_DST_RDY,
                                DMA_UP_SOP, DMA_UP_EOP, DMA_UP_HDR_TYPE,
                                TAG_OFFER_VLD)
   begin
      case (present_state) is

         when S_IDLE =>
            if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1' AND
                DMA_UP_SOP = '1') then
               -- read transaction
               if (DMA_UP_HDR_TYPE = 0) then
                  if (DMA_UP_EOP = '1') then
                     if (TAG_OFFER_VLD = '1') then
                        next_state <= S_IDLE;
                     else
                        next_state <= S_WAIT_FOR_TAG;
                     end if;
                  else
                     next_state <= S_IDLE;
                  end if;
               -- write transaction
               elsif (DMA_UP_HDR_TYPE = 1) then
                  if (DMA_UP_EOP = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WRITE_TRANSACTION;
                  end if;
               else
                  next_state <= S_IDLE;
               end if;            
            else
               next_state <= S_IDLE;
            end if;

         when S_WRITE_TRANSACTION =>
            if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1') then
               if (DMA_UP_EOP = '1') then
                  next_state <= S_IDLE;
               else
                  next_state <= S_WRITE_TRANSACTION;
               end if;
            else
               next_state <= S_WRITE_TRANSACTION;
            end if;

         when S_WAIT_FOR_TAG =>
            if (TAG_OFFER_VLD = '1') then
--               if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1' AND
--                   DMA_UP_SOP = '1') then
--                  if (DMA_UP_EOP = '1') then
                     next_state <= S_IDLE;
--                  else
--                     next_state <= S_IDLE;
--                  end if;
--               else
--                  next_state <= S_IDLE;
--               end if;
            else
               next_state <= S_WAIT_FOR_TAG;
            end if;

--         when others =>
--            next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process(present_state, DMA_UP_SRC_RDY, DMA_DST_RDY,
                            DMA_UP_SOP, DMA_UP_EOP, DMA_UP_HDR_TYPE,
                            TAG_OFFER_VLD)
   begin
      -- default values
      DMA_UP_DST_RDY   <= '0';
      DMA_SRC_RDY      <= '0';
      DMA_SOP          <= '0';
      DMA_EOP          <= '0';
      TAG_OFFER_ACCEPT <= '0';

      case (present_state) is

         when S_IDLE =>
            if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1' AND
                DMA_UP_SOP = '1') then
               -- read transaction
               if (DMA_UP_HDR_TYPE = 0) then
                  if (DMA_UP_EOP = '1') then
                     if (TAG_OFFER_VLD = '1') then
                        DMA_UP_DST_RDY   <= '1';
                        DMA_SRC_RDY      <= '1';
                        DMA_SOP          <= '1';
                        DMA_EOP          <= '1';
                        TAG_OFFER_ACCEPT <= '1';
                     end if;
                  end if;
               -- write transaction
               elsif (DMA_UP_HDR_TYPE = 1) then
                  if (DMA_UP_EOP = '1') then
                     DMA_UP_DST_RDY <= '1';
                     DMA_SRC_RDY    <= '1';
                     DMA_SOP        <= '1';
                     DMA_EOP        <= '1';
                  else
                     DMA_UP_DST_RDY <= '1';
                     DMA_SRC_RDY    <= '1';
                     DMA_SOP        <= '1';
                  end if;
               end if;            
            end if;

         when S_WRITE_TRANSACTION =>
            if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1') then
               if (DMA_UP_EOP = '1') then
                  DMA_UP_DST_RDY <= '1';
                  DMA_SRC_RDY    <= '1';
                  DMA_EOP        <= '1';
               else
                  DMA_UP_DST_RDY <= '1';
                  DMA_SRC_RDY    <= '1';
               end if;
            end if;

         when S_WAIT_FOR_TAG =>
            if (TAG_OFFER_VLD = '1') then
               if (DMA_UP_SRC_RDY = '1' AND DMA_DST_RDY = '1' AND
                   DMA_UP_SOP = '1') then
                  if (DMA_UP_EOP = '1') then
                     DMA_UP_DST_RDY   <= '1';
                     DMA_SRC_RDY      <= '1';
                     DMA_SOP          <= '1';
                     DMA_EOP          <= '1';
                     TAG_OFFER_ACCEPT <= '1';
                  end if;
               end if;
            end if;

--         when others =>
--            null;

      end case;      
   end process output_logic_p;


   -- -------------------------------------------------------------------------
   --                    Assigning values to output ports
   -- -------------------------------------------------------------------------

   -- DMA-like interface to rq_int component
   DMA_DATA       <= DMA_UP_DATA;
   DMA_HDR_DWORDS <= DMA_UP_HDR_DWORDS;
   DMA_HDR_TYPE   <= '0' when (DMA_UP_HDR_TYPE = 0) else
                     '1' when (DMA_UP_HDR_TYPE = 1) else
                     '0';
   DMA_HDR_ADDR   <= DMA_UP_HDR_ADDR;
   PCIE_TAG       <= TAG_OFFER;

   -- interface to tag_manager component
   DMA_HDR_TAG    <= DMA_UP_HDR_TAG;
   DMA_HDR_ID     <= DMA_UP_HDR_ID;  

end architecture dma_up_int_arch;
