-------------------------------------------------------------------------------
-- Title      : BootFPGA Controller
-- Project    : IP Probe
-------------------------------------------------------------------------------
-- File       : BootFPGA_ctrl.vhd
-- Author     : Rune Dahl Jorgensen
-- Company    : Fiberblaze A/S
-- Created    : 2012-12-14
-- Last update: 2013-05-07
-- Platform   : Livorno
-- Standard   : VHDL'93/02
-------------------------------------------------------------------------------
-- Description: SPI interface to Actel BootFPGA
-------------------------------------------------------------------------------
-- Copyright (c) 2012 Fiberblaze A/S
-------------------------------------------------------------------------------
-- Revisions  :
-- Date        Version  Author  Description
-- 2012-12-14  1.0      RDJ     Created
-------------------------------------------------------------------------------
LIBRARY ieee;
USE ieee.std_logic_1164.ALL;
USE ieee.STD_LOGIC_UNSIGNED.ALL;

ENTITY BootFPGA_ctrl IS
   PORT (
      PCIe_clk        : IN  STD_LOGIC;
      reset           : IN  STD_LOGIC;
      -- SPI to BootFPGA
      bf_clk          : OUT STD_LOGIC;  -- SPI clock
      bf_nss          : OUT STD_LOGIC;  -- SPI not slave select
      bf_mosi         : OUT STD_LOGIC;  -- SPI Master data out
      bf_miso         : IN  STD_LOGIC;  -- SPI Slave data out
      -- PCIe i/f
      setup_wr        : IN  STD_LOGIC_VECTOR(63 DOWNTO 0);
      setup_wr_strobe : IN  STD_LOGIC;
      setup_rd        : OUT STD_LOGIC_VECTOR(63 DOWNTO 0)
      );

END ENTITY BootFPGA_ctrl;

ARCHITECTURE Behavioral OF BootFPGA_ctrl IS
   TYPE STATE_TYPE IS (IDLE, WRITE_ST, WRITE_CLOCK_ST, WRITE_WAIT_ST, READ_INIT_WAIT, READ_CLOCK0_ST, READ_CLOCK1_ST, READ_WAIT_DATA_ST, READ_DATA_ST, READ_LAST_ST, READ_RETURN_ST);
   SIGNAL state : STATE_TYPE := IDLE;

   SIGNAL output_reg : STD_LOGIC_VECTOR(47 DOWNTO 0);
   SIGNAL input_reg  : STD_LOGIC_VECTOR(15 DOWNTO 0);
   SIGNAL Command    : STD_LOGIC_VECTOR(3 DOWNTO 0);
   SIGNAL Flash_addr : STD_LOGIC_VECTOR(24 DOWNTO 0);
   SIGNAL Flash_data : STD_LOGIC_VECTOR(15 DOWNTO 0);

   SIGNAL wait_count : STD_LOGIC_VECTOR(6 DOWNTO 0) := (OTHERS => '0');
   SIGNAL wr_count   : STD_LOGIC_VECTOR(1 DOWNTO 0) := (OTHERS => '0');

   -- IO Register signals
   SIGNAL Dout     : STD_LOGIC_VECTOR(15 DOWNTO 0);
   SIGNAL Dout_reg : STD_LOGIC_VECTOR(15 DOWNTO 0);
   SIGNAL Din      : STD_LOGIC_VECTOR(15 DOWNTO 0);
   SIGNAL Tristate : STD_LOGIC;
   SIGNAL Req_in   : STD_LOGIC;
   SIGNAL Req_out  : STD_LOGIC;

   SIGNAL setup_data  : STD_LOGIC_VECTOR(15 DOWNTO 0);
   SIGNAL setup_done  : STD_LOGIC;      -- Single pulse when done
   SIGNAL setup_noack : STD_LOGIC;


   SIGNAL bf_clk_i  : STD_LOGIC;
   SIGNAL bf_nss_i  : STD_LOGIC;
   SIGNAL bf_mosi_i : STD_LOGIC;
   SIGNAL bf_miso_i : STD_LOGIC;
BEGIN  -- ARCHITECTURE Behavioral

   -- Pass internal signals out
   setup_rd(15 DOWNTO 0)  <= setup_data;
   setup_rd(16)           <= setup_done;
   setup_rd(17)           <= setup_noack;
   setup_rd(23 DOWNTO 18) <= (OTHERS => '0');
   setup_rd(31 DOWNTO 24) <= X"01";     -- Boot controller version
   setup_rd(63 DOWNTO 32) <= (OTHERS => '0');

   ----------------------------------------------------------------------
   -- IO FSM
   ----------------------------------------------------------------------
   fsm_proc : PROCESS(PCIe_clk)
   BEGIN
      IF rising_edge(PCIe_clk) THEN
         -- defaults
         -- setup_done <= '0';             -- default not done

         CASE state IS
            WHEN IDLE =>
               bf_clk_i   <= '1';
               bf_nss_i   <= '1';
               bf_mosi_i  <= '1';
               wait_count <= (OTHERS => '0');
               wr_count   <= (OTHERS => '0');
               IF setup_wr_strobe = '1' THEN
                  setup_done  <= '0';             -- default not done
                  -- Split setup vector into minor parts
                  command     <= setup_wr(63 DOWNTO 60);
                  flash_addr  <= setup_wr(56 DOWNTO 32);
                  flash_data  <= setup_wr(15 DOWNTO 0);
                  -- Sample directly to the output shift register
                  output_reg  <= setup_wr(63 DOWNTO 32) & setup_wr(15 DOWNTO 0);
                  -- No Ack signal
                  setup_noack <= '0';
                  -- Select slave
                  bf_nss_i    <= '0';
                  state       <= WRITE_ST;
               END IF;
               
            WHEN WRITE_ST =>
               bf_clk_i   <= '0';
               bf_mosi_i  <= output_reg(output_reg'HIGH);
               output_reg <= output_reg(output_reg'HIGH-1 DOWNTO 0) & '0';
               state      <= WRITE_CLOCK_ST;

            WHEN WRITE_CLOCK_ST =>
               bf_clk_i <= '1';
               IF wait_count = 15 THEN
                  wait_count <= (OTHERS => '0');
                  state      <= WRITE_WAIT_ST;
               ELSE
                  wait_count <= wait_count + 1;
                  state      <= WRITE_ST;
               END IF;

            WHEN WRITE_WAIT_ST =>
               bf_clk_i <= '1';
               IF wait_count = 10 THEN
                  wr_count   <= wr_count + 1;
                  wait_count <= (OTHERS => '0');
                  IF wr_count = 2 THEN
                     CASE Command IS
                        WHEN X"1" =>           -- Asynchronous Read command
                           state <= READ_INIT_WAIT;
                        WHEN X"8" =>           -- Read revision LSB
                           state <= READ_INIT_WAIT;
                        WHEN X"9" =>           -- Read revision MSB
                           state <= READ_INIT_WAIT;
                        WHEN X"A" =>           -- Read Actel Status
                           state <= READ_INIT_WAIT;
                        WHEN X"C" =>           -- Read ScratchPad
                           state <= READ_INIT_WAIT;
                        WHEN OTHERS =>
                           setup_done <= '1';  -- Done signal
                           state      <= IDLE;
                     END CASE;
                  ELSE
                     state <= WRITE_ST;
                  END IF;
               ELSE
                  wait_count <= wait_count + 1;
               END IF;

            WHEN READ_INIT_WAIT =>
               IF wait_count = 70 THEN
                  wait_count <= (OTHERS => '0');
                  state      <= READ_CLOCK0_ST;
               ELSE
                  wait_count <= wait_count + 1;
               END IF;

            WHEN READ_CLOCK0_ST =>
               input_reg <= input_reg(input_reg'HIGH-1 DOWNTO 0) & bf_miso_i;
               bf_clk_i  <= '0';
               bf_mosi_i <= '0';        -- Command = 0 = NOP - used to push out data
               state     <= READ_CLOCK1_ST;

            WHEN READ_CLOCK1_ST =>
               bf_clk_i <= '0';
               state    <= READ_WAIT_DATA_ST;

            WHEN READ_WAIT_DATA_ST =>
               bf_clk_i <= '1';
               state    <= READ_DATA_ST;
               
            WHEN READ_DATA_ST =>
               bf_clk_i <= '1';
               IF wait_count = 15 THEN
                  wait_count <= (OTHERS => '0');
                  state      <= READ_LAST_ST;
               ELSE
                  wait_count <= wait_count + 1;
                  state      <= READ_CLOCK0_ST;
               END IF;

            WHEN READ_LAST_ST =>
               input_reg <= input_reg(input_reg'HIGH-1 DOWNTO 0) & bf_miso_i;
               state     <= READ_RETURN_ST;

            WHEN READ_RETURN_ST =>
               setup_done <= '1';        -- Done signal
               setup_data <= input_reg;  -- Sample the returned data
               state      <= IDLE;

               
         END CASE;
         IF reset = '1' THEN
            state <= IDLE;
            setup_done <= '0';
         ELSE
         END IF;
      END IF;
   END PROCESS;

   iobuf_proc : PROCESS(PCIe_clk)
   BEGIN
      IF rising_edge(PCIe_clk) THEN
         bf_clk    <= bf_clk_i;
         bf_nss    <= bf_nss_i;
         bf_mosi   <= bf_mosi_i;
         bf_miso_i <= bf_miso;
      END IF;
   END PROCESS;
END ARCHITECTURE Behavioral;
