-- pcie.vhd : wrapper for two PCIe gen3 x8 cores and transformation components
--            AXI2MI and AXI2DMA on Fiberblaze FB1CG card
--!
--! \file
--! \brief wrapper for two PCIe gen3 x8 cores and transformation components
--!        AXI2MI and AXI2DMA on Fiberblaze FB1CG card
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;

library unisim;
use unisim.vcomponents.all;


-- ----------------------------------------------------------------------------
--                            Entity declaration
-- ----------------------------------------------------------------------------

entity pcie is
generic (
   --! \brief PCIe vendor ID
   --! \details X"18EC" = CESNET
   VENDOR_ID              : std_logic_vector(15 downto 0) := X"18EC";
   --! \brief PCIe core 0 device ID
   --! \details X"FBC0" = fb-100g1 card, PCIe endpoint 0
   DEVICE0_ID             : std_logic_vector(15 downto 0) := X"FBC0";
   --! \brief PCIe core 1 device ID
   --! \details X"FBC1" = fb-100g1 card, PCIe endpoint 1
   DEVICE1_ID             : std_logic_vector(15 downto 0) := X"FBC1";
   --! \brief PCIe subvendor ID
   --! \details X"18EC" = CESNET
   SUB_VENDOR_ID          : std_logic_vector(15 downto 0) := X"18EC";
   --! \brief PCIe core 0 subdevice ID
   --! \details X"FBC0" = fb-100g1 card, PCIe endpoint 0
   SUB_DEVICE0_ID         : std_logic_vector(15 downto 0) := X"FBC0";
   --! \brief PCIe core 1 subdevice ID
   --! \details X"FBC1" = fb-100g1 card, PCIe endpoint 1
   SUB_DEVICE1_ID         : std_logic_vector(15 downto 0) := X"FBC1";

   --! \brief BAR0 base address for PCIE->MI32 transalation
   BAR0_BASE_ADDR         : std_logic_vector(31 downto 0) := X"01000000";
   --! \brief BAR1 base address for PCIE->MI32 transalation
   BAR1_BASE_ADDR         : std_logic_vector(31 downto 0) := X"02000000";
   --! \brief BAR2 base address for PCIE->MI32 transalation
   BAR2_BASE_ADDR         : std_logic_vector(31 downto 0) := X"03000000";
   --! \brief BAR3 base address for PCIE->MI32 transalation
   BAR3_BASE_ADDR         : std_logic_vector(31 downto 0) := X"04000000";
   --! \brief BAR4 base address for PCIE->MI32 transalation
   BAR4_BASE_ADDR         : std_logic_vector(31 downto 0) := X"05000000";
   --! \brief BAR5 base address for PCIE->MI32 transalation
   BAR5_BASE_ADDR         : std_logic_vector(31 downto 0) := X"06000000";
   --! \brief Expansion ROM base address for PCIE->MI32 transalation
   EXP_ROM_BASE_ADDR      : std_logic_vector(31 downto 0) := X"0A000000";

   --! \brief Number of items in FIFO for data read from FPGA
   AXI2MI_FIFO_SIZE       : integer := 16;

   --! \brief Select whether single or double PCIe cores are used
   --! \details single PCIe = false; double PCIe = true
   DOUBLE_PCIE            : boolean := true
);
port (
   --! \name Input clock and reset (from PCIe ports, 100 MHz)
   -- -------------------------------------------------------------------------
   --! \brief Clock from PCIe0 port
   PCIE0_SYS_CLK          : in  std_logic;
   --! \brief Clock from PCIe1 port
   PCIE1_SYS_CLK          : in  std_logic;
   --! \brief PCIe reset
   PCIE_SYS_RESET         : in  std_logic;

   --! \name PCIe interface
   -- -------------------------------------------------------------------------
   --! \brief Receive data, positive
   PCI_EXP_RXP            : in  std_logic_vector(15 downto 0);
   --! \brief Receive data, negative
   PCI_EXP_RXN            : in  std_logic_vector(15 downto 0);
   --! \brief Transmit data, positive
   PCI_EXP_TXP            : out std_logic_vector(15 downto 0);
   --! \brief Transmit data, negative
   PCI_EXP_TXN            : out std_logic_vector(15 downto 0);

   --! \name Clock and reset for DMA bus from PCIe core 0 (250 MHz)
   --! \details Use this clock and reset only for DMA 0 bus!
   -- -------------------------------------------------------------------------
   --! \brief Clock for DMA 0 bus
   DMA0_CLK               : out std_logic;
   --! \brief Reset for DMA 0 bus
   DMA0_RESET             : out std_logic;

   --! \name DMA UP interface (user application --> host system), PCIe core 0
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   DMA0_UP_DATA           : in  std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when DMA0_UP_SOP is valid.
   DMA0_UP_HDR            : in  std_logic_vector(95 downto 0);
   --! \brief Start of transaction
   --! \details Valid when DMA0_UP_SRC_RDY = DMA0_UP_DST_RDY='1'.
   DMA0_UP_SOP            : in  std_logic;
   --! \brief End of transaction
   --! \details Valid when DMA0_UP_SRC_RDY = DMA0_UP_DST_RDY = '1'.
   DMA0_UP_EOP            : in  std_logic;
   --! \brief Source is ready to transmit data
   DMA0_UP_SRC_RDY        : in  std_logic;
   --! \brief Destination is ready to receive data
   DMA0_UP_DST_RDY        : out std_logic;

   --! \name DMA DOWN interface (host system --> user application), PCIe core 0
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   DMA0_DOWN_DATA         : out std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when DMA0_DOWN_SOP is valid.
   DMA0_DOWN_HDR          : out std_logic_vector(31 downto 0);
   --! \brief Start of transaction
   --! \details Valid when DMA0_DOWN_SRC_RDY = DMA0_DOWN_DST_RDY='1'.
   DMA0_DOWN_SOP          : out std_logic;
   --! \brief End of transaction
   --! \details Valid when DMA0_DOWN_SRC_RDY = DMA0_DOWN_DST_RDY='1'.
   DMA0_DOWN_EOP          : out std_logic;
   --! \brief Source is ready to transmit data
   DMA0_DOWN_SRC_RDY      : out std_logic;
   --! \brief Destination is ready to receive data
   DMA0_DOWN_DST_RDY      : in  std_logic;

   --! \name Clock and reset for DMA bus from PCIe core 1 (250 MHz)
   --! \details Use this clock and reset only for DMA 1 bus!
   -- -------------------------------------------------------------------------
   --! \brief Clock for DMA 1 bus
   DMA1_CLK               : out std_logic;
   --! \brief Reset for DMA 1 bus
   DMA1_RESET             : out std_logic;

   --! \name DMA UP interface (user application --> host system), PCIe core 1
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   DMA1_UP_DATA           : in  std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when DMA1_UP_SOP is valid.
   DMA1_UP_HDR            : in  std_logic_vector(95 downto 0);
   --! \brief Start of transaction
   --! \details Valid when DMA1_UP_SRC_RDY = DMA1_UP_DST_RDY='1'.
   DMA1_UP_SOP            : in  std_logic;
   --! \brief End of transaction
   --! \details Valid when DMA1_UP_SRC_RDY = DMA1_UP_DST_RDY = '1'.
   DMA1_UP_EOP            : in  std_logic;
   --! \brief Source is ready to transmit data
   DMA1_UP_SRC_RDY        : in  std_logic;
   --! \brief Destination is ready to receive data
   DMA1_UP_DST_RDY        : out std_logic;

   --! \name DMA DOWN interface (host system --> user application), PCIe core 1
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   DMA1_DOWN_DATA         : out std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when DMA1_DOWN_SOP is valid.
   DMA1_DOWN_HDR          : out std_logic_vector(31 downto 0);
   --! \brief Start of transaction
   --! \details Valid when DMA1_DOWN_SRC_RDY = DMA1_DOWN_DST_RDY='1'.
   DMA1_DOWN_SOP          : out std_logic;
   --! \brief End of transaction
   --! \details Valid when DMA1_DOWN_SRC_RDY = DMA1_DOWN_DST_RDY='1'.
   DMA1_DOWN_EOP          : out std_logic;
   --! \brief Source is ready to transmit data
   DMA1_DOWN_SRC_RDY      : out std_logic;
   --! \brief Destination is ready to receive data
   DMA1_DOWN_DST_RDY      : in  std_logic;

   --! \name Configuration status interface, both PCIe cores
   -- -------------------------------------------------------------------------
   --! \brief Max_Payload_Size reported by PCIe core 0
   PCIE0_MAX_PAYLOAD      : out std_logic_vector(2 downto 0);
   --! \brief Max_Read_Request_Size reported by PCIe core 0
   PCIE0_MAX_READ_REQ     : out std_logic_vector(2 downto 0);
   --! \brief Max_Payload_Size reported by PCIe core 1
   PCIE1_MAX_PAYLOAD      : out std_logic_vector(2 downto 0);
   --! \brief Max_Read_Request_Size reported by PCIe core 1
   PCIE1_MAX_READ_REQ     : out std_logic_vector(2 downto 0);

   --! \name Configuration interrupt controller interface
   --! \details Ports are internally connected to PCIe core 0.
   -- -------------------------------------------------------------------------
   --! \brief MSI interrupt vector
   CFG_INTERRUPT_MSI_INT  : in  std_logic_vector(31 downto 0);
   --! \brief MSI interrupt sent
   CFG_INTERRUPT_MSI_SENT : out std_logic;

   --! \name Output clock and reset (to user application, 250 MHz)
   --! \details Ports are internally connected to PCIe core 1.
   -- -------------------------------------------------------------------------
   --! \brief Clock for user application
   USER_CLK               : out std_logic;
   --! \brief Reset for user application
   USER_RESET             : out std_logic;

   --! \name MI32 interface - root of the MI32 bus tree
   --! \details Interface is clocked by user-defined clock (MI_CLK); ports are
   --!          internally connected to PCIe core 1.
   -- -------------------------------------------------------------------------
   --! \brief Clock
   MI_CLK                 : in  std_logic;
   --! \brief Reset
   MI_RESET               : in  std_logic;
   --! \brief Data to be written
   MI_DWR                 : out std_logic_vector(31 downto 0);
   --! \brief Address bus
   MI_ADDR                : out std_logic_vector(31 downto 0);
   --! \brief Byte enable signals
   MI_BE                  : out std_logic_vector(3 downto 0);
   --! \brief Read request
   MI_RD                  : out std_logic;
   --! \brief Write request
   MI_WR                  : out std_logic;
   --! \brief Read data
   MI_DRD                 : in  std_logic_vector(31 downto 0);
   --! \brief Address ready
   MI_ARDY                : in  std_logic;
   --! \brief Read data ready
   MI_DRDY                : in  std_logic
);
end entity pcie;


-- ----------------------------------------------------------------------------
--                           Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of pcie is

   --! Components declaration
   -- -------------------------------------------------------------------------

   component pcie3_7x_v3_0
   generic(
      VENDOR_ID                         : std_logic_vector(15 downto 0);
      DEVICE_ID                         : std_logic_vector(15 downto 0);
      SUB_VENDOR_ID                     : std_logic_vector(15 downto 0);
      SUB_DEVICE_ID                     : std_logic_vector(15 downto 0);
      PCIE_CORE                         : integer
   );
   port(
      pci_exp_txn                       : out std_logic_vector(7 downto 0);
      pci_exp_txp                       : out std_logic_vector(7 downto 0);
      pci_exp_rxn                       : in  std_logic_vector(7 downto 0);
      pci_exp_rxp                       : in  std_logic_vector(7 downto 0);
      user_clk                          : out std_logic;
      user_reset                        : out std_logic;
      user_lnk_up                       : out std_logic;
      user_app_rdy                      : out std_logic;
      s_axis_rq_tlast                   : in  std_logic;
      s_axis_rq_tdata                   : in  std_logic_vector(255 downto 0);
      s_axis_rq_tuser                   : in  std_logic_vector(59 downto 0);
      s_axis_rq_tkeep                   : in  std_logic_vector(7 downto 0);
      s_axis_rq_tready                  : out std_logic_vector(3 downto 0);
      s_axis_rq_tvalid                  : in  std_logic;
      m_axis_rc_tdata                   : out std_logic_vector(255 downto 0);
      m_axis_rc_tuser                   : out std_logic_vector(74 downto 0);
      m_axis_rc_tlast                   : out std_logic;
      m_axis_rc_tkeep                   : out std_logic_vector(7 downto 0);
      m_axis_rc_tvalid                  : out std_logic;
      m_axis_rc_tready                  : in  std_logic_vector(21 downto 0);
      m_axis_cq_tdata                   : out std_logic_vector(255 downto 0);
      m_axis_cq_tuser                   : out std_logic_vector(84 downto 0);
      m_axis_cq_tlast                   : out std_logic;
      m_axis_cq_tkeep                   : out std_logic_vector(7 downto 0);
      m_axis_cq_tvalid                  : out std_logic;
      m_axis_cq_tready                  : in  std_logic_vector(21 downto 0);
      s_axis_cc_tdata                   : in  std_logic_vector(255 downto 0);
      s_axis_cc_tuser                   : in  std_logic_vector(32 downto 0);
      s_axis_cc_tlast                   : in  std_logic;
      s_axis_cc_tkeep                   : in  std_logic_vector(7 downto 0);
      s_axis_cc_tvalid                  : in  std_logic;
      s_axis_cc_tready                  : out std_logic_vector(3 downto 0);
      pcie_rq_seq_num                   : out std_logic_vector(3 downto 0);
      pcie_rq_seq_num_vld               : out std_logic;
      pcie_rq_tag                       : out std_logic_vector(5 downto 0);
      pcie_rq_tag_vld                   : out std_logic;
      pcie_cq_np_req                    : in  std_logic;
      pcie_cq_np_req_count              : out std_logic_vector(5 downto 0);
      cfg_phy_link_down                 : out std_logic;
      cfg_phy_link_status               : out std_logic_vector(1 downto 0);
      cfg_negotiated_width              : out std_logic_vector(3 downto 0);
      cfg_current_speed                 : out std_logic_vector(2 downto 0);
      cfg_max_payload                   : out std_logic_vector(2 downto 0);
      cfg_max_read_req                  : out std_logic_vector(2 downto 0);
      cfg_function_status               : out std_logic_vector(7 downto 0);
      cfg_function_power_state          : out std_logic_vector(5 downto 0);
      cfg_vf_status                     : out std_logic_vector(11 downto 0);
      cfg_vf_power_state                : out std_logic_vector(17 downto 0);
      cfg_link_power_state              : out std_logic_vector(1 downto 0);
      cfg_err_cor_out                   : out std_logic;
      cfg_err_nonfatal_out              : out std_logic;
      cfg_err_fatal_out                 : out std_logic;
      cfg_ltr_enable                    : out std_logic;
      cfg_ltssm_state                   : out std_logic_vector(5 downto 0);
      cfg_rcb_status                    : out std_logic_vector(1 downto 0);
      cfg_dpa_substate_change           : out std_logic_vector(1 downto 0);
      cfg_obff_enable                   : out std_logic_vector(1 downto 0);
      cfg_pl_status_change              : out std_logic;
      cfg_tph_requester_enable          : out std_logic_vector(1 downto 0);
      cfg_tph_st_mode                   : out std_logic_vector(5 downto 0);
      cfg_vf_tph_requester_enable       : out std_logic_vector(5 downto 0);
      cfg_vf_tph_st_mode                : out std_logic_vector(17 downto 0);
      cfg_interrupt_int                 : in  std_logic_vector(3 downto 0);
      cfg_interrupt_pending             : in  std_logic_vector(1 downto 0);
      cfg_interrupt_sent                : out std_logic;
      cfg_interrupt_msi_enable          : out std_logic_vector(1 downto 0);
      cfg_interrupt_msi_vf_enable       : out std_logic_vector(5 downto 0);
      cfg_interrupt_msi_mmenable        : out std_logic_vector(5 downto 0);
      cfg_interrupt_msi_mask_update     : out std_logic;
      cfg_interrupt_msi_data            : out std_logic_vector(31 downto 0);
      cfg_interrupt_msi_select          : in  std_logic_vector(3 downto 0);
      cfg_interrupt_msi_int             : in  std_logic_vector(31 downto 0);
      cfg_interrupt_msi_pending_status  : in  std_logic_vector(63 downto 0);
      cfg_interrupt_msi_sent            : out std_logic;
      cfg_interrupt_msi_fail            : out std_logic;
      cfg_interrupt_msi_attr            : in  std_logic_vector(2 downto 0);
      cfg_interrupt_msi_tph_present     : in  std_logic;
      cfg_interrupt_msi_tph_type        : in  std_logic_vector(1 downto 0);
      cfg_interrupt_msi_tph_st_tag      : in  std_logic_vector(8 downto 0);
      cfg_interrupt_msi_function_number : in  std_logic_vector(2 downto 0);
      sys_clk                           : in  std_logic;
      sys_reset                         : in  std_logic
   );
   end component;

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! PCIe core 0 requester request interface signals
   signal pcie0_rq_tdata           : std_logic_vector(255 downto 0);
   signal pcie0_rq_tuser           : std_logic_vector(59 downto 0);
   signal pcie0_rq_tlast           : std_logic;
   signal pcie0_rq_tkeep           : std_logic_vector(7 downto 0);
   signal pcie0_rq_tready_vector   : std_logic_vector(3 downto 0);
   signal pcie0_rq_tready          : std_logic;
   signal pcie0_rq_tvalid          : std_logic;
   signal pcie0_rq_seq_num         : std_logic_vector(3 downto 0);
   signal pcie0_rq_seq_num_vld     : std_logic;
   signal pcie0_rq_tag             : std_logic_vector(5 downto 0);
   signal pcie0_rq_tag_vld         : std_logic;

   --! PCIe core 1 requester request interface signals
   signal pcie1_rq_tdata           : std_logic_vector(255 downto 0);
   signal pcie1_rq_tuser           : std_logic_vector(59 downto 0);
   signal pcie1_rq_tlast           : std_logic;
   signal pcie1_rq_tkeep           : std_logic_vector(7 downto 0);
   signal pcie1_rq_tready_vector   : std_logic_vector(3 downto 0);
   signal pcie1_rq_tready          : std_logic;
   signal pcie1_rq_tvalid          : std_logic;
   signal pcie1_rq_seq_num         : std_logic_vector(3 downto 0);
   signal pcie1_rq_seq_num_vld     : std_logic;
   signal pcie1_rq_tag             : std_logic_vector(5 downto 0);
   signal pcie1_rq_tag_vld         : std_logic;

   --! PCIe core 0 requester completion interface signals
   signal pcie0_rc_tdata           : std_logic_vector(255 downto 0);
   signal pcie0_rc_tuser           : std_logic_vector(74 downto 0);
   signal pcie0_rc_tlast           : std_logic;
   signal pcie0_rc_tkeep           : std_logic_vector(7 downto 0);
   signal pcie0_rc_tvalid          : std_logic;
   signal pcie0_rc_tready_vector   : std_logic_vector(21 downto 0);
   signal pcie0_rc_tready          : std_logic;

   --! PCIe core 1 requester completion interface signals
   signal pcie1_rc_tdata           : std_logic_vector(255 downto 0);
   signal pcie1_rc_tuser           : std_logic_vector(74 downto 0);
   signal pcie1_rc_tlast           : std_logic;
   signal pcie1_rc_tkeep           : std_logic_vector(7 downto 0);
   signal pcie1_rc_tvalid          : std_logic;
   signal pcie1_rc_tready_vector   : std_logic_vector(21 downto 0);
   signal pcie1_rc_tready          : std_logic;

   --! Completer request interface signals
   signal pcie_cq_tdata            : std_logic_vector(255 downto 0);
   signal pcie_cq_tuser            : std_logic_vector(84 downto 0);
   signal pcie_cq_tlast            : std_logic;
   signal pcie_cq_tkeep            : std_logic_vector(7 downto 0);
   signal pcie_cq_tvalid           : std_logic;
   signal pcie_cq_tready_vector    : std_logic_vector(21 downto 0);
   signal pcie_cq_tready           : std_logic;
   signal pcie_cq_np_req           : std_logic;
   signal pcie_cq_np_req_count     : std_logic_vector(5 downto 0);

   --! Completer completion interface signals
   signal pcie_cc_tdata            : std_logic_vector(255 downto 0);
   signal pcie_cc_tuser            : std_logic_vector(32 downto 0);
   signal pcie_cc_tlast            : std_logic;
   signal pcie_cc_tkeep            : std_logic_vector(7 downto 0);
   signal pcie_cc_tvalid           : std_logic;
   signal pcie_cc_tready_vector    : std_logic_vector(3 downto 0);
   signal pcie_cc_tready           : std_logic;

   --! User logic clk and reset signals for PCIe 0
   signal pcie0_user_clk           : std_logic;
   signal pcie0_user_reset         : std_logic;
   signal pcie0_user_lnk_up        : std_logic;
   signal pcie0_user_reset_shifter : std_logic_vector(15 downto 0);
   signal pcie0_user_reset_out     : std_logic;

   --! User logic clk and reset signals for PCIe 1
   signal pcie1_user_clk           : std_logic;
   signal pcie1_user_reset         : std_logic;
   signal pcie1_user_lnk_up        : std_logic;
   signal pcie1_user_reset_shifter : std_logic_vector(15 downto 0);
   signal pcie1_user_reset_out     : std_logic;

   --! Configuration status interface signals
   signal pcie1_max_payload_out    : std_logic_vector(2 downto 0);

   --! Configuration interrupt controller interface signals
   signal interrupt_msi_int        : std_logic_vector(31 downto 0);
   signal interrupt_msi_sent       : std_logic;

   --! Output MI32 interface of AXI2MI component
   signal axi2mi_mi_dwr            : std_logic_vector(31 downto 0);
   signal axi2mi_mi_addr           : std_logic_vector(31 downto 0);
   signal axi2mi_mi_be             : std_logic_vector(3 downto 0);
   signal axi2mi_mi_rd             : std_logic;
   signal axi2mi_mi_wr             : std_logic;
   signal axi2mi_mi_drd            : std_logic_vector(31 downto 0);
   signal axi2mi_mi_ardy           : std_logic;
   signal axi2mi_mi_drdy           : std_logic;


-- ----------------------------------------------------------------------------
--                              Architecture body
-- ----------------------------------------------------------------------------

begin


   ----------------------------------------------------------------------------
   --                     PCIe core 1 (always instantiated)
   ----------------------------------------------------------------------------

   --! PCIe core 1 instantiation
   -- -------------------------------------------------------------------------

   --! auxiliary signals for connection of some ports
   pcie1_rq_tready        <= pcie1_rq_tready_vector(0);
   pcie1_rc_tready_vector <= (others => pcie1_rc_tready);
   pcie_cq_tready_vector  <= (others => pcie_cq_tready);
   pcie_cc_tready         <= pcie_cc_tready_vector(0);

   pcie1_i : pcie3_7x_v3_0
   generic map (
      VENDOR_ID                         => VENDOR_ID,
      DEVICE_ID                         => DEVICE1_ID,
      SUB_VENDOR_ID                     => SUB_VENDOR_ID,
      SUB_DEVICE_ID                     => SUB_DEVICE1_ID,
      PCIE_CORE                         => 1
   )
   port map (
      pci_exp_txn                       => PCI_EXP_TXN(15 downto 8),
      pci_exp_txp                       => PCI_EXP_TXP(15 downto 8),
      pci_exp_rxn                       => PCI_EXP_RXN(15 downto 8),
      pci_exp_rxp                       => PCI_EXP_RXP(15 downto 8),
      user_clk                          => pcie1_user_clk,
      user_reset                        => pcie1_user_reset,
      user_lnk_up                       => pcie1_user_lnk_up,
      user_app_rdy                      => open,
      s_axis_rq_tlast                   => pcie1_rq_tlast,
      s_axis_rq_tdata                   => pcie1_rq_tdata,
      s_axis_rq_tuser                   => pcie1_rq_tuser,
      s_axis_rq_tkeep                   => pcie1_rq_tkeep,
      s_axis_rq_tready                  => pcie1_rq_tready_vector,
      s_axis_rq_tvalid                  => pcie1_rq_tvalid,
      m_axis_rc_tdata                   => pcie1_rc_tdata,
      m_axis_rc_tuser                   => pcie1_rc_tuser,
      m_axis_rc_tlast                   => pcie1_rc_tlast,
      m_axis_rc_tkeep                   => pcie1_rc_tkeep,
      m_axis_rc_tvalid                  => pcie1_rc_tvalid,
      m_axis_rc_tready                  => pcie1_rc_tready_vector,
      m_axis_cq_tdata                   => pcie_cq_tdata,
      m_axis_cq_tuser                   => pcie_cq_tuser,
      m_axis_cq_tlast                   => pcie_cq_tlast,
      m_axis_cq_tkeep                   => pcie_cq_tkeep,
      m_axis_cq_tvalid                  => pcie_cq_tvalid,
      m_axis_cq_tready                  => pcie_cq_tready_vector,
      s_axis_cc_tdata                   => pcie_cc_tdata,
      s_axis_cc_tuser                   => pcie_cc_tuser,
      s_axis_cc_tlast                   => pcie_cc_tlast,
      s_axis_cc_tkeep                   => pcie_cc_tkeep,
      s_axis_cc_tvalid                  => pcie_cc_tvalid,
      s_axis_cc_tready                  => pcie_cc_tready_vector,
      pcie_rq_seq_num                   => pcie1_rq_seq_num,
      pcie_rq_seq_num_vld               => pcie1_rq_seq_num_vld,
      pcie_rq_tag                       => pcie1_rq_tag,
      pcie_rq_tag_vld                   => pcie1_rq_tag_vld,
      pcie_cq_np_req                    => pcie_cq_np_req,
      pcie_cq_np_req_count              => pcie_cq_np_req_count,
      cfg_phy_link_down                 => open,
      cfg_phy_link_status               => open,
      cfg_negotiated_width              => open,
      cfg_current_speed                 => open,
      cfg_max_payload                   => pcie1_max_payload_out,
      cfg_max_read_req                  => PCIE1_MAX_READ_REQ,
      cfg_function_status               => open,
      cfg_function_power_state          => open,
      cfg_vf_status                     => open,
      cfg_vf_power_state                => open,
      cfg_link_power_state              => open,
      cfg_err_cor_out                   => open,
      cfg_err_nonfatal_out              => open,
      cfg_err_fatal_out                 => open,
      cfg_ltr_enable                    => open,
      cfg_ltssm_state                   => open,
      cfg_rcb_status                    => open,
      cfg_dpa_substate_change           => open,
      cfg_obff_enable                   => open,
      cfg_pl_status_change              => open,
      cfg_tph_requester_enable          => open,
      cfg_tph_st_mode                   => open,
      cfg_vf_tph_requester_enable       => open,
      cfg_vf_tph_st_mode                => open,
      cfg_interrupt_int                 => "0000",
      cfg_interrupt_pending             => "00",
      cfg_interrupt_sent                => open,
      cfg_interrupt_msi_enable          => open,
      cfg_interrupt_msi_vf_enable       => open,
      cfg_interrupt_msi_mmenable        => open,
      cfg_interrupt_msi_mask_update     => open,
      cfg_interrupt_msi_data            => open,
      cfg_interrupt_msi_select          => "0000",
      cfg_interrupt_msi_int             => interrupt_msi_int,
      cfg_interrupt_msi_pending_status  => X"0000000000000000",
      cfg_interrupt_msi_sent            => interrupt_msi_sent,
      cfg_interrupt_msi_fail            => open,
      cfg_interrupt_msi_attr            => "000",
      cfg_interrupt_msi_tph_present     => '0',
      cfg_interrupt_msi_tph_type        => "00",
      cfg_interrupt_msi_tph_st_tag      => "000000000",
      cfg_interrupt_msi_function_number => "000",
      sys_clk                           => PCIE1_SYS_CLK,
      sys_reset                         => PCIE_SYS_RESET
   );

   --! Reset logic
   -- -------------------------------------------------------------------------

   pcie1_reset_logic_p : process(pcie1_user_clk)
   begin
      if ((pcie1_user_clk'event) AND (pcie1_user_clk = '1')) then
         if ((pcie1_user_reset = '1') OR
             (pcie1_user_lnk_up = '0')) then
            pcie1_user_reset_shifter <= (others => '1');
         else
            pcie1_user_reset_shifter <= pcie1_user_reset_shifter(pcie1_user_reset_shifter'high-1 downto 0) & '0';
         end if;
         pcie1_user_reset_out <= pcie1_user_reset_shifter(pcie1_user_reset_shifter'high);
      end if;
   end process pcie1_reset_logic_p;

   --! MSI_INT synchronization
   -- -------------------------------------------------------------------------

   msi_int_sync : entity work.ASYNC_BUS_HANDSHAKE
   generic map(
      DATA_WIDTH => 32
   )
   port map(
      --! A clock domain
      ACLK       => MI_CLK,
      ARST       => MI_RESET,
      ADATAIN    => CFG_INTERRUPT_MSI_INT,
      ASEND      => '1',
      AREADY     => open,

      --! B clock domain
      BCLK       => pcie1_user_clk,
      BRST       => pcie1_user_reset,
      BDATAOUT   => interrupt_msi_int,
      BLOAD      => '1',
      BVALID     => open
   );

   --! MSI_SENT synchronization
   -- -------------------------------------------------------------------------

   msi_sent_sync : entity work.ASYNC_GENERAL
   generic map(
      TWO_REG             => false,
      DETECT_RISING_EDGE  => true,
      DETECT_FALLING_EDGE => false
   )
   port map(
      --! A clock domain
      ACLK                => pcie1_user_clk,
      ARST                => pcie1_user_reset,
      ADATAIN             => interrupt_msi_sent,
      AREADY              => open,

      --! B clock domain
      BCLK                => MI_CLK,
      BRST                => MI_RESET,
      BDATAOUT            => CFG_INTERRUPT_MSI_SENT
   );

   --! AXI2DMA instantiation
   -- -------------------------------------------------------------------------

   pcie1_axi2dma_i : entity work.axi2dma
   port map (
      --! Common signals
      CLK              => pcie1_user_clk,
      RESET            => pcie1_user_reset_out,

      --! Requester Request Interface (RQ)
      RQ_DATA          => pcie1_rq_tdata,
      RQ_USER          => pcie1_rq_tuser,
      RQ_LAST          => pcie1_rq_tlast,
      RQ_KEEP          => pcie1_rq_tkeep,
      RQ_READY         => pcie1_rq_tready,
      RQ_VALID         => pcie1_rq_tvalid,
      RQ_SEQ_NUM       => pcie1_rq_seq_num,
      RQ_SEQ_NUM_VLD   => pcie1_rq_seq_num_vld,
      RQ_TAG           => pcie1_rq_tag,
      RQ_TAG_VLD       => pcie1_rq_tag_vld,

      --! Requester Completion Interface (RC)
      RC_DATA          => pcie1_rc_tdata,
      RC_USER          => pcie1_rc_tuser,
      RC_LAST          => pcie1_rc_tlast,
      RC_KEEP          => pcie1_rc_tkeep,
      RC_VALID         => pcie1_rc_tvalid,
      RC_READY         => pcie1_rc_tready,

      --! DMA UP Interface (from user application to host system)
      DMA_UP_DATA      => DMA1_UP_DATA,
      DMA_UP_HDR       => DMA1_UP_HDR,
      DMA_UP_SOP       => DMA1_UP_SOP,
      DMA_UP_EOP       => DMA1_UP_EOP,
      DMA_UP_SRC_RDY   => DMA1_UP_SRC_RDY,
      DMA_UP_DST_RDY   => DMA1_UP_DST_RDY,

      --! DMA DOWN Interface (from host system to user application)
      DMA_DOWN_DATA    => DMA1_DOWN_DATA,
      DMA_DOWN_HDR     => DMA1_DOWN_HDR,
      DMA_DOWN_SOP     => DMA1_DOWN_SOP,
      DMA_DOWN_EOP     => DMA1_DOWN_EOP,
      DMA_DOWN_SRC_RDY => DMA1_DOWN_SRC_RDY,
      DMA_DOWN_DST_RDY => DMA1_DOWN_DST_RDY
   );

   --! AXI2MI instantiation
   --! Instantiated only for PCIe core 1
   -- -------------------------------------------------------------------------

   axi2mi_i : entity work.axi2mi
   generic map (
      BAR0_BASE_ADDR    => BAR0_BASE_ADDR,
      BAR1_BASE_ADDR    => BAR1_BASE_ADDR,
      BAR2_BASE_ADDR    => BAR2_BASE_ADDR,
      BAR3_BASE_ADDR    => BAR3_BASE_ADDR,
      BAR4_BASE_ADDR    => BAR4_BASE_ADDR,
      BAR5_BASE_ADDR    => BAR5_BASE_ADDR,
      EXP_ROM_BASE_ADDR => EXP_ROM_BASE_ADDR,
      FIFO_SIZE         => AXI2MI_FIFO_SIZE
   )
   port map (
      --! Common signals
      CLK               => pcie1_user_clk,
      RESET             => pcie1_user_reset_out,

      --! Completer Request Interface (CQ)
      CQ_DATA           => pcie_cq_tdata,
      CQ_USER           => pcie_cq_tuser,
      CQ_LAST           => pcie_cq_tlast,
      CQ_KEEP           => pcie_cq_tkeep,
      CQ_VALID          => pcie_cq_tvalid,
      CQ_READY          => pcie_cq_tready,
      CQ_NP_REQ         => pcie_cq_np_req,
      CQ_NP_REQ_COUNT   => pcie_cq_np_req_count,

      --! Completer Completion Interface (CC)
      CC_DATA           => pcie_cc_tdata,
      CC_USER           => pcie_cc_tuser,
      CC_LAST           => pcie_cc_tlast,
      CC_KEEP           => pcie_cc_tkeep,
      CC_VALID          => pcie_cc_tvalid,
      CC_READY          => pcie_cc_tready,

      --! Configuration Status Interface
      MAX_PAYLOAD_SIZE  => pcie1_max_payload_out,

      --! MI32 interface
      MI_DWR            => axi2mi_mi_dwr,
      MI_ADDR           => axi2mi_mi_addr,
      MI_BE             => axi2mi_mi_be,
      MI_RD             => axi2mi_mi_rd,
      MI_WR             => axi2mi_mi_wr,
      MI_DRD            => axi2mi_mi_drd,
      MI_ARDY           => axi2mi_mi_ardy,
      MI_DRDY           => axi2mi_mi_drdy
   );

   --! MI32_async instantiation
   -- -------------------------------------------------------------------------

   mi_async_i : entity work.mi32_async_handshake
   port map (
      -- Master interface
      CLK_M     => pcie1_user_clk,
      RESET_M   => pcie1_user_reset_out,
      MI_M_DWR  => axi2mi_mi_dwr,
      MI_M_ADDR => axi2mi_mi_addr,
      MI_M_BE   => axi2mi_mi_be,
      MI_M_RD   => axi2mi_mi_rd,
      MI_M_WR   => axi2mi_mi_wr,
      MI_M_DRD  => axi2mi_mi_drd,
      MI_M_ARDY => axi2mi_mi_ardy,
      MI_M_DRDY => axi2mi_mi_drdy,

      -- Slave interface
      CLK_S     => MI_CLK,
      RESET_S   => MI_RESET,
      MI_S_DWR  => MI_DWR,
      MI_S_ADDR => MI_ADDR,
      MI_S_BE   => MI_BE,
      MI_S_RD   => MI_RD,
      MI_S_WR   => MI_WR,
      MI_S_DRD  => MI_DRD,
      MI_S_ARDY => MI_ARDY,
      MI_S_DRDY => MI_DRDY
   );

   --! Output ports mapping
   -- -------------------------------------------------------------------------

   --! Clock and reset for DMA 1 bus
   DMA1_CLK          <= pcie1_user_clk;
   DMA1_RESET        <= pcie1_user_reset_out;

   --! Configuration status interface
   PCIE1_MAX_PAYLOAD <= pcie1_max_payload_out;


   -- -------------------------------------------------------------------------
   --            Generate when single PCIe core is instantiated
   -- -------------------------------------------------------------------------

   --! When single PCIe core is utilized, only the PCIe core 1 (PCIE_X0Y1)
   --! is instantiated.

   single_pcie_gen : if (NOT DOUBLE_PCIE) generate

      --! Mapping of output ports related do PCIe core 0 (tying to constants)
      -- ----------------------------------------------------------------------

      --! PCIe interface is not imitated
      --! Imitation of this interface (tying corresponding ports to '0') causes
      --! errors during implementation
      --! Relevant ports:
      --!    PCI_EXP_TXP(7 downto 0)
      --!    PCI_EXP_TXN(7 downto 0)

      --! Clock and reset for DMA 0 bus
      DMA0_CLK                <= pcie1_user_clk;
      DMA0_RESET              <= pcie1_user_reset_out;

      --! DMA UP interface (user application --> host system), PCIe core 0
      DMA0_UP_DST_RDY         <= '0';

      --! DMA DOWN interface (host system --> user application), PCIe core 0
      DMA0_DOWN_DATA          <= (others => '0');
      DMA0_DOWN_HDR           <= (others => '0');
      DMA0_DOWN_SOP           <= '0';
      DMA0_DOWN_EOP           <= '0';
      DMA0_DOWN_SRC_RDY       <= '0';

      --! \name Configuration status interface, both PCIe cores
      PCIE0_MAX_PAYLOAD       <= (others => '0');
      PCIE0_MAX_READ_REQ      <= (others => '0');

   end generate single_pcie_gen;


   -- -------------------------------------------------------------------------
   --      PCIe core 0 (generate when double PCIe cores are instantiated)
   -- -------------------------------------------------------------------------

   double_pcie_gen : if (DOUBLE_PCIE) generate

      --! PCIe core 0 instantiation
      -------------------------------------------------------------------------

      --! auxiliary signals for connection of some ports
      pcie0_rq_tready        <= pcie0_rq_tready_vector(0);
      pcie0_rc_tready_vector <= (others => pcie0_rc_tready);

      pcie0_i : pcie3_7x_v3_0
      generic map (
         VENDOR_ID                         => VENDOR_ID,
         DEVICE_ID                         => DEVICE0_ID,
         SUB_VENDOR_ID                     => SUB_VENDOR_ID,
         SUB_DEVICE_ID                     => SUB_DEVICE0_ID,
         PCIE_CORE                         => 0
      )
      port map (
         pci_exp_txn                       => PCI_EXP_TXN(7 downto 0),
         pci_exp_txp                       => PCI_EXP_TXP(7 downto 0),
         pci_exp_rxn                       => PCI_EXP_RXN(7 downto 0),
         pci_exp_rxp                       => PCI_EXP_RXP(7 downto 0),
         user_clk                          => pcie0_user_clk,
         user_reset                        => pcie0_user_reset,
         user_lnk_up                       => pcie0_user_lnk_up,
         user_app_rdy                      => open,
         s_axis_rq_tlast                   => pcie0_rq_tlast,
         s_axis_rq_tdata                   => pcie0_rq_tdata,
         s_axis_rq_tuser                   => pcie0_rq_tuser,
         s_axis_rq_tkeep                   => pcie0_rq_tkeep,
         s_axis_rq_tready                  => pcie0_rq_tready_vector,
         s_axis_rq_tvalid                  => pcie0_rq_tvalid,
         m_axis_rc_tdata                   => pcie0_rc_tdata,
         m_axis_rc_tuser                   => pcie0_rc_tuser,
         m_axis_rc_tlast                   => pcie0_rc_tlast,
         m_axis_rc_tkeep                   => pcie0_rc_tkeep,
         m_axis_rc_tvalid                  => pcie0_rc_tvalid,
         m_axis_rc_tready                  => pcie0_rc_tready_vector,
         m_axis_cq_tdata                   => open,
         m_axis_cq_tuser                   => open,
         m_axis_cq_tlast                   => open,
         m_axis_cq_tkeep                   => open,
         m_axis_cq_tvalid                  => open,
         m_axis_cq_tready                  => "0000000000000000000000",
         s_axis_cc_tdata                   => X"0000000000000000000000000000000000000000000000000000000000000000",
         s_axis_cc_tuser                   => "000000000000000000000000000000000",
         s_axis_cc_tlast                   => '0',
         s_axis_cc_tkeep                   => "00000000",
         s_axis_cc_tvalid                  => '0',
         s_axis_cc_tready                  => open,
         pcie_rq_seq_num                   => pcie0_rq_seq_num,
         pcie_rq_seq_num_vld               => pcie0_rq_seq_num_vld,
         pcie_rq_tag                       => pcie0_rq_tag,
         pcie_rq_tag_vld                   => pcie0_rq_tag_vld,
         pcie_cq_np_req                    => '0',
         pcie_cq_np_req_count              => open,
         cfg_phy_link_down                 => open,
         cfg_phy_link_status               => open,
         cfg_negotiated_width              => open,
         cfg_current_speed                 => open,
         cfg_max_payload                   => PCIE0_MAX_PAYLOAD,
         cfg_max_read_req                  => PCIE0_MAX_READ_REQ,
         cfg_function_status               => open,
         cfg_function_power_state          => open,
         cfg_vf_status                     => open,
         cfg_vf_power_state                => open,
         cfg_link_power_state              => open,
         cfg_err_cor_out                   => open,
         cfg_err_nonfatal_out              => open,
         cfg_err_fatal_out                 => open,
         cfg_ltr_enable                    => open,
         cfg_ltssm_state                   => open,
         cfg_rcb_status                    => open,
         cfg_dpa_substate_change           => open,
         cfg_obff_enable                   => open,
         cfg_pl_status_change              => open,
         cfg_tph_requester_enable          => open,
         cfg_tph_st_mode                   => open,
         cfg_vf_tph_requester_enable       => open,
         cfg_vf_tph_st_mode                => open,
         cfg_interrupt_int                 => "0000",
         cfg_interrupt_pending             => "00",
         cfg_interrupt_sent                => open,
         cfg_interrupt_msi_enable          => open,
         cfg_interrupt_msi_vf_enable       => open,
         cfg_interrupt_msi_mmenable        => open,
         cfg_interrupt_msi_mask_update     => open,
         cfg_interrupt_msi_data            => open,
         cfg_interrupt_msi_select          => "0000",
         cfg_interrupt_msi_int             => "00000000000000000000000000000000",
         cfg_interrupt_msi_pending_status  => X"0000000000000000",
         cfg_interrupt_msi_sent            => open,
         cfg_interrupt_msi_fail            => open,
         cfg_interrupt_msi_attr            => "000",
         cfg_interrupt_msi_tph_present     => '0',
         cfg_interrupt_msi_tph_type        => "00",
         cfg_interrupt_msi_tph_st_tag      => "000000000",
         cfg_interrupt_msi_function_number => "000",
         sys_clk                           => PCIE0_SYS_CLK,
         sys_reset                         => PCIE_SYS_RESET
      );

      --! Reset logic
      -- ----------------------------------------------------------------------

      pcie0_reset_logic_p : process(pcie0_user_clk)
      begin
         if ((pcie0_user_clk'event) AND (pcie0_user_clk = '1')) then
            if ((pcie0_user_reset = '1') OR
                (pcie0_user_lnk_up = '0')) then
               pcie0_user_reset_shifter <= (others => '1');
            else
               pcie0_user_reset_shifter <= pcie0_user_reset_shifter(pcie0_user_reset_shifter'high-1 downto 0) & '0';
            end if;
            pcie0_user_reset_out <= pcie0_user_reset_shifter(pcie0_user_reset_shifter'high);
         end if;
      end process pcie0_reset_logic_p;

      --! AXI2DMA instantiation
      -- ----------------------------------------------------------------------

      pcie0_axi2dma_i : entity work.axi2dma
      port map (
         --! Common signals
         CLK              => pcie0_user_clk,
         RESET            => pcie0_user_reset_out,

         --! Requester Request Interface (RQ)
         RQ_DATA          => pcie0_rq_tdata,
         RQ_USER          => pcie0_rq_tuser,
         RQ_LAST          => pcie0_rq_tlast,
         RQ_KEEP          => pcie0_rq_tkeep,
         RQ_READY         => pcie0_rq_tready,
         RQ_VALID         => pcie0_rq_tvalid,
         RQ_SEQ_NUM       => pcie0_rq_seq_num,
         RQ_SEQ_NUM_VLD   => pcie0_rq_seq_num_vld,
         RQ_TAG           => pcie0_rq_tag,
         RQ_TAG_VLD       => pcie0_rq_tag_vld,

         --! Requester Completion Interface (RC)
         RC_DATA          => pcie0_rc_tdata,
         RC_USER          => pcie0_rc_tuser,
         RC_LAST          => pcie0_rc_tlast,
         RC_KEEP          => pcie0_rc_tkeep,
         RC_VALID         => pcie0_rc_tvalid,
         RC_READY         => pcie0_rc_tready,

         --! DMA UP Interface (from user application to host system)
         DMA_UP_DATA      => DMA0_UP_DATA,
         DMA_UP_HDR       => DMA0_UP_HDR,
         DMA_UP_SOP       => DMA0_UP_SOP,
         DMA_UP_EOP       => DMA0_UP_EOP,
         DMA_UP_SRC_RDY   => DMA0_UP_SRC_RDY,
         DMA_UP_DST_RDY   => DMA0_UP_DST_RDY,

         --! DMA DOWN Interface (from host system to user application)
         DMA_DOWN_DATA    => DMA0_DOWN_DATA,
         DMA_DOWN_HDR     => DMA0_DOWN_HDR,
         DMA_DOWN_SOP     => DMA0_DOWN_SOP,
         DMA_DOWN_EOP     => DMA0_DOWN_EOP,
         DMA_DOWN_SRC_RDY => DMA0_DOWN_SRC_RDY,
         DMA_DOWN_DST_RDY => DMA0_DOWN_DST_RDY
      );

      --! Output ports mapping
      -- ----------------------------------------------------------------------

      --! Clock and reset for DMA 0 bus
      DMA0_CLK   <= pcie0_user_clk;
      DMA0_RESET <= pcie0_user_reset_out;

   end generate double_pcie_gen;


   -- -------------------------------------------------------------------------
   --                          Output ports mapping
   -- -------------------------------------------------------------------------

   --! User clock and reset interface
   USER_CLK   <= pcie1_user_clk;
   USER_RESET <= pcie1_user_reset_out;


end architecture full;
