-- dma_switch.vhd: DMA Switch
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

-- -----------------------------------------------------------------------------
--                              Entity declaration
-- -----------------------------------------------------------------------------
entity DMA_SWITCH is
   generic (
      DATA_WIDTH        : integer := 512;
      HDR_WIDTH         : integer := 32
   );
   port (
      CLK               : in std_logic;
      RESET             : in std_logic;

      DMA_TX_DATA       : out std_logic_vector(DATA_WIDTH-1 downto 0);
      DMA_TX_HDR        : out std_logic_vector(HDR_WIDTH-1 downto 0);
      DMA_TX_SOP        : out std_logic;
      DMA_TX_EOP        : out std_logic;
      DMA_TX_SRC_RDY    : out std_logic;
      DMA_TX_DST_RDY    : in  std_logic;

      DMA_RX0_DATA      : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      DMA_RX0_HDR       : in  std_logic_vector(HDR_WIDTH-1 downto 0);
      DMA_RX0_SOP       : in  std_logic;
      DMA_RX0_EOP       : in  std_logic;
      DMA_RX0_SRC_RDY   : in  std_logic;
      DMA_RX0_DST_RDY   : out std_logic;

      DMA_RX1_DATA      : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      DMA_RX1_HDR       : in  std_logic_vector(HDR_WIDTH-1 downto 0);
      DMA_RX1_SOP       : in  std_logic;
      DMA_RX1_EOP       : in  std_logic;
      DMA_RX1_SRC_RDY   : in  std_logic;
      DMA_RX1_DST_RDY   : out std_logic
   );
end entity DMA_SWITCH;

-- -----------------------------------------------------------------------------
--                           Architecture declaration
-- -----------------------------------------------------------------------------
architecture behavioral of DMA_SWITCH is


   type t_state is (S_INIT0, S_INIT1, S_RX0, S_RX1);
   signal present_state, next_state : t_state;

begin

   -- --------------- Sync logic -------------------------------------------
   sync_logic : process(CLK, RESET, next_state)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state  <= S_INIT0;
         else
            present_state  <= next_state;
         end if;
      end if;
   end process sync_logic;

   -- ------------------ Next state logic -------------------------------------
   next_state_logic : process(DMA_RX0_SRC_RDY, DMA_RX1_SRC_RDY, DMA_TX_DST_RDY, DMA_RX0_EOP, DMA_RX1_EOP, present_state)
   begin
      next_state <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT0 =>
            if(DMA_RX0_SRC_RDY = '1') then
               next_state <= S_RX0;

               if(DMA_TX_DST_RDY = '1' and DMA_RX0_EOP = '1') then
                  next_state <= S_INIT1;
               end if;
            elsif(DMA_RX1_SRC_RDY = '1') then
               next_state <= S_RX1;

               if(DMA_TX_DST_RDY = '1' and DMA_RX1_EOP = '1') then
                  next_state <= S_INIT0;
               end if;
            end if;
         -- ---------------------------------------------
         when S_INIT1 =>
            if(DMA_RX1_SRC_RDY = '1') then
               next_state <= S_RX1;

               if(DMA_TX_DST_RDY = '1' and DMA_RX1_EOP = '1') then
                  next_state <= S_INIT0;
               end if;

            elsif(DMA_RX0_SRC_RDY = '1') then
               next_state <= S_RX0;

               if(DMA_TX_DST_RDY = '1' and DMA_RX0_EOP = '1') then
                  next_state <= S_INIT1;
               end if;
            end if;
         -- ---------------------------------------------
         when S_RX0 =>
            if(DMA_RX0_SRC_RDY = '1' and DMA_TX_DST_RDY = '1' and DMA_RX0_EOP = '1') then
               next_state  <= S_INIT1;
            end if;
         -- ---------------------------------------------
         when S_RX1 =>
            if(DMA_RX1_SRC_RDY = '1' and DMA_TX_DST_RDY = '1' and DMA_RX1_EOP = '1') then
               next_state  <= S_INIT0;
            end if;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic -----------------------------------------
   output_logic: process(present_state, DMA_RX0_DATA, DMA_RX0_HDR, DMA_RX0_SOP, DMA_RX0_EOP, DMA_RX1_DATA, DMA_RX1_HDR, DMA_RX1_SOP, DMA_RX1_EOP, DMA_RX0_SRC_RDY, DMA_RX1_SRC_RDY, DMA_TX_DST_RDY)
   begin

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT0 =>
            DMA_TX_SRC_RDY       <= DMA_RX0_SRC_RDY or DMA_RX1_SRC_RDY;
            DMA_RX0_DST_RDY      <= DMA_TX_DST_RDY;
            DMA_RX1_DST_RDY      <= DMA_TX_DST_RDY and not DMA_RX0_SRC_RDY;

            if(DMA_RX0_SRC_RDY = '1') then
               DMA_TX_DATA       <= DMA_RX0_DATA;
               DMA_TX_HDR        <= DMA_RX0_HDR;
               DMA_TX_SOP        <= DMA_RX0_SOP;
               DMA_TX_EOP        <= DMA_RX0_EOP;
            else
               DMA_TX_DATA       <= DMA_RX1_DATA;
               DMA_TX_HDR        <= DMA_RX1_HDR;
               DMA_TX_SOP        <= DMA_RX1_SOP;
               DMA_TX_EOP        <= DMA_RX1_EOP;
            end if;
         -- ---------------------------------------------
         when S_INIT1 =>
            DMA_TX_SRC_RDY       <= DMA_RX1_SRC_RDY or DMA_RX0_SRC_RDY;
            DMA_RX0_DST_RDY      <= DMA_TX_DST_RDY and not DMA_RX1_SRC_RDY;
            DMA_RX1_DST_RDY      <= DMA_TX_DST_RDY;

            if(DMA_RX1_SRC_RDY = '1') then
               DMA_TX_DATA       <= DMA_RX1_DATA;
               DMA_TX_HDR        <= DMA_RX1_HDR;
               DMA_TX_SOP        <= DMA_RX1_SOP;
               DMA_TX_EOP        <= DMA_RX1_EOP;
            else
               DMA_TX_DATA       <= DMA_RX0_DATA;
               DMA_TX_HDR        <= DMA_RX0_HDR;
               DMA_TX_SOP        <= DMA_RX0_SOP;
               DMA_TX_EOP        <= DMA_RX0_EOP;
            end if;
         -- ---------------------------------------------
         when S_RX0 =>
            DMA_TX_SRC_RDY       <= DMA_RX0_SRC_RDY;
            DMA_RX0_DST_RDY      <= DMA_TX_DST_RDY;
            DMA_RX1_DST_RDY      <= '0';

            DMA_TX_DATA          <= DMA_RX0_DATA;
            DMA_TX_HDR           <= DMA_RX0_HDR;
            DMA_TX_SOP           <= DMA_RX0_SOP;
            DMA_TX_EOP           <= DMA_RX0_EOP;
         -- ---------------------------------------------
         when S_RX1 =>
            DMA_TX_SRC_RDY       <= DMA_RX1_SRC_RDY;
            DMA_RX1_DST_RDY      <= DMA_TX_DST_RDY;
            DMA_RX0_DST_RDY      <= '0';

            DMA_TX_DATA          <= DMA_RX1_DATA;
            DMA_TX_HDR           <= DMA_RX1_HDR;
            DMA_TX_SOP           <= DMA_RX1_SOP;
            DMA_TX_EOP           <= DMA_RX1_EOP;
         -- ---------------------------------------------
      end case;
   end process;

end architecture;
