-- dma_desc_manager.vhd: Descriptor manager for DMA controllers
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_desc_manager is
   generic (
      --! Channel count, needs to be a power of 2
      CHANNELS          : integer := 8;
      DATA_ALIGN        : integer := 8;
      DMA_ID            : integer := 8
   );
   port (
      --! Common signals
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Enable descriptor download per channel
      ENABLE            : in  std_logic_vector(CHANNELS-1 downto 0);
      --! Base adress of memory area, which contains initial descriptors
      CONFIG_ADDR       : in  std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

      --! Update descriptor, or get next
      --* Do not activate, when RDY is not 1
      DESC_UPDATE       : in  std_logic;
      --! Data size - increment by
      DESC_UPDATE_LEN   : in  std_logic_vector(PAGE_WIDTH downto 0);
      --! Channel for operation
      DESC_CHANNEL      : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      --! Descriptor for current channel is ready
      DESC_RDY          : out std_logic;
      --! Memory descriptor
      DESC_DATA         : out std_logic_vector(DMA_DESC_WIDTH-1 downto 0);

      --! Interface for generating DMA requests
      DMA_UP_DATA       : out std_logic_vector(511 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! Interface for receiving completed requests
      DMA_DOWN_DATA     : in  std_logic_vector(511 downto 0);
      DMA_DOWN_HDR      : in  std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
      DMA_DOWN_SOP      : in  std_logic;
      DMA_DOWN_EOP      : in  std_logic;
      DMA_DOWN_SRC_RDY  : in  std_logic;
      DMA_DOWN_DST_RDY  : out std_logic
   );
end;

architecture behavioral of dma_desc_manager is

   type t_state is (S_INIT, S_STOP, S_RUN, S_NEXT, S_WAIT);
   signal present_state, next_state : t_state;

   type   t_ptr                     is array(CHANNELS-1 downto 0) of std_logic_vector(63 downto 0);
   signal reg_address               : t_ptr;

   signal init                      : std_logic_vector(CHANNELS-1 downto 0);
   signal reg_write                 : std_logic;

   signal fifo_write                : std_logic;
   signal fifo_read                 : std_logic;
   signal fifo_full                 : std_logic_vector(CHANNELS-1 downto 0);
   signal fifo_empty                : std_logic_vector(CHANNELS-1 downto 0);
   signal fifo_data                 : std_logic_vector(DMA_DESC_WIDTH-1 downto 0);

   signal reg_channel               : std_logic_vector(log2(CHANNELS)-1 downto 0) := (others => '0');
   signal reg_update_channel        : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal update_channel            : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal iReg_channel              : integer := 0;
   signal iDESC_CHANNEL             : integer := 0;
   signal iUPDATE_CHANNEL           : integer := 0;

   signal page_full                 : std_logic;
   signal reg_update                : std_logic := '0';
   signal reg_update_len            : std_logic_vector(PAGE_WIDTH downto 0);

   signal desc_reg_we               : std_logic;
   signal desc_reg_rdy              : std_logic_vector(CHANNELS-1 downto 0) := (others => '0');
   signal desc_reg                  : std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
   signal desc_reg_w                : std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
   signal reg_desc_reg              : std_logic_vector(PAGE_WIDTH-1 downto log2(DATA_ALIGN));

   signal desc_count_we             : std_logic;
   signal desc_count_w              : std_logic_vector(PAGE_WIDTH-2 downto 0);
   signal desc_count_sub_sel        : std_logic_vector(PAGE_WIDTH-2 downto 0);
   signal desc_count                : std_logic_vector(PAGE_WIDTH-2 downto 0);

   signal zeros                     : std_logic_vector(511 downto 0) := (others => '0');

begin

   -- ---------------------------------------------------------
   -- Descriptor manager UPDATE process
   --

   -- In second cycle of UPDATE is  a) current descriptor updated and its validity checked (desc_reg_rdy)   OR
   --                               b) new descriptor readen from FIFO, writen to REG and sent to DESC_DATA output

   -- INFO: RX and TX DMA Ctrl holds same channel one cycle after DESC_UPDATE (state S_NEXT after S_DMA_REQ)
   -- Thus, no need to latch and mux channel number to update.
   --update_channel    <= reg_update_channel when reg_update = '1' else DESC_CHANNEL;
   update_channel    <= DESC_CHANNEL;
   iDESC_CHANNEL     <= conv_integer(DESC_CHANNEL);
   iUPDATE_CHANNEL   <= conv_integer(update_channel);

   -- Descriptor manager can't work one cycle after DESC_UPDATE (=> not reg_update)
   -- Descriptor is ready to update when is valid value in FIFO or in REG
   DESC_RDY          <= ((not fifo_empty(iDESC_CHANNEL)) or desc_reg_rdy(iDESC_CHANNEL)) and not reg_update;

   -- Descriptor mux - switch between data from FIFO and REG
   DESC_DATA         <= desc_reg when desc_reg_rdy(iDESC_CHANNEL) = '1' else fifo_data(DMA_DESC_WIDTH-1 downto PAGE_WIDTH) & zeros(PAGE_WIDTH-1 downto 0);

   -- Read from FIFO, when descriptor is not ready (because init or full page)
   fifo_read         <= '1' when reg_update = '1' and desc_reg_rdy(conv_integer(DESC_CHANNEL)) = '0' else '0';
   -- Page full flag
   page_full         <= '1' when reg_update = '1' and (reg_desc_reg(PAGE_WIDTH-1 downto log2(DATA_ALIGN)) + reg_update_len(PAGE_WIDTH-1 downto log2(DATA_ALIGN))) = std_logic_vector(to_unsigned(0, 11-log2(DATA_ALIGN))) else '0';

   -- New descriptor write mux - select between desc from FIFO and updated REG
   desc_reg_we       <= reg_update;

   -- INFO: The upper 32b of adddress are constant. Aggregation can't be used for aggregating addresses, which crosssing 4GB boundaries (changing bits >= 32).
   desc_reg_w        <= fifo_data(DMA_DESC_WIDTH-1 downto 32) & (fifo_data(31 downto PAGE_WIDTH) & zeros(PAGE_WIDTH-1 downto 0) + reg_update_len) when fifo_read = '1' else
                         desc_reg(DMA_DESC_WIDTH-1 downto 32) & (desc_reg(31 downto log2(DATA_ALIGN)) + reg_update_len(PAGE_WIDTH downto log2(DATA_ALIGN))) & zeros(log2(DATA_ALIGN)-1 downto 0);

   -- The old implementation causes critical paths:
--   desc_reg_w        <= fifo_data(DMA_DESC_WIDTH-1 downto PAGE_WIDTH) & zeros(PAGE_WIDTH-1 downto 0) + reg_update_len when fifo_read = '1' else
--                        (desc_reg(DMA_DESC_WIDTH-1 downto log2(DATA_ALIGN)) + reg_update_len(PAGE_WIDTH downto log2(DATA_ALIGN))) & zeros(log2(DATA_ALIGN)-1 downto 0);

   -- Write count of adjacent pages
   desc_count_we     <= fifo_read or page_full;
   desc_count_sub_sel<= desc_count when fifo_read = '0' else fifo_data(PAGE_WIDTH-1 downto 1);
   desc_count_w      <= desc_count_sub_sel - 1 when fifo_read = '0' or reg_update_len = PAGE_SIZE else fifo_data(PAGE_WIDTH-1 downto 1);

   reg_update_p : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         reg_update        <= DESC_UPDATE;
         reg_update_channel<= DESC_CHANNEL;
         reg_update_len    <= DESC_UPDATE_LEN;
         reg_desc_reg      <= desc_reg(PAGE_WIDTH-1 downto log2(DATA_ALIGN));
      end if;
   end process;

   desc_reg_rdy_p : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(init(iUPDATE_CHANNEL) = '1') then
            desc_reg_rdy(iUPDATE_CHANNEL) <= '0';
         elsif(fifo_read = '1') then
            desc_reg_rdy(iUPDATE_CHANNEL) <= '1';
         elsif(page_full = '1' and conv_integer(desc_count) = 0) then
            desc_reg_rdy(iUPDATE_CHANNEL) <= '0';
         end if;
      end if;
   end process;

   -- ----------------------------------------------------------------------
   -- Write processes - Descriptor Download

   init                             <= not ENABLE;
   iReg_channel                     <= conv_integer(reg_channel);

   DMA_UP_SOP                       <= '1';
   DMA_UP_EOP                       <= '1';
   DMA_UP_DATA                      <= (others => '0');

   DMA_UP_HDR(DMA_REQUEST_GLOBAL)   <= reg_address(iReg_channel);
   DMA_UP_HDR(DMA_REQUEST_LENGTH)   <= "00000000010";
   DMA_UP_HDR(DMA_REQUEST_TYPE)     <= DMA_TYPE_READ;
   DMA_UP_HDR(DMA_REQUEST_TAG)      <= "00000000";
   DMA_UP_HDR(DMA_REQUEST_UNITID)   <= conv_std_logic_vector(DMA_ID, 8);

   DMA_DOWN_DST_RDY                 <= '1';

   -- When is data descriptor pointer, write value in next address register
   reg_write                        <= DMA_DOWN_SRC_RDY and     DMA_DOWN_DATA(0);
   -- When is data direct descriptor, write value in fifo
   fifo_write                       <= DMA_DOWN_SRC_RDY and not DMA_DOWN_DATA(0);

   -- Addresses for next desciptor download
   gen_reg_address : for i in 0 to CHANNELS-1 generate
      reg_address_p : process(CLK)
      begin
         if(CLK'event AND CLK = '1') then
            if(present_state = S_NEXT) then
               reg_channel    <= reg_channel + 1;
            end if;

            if(ENABLE(i) = '0') then
               reg_address(i) <= CONFIG_ADDR + i*8;
            elsif(reg_write = '1' and i = iReg_channel) then
               reg_address(i) <= DMA_DOWN_DATA(DMA_DESC_WIDTH-1 downto 1) & '0';
            elsif(fifo_write = '1' and i = iReg_channel) then
               reg_address(i) <= reg_address(i) + 8;
            end if;
         end if;
      end process;
   end generate;

   -- --------------- Sync logic -------------------------------------------
   sync_logic : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(RESET = '1') then
            present_state  <= S_INIT;
         else
            present_state  <= next_state;
         end if;
      end if;
   end process sync_logic;

   -- ------------------ Next state logic ----------------------------------
   next_state_logic : process(present_state, iReg_channel, enable, fifo_full, DMA_UP_DST_RDY, DMA_DOWN_SRC_RDY)
   begin
      next_state <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
            if(ENABLE(iReg_channel) = '1' and fifo_full(iReg_channel) = '0') then
               next_state  <= S_RUN;
            else
               next_state  <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when S_RUN =>
            if(DMA_UP_DST_RDY = '1') then
               next_state  <= S_WAIT;
            end if;
         -- ---------------------------------------------
         when S_WAIT =>
            if(DMA_DOWN_SRC_RDY = '1') then
               next_state  <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when others =>
            next_state     <= S_INIT;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic --------------------------------------
   output_logic : process(present_state)
   begin
      DMA_UP_SRC_RDY       <= '0';

      case (present_state) is
         -- ---------------------------------------------
         when S_RUN =>
            DMA_UP_SRC_RDY <= '1';
         -- ---------------------------------------------
         when others =>
      end case;
   end process;

   -- ----------------------------------------------------------------------
   -- FIFO, REG and desc_count memory

   nfifo_i : entity work.NFIFO
   generic map(
      DATA_WIDTH     => DMA_DESC_WIDTH,
      FLOWS          => CHANNELS,
      BLOCK_SIZE     => 4,
      LUT_MEMORY     => true,
      OUTPUT_REG     => false
   )
   port map(
      CLK            => CLK,
      RESET          => RESET,
      INIT           => init,

      DATA_IN        => DMA_DOWN_DATA(DMA_DESC_WIDTH-1 downto 0),
      WR_BLK_ADDR    => reg_channel,
      WRITE          => fifo_write,
      FULL           => fifo_full,

      DATA_OUT       => fifo_data,
      DATA_VLD       => open,
      RD_BLK_ADDR    => update_channel,
      READ           => fifo_read,
      PIPE_EN        => '1',
      EMPTY          => fifo_empty,
      STATUS         => open
   );

   desc_count_i : entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => PAGE_WIDTH-1,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => desc_count_w,
      WE             => desc_count_we,
      ADDR           => update_channel,
      DO             => desc_count
   );

   desc_reg_i : entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => DMA_DESC_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => desc_reg_w,
      WE             => desc_reg_we,
      ADDR           => update_channel,
      DO             => desc_reg
   );

end architecture;
