-- dma_ctrl_rxtx_newdata.vhd: New Data Process of TX DMA Controller
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- TODO: Optimize next channel logic, fix interrupts

library ieee;
library work;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_ctrl_tx_newdata is
   generic (
      CHANNELS          : integer;
      BUFFER_SIZE       : integer;
      DATA_ALIGN        : integer
   );
  -- interface description
   port (
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Interface from DMA Buffer
      PACKET_SENT       : in  std_logic;
      PACKET_CHANNEL    : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      PACKET_LENGTH     : in  std_logic_vector(log2(BUFFER_SIZE) downto 0);

      --! Global signals of NewData process
      CHANNEL           : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      DMA_INTERRUPT     : out std_logic;
      SW_STATUS         : out std_logic_vector(7 downto 0);
      --! Interface for monitoring changes
      SW_CONTROL        : in  std_logic_vector(1 downto 0);
      SW_CHANNEL        : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      SW_CONTROL_WR     : in  std_logic;
      SW_PTR_WR         : in  std_logic;

      --! Controller state
      ENABLE            : out std_logic_vector(CHANNELS-1 downto 0);

      --! Interface from Descriptor Manager
      DESC_DATA         : in  std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
      DESC_RDY          : in  std_logic;

      --! Interface to Request unit
      DMA_RDY           : in  std_logic;
      DMA_REQ           : out std_logic;
      DMA_LENGTH        : out std_logic_vector(PAGE_WIDTH - log2(DATA_ALIGN) downto 0);
      DMA_CHANNEL       : out std_logic_vector(log2(CHANNELS) - 1 downto 0);
      DMA_HWADDRESS     : out std_logic_vector(log2(BUFFER_SIZE)-1 downto 0);
      DMA_POINTER       : out std_logic_vector(31 downto 0);

      MAXREQUEST        : in  std_logic_vector(PAGE_WIDTH downto 0);
      INTERRUPT_EN      : in  std_logic;
      INTERRUPT_RST     : out std_logic;
      INTERRUPT         : in  std_logic_vector(31 downto 0);
      SWBUFFER_MASK     : in  std_logic_vector(31 downto 0);
      SWENDPTR          : in  std_logic_vector(31 downto 0)
    );
end;

architecture behavioral of dma_ctrl_tx_newdata is

   type t_state is (S_INIT, S_START, S_STOP, S_RUN, S_COMP, S_DMA_REQ, S_NEXT, S_RESET);
   signal present_state, next_state, prev_state : t_state;

   constant HWPTR_WIDTH       : integer := log2(BUFFER_SIZE) + 1;
   constant SWPTR_WIDTH       : integer := 32;

   constant CR_RUN            : integer := 0;
   constant CR_DISCARD        : integer := 1;
   constant zeros             : std_logic_vector(127 downto 0) := (others => '0');

   --! General signals
   signal enable_set          : std_logic;
   signal enable_rst          : std_logic;

   signal reg_enable          : std_logic_vector(CHANNELS-1 downto 0) := (others => '0');

   signal reg_channel         : std_logic_vector(log2(CHANNELS)-1 downto 0) := (others => '0');
   signal iReg_channel        : integer := 0;

   signal dma_exec            : std_logic;

   --! Next channel signals
   signal controlFlagRst      : std_logic;
   signal reg_freespaceFlag   : std_logic_vector(CHANNELS-1 downto 0) := (others => '1');
   signal reg_controlFlag     : std_logic_vector(CHANNELS-1 downto 0) := (others => '0');
   signal channel_enable      : std_logic_vector(CHANNELS-1 downto 0);
   signal channel_enable_rotated: std_logic_vector(CHANNELS-1 downto 0);
   signal next_channel_out    : std_logic_vector(log2(CHANNELS)-1 downto 0);

   signal hwStrPtr            : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwStrPtr_R          : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal hwEndPtr            : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtr_w          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtr_we         : std_logic;

   signal hwStrPtr_w          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwStrPtr_ch         : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal hwStrPtr_we         : std_logic;

   --! SW pointers
   signal swStrPtr_w          : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal swStrPtr_we         : std_logic;
   signal reg_swStrPtr        : std_logic_vector(SWPTR_WIDTH-1 downto 0);

   signal swBuffer_data       : std_logic;
   signal swDataspace         : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_swDataspace     : std_logic_vector(SWPTR_WIDTH-1 downto 0);

   --! Control signals
   signal reg_hwDiff          : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal dataLength          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal reg_dataLength      : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal transLength         : std_logic_vector(PAGE_WIDTH downto 0);
   signal reg_transLength     : std_logic_vector(PAGE_WIDTH downto 0);

   signal maxLength           : std_logic_vector(PAGE_WIDTH downto 0);
   signal reg_maxLength       : std_logic_vector(PAGE_WIDTH downto 0);

   signal pageSpace           : std_logic_vector(PAGE_WIDTH downto 0);

   signal reg_interrupt       : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_interrupt_w     : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_interrupt_we    : std_logic;
   signal hwEmpty             : std_logic;

begin

   iReg_channel      <= conv_integer(reg_channel);
   pageSpace         <= PAGE_SIZE - ('0' & DESC_DATA(PAGE_WIDTH-1 downto 0));
   maxLength         <= MAXREQUEST and not (conv_std_logic_vector(DATA_ALIGN - 1, PAGE_WIDTH+1)) when MAXREQUEST > DATA_ALIGN else conv_std_logic_vector(DATA_ALIGN, PAGE_WIDTH+1);
   dataLength        <= BUFFER_SIZE - (reg_hwDiff);
   transLength       <= reg_maxLength when reg_dataLength > reg_maxLength else reg_dataLength(PAGE_WIDTH downto 0);
   swDataSpace       <= (swEndPtr - reg_swStrPtr) and SWBUFFER_MASK;

   hwEmpty           <= '1' when hwEndPtr = hwStrPtr and reg_freespaceFlag(iReg_channel) = '1' else '0';
   swbuffer_data     <= '1' when swEndPtr /= reg_swStrPtr else '0';

   reg_interrupt_we  <= '1' when (dma_exec ='1' and reg_interrupt(0) = '1') or (INTERRUPT_EN = '1' and present_state = S_RUN) else '0';
   reg_interrupt_w   <= INTERRUPT when reg_interrupt(0) = '1' and (reg_interrupt(SWPTR_WIDTH-1 downto 1) & '0') <= reg_transLength else ((reg_interrupt(SWPTR_WIDTH-1 downto 1) & '0') - reg_transLength) and X"FFFFFFFE";

   CHANNEL           <= reg_channel;
   ENABLE            <= reg_enable;
   DMA_INTERRUPT     <= dma_exec and INTERRUPT(0);
   INTERRUPT_RST     <= INTERRUPT(0) and not reg_interrupt(0) and dma_exec;
   DMA_POINTER       <= swStrPtr_w;
   SW_STATUS         <= "00" & DMA_RDY & hwEmpty & swBuffer_data & reg_freespaceFlag(iReg_channel) & DESC_RDY & reg_enable(iReg_channel);

   DMA_REQ           <= dma_exec;
   DMA_CHANNEL       <= reg_channel;
   DMA_LENGTH        <= reg_transLength(PAGE_WIDTH downto log2(DATA_ALIGN));
   DMA_HWADDRESS     <= hwEndPtr(HWPTR_WIDTH-2 downto 0);

   -- --------------- Sync logic -------------------------------------------
   sync_logic : process(CLK, RESET, next_state)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state  <= S_RESET;
         else
            prev_state     <= present_state;
            present_state  <= next_state;
         end if;
      end if;
   end process sync_logic;

   -- ------------------ Next state logic -------------------------------------
   next_state_logic : process(present_state, SW_CONTROL, reg_freespaceFlag, swbuffer_data, DESC_RDY, DMA_RDY, reg_channel, hwStrPtr, hwEndPtr, reg_enable, dataLength, iReg_channel, hwEmpty, packet_sent)
   begin
      next_state <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
            if (SW_CONTROL(CR_RUN) = '1' and reg_enable(iReg_channel) = '0') then
               next_state  <= S_START;
            elsif (SW_CONTROL(CR_RUN) = '0' and reg_enable(iReg_channel) = '1') then -- and hwEmpty = '1' and swbuffer_data = '0') then
               next_state  <= S_STOP;
            elsif (reg_enable(iReg_channel) = '1' and reg_freespaceFlag(ireg_channel) = '1' and swbuffer_data = '1' and DESC_RDY = '1') then
               next_state  <= S_RUN;
            else
               next_state  <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when S_START =>
            if(PACKET_SENT = '0') then
               next_state <= S_INIT;
            end if;
         -- ---------------------------------------------
         when S_STOP =>
            next_state     <= S_NEXT;
         -- ---------------------------------------------
         when S_RUN =>
            if(dataLength = 0) then
               next_state  <= S_NEXT;
            else
               next_state  <= S_COMP;
            end if;
         -- ---------------------------------------------
         when S_COMP =>
            if(DMA_RDY = '1') then
               next_state  <= S_DMA_REQ;
            end if;
         -- ---------------------------------------------
         when S_DMA_REQ =>
               next_state  <= S_NEXT;
         -- ---------------------------------------------
         when others =>
            next_state     <= S_INIT;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic -----------------------------------------
   output_logic: process(present_state, reg_swStrPtr, reg_transLength, SWBUFFER_MASK, PACKET_SENT, PACKET_CHANNEL, PACKET_LENGTH, hwStrPtr_R, dma_exec, hwEndPtr, reg_channel)
   begin
      controlFlagRst       <= '0';

      enable_rst           <= '0';
      enable_set           <= '0';

      hwEndPtr_w           <= hwEndPtr + (zeros(HWPTR_WIDTH-1 downto PAGE_WIDTH+1) & reg_transLength);
      hwEndPtr_we          <= '0';

      hwStrPtr_we          <= PACKET_SENT;
      hwStrPtr_ch          <= PACKET_CHANNEL;
      hwStrPtr_w           <= hwStrPtr_R + PACKET_LENGTH;

      swStrPtr_w           <= (reg_swStrPtr + reg_transLength) and SWBUFFER_MASK;
      swStrPtr_we          <= '0';

      dma_exec             <= '0';

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
         -- ---------------------------------------------
         when S_STOP =>
            controlFlagRst       <= '1';
            enable_rst           <= '1';
         -- ---------------------------------------------
         when S_START =>
            controlFlagRst       <= '1';
            if(PACKET_SENT = '0') then
               enable_set        <= '1';

               hwStrPtr_ch       <= reg_channel;

               hwEndPtr_w        <= (others => '0');
               hwStrPtr_w        <= (others => '0');
               swStrPtr_w        <= (others => '0');

               hwEndPtr_we       <= '1';
               hwStrPtr_we       <= '1';
               swStrPtr_we       <= '1';
            end if;
         -- ---------------------------------------------
         when S_DMA_REQ =>
            dma_exec             <= '1';

            hwEndPtr_we          <= '1';
            swStrPtr_we          <= '1';
         -- ---------------------------------------------
         when others =>
      end case;
   end process;

   -- ------------- FSM registers ------------------------------
   reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         reg_hwDiff        <= hwEndPtr - hwStrPtr;

         if(present_state = S_NEXT) then
            reg_channel       <= reg_channel + 1;
         end if;

         if(present_state = S_INIT) then
            reg_swDataspace   <= swDataSpace;

            if(pageSpace < maxLength) then
               reg_maxLength  <= pageSpace;
            else
               reg_maxLength  <= maxLength;
            end if;
         end if;

         if(present_state = S_RUN) then
            if(reg_swDataSpace < dataLength) then
               reg_dataLength    <= reg_swDataSpace(HWPTR_WIDTH-1 downto 0);
            else
               reg_dataLength    <= dataLength;
            end if;
         end if;

         if(present_state = S_COMP) then
            reg_transLength      <= transLength;
         end if;
      end if;
   end process;

   -- ----------- Next channel process ------------------------
   next_channel_fod : entity work.first_one_detector
   generic map (
      DATA_WIDTH        => CHANNELS
   )
   port map(
      MASK              => channel_enable_rotated,
      FIRST_ONE_BINARY  => next_channel_out
   );

   g: for i in 0 to CHANNELS-1 generate
      channel_enable_rotated(i)  <= channel_enable((i + iReg_channel + 1) mod CHANNELS);
      channel_enable(i)          <= reg_freespaceFlag(i) or reg_controlFlag(i);
   end generate;

   -- -------- Change flags for next channel select  -----------
   gen_freespaceFlag: for i in 0 to CHANNELS-1 generate
      reg_freespaceFlagp : process (CLK)
      begin
         if(CLK = '1' and CLK'event) then
            if((present_state = S_START and PACKET_SENT = '0' and ireg_channel = i) or (i = PACKET_CHANNEL and PACKET_SENT = '1') or present_state = S_RESET) then
               reg_freespaceFlag(i) <= '1';
            elsif(i = reg_channel and hwEndPtr(HWPTR_WIDTH-1) /= hwStrPtr(HWPTR_WIDTH-1) and hwEndPtr(HWPTR_WIDTH-2 downto 0) = hwStrPtr(HWPTR_WIDTH-2 downto 0) and prev_state = S_DMA_REQ) then
               reg_freespaceFlag(i) <= '0';
            end if;
         end if;

         if(CLK = '1' and CLK'event) then
            if(i = SW_CHANNEL and SW_CONTROL_WR = '1') then
               reg_controlFlag(i) <= '1';
            elsif((i = reg_channel and controlFlagRst = '1') or present_state = S_RESET) then
               reg_controlFlag(i) <= '0';
            end if;
         end if;
      end process;
   end generate;

   -- ---------- Enable / Discard reg process ------------------
   reg_enablep : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(present_state = S_RESET ) then
            reg_enable   <= (others => '0');
         elsif(enable_set = '1') then
            reg_enable(ireg_channel) <= '1';
         elsif(enable_rst = '1') then
            reg_enable(iReg_channel) <= '0';
         end if;
      end if;
   end process reg_enablep;

   -- ---------------   Registers   -----------------------------

   hwEndPtr_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => hwEndPtr_W,
      WE             => hwEndPtr_we,
      ADDR           => reg_channel,
      DO             => hwEndPtr
   );

   hwStrPtr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => hwStrPtr_W,
      WE             => hwStrPtr_we,
      ADDRA          => hwStrPtr_ch,
      DOA            => hwStrPtr_R,

      ADDRB          => reg_channel,
      DOB            => hwStrPtr
   );

   swStrPtr_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => SWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => swStrPtr_w,
      WE             => swStrPtr_we,
      ADDR           => reg_channel,
      DO             => reg_swStrPtr
   );

   interrupt_reg_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => SWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => reg_interrupt_w,
      WE             => reg_interrupt_we,
      ADDR           => reg_channel,
      DO             => reg_interrupt
   );

end architecture;
