-- dma_ctrl_rx.vhd: Generic RX DMA Controller - Top level
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library IEEE;
use IEEE.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.math_pack.all;

use work.dma_pkg.all;

entity dma_ctrl_rx is
   generic (
      --! Channel count, needs to be a power of 2
      CHANNELS          : integer := 8;
      --! Size of HW Buffer in bytes
      BUFFER_SIZE       : integer := 8192;
      --! Buffer count. It depends on data width and align
      BUFFERS           : integer := 8;
      --! Data align
      DATA_ALIGN        : integer := 8;
      --! DMA module is used in 2 PCIe configuration
      DOUBLE_PCIE       : boolean := false
   );
   port (
      --! Common signals
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Reflects state of an channel
      ENABLE            : out std_logic_vector(CHANNELS-1 downto 0);
      --! Enable packet receiving - for channel stopping
      ENABLE_BUFFER     : out std_logic_vector(CHANNELS-1 downto 0);
      --! Discarding is configurable for each channel - enabled / disabled
      DISCARD           : out std_logic_vector(CHANNELS-1 downto 0);
      --! Signal for generating an interrupt
      INTERRUPT         : out std_logic;

      --! RAM address for pointer update
      CONFIG_ADDR       : in  std_logic_vector(DMA_ADDR_WIDTH-1 downto 0);

      --! Interface from DMA Buffer - when new data arrives.
      PACKET_NEW        : in  std_logic;
      PACKET_DISC       : in  std_logic;
      PACKET_CHANNEL    : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      PACKET_LENGTH     : in  std_logic_vector(log2(BUFFER_SIZE) downto 0);

      --! Interface to DMA Buffer - when data needs to be transfered.
      BUFFER_READ       : out std_logic_vector(log2(BUFFERS) downto 0);
      BUFFER_CHANNEL    : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      BUFFER_DATA       : in  std_logic_vector(511 downto 0);
      BUFFER_DV         : in  std_logic;

      --! Interface from Descriptor manager
      DESC_DATA         : in  std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
      DESC_RDY          : in  std_logic;
      DESC_UPDATE       : out std_logic;
      DESC_UPDATE_LEN   : out std_logic_vector(PAGE_WIDTH downto 0);
      DESC_CHANNEL      : out std_logic_vector(log2(CHANNELS)-1 downto 0);

      --! Interface for generating DMA requests
      DMA_UP_DATA       : out std_logic_vector(511 downto 0);
      DMA_UP_HDR        : out std_logic_vector(DMA_UPHDR_WIDTH-1 downto 0);
      DMA_UP_SOP        : out std_logic;
      DMA_UP_EOP        : out std_logic;
      DMA_UP_SRC_RDY    : out std_logic;
      DMA_UP_DST_RDY    : in  std_logic;

      --! Interface for receiving completed requests
      DMA_DOWN_DATA     : in  std_logic_vector(511 downto 0);
      DMA_DOWN_HDR      : in  std_logic_vector(DMA_DOWNHDR_WIDTH-1 downto 0);
      DMA_DOWN_SOP      : in  std_logic;
      DMA_DOWN_EOP      : in  std_logic;
      DMA_DOWN_SRC_RDY  : in  std_logic;
      DMA_DOWN_DST_RDY  : out std_logic;

      --! SW Access for configuration
      MI_ADDR           : in  std_logic_vector(31 downto 0);
      MI_ARDY           : out std_logic;
      MI_BE             : in  std_logic_vector(3 downto 0);
      MI_DRD            : out std_logic_vector(31 downto 0);
      MI_DRDY           : out std_logic;
      MI_DWR            : in  std_logic_vector(31 downto 0);
      MI_RD             : in  std_logic;
      MI_WR             : in  std_logic
   );
end;

architecture behavioral of dma_ctrl_rx is

   constant zeros                   : std_logic_vector(63 downto 0) := (others => '0');
   constant SWPTR_WIDTH             : integer := 32;

   --! Pending FIFO constants
   constant FIFO_WIDTH              : integer := log2(CHANNELS) + SWPTR_WIDTH + 1 + 1;
   constant FIFO_COMPLETE           : integer := 0;
   constant FIFO_INTERRUPT          : integer := 1;
   subtype  FIFO_CHANNEL            is natural range log2(CHANNELS)+2-1 downto 2;
   subtype  FIFO_POINTER            is natural range log2(CHANNELS)+2+SWPTR_WIDTH-1 downto log2(CHANNELS)+2;


   --! New data process signals
   signal newdata_control           : std_logic_vector(1 downto 0);
   signal newdata_status            : std_logic_vector(7 downto 0);
   signal newdata_channel           : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal newdata_swstrptr          : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_swbufmask         : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_interrupt         : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal newdata_interrupt_en      : std_logic;
   signal newdata_maxreq            : std_logic_vector(PAGE_WIDTH downto 0);

   signal ctrl_enable               : std_logic_vector(CHANNELS-1 downto 0);
   signal ctrl_enable_buffer        : std_logic_vector(CHANNELS-1 downto 0);
   signal ctrl_discard              : std_logic_vector(CHANNELS-1 downto 0);

   signal reg_interrupt_en          : std_logic_vector(CHANNELS-1 downto 0);
   signal interrupt_rst             : std_logic;

   --! DMA Request done process signals
   signal dma_length                : std_logic_vector(PAGE_WIDTH - log2(DATA_ALIGN) downto 0);
   signal dma_channel               : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal dma_interrupt             : std_logic;
   signal dma_req                   : std_logic;
   signal dma_rdy                   : std_logic;
   signal dma_done                  : std_logic;
   signal dma_complete              : std_logic;
   signal dma_pointer               : std_logic_vector(SWPTR_WIDTH-1 downto 0);

   --! Update process signals
   signal dma_fifo_datain           : std_logic_vector(FIFO_WIDTH-1 downto 0);
   signal dma_fifo_dataout          : std_logic_vector(FIFO_WIDTH-1 downto 0);

   signal update_dma_complete       : std_logic;
   signal update_dma_donecomplete   : std_logic;
   signal update_dma_interrupt      : std_logic;
   signal update_dma_channel        : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal update_dma_pointer        : std_logic_vector(SWPTR_WIDTH-1 downto 0);

   signal update_timeout_channel    : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal update_timeout            : std_logic_vector(31 downto 0);

   signal update_interrupt          : std_logic;
   signal update_interrupt_channel  : std_logic_vector(log2(CHANNELS)-1 downto 0);

   --! Software - MI signals
   signal reg_control               : std_logic_vector(1 downto 0);
   signal reg_status                : std_logic_vector(7 downto 0);
   signal reg_swstrptr              : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_swendptr              : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_swbufmask             : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_interrupt             : std_logic_vector(SWPTR_WIDTH-1 downto 0);
   signal reg_timeout               : std_logic_vector(31 downto 0);
   signal reg_maxreq                : std_logic_vector(PAGE_WIDTH downto 0);
   signal sw_regsel                 : std_logic_vector(15 downto 0);
   signal sw_channel                : std_logic_vector(log2(CHANNELS)-1 downto 0);
   signal sw_interrupt_en           : std_logic;

   --! DMA Bus Signals
   signal dma_up0_data              : std_logic_vector(511 downto 0);
   signal dma_up0_hdr               : std_logic_vector(95 downto 0);
   signal dma_up0_sop               : std_logic;
   signal dma_up0_eop               : std_logic;
   signal dma_up0_src_rdy           : std_logic;
   signal dma_up0_dst_rdy           : std_logic;

   signal dma_up1_data              : std_logic_vector(511 downto 0);
   signal dma_up1_hdr               : std_logic_vector(95 downto 0);
   signal dma_up1_sop               : std_logic;
   signal dma_up1_eop               : std_logic;
   signal dma_up1_src_rdy           : std_logic;
   signal dma_up1_dst_rdy           : std_logic;

begin

   ENABLE                  <= ctrl_enable;
   ENABLE_BUFFER           <= ctrl_enable_buffer;
   DISCARD                 <= ctrl_discard;

   DESC_CHANNEL            <= newdata_channel;
   DESC_UPDATE             <= dma_req;
   DESC_UPDATE_LEN         <= dma_length & zeros(log2(DATA_ALIGN)-1 downto 0);

   INTERRUPT               <= update_interrupt and reg_interrupt_en(conv_integer(update_interrupt_channel));

   update_dma_donecomplete <= dma_done and update_dma_complete;
   newdata_interrupt_en    <= reg_interrupt_en(conv_integer(newdata_channel));

   newdata_i : entity work.dma_ctrl_rx_newdata
   generic map(
      CHANNELS          => CHANNELS,
      BUFFER_SIZE       => BUFFER_SIZE,
      DATA_ALIGN        => DATA_ALIGN
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      PACKET_NEW        => PACKET_NEW,
      PACKET_DISC       => PACKET_DISC,
      PACKET_CHANNEL    => PACKET_CHANNEL,
      PACKET_LENGTH     => PACKET_LENGTH,

      DMA_REQ           => dma_req,
      DMA_RDY           => dma_rdy,
      DMA_CHANNEL       => dma_channel,
      DMA_LENGTH        => dma_length,
      DMA_COMPLETE      => dma_complete,

      ENABLE            => ctrl_enable,
      ENABLE_BUFFER     => ctrl_enable_buffer,
      DISCARD           => ctrl_discard,

      SW_CHANNEL        => sw_channel,
      SW_CONTROL_WR     => sw_regsel(0),
      SW_PTR_WR         => sw_regsel(2),

      SW_CONTROL        => newdata_control,
      SW_STATUS         => newdata_status,
      CHANNEL           => newdata_channel,
      DESC_RDY          => DESC_RDY,
      DESC_DATA         => DESC_DATA,

      SWSTRPTR          => newdata_swstrptr,
      MAXREQUEST        => newdata_maxreq,
      INTERRUPT_EN      => newdata_interrupt_en,
      INTERRUPT         => newdata_interrupt,
      INTERRUPT_RST     => interrupt_rst,
      GEN_INTERRUPT     => dma_interrupt,
      SWENDPTR          => dma_pointer,
      SWBUFFER_MASK     => newdata_swbufmask
   );

   request_i : entity work.dma_ctrl_rx_request
   generic map(
      CHANNELS          => CHANNELS,
      BUFFERS           => BUFFERS,
      DATA_ALIGN        => DATA_ALIGN,
      UNIT_ID           => X"00"
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      BUFFER_READ       => BUFFER_READ,
      BUFFER_CHANNEL    => BUFFER_CHANNEL,
      BUFFER_DATA       => BUFFER_DATA,
      BUFFER_DV         => BUFFER_DV,

      DESC_DATA         => DESC_DATA,
      DMA_REQ           => dma_req,
      DMA_RDY           => dma_rdy,
      DMA_LENGTH        => dma_length,
      DMA_CHANNEL       => dma_channel,
      DMA_DONE          => dma_done,

      DMA_UP_DATA       => dma_up0_data,
      DMA_UP_HDR        => dma_up0_hdr,
      DMA_UP_SOP        => dma_up0_sop,
      DMA_UP_EOP        => dma_up0_eop,
      DMA_UP_SRC_RDY    => dma_up0_src_rdy,
      DMA_UP_DST_RDY    => dma_up0_dst_rdy
   );

   update_i : entity work.dma_ctrl_update
   generic map(
      CHANNELS          => CHANNELS,
      UNIT_ID           => X"02",
      FLUSH             => DOUBLE_PCIE
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      ENABLE            => ctrl_enable,

      UPDATE_ADDR       => CONFIG_ADDR,

      UPDATE            => update_dma_donecomplete,
      UPDATE_CHANNEL    => update_dma_channel,
      UPDATE_VALUE      => update_dma_pointer,
      UPDATE_INTERRUPT  => update_dma_interrupt,

      TIMEOUT           => update_timeout,
      TIMEOUT_CHANNEL   => update_timeout_channel,
      INTERRUPT         => update_interrupt,
      INTERRUPT_CHANNEL => update_interrupt_channel,

      DMA_UP_DATA       => dma_up1_data,
      DMA_UP_HDR        => dma_up1_hdr,
      DMA_UP_SOP        => dma_up1_sop,
      DMA_UP_EOP        => dma_up1_eop,
      DMA_UP_SRC_RDY    => dma_up1_src_rdy,
      DMA_UP_DST_RDY    => dma_up1_dst_rdy,

      DMA_DOWN_DATA     => DMA_DOWN_DATA,
      DMA_DOWN_HDR      => DMA_DOWN_HDR,
      DMA_DOWN_SOP      => DMA_DOWN_SOP,
      DMA_DOWN_EOP      => DMA_DOWN_EOP,
      DMA_DOWN_SRC_RDY  => DMA_DOWN_SRC_RDY,
      DMA_DOWN_DST_RDY  => DMA_DOWN_DST_RDY
   );

   switch_i: entity work.dma_switch
   generic map(
      DATA_WIDTH        => 512,
      HDR_WIDTH         => 96
   )
   PORT map (
      CLK               => CLK,
      RESET             => RESET,

      DMA_TX_DATA       => DMA_UP_DATA,
      DMA_TX_HDR        => DMA_UP_HDR,
      DMA_TX_SOP        => DMA_UP_SOP,
      DMA_TX_EOP        => DMA_UP_EOP,
      DMA_TX_SRC_RDY    => DMA_UP_SRC_RDY,
      DMA_TX_DST_RDY    => DMA_UP_DST_RDY,

      DMA_RX0_DATA      => dma_up0_data,
      DMA_RX0_HDR       => dma_up0_hdr,
      DMA_RX0_SOP       => dma_up0_sop,
      DMA_RX0_EOP       => dma_up0_eop,
      DMA_RX0_SRC_RDY   => dma_up0_src_rdy,
      DMA_RX0_DST_RDY   => dma_up0_dst_rdy,

      DMA_RX1_DATA      => dma_up1_data,
      DMA_RX1_HDR       => dma_up1_hdr,
      DMA_RX1_SOP       => dma_up1_sop,
      DMA_RX1_EOP       => dma_up1_eop,
      DMA_RX1_SRC_RDY   => dma_up1_src_rdy,
      DMA_RX1_DST_RDY   => dma_up1_dst_rdy
   );

   dma_fifo_complete: entity work.FIFO
   generic map(
      DATA_WIDTH        => FIFO_WIDTH,
      ITEMS             => 8
   )
   port map(
      RESET             => RESET,
      CLK               => CLK,
      DATA_IN           => dma_fifo_datain,
      WRITE_REQ         => dma_req,
      FULL              => open,
      LSTBLK            => open,

      DATA_OUT          => dma_fifo_dataout,
      READ_REQ          => dma_done,
      EMPTY             => open
   );

   dma_fifo_datain(FIFO_COMPLETE)   <= dma_complete;
   dma_fifo_datain(FIFO_INTERRUPT)  <= dma_interrupt;
   dma_fifo_datain(FIFO_CHANNEL)    <= dma_channel;
   dma_fifo_datain(FIFO_POINTER)    <= dma_pointer;

   update_dma_complete              <= dma_fifo_dataout(FIFO_COMPLETE);
   update_dma_interrupt             <= dma_fifo_dataout(FIFO_INTERRUPT);
   update_dma_channel               <= dma_fifo_dataout(FIFO_CHANNEL);
   update_dma_pointer               <= dma_fifo_dataout(FIFO_POINTER);

   gen_interrupt: for i in 0 to CHANNELS-1 generate
      reg_swendptrp : process (CLK)
      begin
         if(CLK = '1' and CLK'event) then
            if(RESET = '1') then
               reg_interrupt_en(i) <= '0';
            elsif(sw_regsel(5) = '1' and sw_channel = i) then
               reg_interrupt_en(i) <= MI_DWR(0);
            elsif(update_interrupt = '1' and update_interrupt_channel = i) then
               reg_interrupt_en(i) <= '0';
            end if;
         end if;
      end process;
   end generate;

   -- ------------   SW Registers   -----------------------------

   MI_DRDY                 <= MI_RD;
   MI_ARDY                 <= MI_WR or MI_RD;
   sw_channel              <= MI_ADDR(log2(CHANNELS) + 5 downto 6);

   addrdec_p: process(MI_ADDR, MI_WR)
   begin
      sw_regsel <= (others => '0');
      for i in 0 to 15 loop
         if(MI_WR = '1' and conv_std_logic_vector(i, 4) = MI_ADDR(5 downto 2)) then
            sw_regsel(i) <= '1';
         end if;
      end loop;
   end process;

   with MI_ADDR(5 downto 2) select
      MI_DRD   <= zeros(31 downto 2) & reg_control    when X"0",
                  zeros(31 downto 8) & reg_status     when X"1",
                                       reg_swstrptr   when X"2",
                                       reg_swendptr   when X"3",
                                       reg_swbufmask  when X"4",
                                       reg_interrupt  when X"5",
                                       reg_timeout    when X"6",
                  zeros(31 downto PAGE_WIDTH+1) & reg_maxreq when X"7",
                                       X"00000000"    when others;

   reg_control_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 2,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,

      DI     			=> MI_DWR(1 downto 0),
      WE      			=> sw_regsel(0),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_control,
      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_control
   );

   reg_status_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 8,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,

      DI     			=> newdata_status,
      WE      			=> '1',
      ADDRA   			=> newdata_channel,
      DOA     			=> open,

      ADDRB   			=> sw_channel,
      DOB     			=> reg_status
   );

   reg_swstrptr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI(31 downto 1)=> MI_DWR(31 downto 1),
      DI(0)    		=> '0',
      WE      			=> sw_regsel(2),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_swstrptr,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_swstrptr
   );

   reg_swendptr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI             => update_dma_pointer,
      WE      			=> update_dma_donecomplete,
      ADDRA   			=> update_dma_channel,
      DOA     			=> open,

      ADDRB   			=> sw_channel,
      DOB     			=> reg_swendptr
   );

   reg_swbufmask_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> SWPTR_WIDTH,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR(31 downto 0),
      WE      			=> sw_regsel(4),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_swbufmask,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_swbufmask
   );

   reg_interrupt_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR,
      WE      			=> sw_regsel(5),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_interrupt,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_interrupt
   );

   reg_timeout_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> 32,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR,
      WE      			=> sw_regsel(6),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_timeout,

      ADDRB   			=> update_timeout_channel,
      DOB     			=> update_timeout
   );

   reg_maxreq_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH   	=> PAGE_WIDTH+1,
      ITEMS        	=> CHANNELS,
      DISTMEM_TYPE 	=> 16
   )
   port map(
      WCLK    			=> CLK,
      RESET   			=> RESET,
      DI      			=> MI_DWR(PAGE_WIDTH downto 0),
      WE      			=> sw_regsel(7),
      ADDRA   			=> sw_channel,
      DOA     			=> reg_maxreq,

      ADDRB   			=> newdata_channel,
      DOB     			=> newdata_maxreq
   );


end architecture;
