-- cc_int.vhd: Completer Completion interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use ieee.numeric_std.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture cc_int_arch of cc_int is

   -- Types declaration -------------------------------------------------------
   type state_type is (S_IDLE, 
                       S_COMPL_0, S_COMPL_1, S_COMPL_2, S_COMPL_3,
                                  S_COMPL_4, S_COMPL_5, S_COMPL_6, S_COMPL_7,
                       S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL,
                       S_WAIT_FOR_CC_INT); 
 
   -- Signals declaration -----------------------------------------------------
   -- FIFO connection
   signal fifo_data_in              : std_logic_vector(35 downto 0);
   signal fifo_data_out             : std_logic_vector(35 downto 0);
   signal fifo_dv                   : std_logic;
   signal fifo_rd                   : std_logic;
   signal fifo_empty                : std_logic;

   -- Internal representation of DATA and EOP inputs
   signal data_int                  : std_logic_vector(31 downto 0);
   signal eop_int                   : std_logic;

   -- Non-constant values from completion descriptor
   signal max_payload               : std_logic_vector(12 downto 0);
   signal reg_max_payload           : std_logic_vector(12 downto 0);
   signal byte_count_total          : std_logic_vector(12 downto 0);
   signal byte_count                : std_logic_vector(12 downto 0);
   signal dword_count               : std_logic_vector(10 downto 0);
   signal lower_address_init        : std_logic_vector(6 downto 0);
   signal lower_address             : std_logic_vector(6 downto 0);

   -- Counter of remaining Dwords before completion split
   signal dwords_to_split_cnt       : std_logic_vector(10 downto 0);
   signal dwords_to_split_cnt_dec   : std_logic;
   
   -- FSM signals
   signal previous_state            : state_type;
   signal present_state             : state_type;
   signal next_state                : state_type;

   -- Auxiliary signals
   signal completion_data           : std_logic_vector(255 downto 0);
   signal completion_keep           : std_logic_vector(7 downto 0);
   signal split_completion_sent     : std_logic;

   -- Auxiliary registers
   signal reg_completion_data       : std_logic_vector(255 downto 0);
   signal reg_completion_keep       : std_logic_vector(7 downto 0);
   signal reg_split_completion_sent : std_logic;


begin


   -- -------------------------------------------------------------------------
   --  FIFO instantiation
   -- -------------------------------------------------------------------------

   -- composition of FIFO data input
   fifo_data_in(31 downto  0) <= DATA;
   fifo_data_in(32          ) <= EOP;
   fifo_data_in(35 downto 33) <= (others => '0');

   -- FIFO instantiation
   fifo_i : entity work.fifo_bram
   generic map(
      ITEMS         => FIFO_SIZE,
      BRAM_TYPE     => 36,
      DATA_WIDTH    => 36,
      AUTO_PIPELINE => true
   )
   port map(
      CLK           => CLK,
      RESET         => RESET,

      -- Write interface
      DI            => fifo_data_in,
      WR            => WR,
      FULL          => FULL,
      LSTBLK        => open,

      -- Read interface
      DO            => fifo_data_out,
      DV            => fifo_dv,
      RD            => fifo_rd,
      EMPTY         => open
   );

   -- fifo_empty derived from fifo_dv
   fifo_empty <= not fifo_dv;

   -- decompositon of FIFO data output
   data_int <= fifo_data_out(31 downto 0);
   eop_int  <= fifo_data_out(32); 


   -- -------------------------------------------------------------------------
   --  Counters and non-constant values of completion descriptor
   -- -------------------------------------------------------------------------
   
   -- MAX_PAYLOAD_SIZE translation 
   max_payload <= "0000010000000" when (MAX_PAYLOAD_SIZE = "000") else
                  "0000100000000" when (MAX_PAYLOAD_SIZE = "001") else
                  "0001000000000" when (MAX_PAYLOAD_SIZE = "010") else
                  "0010000000000" when (MAX_PAYLOAD_SIZE = "011") else
                  "0100000000000" when (MAX_PAYLOAD_SIZE = "100") else
                  "1000000000000" when (MAX_PAYLOAD_SIZE = "101") else
                  (others => '0');

   -- registering translated max_payload value
   reg_max_payload_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         reg_max_payload <= max_payload; 
      end if;
   end process reg_max_payload_p;
 
   -- Byte count --------------------------------------------------------------
   -- total byte count computation
   byte_count_total <= "0000000000100" when
                          (std_match(USER_FIRST_BE,"1--1") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000011" when
                          (std_match(USER_FIRST_BE,"01-1") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000011" when
                          (std_match(USER_FIRST_BE,"1-10") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000010" when
                          (std_match(USER_FIRST_BE,"0011") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000010" when
                          (std_match(USER_FIRST_BE,"0110") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000010" when
                          (std_match(USER_FIRST_BE,"1100") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000001" when
                          (std_match(USER_FIRST_BE,"0001") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000001" when
                          (std_match(USER_FIRST_BE,"0010") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000001" when
                          (std_match(USER_FIRST_BE,"0100") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000001" when
                          (std_match(USER_FIRST_BE,"1000") AND
                           std_match(USER_LAST_BE,"0000")) else
                       "0000000000001" when
                          (std_match(USER_FIRST_BE,"0000") AND
                           std_match(USER_LAST_BE,"0000")) else
                       DATA_DWORD_COUNT & "00" when
                          (std_match(USER_FIRST_BE,"---1") AND
                           std_match(USER_LAST_BE,"1---")) else
                       (DATA_DWORD_COUNT & "00") - 1 when
                          (std_match(USER_FIRST_BE,"---1") AND
                           std_match(USER_LAST_BE,"01--")) else
                       (DATA_DWORD_COUNT & "00") - 2 when
                          (std_match(USER_FIRST_BE,"---1") AND
                           std_match(USER_LAST_BE,"001-")) else
                       (DATA_DWORD_COUNT & "00") - 3 when
                          (std_match(USER_FIRST_BE,"---1") AND
                           std_match(USER_LAST_BE,"0001")) else
                       (DATA_DWORD_COUNT & "00") - 1 when
                          (std_match(USER_FIRST_BE,"--10") AND
                           std_match(USER_LAST_BE,"1---")) else
                       (DATA_DWORD_COUNT & "00") - 2 when
                          (std_match(USER_FIRST_BE,"--10") AND
                           std_match(USER_LAST_BE,"01--")) else
                       (DATA_DWORD_COUNT & "00") - 3 when
                          (std_match(USER_FIRST_BE,"--10") AND
                           std_match(USER_LAST_BE,"001-")) else
                       (DATA_DWORD_COUNT & "00") - 4 when
                          (std_match(USER_FIRST_BE,"--10") AND
                           std_match(USER_LAST_BE,"0001")) else
                       (DATA_DWORD_COUNT & "00") - 2 when
                          (std_match(USER_FIRST_BE,"-100") AND
                           std_match(USER_LAST_BE,"1---")) else
                       (DATA_DWORD_COUNT & "00") - 3 when
                          (std_match(USER_FIRST_BE,"-100") AND
                           std_match(USER_LAST_BE,"01--")) else
                       (DATA_DWORD_COUNT & "00") - 4 when
                          (std_match(USER_FIRST_BE,"-100") AND
                           std_match(USER_LAST_BE,"001-")) else
                       (DATA_DWORD_COUNT & "00") - 5 when
                          (std_match(USER_FIRST_BE,"-100") AND
                           std_match(USER_LAST_BE,"0001")) else
                       (DATA_DWORD_COUNT & "00") - 3 when
                          (std_match(USER_FIRST_BE,"1000") AND
                           std_match(USER_LAST_BE,"1---")) else
                       (DATA_DWORD_COUNT & "00") - 4 when
                          (std_match(USER_FIRST_BE,"1000") AND
                           std_match(USER_LAST_BE,"01--")) else
                       (DATA_DWORD_COUNT & "00") - 5 when
                          (std_match(USER_FIRST_BE,"1000") AND
                           std_match(USER_LAST_BE,"001-")) else
                       (DATA_DWORD_COUNT & "00") - 6 when
                          (std_match(USER_FIRST_BE,"1000") AND
                           std_match(USER_LAST_BE,"0001")) else
                       (others => '0');
   
   -- byte count (decremented when split completion occurs)
   byte_count_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            byte_count <= (others => '0');
         else
            if (READ_REQ = '1') then
               byte_count <= byte_count_total;
            elsif (UNSUPPORTED_REQ = '1') then
               byte_count <= (others => '0');
            elsif (split_completion_sent = '1') then
               byte_count <= byte_count - reg_max_payload;
            end if;
         end if;
      end if;
   end process byte_count_p;

   -- Dword count -------------------------------------------------------------
   -- derived from byte count
   dword_count <= reg_max_payload(12 downto 2) when
                    (byte_count > reg_max_payload) else
                  byte_count(12 downto 2);

   -- Lower address -----------------------------------------------------------
   -- initial address of the first completion data
   lower_address_init <= DATA_LOWER_ADDR & "01" when
                            USER_FIRST_BE(1 downto 0) = "10" else
                         DATA_LOWER_ADDR & "10" when
                            USER_FIRST_BE(2 downto 0) = "100" else
                         DATA_LOWER_ADDR & "11" when
                            USER_FIRST_BE(3 downto 0) = "1000" else
                         DATA_LOWER_ADDR & "00";

   -- lower address
   lower_address_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            lower_address <= (others => '0');
         else
            if (READ_REQ = '1') then
               lower_address <= lower_address_init;
            elsif (UNSUPPORTED_REQ = '1') then
               lower_address <= (others => '0');
            elsif (split_completion_sent = '1') then
               lower_address <= conv_std_logic_vector(
                                   CONV_INTEGER(lower_address + reg_max_payload),
                                   7
                                );
            end if;
         end if;
      end if; 
   end process lower_address_p; 

   -- Counter of Dwords before the completion split ---------------------------
   dwords_to_split_cnt_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            dwords_to_split_cnt <= (others => '0');
         else
            if ((READ_REQ = '1') OR (split_completion_sent = '1')) then
               dwords_to_split_cnt <= reg_max_payload(12 downto 2);
            elsif (dwords_to_split_cnt_dec = '1') then
               dwords_to_split_cnt <= dwords_to_split_cnt - 1;
            end if;
         end if;
      end if;
   end process dwords_to_split_cnt_p;
                             

   -- -------------------------------------------------------------------------
   --  FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (RESET = '1') then
            previous_state <= S_IDLE;
            present_state  <= S_IDLE;
         else
            previous_state <= present_state;
            present_state  <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process (present_state, READ_REQ, UNSUPPORTED_REQ,
                                 fifo_empty, eop_int, CC_READY,
                                 dwords_to_split_cnt)
   begin
      -- default value
      next_state <= S_IDLE;      

      case (present_state) is

         when S_IDLE =>
            if (READ_REQ = '1') then
               next_state <= S_COMPL_3;
            elsif (UNSUPPORTED_REQ = '1') then
               next_state <= S_WAIT_FOR_CC_INT;
            else
               next_state <= S_IDLE;
            end if;

         when S_COMPL_0 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_1;
               end if;
            else
               next_state <= S_COMPL_0;
            end if;

         when S_COMPL_1 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_2;
               end if;
            else
               next_state <= S_COMPL_1;
            end if;

         when S_COMPL_2 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  if (dwords_to_split_cnt = 1) then
                     if (CC_READY = '1') then
                        next_state <= S_COMPL_3;
                     else
                        next_state <= S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL;
                     end if;
                  else
                     next_state <= S_COMPL_3;
                  end if;
               end if;
            else
               next_state <= S_COMPL_2;
            end if;

         when S_COMPL_3 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_4;
               end if;
            else
               next_state <= S_COMPL_3;
            end if;

         when S_COMPL_4 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_5;
               end if;
            else
               next_state <= S_COMPL_4;
            end if;

         when S_COMPL_5 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_6;
               end if;
            else
               next_state <= S_COMPL_5;
            end if;

         when S_COMPL_6 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  next_state <= S_COMPL_7;
               end if;
            else
               next_state <= S_COMPL_6;
            end if;

         when S_COMPL_7 =>
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  if (CC_READY = '1') then
                     next_state <= S_IDLE;
                  else
                     next_state <= S_WAIT_FOR_CC_INT;
                  end if;
               else
                  if (CC_READY = '1') then
                     next_state <= S_COMPL_0;
                  else
                     next_state <= S_COMPL_7;
                  end if;
               end if;
            else
               next_state <= S_COMPL_7;
            end if;

         when S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL =>
            if (CC_READY = '1') then
               next_state <= S_COMPL_3;
            else
               next_state <= S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL;
            end if;

         when S_WAIT_FOR_CC_INT =>
            if (CC_READY = '1') then
               next_state <= S_IDLE;
            else
               next_state <= S_WAIT_FOR_CC_INT;
            end if;

--          when others =>
--             next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process (REG_CQ_DESCRIPTOR, REG_CQ_USER_FIRST_BE,
                             REG_CQ_USER_LAST_BE, REG_CQ_USER_TPH_PRESENT,
                             REG_CQ_USER_TPH_TYPE, REG_CQ_USER_TPH_ST_TAG,
                             READ_REQ, eop_int, CC_READY, present_state,
                             previous_state, data_int, fifo_empty, byte_count,
                             dword_count, lower_address, dwords_to_split_cnt,
                             reg_completion_data, reg_completion_keep,
                             reg_split_completion_sent)
   begin
      -- default values
      COMPLETION_SENT         <= '0';
      split_completion_sent   <= '0';
      completion_data         <= reg_completion_data;
      CC_LAST                 <= '0';
      completion_keep         <= reg_completion_keep;
      CC_VALID                <= '0';
      dwords_to_split_cnt_dec <= '0';
      fifo_rd                 <= '0';

      case (present_state) is

         when S_IDLE => 
            completion_data <= (others => '0');
            completion_keep <= (others => '0');
            if (READ_REQ = '1' AND fifo_empty = '0') then
               fifo_rd <= '1';
            end if;

         when S_COMPL_0 =>
            completion_data(31 downto 0)   <= data_int;
            completion_data(255 downto 32) <= (others => '0');
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00000001";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_1 =>
            completion_data(63 downto 32) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00000011";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_2 =>
            completion_data(95 downto 64) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00000111";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               if (dwords_to_split_cnt = 1) then
                  completion_keep       <= "00000111";
                  if (CC_READY = '1') then
                     split_completion_sent <= '1';
                     CC_LAST               <= '1';
                     CC_VALID              <= '1';                        
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_3 =>
            if ((previous_state = S_IDLE) OR
                (previous_state = S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL) OR
                ((previous_state = S_COMPL_2) AND 
                 (reg_split_completion_sent = '1'))) then
               completion_data <=
                  (255 downto 128 => '0')           & -- empty space in completion payload
                  data_int                          & -- payload data
                  '0'                               & -- force ECRC
                  REG_CQ_DESCRIPTOR(126 downto 124) & -- attributes
                  REG_CQ_DESCRIPTOR(123 downto 121) & -- transaction class
                  '0'                               & -- completer ID enable
                  "00000000"                        & -- completer bus number
                  REG_CQ_DESCRIPTOR(111 downto 104) & -- target function/device
                                                      --    number
                  REG_CQ_DESCRIPTOR(103 downto  96) & -- tag
                  REG_CQ_DESCRIPTOR( 95 downto  80) & -- requester ID
                  '0'                               & -- RESERVED
                  '0'                               & -- poisoned completion
                  "000"                             & -- completion status
                                                      --    (successful
                                                      --     completion)
                  dword_count                       & -- Dword count
                  "00"                              & -- RESERVED
                  '0'                               & -- locked read completion
                  byte_count                        & -- byte count
                  "000000"                          & -- RESERVED
                  REG_CQ_DESCRIPTOR(1 downto 0)     & -- address type
                  '0'                               & -- RESERVED
                  lower_address;                      -- lower address
            else
               completion_data(127 downto 96) <= data_int;
            end if;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00001111";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_4 =>
            completion_data(159 downto 128) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00011111";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_5 =>
            completion_data(191 downto 160) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "00111111";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_6 =>
            completion_data(223 downto 192) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep <= "01111111";
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               end if;
               dwords_to_split_cnt_dec <= '1';
               fifo_rd                 <= '1';
            end if;

         when S_COMPL_7 =>
            completion_data(255 downto 224) <= data_int;
            if (fifo_empty = '0') then
               if (eop_int = '1') then
                  completion_keep         <= "11111111";
                  dwords_to_split_cnt_dec <= '1';
                  fifo_rd                 <= '1';
                  if (CC_READY = '1') then
                     COMPLETION_SENT <= '1';
                     CC_LAST         <= '1';
                     CC_VALID        <= '1';
                  end if;
               else
                  if (CC_READY = '1') then
                     completion_keep         <= "11111111";
                     CC_VALID                <= '1';
                     dwords_to_split_cnt_dec <= '1';
                     fifo_rd                 <= '1';
                  end if;
               end if;
            end if;

         when S_WAIT_FOR_CC_INT_BEFORE_SPLIT_COMPL =>
            if (CC_READY = '1') then
               split_completion_sent <= '1';
               CC_LAST               <= '1';
               CC_VALID              <= '1';                        
            end if;

         when S_WAIT_FOR_CC_INT =>
            if (previous_state = S_IDLE) then
               completion_data <=
                  REG_CQ_DESCRIPTOR(127 downto 0)   & -- request descriptor
                  "00000000"                        & -- RESERVED
                  REG_CQ_USER_TPH_ST_TAG            & -- tph_st_tag
                  "00000"                           & -- RESERVED
                  REG_CQ_USER_TPH_TYPE              & -- tph_type
                  REG_CQ_USER_TPH_PRESENT           & -- tph_present
                  REG_CQ_USER_LAST_BE               & -- last_be
                  REG_CQ_USER_FIRST_BE              & -- first_be
                  '0'                               & -- force ECRC
                  REG_CQ_DESCRIPTOR(126 downto 124) & -- attributes
                  REG_CQ_DESCRIPTOR(123 downto 121) & -- transaction class
                  '0'                               & -- completer ID enable
                  "00000000"                        & -- completer bus number
                  REG_CQ_DESCRIPTOR(111 downto 104) & -- target function/device
                                                      --    number
                  REG_CQ_DESCRIPTOR(103 downto  96) & -- tag
                  REG_CQ_DESCRIPTOR( 95 downto  80) & -- requester ID
                  '0'                               & -- RESERVED
                  '0'                               & -- poisoned completion
                  "001"                             & -- completion status
                                                      --    (unsupported
                                                      --     request)
                  dword_count                       & -- Dword count
                  "00"                              & -- RESERVED
                  '0'                               & -- locked read completion
                  byte_count                        & -- byte count
                  "000000"                          & -- RESERVED
                  REG_CQ_DESCRIPTOR(1 downto 0)     & -- address type
                  '0'                               & -- RESERVED
                  lower_address;                      -- lower address
            end if;
            if (CC_READY = '1') then
               COMPLETION_SENT <= '1';
               CC_LAST         <= '1';
               CC_VALID        <= '1';
            end if;

--         when others =>
--            null;

      end case;      
      
   end process output_logic_p;
 
   -- Auxiliary registers -----------------------------------------------------
   -- reg_completion_data
   reg_completion_data_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         reg_completion_data <= completion_data;
      end if;
   end process reg_completion_data_p;
   
   -- reg_completion_keep
   reg_completion_keep_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            reg_completion_keep <= (others => '0');
         else
            reg_completion_keep <= completion_keep;
         end if;
      end if;
   end process reg_completion_keep_p;

   -- reg_split_completion_sent
   reg_split_completion_sent_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            reg_split_completion_sent <= '0';
         else
            reg_split_completion_sent <= split_completion_sent;
         end if;
      end if;
   end process reg_split_completion_sent_p;


   -- -------------------------------------------------------------------------
   --  Assigning values to output ports
   -- -------------------------------------------------------------------------
   
   -- CC interface
   CC_DATA <= completion_data;
   CC_KEEP <= completion_keep;

end architecture cc_int_arch;
