-- asfifo_arch.vhd
--!
--! \file
--! \brief Asynchronous fifo in BRAMs for 7series FPGAs
--! \author Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2013
--!
--! \section License
--!
--! Copyright (C) 2013 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

library UNISIM;
use UNISIM.vcomponents.all;

library UNIMACRO;
use UNIMACRO.vcomponents.all;

--! package with log2 function.
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture FULL of ASFIFO_BRAM_7SERIES_MAIN is

   --! Constants declaration
   -- -------------------------------------------------------------------------

   constant FULL_FIFOS        : integer := DATA_WIDTH / 72;
   constant REMINDER_WIDTH    : integer := DATA_WIDTH - (FULL_FIFOS * 72);

   --! Types declaration
   -- -------------------------------------------------------------------------

   type count_t is array (0 to FULL_FIFOS) of std_logic_vector(8 downto 0);

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! \name FIFO data signals
   signal in_rem              : std_logic_vector(72-1 downto 0);
   signal out_rem             : std_logic_vector(72-1 downto 0);
  
   --! \name Misc control signals
   signal reset_async         : std_logic; -- Ignore timing at this signal
   signal rdcount             : count_t;
   signal wrcount             : count_t;
   signal fifo_full           : std_logic_vector(FULL_FIFOS downto 0);
   signal fifo_empty          : std_logic_vector(FULL_FIFOS downto 0);
   signal fifo_afull          : std_logic_vector(FULL_FIFOS downto 0);
   signal fifo_aempty         : std_logic_vector(FULL_FIFOS downto 0);

   --! \name Internal control signals
   signal sig_full            : std_logic;
   signal sig_empty           : std_logic;
   signal sig_afull           : std_logic;
   signal sig_aempty          : std_logic;
   signal sig_rd              : std_logic;
   signal sig_wr              : std_logic;

   --! \name Synchronized asynchronous reset signals
   signal reset_async_rd      : std_logic;
   signal reset_async_wr      : std_logic;

   --! \name Safe reset logic
   signal rst_rd_sync_wr      : std_logic;
   signal wait_cnt_wr         : std_logic_vector(2 downto 0) := (others => '0');
   signal wait_cnt_wr_ce      : std_logic;

   signal rst_wr_sync_rd      : std_logic;
   signal wait_cnt_rd         : std_logic_vector(2 downto 0) := (others => '0');
   signal wait_cnt_rd_ce      : std_logic;

   --! Attributes declaration
   -- -------------------------------------------------------------------------

   --! dont_touch is required because of constraints
   attribute dont_touch : string;
   attribute dont_touch of reset_async : signal is "true";


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                       Safe reset implementation
   -- -------------------------------------------------------------------------

   --! Generate when safe reset IS enabled
   -- -------------------------------------------------------------------------

   safe_reset_gen : if SAFE_RESET generate

      --! RD domain safe reset logic
      -- ----------------------------------------------------------------------

      --! Synchronization of RST_WR to RD clock domain
      rst_wr_sync_rd_i : entity work.ASYNC_RESET
      port map (
         CLK        => CLK_RD,
         ASYNC_RST  => RST_WR,
         OUT_RST(0) => rst_wr_sync_rd
      );

      --! Counter for prolonging RD reset
      reset_wait_cnt_rd : process(CLK_RD)
      begin
         if rising_edge(CLK_RD) then
            if (RST_RD = '1') OR (rst_wr_sync_rd = '1') then
               wait_cnt_rd <= (others => '0');
            elsif (wait_cnt_rd_ce = '1') then
               wait_cnt_rd <= wait_cnt_rd + 1;
            end if;
         end if;
      end process reset_wait_cnt_rd;

      --! wait_cnt_rd counting enable
      wait_cnt_rd_ce <= '1' when wait_cnt_rd < "101" else
                        '0';

      --! WR domain safe reset logic
      -- ----------------------------------------------------------------------

      --! Synchronization of RST_RD to WR clock domain
      rst_rd_sync_wr_i : entity work.ASYNC_RESET
      port map (
         CLK        => CLK_WR,
         ASYNC_RST  => RST_RD,
         OUT_RST(0) => rst_rd_sync_wr
      );

      --! Counter for prolonging WR reset
      reset_wait_cnt_wr : process(CLK_WR)
      begin
         if rising_edge(CLK_WR) then
            if (RST_WR = '1') OR (rst_rd_sync_wr = '1') then
               wait_cnt_wr <= (others => '0');
            elsif (wait_cnt_wr_ce = '1') then
               wait_cnt_wr <= wait_cnt_wr + 1;
            end if;
         end if;
      end process reset_wait_cnt_wr;

      --! wait_cnt_wr counting enable
      wait_cnt_wr_ce <= '1' when wait_cnt_wr < "101" else
                        '0';

      --! Generation of safe asynchronous reset
      -- ----------------------------------------------------------------------

      reset_async <= wait_cnt_rd_ce OR wait_cnt_wr_ce;

   end generate safe_reset_gen;

   --! Generate when safe reset IS NOT enabled
   -- -------------------------------------------------------------------------

   nonsafe_reset_gen : if not SAFE_RESET generate

      --! Asynchronous reset composition
      reset_async <= RST_WR or RST_RD;

   end generate nonsafe_reset_gen;


   -- -------------------------------------------------------------------------
   --                       Control signals handling
   -- -------------------------------------------------------------------------

   --! Control signals merging using OR-reduce
   -- -------------------------------------------------------------------------

   --! Full signals
   FULL_OR : entity work.GEN_OR
   generic map (
      OR_WIDTH => FULL_FIFOS + 1
   )
   port map (
      DI       => fifo_full,
      DO       => sig_full
   );

   --! Empty signals
   EMPTY_OR : entity work.GEN_OR
   generic map (
      OR_WIDTH => FULL_FIFOS + 1
   )
   port map (
      DI       => fifo_empty,
      DO       => sig_empty
   );

   --! Almost full signals
   AFULL_OR : entity work.GEN_OR
   generic map (
      OR_WIDTH => FULL_FIFOS + 1
   )
   port map (
      DI       => fifo_afull,
      DO       => sig_afull
   );

   --! Almost empty signals
   AEMPTY_OR : entity work.GEN_OR
   generic map (
      OR_WIDTH => FULL_FIFOS + 1
   )
   port map (
      DI       => fifo_aempty,
      DO       => sig_aempty
   );

   --! Backward reset synchronization to RD and WR clock domains
   -- ----------------------------------------------------------------------

   --! Synchronization to RD clock domain
   reset_async_rd_i : entity work.ASYNC_RESET
   port map (
      CLK        => CLK_RD,
      ASYNC_RST  => reset_async,
      OUT_RST(0) => reset_async_rd
   );

   --! Synchronization to WR clock domain
   reset_async_wr_i : entity work.ASYNC_RESET
   port map (
      CLK        => CLK_WR,
      ASYNC_RST  => reset_async,
      OUT_RST(0) => reset_async_wr
   );

   --! Handling control inputs and outputs
   -- -------------------------------------------------------------------------

   --! Control ports connection
   FULL   <= sig_full OR reset_async_wr;
   EMPTY  <= sig_empty OR reset_async_rd;
   AFULL  <= sig_afull OR reset_async_wr;
   AEMPTY <= sig_aempty OR reset_async_rd;
   sig_rd <= RD AND NOT reset_async_rd;
   sig_wr <= WR AND NOT reset_async_wr;


   -- -------------------------------------------------------------------------
   --                  Fully utilized ASFIFOs instantiation
   -- -------------------------------------------------------------------------

   full_fifos_gen : for i in 0 to FULL_FIFOS - 1 generate
      --! \brief Async FIFOs components with fully used width
      --! \details 36KB FIFO (First In, First Out) Block RAM Memory \n
      --! Virtex-7 \n
      --! Xilinx HDL Libraries Guide, version 14.6 \n

      FIFO_DUALCLOCK_MACRO_inst : FIFO_DUALCLOCK_MACRO
      generic map (
         DEVICE                  => DEVICE,                  -- Target Device: "VIRTEX5", "VIRTEX6", "7SERIES"
         ALMOST_FULL_OFFSET      => ALMOST_FULL_OFFSET,      -- Sets almost full threshold
         ALMOST_EMPTY_OFFSET     => ALMOST_EMPTY_OFFSET,     -- Sets the almost empty threshold
         DATA_WIDTH              => 72,                      -- Valid values are 1-72 (37-72 only valid when FIFO_SIZE="36Kb")
         FIFO_SIZE               => "36Kb",                  -- Target BRAM, "18Kb" or "36Kb"
         FIRST_WORD_FALL_THROUGH => FIRST_WORD_FALL_THROUGH) -- Sets the FIFO FWFT to TRUE or FALSE
      port map (
         ALMOSTEMPTY             => fifo_aempty(i),             -- 1-bit output almost empty
         ALMOSTFULL              => fifo_afull(i),              -- 1-bit output almost full
         DO                      => DO((i+1)*72-1 downto i*72), -- Output data, width defined by DATA_WIDTH parameter
         EMPTY                   => fifo_empty(i),              -- 1-bit output empty
         FULL                    => fifo_full(i),               -- 1-bit output full
         RDCOUNT                 => rdcount(i),                 -- Output read count, width determined by FIFO depth
         RDERR                   => open,                       -- 1-bit output read error
         WRCOUNT                 => wrcount(i),                 -- Output write count, width determined by FIFO depth
         WRERR                   => open,                       -- 1-bit output write error
         DI                      => DI((i+1)*72-1 downto i*72), -- Input data, width defined by DATA_WIDTH parameter
         RDCLK                   => CLK_RD,                     -- 1-bit input read clock
         RDEN                    => sig_rd,                     -- 1-bit input read enable
         RST                     => reset_async,                -- 1-bit input reset
         WRCLK                   => CLK_WR,                     -- 1-bit input write clock
         WREN                    => sig_wr                      -- 1-bit input write enable
      );

   end generate full_fifos_gen;


   -- -------------------------------------------------------------------------
   --                  Partly utilized ASFIFOs instantiation
   -- -------------------------------------------------------------------------

   --! There is one partly utilized ASFIFO
   -- -------------------------------------------------------------------------

   longer_reminder_gen : if REMINDER_WIDTH > 0 generate

      --! Input data to the partly utilized ASFIFO
      in_rem <= (71 downto REMINDER_WIDTH => '0') & DI(DATA_WIDTH-1 downto DATA_WIDTH-REMINDER_WIDTH);

      --! FIFO_DUALCLOCK_MACRO instantiation
      FIFO_DUALCLOCK_MACRO_inst : FIFO_DUALCLOCK_MACRO
      generic map (
         DEVICE                  => DEVICE,                 -- Target Device: "VIRTEX5", "VIRTEX6", "7SERIES"
         ALMOST_FULL_OFFSET      => ALMOST_FULL_OFFSET,     -- Sets almost full threshold
         ALMOST_EMPTY_OFFSET     => ALMOST_EMPTY_OFFSET,    -- Sets the almost empty threshold
         DATA_WIDTH              => 72,                     -- Valid values are 1-72 (37-72 only valid when FIFO_SIZE="36Kb")
         FIFO_SIZE               => "36Kb",                 -- Target BRAM, "18Kb" or "36Kb"
         FIRST_WORD_FALL_THROUGH => FIRST_WORD_FALL_THROUGH -- Sets the FIFO FWFT to TRUE or FALSE
      )
      port map (
         ALMOSTEMPTY             => fifo_aempty(FULL_FIFOS), -- 1-bit output almost empty
         ALMOSTFULL              => fifo_afull(FULL_FIFOS),  -- 1-bit output almost full
         DO                      => out_rem,                 -- Output data, width defined by DATA_WIDTH parameter
         EMPTY                   => fifo_empty(FULL_FIFOS),  -- 1-bit output empty
         FULL                    => fifo_full(FULL_FIFOS),   -- 1-bit output full
         RDCOUNT                 => rdcount(FULL_FIFOS),     -- Output read count, width determined by FIFO depth
         RDERR                   => open,                    -- 1-bit output read error
         WRCOUNT                 => wrcount(FULL_FIFOS),     -- Output write count, width determined by FIFO depth
         WRERR                   => open,                    -- 1-bit output write error
         DI                      => in_rem,                  -- Input data, width defined by DATA_WIDTH parameter
         RDCLK                   => CLK_RD,                  -- 1-bit input read clock
         RDEN                    => sig_rd,                  -- 1-bit input read enable
         RST                     => reset_async,             -- 1-bit input reset
         WRCLK                   => CLK_WR,                  -- 1-bit input write clock
         WREN                    => sig_wr                   -- 1-bit input write enable
      );

      --! Output data from the partly utilized ASFIFO
      DO(DATA_WIDTH-1 downto DATA_WIDTH-REMINDER_WIDTH) <= out_rem(REMINDER_WIDTH-1 downto 0);

   end generate longer_reminder_gen;

   --! There is no partly utilized ASFIFO
   -- -------------------------------------------------------------------------

   no_reminder_gen : if REMINDER_WIDTH = 0 generate

     fifo_empty(FULL_FIFOS)  <= '0'; -- fake FIFO is never empty nor full
     fifo_full(FULL_FIFOS)   <= '0';
     fifo_aempty(FULL_FIFOS) <= '0';
     fifo_afull(FULL_FIFOS)  <= '0';

   end generate no_reminder_gen;


end architecture;
