-- tx_lane_fifo.vhd : FIFO for 40/100GBASE-R PCS - rate compensation, idle block
--                    dropping
--
-- Copyright (C) 2012 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity pcs_tx_fifo is
    generic (
       NUM_LANES : natural := 8
   );
   port (
      RESET_D : in std_logic; -- D-clock reset
      CLK   : in std_logic; -- D clock
      D     : in std_logic_vector(NUM_LANES*66-1 downto 0);  -- Input data
      --
      RESET_Q : in std_logic; -- Q-clock reset
      TXCLK : in std_logic; -- Q clock
      RE    : in std_logic; -- Read enable
      Q     : out std_logic_vector(NUM_LANES*66-1 downto 0) := (others => '0');  -- Output data
      -- Debug
      FIFO_EMPTY_O : out std_logic;
      FIFO_FULL_O  : out std_logic;
      FIFO_DIN_O   : out std_logic_vector(NUM_LANES*66-1 downto 0);
      DROP_O       : out std_logic;
      INDEX_O      : out std_logic_vector(3 downto 0)
   );
end pcs_tx_fifo;

architecture behavioral of pcs_tx_fifo is

signal discard : std_logic;
signal idle    : std_logic_vector(NUM_LANES-1 downto 0);
signal fifo_din   : std_logic_vector(66*NUM_LANES-1 downto 0);
signal fifo_dout  : std_logic_vector(NUM_LANES*66-1 downto 0);
signal fifo_wen   : std_logic;
signal fifo_full  : std_logic;
signal fifo_empty : std_logic;
signal fifo_ren   : std_logic;
signal fifo_afull : std_logic;
signal fifo_aempty : std_logic;
signal idle_found : std_logic;
signal drop       : std_logic;
signal drop_index : natural range 0 to NUM_LANES-1;
signal sh_din     : std_logic_vector(66*NUM_LANES-1 downto 0);
signal sh_drop    : std_logic;
signal sh_index   : natural range 0 to NUM_LANES-1;


begin

IDLE_DETECT_LOGIC: process(D)
begin
   idle_found <= '0';
   -- Detect IDLE characters on individual lanes
   for i in 0 to NUM_LANES-1 loop
      if (D(1+i*66 downto i*66) = "01") and (D(9+i*66 downto i*66+2) = X"1E") then
         idle(i) <= '1';
         idle_found <= '1';
      else
         idle(i) <= '0';
      end if;
   end loop;
end process;

GEN_DROP_INDEX : process(idle)
begin
   drop_index <= 0;
   for i in 0 to NUM_LANES-1 loop
      if idle(i) = '1' then
         drop_index <= i;
      end if;
   end loop;
end process;

drop <= discard and idle_found;

TX_PIPE: process(CLK)
begin
   if CLK'event and CLK = '1' then
      discard  <= fifo_afull;
      sh_din   <= D;
      sh_drop  <= drop;
      sh_index <= drop_index;
   end if;
end process;

GEN_MULTILANE_DROP: if (NUM_LANES > 1) generate
    BLOCK_DROP: entity work.block_shifter
    generic map (
        NUM_LANES => NUM_LANES
    )
    port map (
        RESET => RESET_D,
        CLK   => CLK,
        D     => sh_din,
        DROP  => sh_drop,
        IDX   => sh_index,
        INS   => '0',
        --
        Q_VAL => fifo_wen,
        Q     => fifo_din
    );
end generate;

GEN_SINGLELANE_DROP: if (NUM_LANES = 1) generate
    fifo_din <= sh_din;
    fifo_wen <= (not(sh_drop) or not(idle_found)) and not(fifo_full);
end generate;


ASFIFO: entity work.ASFIFO_BRAM_7SERIES
generic map (
  DATA_WIDTH              => 66*NUM_LANES,
  FIRST_WORD_FALL_THROUGH => true,
  ALMOST_FULL_OFFSET      => X"0018",
  ALMOST_EMPTY_OFFSET     => X"0008"
  -- SAFE_RESET            => false
)
port map (
  CLK_WR   => CLK,
  RST_WR   => RESET_D,
  DI       => fifo_din,
  WR       => fifo_wen,
  AFULL    => fifo_afull,
  FULL     => fifo_full,
  --
  CLK_RD   => TXCLK,
  RST_RD   => RESET_Q,
  DO       => fifo_dout,
  RD       => fifo_ren,
  AEMPTY   => fifo_aempty,
  EMPTY    => fifo_empty
);

fifo_ren <= RE and not(fifo_empty);


OUT_REG: process(TXCLK)
begin
   if TXCLK'event and TXCLK = '1' then
      if RE = '1' then
         Q <= fifo_dout;
      end if;
   end if;
end process;

FIFO_FULL_O  <= fifo_full;
FIFO_EMPTY_O <= fifo_empty;
FIFO_DIN_O   <= fifo_din;
DROP_O       <= sh_drop;
INDEX_O      <= conv_std_logic_vector(sh_index,4);

end behavioral;
