-- mgmt.vhd : 40/100GBASE-R management (status/control)
--                     
-- Copyright (C) 2011 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:
-- Address space is similar to standard MDIO register mapping, while the lower 
-- 16 address bits are register number, upper 5 bits are device (1=PMA, 3=PCS)
-- Detailed address space:
-- PMA:  
-- MI address  upper 16b  &  lower 16b
-- 0x10000    PMA status1 & control1
-- 0x10004    PMA device identifier 2 & 1
-- 0x10008    PMA devices in package & speed capability
-- ...
--
-- 1.0      PMA/PMD control 1
-- 1.1      PMA/PMD status 1
-- 1.2, 1.3 PMA/PMD device identifier
-- 1.4      PMA/PMD speed ability
-- 1.5, 1.6 PMA/PMD devices in package
-- 1.7      PMA/PMD control 2
-- 1.8      PMA/PMD status 2
-- 1.9      PMA/PMD transmit disable
-- 1.10     PMD receive signal detect
-- 1.11     PMA/PMD extended ability register
-- 1.12     10G-EPON PMA/PMD P2MP ability register 45.2.1.11
-- 1.13     40G/100G PMA/PMD extended ability register 45.2.1.11a
-- 1.14,1.15 PMA/PMD package identifier
-- 1.1500    Test pattern ability 45.2.1.95
-- 1.1501    PRBS pattern testing control 45.2.1.96
-- 1.1510    Square wave testing control 45.2.1.97
-- 1.1600 - 1609 PRBS Tx error counters, lane 0 through lane 9 45.2.1.98
-- 1.1700 - 1709 PRBS Rx error counters, lane 0 through lane 9 45.2.1.99
-- PCS:
--   TODO

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity mgmt is
   generic (
      NUM_LANES : natural range 0 to 20 := 4; -- Max 20
      PMA_LANES : natural range 0 to 10 := 4; -- Max 10      
      SPEED100G     : std_logic := '0';       -- Speed is 100G when '1'
      GBASE40_ABLE  : std_logic := '1';       -- 40GE capable
      GBASE100_ABLE : std_logic := '0'        -- 100GE capable
   );
   port (
      RESET       : in  std_logic; 
      -- MI32 interface
      MI_CLK      : in  std_logic;
      MI_DWR      : in  std_logic_vector(31 downto 0);
      MI_ADDR     : in  std_logic_vector(31 downto 0);
      MI_RD       : in  std_logic;
      MI_WR       : in  std_logic;
      MI_BE       : in  std_logic_vector( 3 downto 0);
      MI_DRD      : out std_logic_vector(31 downto 0);
      MI_ARDY     : out std_logic;
      MI_DRDY     : out std_logic;
      -- 
      PCSCLK      : in  std_logic; -- PCS clock (156.25MHz)
      PMACLK      : in  std_logic; -- PMA clock (159.xxMHz)
      -- PCS control/status
      HI_BER        : in std_logic; -- BER monitor HI BER
      BLK_LOCK      : in std_logic_vector(NUM_LANES-1 downto 0); -- Block sync lock for each lane
      LINKSTATUS    : in std_logic; -- RX link status (=aligned and !hi_ber)
      BER_COUNT     : in std_logic_vector(21 downto 0);  -- BER monitor number of errored blocks for each lane
      BER_COUNT_CLR : out std_logic;                     -- Clear block count in the block sync
      BLK_ERR_CNTR  : in  std_logic_vector(21 downto 0); -- Block decode error counter
      BLK_ERR_CLR   : out std_logic;                     -- Clear errored block counter in the decoder
      SCR_BYPASS    : out std_logic_vector(1 downto 0);  -- Bypass the RX scrambler (bit 0) and TX scrambler (bit 1)
      PCS_RESET     : out std_logic;                     -- Reset the PCS block
      PCS_LPBCK     : out std_logic;                     -- PCS loopback enable 
      -- Lane align
      ALGN_LOCKED   : in  std_logic;
      BIP_ERR_CNTRS : in  std_logic_vector(NUM_LANES*16-1 downto 0); -- BIP error counters
      BIP_ERR_CLR   : out std_logic_vector(NUM_LANES-1 downto 0);      -- Clear BIP error counter for individual lane
      LANE_MAP      : in  std_logic_vector(NUM_LANES*5-1 downto 0);  -- PCS lane mapping
      LANE_ALIGN    : in  std_logic_vector(NUM_LANES-1 downto 0);      -- PCS lane align for individual lanes
      -- PMA control
      PMA_LOPWR     : out std_logic; -- Turn the PMA into lo-power state
      PMA_LPBCK     : out std_logic;
      PMA_REM_LPBCK : out std_logic;
      PMA_RESET     : out std_logic;
      PMA_RETUNE    : out std_logic;
      PMA_PTRN_EN   : out std_logic := '0'; -- Pattern generator enable
      PMA_TX_DIS    : out std_logic_vector(PMA_LANES-1 downto 0); -- PMA transmitt disable for individual lanes
      PMA_RX_OK     : in  std_logic_vector(PMA_LANES-1 downto 0); -- PMA RX link ok
      PMD_SIG_DET   : in  std_logic_vector(PMA_LANES-1 downto 0)  --
   );
end mgmt;

architecture behavioral of mgmt is

signal mi_drd_i     : std_logic_vector(31 downto 0);
signal pma_mode     : std_logic;
signal pma_fault    : std_logic;
signal pma_rxl_stat : std_logic;
signal pma_rst      : std_logic := '0'; 
signal pcs_fault    : std_logic;
signal pcs_rxl_stat : std_logic;
signal pcs_rxl_stat_l : std_logic;
signal pcs_rst      : std_logic := '0'; 
signal pcs_lpbk     : std_logic := '0'; 
signal pcs_low_pwr  : std_logic := '0'; 
signal pcs_blk_lock_g : std_logic;
signal pcs_blk_lock_l : std_logic; -- latched version
signal pcs_hi_ber   : std_logic;
signal pcs_hi_ber_l : std_logic; -- latched version
signal pcs_blk_lock : std_logic_vector(19 downto 0);
signal ber_count_r  : std_logic_vector(21 downto 0);
signal blk_err_cntr_r : std_logic_vector(21 downto 0);
signal ber_count_l    : std_logic_vector(15 downto 0);
signal blk_err_cntr_l : std_logic_vector(13 downto 0);
signal tx_fault     : std_logic; -- latching 
signal rx_fault     : std_logic; -- latching 
signal low_pwr      : std_logic := '0';  
signal pma_rem_lpbk : std_logic := '0'; 
signal pma_loc_lpbk : std_logic := '0'; 
signal tx_dis       : std_logic_vector(9 downto 0) := (others => '0'); 
signal tx_dis_g     : std_logic; -- global PMD TX disable
signal rx_sig_det   : std_logic_vector(9 downto 0); 
signal rx_sig_det_g : std_logic; -- global signal detect
signal bip_ercntr_r : std_logic_vector(20*16-1 downto 0);
signal lane_map_i   : std_logic_vector(20*5-1 downto 0);
signal lane_align_i : std_logic_vector(20-1 downto 0);
signal scr_bypass_r : std_logic_vector(1 downto 0); -- 
signal r333_rd      : std_logic; -- register rx.yy read
signal r333_dly     : std_logic_vector(3 downto 0); -- register rx.yy read
signal r333_rd_r    : std_logic; -- register rx.yy read registred
signal r333_clr_sync : std_logic; -- 
signal r333_clr     : std_logic; -- 
signal r301_rd      : std_logic; -- register rx.yy read
signal r301_dly     : std_logic_vector(3 downto 0);
signal r301_rd_r    : std_logic; -- register rx.yy read registred
signal r301_clr_sync : std_logic; -- 
signal r301_clr     : std_logic; -- 
signal r3200_rd     : std_logic_vector(19 downto 0); -- register r3.200..219 read 
signal r3200_dly    : std_logic_vector(20*4-1 downto 0);
signal r3200_rd_r   : std_logic_vector(19 downto 0); -- register r3.200..219 read 

function AND_REDUCE (D : in std_logic_vector) return std_logic is
variable tmp : std_logic;
begin
   tmp := D(0);
   for i in 1 to D'high loop
      tmp := tmp and D(i);
   end loop;
   return tmp;
end function AND_REDUCE;

function OR_REDUCE (D : in std_logic_vector) return std_logic is
variable tmp : std_logic;
begin
   tmp := D(0);
   for i in 1 to D'high loop
      tmp := tmp or D(i);
   end loop;
   return tmp;
end function OR_REDUCE;

begin

rx_sig_det(PMD_SIG_DET'high downto 0) <= PMD_SIG_DET;
rx_sig_det_g <= AND_REDUCE(PMD_SIG_DET);
pma_rxl_stat <= AND_REDUCE(PMA_RX_OK);
pma_fault    <= tx_fault or rx_fault;
tx_fault     <= '0';
rx_fault     <= (not pma_rxl_stat) or (not rx_sig_det_g);

UNUSED_FLAGS: if NUM_LANES < 20 generate
   rx_sig_det(rx_sig_det'high downto PMD_SIG_DET'high+1) <= (others => '0');
   lane_map_i(lane_map_i'high downto LANE_MAP'high+1) <= (others => '0');
   lane_align_i(lane_align_i'high downto LANE_ALIGN'high+1) <= (others => '0');
   bip_ercntr_r(bip_ercntr_r'high downto BIP_ERR_CNTRS'high+1) <= (others => '0');
   pcs_blk_lock(pcs_blk_lock'high downto BLK_LOCK'high+1) <= (others => '0');
end generate;

lane_map_i(LANE_MAP'high downto 0) <= LANE_MAP;
lane_align_i(LANE_ALIGN'high downto 0) <= LANE_ALIGN;
pcs_blk_lock(BLK_LOCK'high downto 0) <= BLK_LOCK;
pcs_blk_lock_g <= AND_REDUCE(BLK_LOCK) and ALGN_LOCKED; -- ALGN_LOCKED should be included according to 802.2ba
pcs_hi_ber     <= HI_BER;
pcs_rxl_stat   <= LINKSTATUS;

pcs_fault    <= (not pcs_blk_lock_g) or (pcs_hi_ber) or (not pcs_rxl_stat);

ADDR_DECODE: process(MI_ADDR,MI_BE,low_pwr,pma_rem_lpbk,pma_loc_lpbk,pma_fault,pma_rxl_stat,pma_mode,tx_fault,rx_fault,
                     tx_dis,tx_dis_g,rx_sig_det,rx_sig_det_g,pma_rst, pcs_rst, pcs_lpbk, pcs_low_pwr, pcs_fault, 
                     pcs_rxl_stat_l, pcs_rxl_stat, pcs_hi_ber, pcs_blk_lock_g, pcs_blk_lock_l, pcs_hi_ber_l, 
                     ber_count_r, blk_err_cntr_r, ber_count_l, blk_err_cntr_l, pcs_blk_lock, lane_align_i, 
                     bip_ercntr_r, lane_map_i, scr_bypass_r)
begin
   r333_rd  <= '0';
   r301_rd  <= '0';
   r3200_rd <= (others => '0');
   mi_drd_i <= (others => '0');
   if MI_ADDR(19 downto 16) = X"1" then -- select PMA (device 0x1)
      case MI_ADDR(5 downto 2) is
          when "0000" => -- PMA status1 & control 1        11       10:7     6:3         2            1             0
             mi_drd_i(15 downto  0) <= pma_rst & "010" & low_pwr & "0000" & "1001" & SPEED100G & pma_rem_lpbk & pma_loc_lpbk; -- r1.0
             mi_drd_i(31 downto 16) <= X"00" & pma_fault & "0000" & pma_rxl_stat & "10";                                      -- r1.1
          when "0001" => -- PMA device identifier
             mi_drd_i(15 downto  0) <= X"18EC"; -- r1.2
             mi_drd_i(31 downto 16) <= X"0000"; -- r1.3
          when "0010" => -- PMA devices in package 0 & speed ability
             mi_drd_i(15 downto  0) <= "000000" & GBASE100_ABLE & GBASE40_ABLE & X"00";  -- 40GE or 100GE capable  -- r1.4
             mi_drd_i(31 downto 16) <= "0000000000001010";  -- Devices in package (PMA & PCS) -- 0x000A -- r1.5
          when "0011" => -- PMA  control 2 & devices in package (high word)
             mi_drd_i(15 downto 0)  <= X"0000"; -- Devices in package -- r1.6
             mi_drd_i(31 downto 18) <= "000000000010"&SPEED100G&'0'; -- r1.7
             if (PMA_LANES = 4) then
                mi_drd_i(17) <= '1'; -- -ER4/-LR4/-SR4
             else
                mi_drd_i(17) <= '0'; -- -SR10/-CR10
             end if;
             mi_drd_i(16) <= pma_mode; -- /LR/SR/CR/KR
          when "0100" => -- PMA transmit disable & status 2 -- 0x...C
             mi_drd_i(15 downto 0)  <= "1011" & tx_fault & rx_fault & "0100000001"; -- Status 2 - r1.8
             mi_drd_i(31 downto 16) <= "00000" & tx_dis & tx_dis_g; -- PMD transmit disable - r1.9
          when "0101" => -- PMA/PMD extended ability registers & receive signal detect
             mi_drd_i(15 downto 0)  <= "00000" & rx_sig_det & rx_sig_det_g; -- Receive signal detect r1.10
             mi_drd_i(31 downto 16) <= "0000010000000000"; -- Extended ability register r1.11
          when "0110" => -- 10G-EPON PMA/PMD P2MP ability register  & 40G/100G PMA/PMD extended ability register
             mi_drd_i(15 downto 0)  <= X"0000";  -- 10G-EPON PMA/PMD P2MP ability register r1.12
             mi_drd_i(31 downto 16) <= "1000" & GBASE100_ABLE & GBASE100_ABLE & GBASE100_ABLE & GBASE100_ABLE & 
                                       "0000" & GBASE40_ABLE & GBASE40_ABLE & GBASE40_ABLE & '0'; -- 40G/100G PMA/PMD extended ability register r1.13
          when "0111" => -- PMA/PMD package identifier
             mi_drd_i(15 downto 0)  <= X"0000"; -- r1.14
             mi_drd_i(31 downto 16) <= X"0000"; -- r1.15 
          -- TODO:
          -- 1.1500    Test pattern ability 45.2.1.95
          -- 1.1501    PRBS pattern testing control 45.2.1.96
          -- 1.1510    Square wave testing control 45.2.1.97
          -- 1.1600 - 1609 PRBS Tx error counters, lane 0 through lane 9 45.2.1.98
          -- 1.1700 - 1709 PRBS Rx error counters, lane 0 through lane 9 45.2.1.99
          when others => 
             mi_drd_i <= (others => '0');
      end case;
   end if;
   --- PCS registers -------------------------------------------------------
   if MI_ADDR(19 downto 16) = X"3" then -- select PCS (device 0x3)
      if MI_ADDR(9 downto 7) = "000" then
         case MI_ADDR(6 downto 2) is
            when "00000" => -- 0x0000
               mi_drd_i(15 downto  0) <= pcs_rst & pcs_lpbk & "10" & pcs_low_pwr & "00001" & "0" & GBASE100_ABLE & GBASE40_ABLE & GBASE40_ABLE & scr_bypass_r;-- 3.0 PCS control 1 -- r3.0
               mi_drd_i(31 downto 16) <= X"00" & pcs_fault & "0000" & pcs_rxl_stat_l & "00"; -- 3.1 PCS status 1 -- r3.1
               r301_rd <= MI_BE(3) or MI_BE(2);
            when "00001" => -- 0x0004
               mi_drd_i(15 downto  0) <= X"18EC"; -- 3.2 PCS device identifier  -- r3.2
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.3 PCS device identifier  -- r3.3
            when "00010" => -- 0x0008
               mi_drd_i(15 downto  0) <= X"0000"; -- 3.4 PCS speed ability      -- r3.4
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.5 PCS devices in package -- r3.5
            when "00011" => -- 0x000C
               mi_drd_i(15 downto  0) <= X"0000"; -- 3.6 PCS devices in package -- r3.6
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.7 10G PCS control 2      -- r3.7
            when "00100" => -- 0x0010
               mi_drd_i(15 downto  0) <= X"0000"; -- 3.8 10G PCS status 2       -- r3.8
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.9 rsvd    
            when "00111" =>  -- 0x001C
               mi_drd_i(15 downto  0) <= X"0000"; -- 3.14 PCS package identifier 
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.15 PCS package identifier 
            when "01100" => -- 0x0030: 3.24 - 10GBASE-X PCS status & test control 
               mi_drd_i(15 downto  0) <= X"0000";
               mi_drd_i(31 downto 16) <= X"0000";
            when "10000" => -- 0x0040: 3.32, 3.33 GBASE-R and GBASE-T PCS status 1 & 2 -- 0x...40
               mi_drd_i(15 downto  0) <= "000" & pcs_rxl_stat & X"00" & "00" & pcs_hi_ber & pcs_blk_lock_g; -- 3.32  GBASE-R and GBASE-T PCS status 1
               mi_drd_i(31 downto 16) <= pcs_blk_lock_l & pcs_hi_ber_l & ber_count_r(5 downto 0) & blk_err_cntr_r(7 downto 0); -- 3.33  10GBASE-R and 10GBASE-T PCS status 2
               r333_rd <= MI_BE(3) or MI_BE(2);
            when "10001" => -- 3.34, 10GBASE-R PCS test pattern seed A -- 0x...44
               mi_drd_i(15 downto  0) <= X"0000";
               mi_drd_i(31 downto 16) <= X"0000";
            when "10010" => -- 3.36, 10GBASE-R PCS test pattern seed A -- 0x...48
               mi_drd_i(15 downto  0) <= X"0000";
               mi_drd_i(31 downto 16) <= X"0000";
            when "10011" => -- 3.38, 10GBASE-R PCS test pattern seed B -- 0x...4C
               mi_drd_i(15 downto  0) <= X"0000";
               mi_drd_i(31 downto 16) <= X"0000";
            when "10100" => -- 3.40, 10GBASE-R PCS test pattern seed B -- 0x...50
               mi_drd_i(15 downto  0) <= X"0000"; 
               mi_drd_i(31 downto 16) <= X"0000"; 
            when "10101" => -- 0x...54
               mi_drd_i(15 downto  0) <= X"0000"; -- 3.42  10GBASE-R PCS test pattern control      
               mi_drd_i(31 downto 16) <= X"0000"; -- 3.43  10GBASE-R PCS test pattern error counter
            when "10110" => -- 0x...58
               mi_drd_i(15 downto  0) <= ber_count_l; -- 3.44  BER high order counter(21:6)
               mi_drd_i(31 downto 16) <= "10" & blk_err_cntr_l; -- 3.45  Errored blocks high order counter
            when "11001" => -- 3.50, 3.51 Multi-lane BASE-R PCS block lock status 1 through 4 -- 0x...64
               mi_drd_i(15 downto  0) <= "000" & pcs_blk_lock_g & X"0" & pcs_blk_lock(7 downto 0);
               mi_drd_i(31 downto 16) <= X"0" & pcs_blk_lock(19 downto 8); 
            when "11010" => -- 3.52, 3.53 Multi-lane BASE-R PCS alignment status 1 through 4 -- 0x...68
               mi_drd_i(15 downto  0) <= X"00" & lane_align_i(7 downto 0); -- aligned flags for each lane
               mi_drd_i(31 downto 16) <= X"0" & lane_align_i(19 downto 8); -- aligned flags for each lane
            when others => 
               mi_drd_i <= (others => '0');
         end case;
      elsif MI_ADDR(9 downto 7) = "011" then -- xxx200 -- 0x...190
         case MI_ADDR(5 downto 2) is
            -- BIP error counters, lanes 0 through 19
            when "0100" => 
               mi_drd_i(15 downto  0) <= bip_ercntr_r(1*16-1 downto 0*16); -- 3.200 BIP error counters, lane 0
               mi_drd_i(31 downto 16) <= bip_ercntr_r(2*16-1 downto 1*16); -- 3.201 BIP error counters, lane 1
               r3200_rd(0) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(1) <= MI_BE(2) or MI_BE(3);
            when "0101" => 
               mi_drd_i(15 downto  0) <= bip_ercntr_r(3*16-1 downto 2*16); -- 3.202 BIP error counters, lane 2
               mi_drd_i(31 downto 16) <= bip_ercntr_r(4*16-1 downto 3*16); -- 3.203 BIP error counters, lane 3               
               r3200_rd(2) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(3) <= MI_BE(2) or MI_BE(3);
            when "0110" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(5*16-1 downto 4*16); -- 3.204 BIP error counters, lane 4
               mi_drd_i(31 downto 16) <= bip_ercntr_r(6*16-1 downto 5*16); -- 3.205 BIP error counters, lane 5
               r3200_rd(4) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(5) <= MI_BE(2) or MI_BE(3);
            when "0111" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(7*16-1 downto 6*16); -- 3.206 BIP error counters, lane 6
               mi_drd_i(31 downto 16) <= bip_ercntr_r(8*16-1 downto 7*16); -- 3.207 BIP error counters, lane 7
               r3200_rd(6) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(7) <= MI_BE(2) or MI_BE(3);
            when "1000" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(9*16-1  downto 8*16); -- 3.208 BIP error counters, lane 8
               mi_drd_i(31 downto 16) <= bip_ercntr_r(10*16-1 downto 9*16); -- 3.209 BIP error counters, lane 9
               r3200_rd(8) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(9) <= MI_BE(2) or MI_BE(3);
            when "1001" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(11*16-1 downto 10*16); -- 3.210 BIP error counters, lane 10
               mi_drd_i(31 downto 16) <= bip_ercntr_r(12*16-1 downto 11*16); -- 3.211 BIP error counters, lane 11
               r3200_rd(10) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(11) <= MI_BE(2) or MI_BE(3);
            when "1010" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(13*16-1 downto 12*16); -- 3.212 BIP error counters, lane 12
               mi_drd_i(31 downto 16) <= bip_ercntr_r(14*16-1 downto 13*16); -- 3.213 BIP error counters, lane 13
               r3200_rd(12) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(13) <= MI_BE(2) or MI_BE(3);
            when "1011" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(15*16-1 downto 14*16); -- 3.214 BIP error counters, lane 14
               mi_drd_i(31 downto 16) <= bip_ercntr_r(16*16-1 downto 15*16); -- 3.215 BIP error counters, lane 15
               r3200_rd(14) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(15) <= MI_BE(2) or MI_BE(3);
            when "1100" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(17*16-1 downto 16*16); -- 3.216 BIP error counters, lane 16
               mi_drd_i(31 downto 16) <= bip_ercntr_r(18*16-1 downto 17*16); -- 3.217 BIP error counters, lane 17
               r3200_rd(16) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(17) <= MI_BE(2) or MI_BE(3);
            when "1101" =>
               mi_drd_i(15 downto  0) <= bip_ercntr_r(19*16-1 downto 18*16); -- 3.218 BIP error counters, lane 18
               mi_drd_i(31 downto 16) <= bip_ercntr_r(20*16-1 downto 19*16); -- 3.219 BIP error counters, lane 19
               r3200_rd(18) <= MI_BE(0) or MI_BE(1); 
               r3200_rd(19) <= MI_BE(2) or MI_BE(3);
            when others => 
               mi_drd_i <= (others => '0');
         end case;
      elsif MI_ADDR(9 downto 7) = "110" then  -- 0x...320
         case MI_ADDR(6 downto 2) is
            when "01000" => -- 3.400 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(1*5-1 downto 0*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(2*5-1 downto 1*5);
            when "01001" => -- 3.402 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(3*5-1 downto 2*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(4*5-1 downto 3*5);
            when "01010" => -- 3.404 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(5*5-1 downto 4*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(6*5-1 downto 5*5);
            when "01011" => -- 3.406 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(7*5-1 downto 6*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(8*5-1 downto 7*5);
            when "01100" => -- 3.408 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(9*5-1 downto 8*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(10*5-1 downto 9*5);
            when "01101" => -- 3.410 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(11*5-1 downto 10*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(12*5-1 downto 11*5);
            when "01110" => -- 3.412 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(13*5-1 downto 12*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(14*5-1 downto 13*5);
            when "01111" => -- 3.414 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(15*5-1 downto 14*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(16*5-1 downto 15*5);
            when "10000" => -- 3.416 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(17*5-1 downto 16*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(18*5-1 downto 17*5);
            when "10001" => -- 3.418 PCS lane mapping registers, lanes 0 through 19
               mi_drd_i(15 downto  0) <= "00000000000" & lane_map_i(19*5-1 downto 18*5);
               mi_drd_i(31 downto 16) <= "00000000000" & lane_map_i(20*5-1 downto 19*5);
            when others => 
               mi_drd_i <= (others => '0');
         end case;
      end if;
   end if;
end process;

MI_READ_REGS: process(MI_CLK)
begin
   if MI_CLK'event and MI_CLK = '1' then
      MI_DRD    <= mi_drd_i;
      MI_DRDY   <= MI_RD;
      r301_dly(0) <= r301_rd and MI_RD;
      r333_dly(0) <= r333_rd and MI_RD;
      for i in 0 to 19 loop
         r3200_dly(i*4) <= r3200_rd(i) and MI_RD;
      end loop;
      -- Enlarge the pulse (shift)
      for j in r301_dly'high downto 1 loop
         r301_dly(j) <= r301_dly(j-1);
         r333_dly(j) <= r333_dly(j-1);
         for i in 0 to 19 loop
            r3200_dly(i*4+j) <= r3200_dly(i*4+j-1);
         end loop;
      end loop;
      -- Do OR reduction to generate 4x larger pulse
      r301_rd_r <= r301_dly(0) or r301_dly(1) or r301_dly(2) or r301_dly(3);
      r333_rd_r <= r333_dly(0) or r333_dly(1) or r333_dly(2) or r333_dly(3);
      for i in 0 to 19 loop
         r3200_rd_r(i) <= r3200_dly(i*4) or r3200_dly(i*4+1) or r3200_dly(i*4+2) or r3200_dly(i*4+3);
      end loop;
   end if;
end process;

MI_ARDY <= MI_RD or MI_WR;

MI_WRITE: process(MI_CLK)
begin
   if MI_CLK'event and MI_CLK = '1' then
      PMA_RETUNE <= '0';
      if RESET = '1' then 
         scr_bypass_r <= "00";
         pcs_lpbk     <= '0';
         pcs_rst      <= '0';
         pma_loc_lpbk <= '0';
         pma_rem_lpbk <= '0';
         pma_rst      <= '0';
         low_pwr      <= '0';
         tx_dis_g     <= '0';
         tx_dis       <= (others => '0');
      else
      --- PMA registers -------------------------------------------------------
      if (MI_ADDR(19 downto 16) = X"1") and (MI_WR = '1') then -- select PMA (device 0x1)
         if MI_ADDR(9 downto 7) = "000" then
            case MI_ADDR(6 downto 2) is -- 1.0 PMA control 1
               when "00000" => -- PMA control
                  if (MI_BE(0) = '1') then
                     pma_loc_lpbk  <= MI_DWR(0);
                     pma_rem_lpbk  <= MI_DWR(1);
                     low_pwr       <= MI_DWR(11);
                     -- PMA_PTRN_EN   <= MI_DWR(13); -- Pattern generator enable - for debugg only
                     PMA_RETUNE    <= MI_DWR(14);
                     pma_rst       <= MI_DWR(15);
                  end if;
               when "00011" => -- PMA  control 2 & devices in package (high word)
                  if (MI_BE(2) = '1') then
                     pma_mode  <= MI_DWR(16);
                  end if;
               when "00100" => -- PMA transmit disable 
                  if (MI_BE(2) = '1') then
                     tx_dis_g <= MI_DWR(16);
                     tx_dis   <= MI_DWR(26 downto 17);
                  end if;
               when others => null;
            end case;
         end if;
      --- PCS registers -------------------------------------------------------
      elsif (MI_ADDR(19 downto 16) = X"3") and (MI_WR = '1') then -- select PCS (device 0x3)
         if MI_ADDR(9 downto 7) = "000" then
            case MI_ADDR(6 downto 2) is -- 3.0 -- 3.0 PCS control 1
               when "00000" => 
                  if (MI_BE(0) = '1') then
                     scr_bypass_r <= MI_DWR(1 downto 0);
                     pcs_lpbk     <= MI_DWR(14);
                     pcs_rst      <= MI_DWR(15);
                  end if;
               when "10001" => -- 3.34, 10GBASE-R PCS test pattern seed A
               when "10010" => -- 3.36, 10GBASE-R PCS test pattern seed A
               when "10011" => -- 3.38, 10GBASE-R PCS test pattern seed B
               when "10100" => -- 3.40, 10GBASE-R PCS test pattern seed B
               -- TODO:
               -- 1.1501    PRBS pattern testing control 45.2.1.96
               -- 1.1510    Square wave testing control 45.2.1.97
               when others => null;
            end case;
         end if;
      end if;
      end if;
   end if;
end process;

RECLOCK_TO_MI: process(MI_CLK)
begin
   if MI_CLK'event and MI_CLK = '1' then
      bip_ercntr_r(BIP_ERR_CNTRS'high downto 0) <= BIP_ERR_CNTRS;
      ber_count_r    <= BER_COUNT;
      blk_err_cntr_r <= BLK_ERR_CNTR;
      
      if (r333_rd = '1') and (MI_RD = '1') then
         ber_count_l    <= ber_count_r(21 downto 6);
         blk_err_cntr_l <= blk_err_cntr_r(21 downto 8);
      end if;
     
   end if;
end process;

RECLOCK_TO_PCS: process(PCSCLK)
begin
   if PCSCLK'event and PCSCLK = '1' then
      -- Synchronize MI READ pulses to the PCSCLK domain
      r333_clr_sync <= r333_rd_r;
      r333_clr      <= r333_clr_sync;
      r301_clr_sync <= r301_rd_r;
      r301_clr      <= r301_clr_sync;
      -- PCS status latching flags --------------
      -- pcs block lock - latching low
      if (pcs_blk_lock_g = '0') then
         pcs_blk_lock_l <= '0';
      elsif (r333_clr = '1') then
         pcs_blk_lock_l <= '1';
      end if;
      -- pcs high BER - latching high
      if (pcs_hi_ber = '1') then
         pcs_hi_ber_l <= '1';
      elsif (r333_clr = '1') then
         pcs_hi_ber_l <= '0';
      end if;
      -- pcs RX link status - latching low
      if pcs_rxl_stat = '0' then
         pcs_rxl_stat_l <= '0';
      elsif (r301_clr) = '1' then
         pcs_rxl_stat_l <= '1';
      end if;
   end if;
end process;

RECLOCK_TO_PMA: process(PMACLK)
begin
   if PMACLK'event and PMACLK = '1' then
   end if;
end process;

BER_COUNT_CLR <= r333_rd_r; -- clear block sync BER count
BLK_ERR_CLR   <= r333_clr; -- clear decoder block error 
BIP_ERR_CLR   <= r3200_rd_r(BIP_ERR_CLR'high downto 0);
SCR_BYPASS    <= scr_bypass_r;
PCS_LPBCK     <= pcs_lpbk;
PCS_RESET     <= pcs_rst;
PMA_LPBCK     <= pma_loc_lpbk;
PMA_REM_LPBCK <= pma_rem_lpbk;
PMA_RESET     <= pma_rst;
PMA_LOPWR     <= low_pwr;
PMA_TX_DIS    <= tx_dis(PMA_TX_DIS'range) when tx_dis_g = '0' else
                 (others => '1');

end behavioral;

