-- gtz_ctle_tuning.vhd : CTLE tuning FSM
--               
-- Copyright (C) 2014 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


entity gtz_ctle_tuning is
   port (
        NEARENDLOOPBACKEN_I        : in std_logic_vector(7 downto 0);
        REDO_CHANNEL_TUNING_I      : in std_logic;
        OCTAL_CTLE_TUNING_RESET_I  : in std_logic;
        OCTAL_CTLE_INIT_ENABLE_I   : in std_logic := '0'; -- initialization only - no tuning, write predefined CTLE cefs
        OCTAL_CTLE_TUNING_ENABLE_I : in std_logic; -- do the complete tuning sequence
        CHAN_ID_I                  : in std_logic_vector(7 downto 0) := "11111111"; -- Channels to be tunned or initializated
        CHANNEL_CTLE_TUNING_DONE_O : out std_logic_vector(7 downto 0);
        CTLE_TUNING_STATE_O        : out std_logic_vector(7 downto 0);
        -- read_chan_id_o : out std_logic_vector(3 downto 0);
        -- DRP interface
        DRPCLK_I  : in std_logic;
        DRPDO_I   : in std_logic_vector(31 downto 0);
        DRPDI_O   : out std_logic_vector(31 downto 0);
        DRPADDR_O : out std_logic_vector(15 downto 0);
        DRPRDY_I  : in std_logic;
        DRPEN_O   : out std_logic;
        DRPWE_O   : out std_logic
   );
end gtz_ctle_tuning;

architecture behavioral of gtz_ctle_tuning is

type t_ctle_reg_addr is array (3 downto 0) of std_logic_vector(15 downto 0);
type t_coef_vals     is array (3 downto 0) of std_logic_vector(31 downto 0);
constant CTLE_REG : t_ctle_reg_addr := (X"0681", X"0481", X"0281", X"0081");
constant COEF_REG : t_ctle_reg_addr := (X"0667", X"0467", X"0267", X"0067");
constant COEFS    : t_coef_vals     := (X"020FEF67",
                                        X"020FFE68",
                                        X"020FFE61",
                                        X"020FDF69");

type T_TUNING_STATE is ( CH_SEL, 
                         CWRITE,CWRITE_WAIT,
                         INIT_0,INIT_0_WAIT,
                         INIT_1,INIT_1_WAIT,
                         INIT_2,INIT_2_WAIT,
                         INIT_3,INIT_3_WAIT,
                         TUNE_0,TUNE_0_WAIT,
                         TUNE_1,TUNE_1_WAIT,
                         TUNE_2,TUNE_2_WAIT,
                         FINISH_0,FINISH_0_WAIT,
                         FINISH_1,FINISH_1_WAIT,
                         FINISH_2,FINISH_2_WAIT,
                         DONE_TEST,DONE_TEST_WAIT,
                         CH_DONE, DONE);
                         
signal tuning_state : T_TUNING_STATE := CH_SEL;
signal current_chan : natural range 0 to 3;
signal loopback_r   : std_logic_vector(NEARENDLOOPBACKEN_I'range);
signal loopback_r2  : std_logic_vector(NEARENDLOOPBACKEN_I'range);
signal redo_channel_tuning_r  : std_logic;
signal redo_channel_tuning_r2 : std_logic;
signal drprdy_r    : std_logic;
signal drprdy_rise : std_logic;
signal do_tuning  : std_logic;
signal done_i      : std_logic_vector(CHANNEL_CTLE_TUNING_DONE_O'range);
     
begin

process(DRPCLK_I)
begin
   if DRPCLK_I'event and DRPCLK_I = '1' then
      loopback_r     <= NEARENDLOOPBACKEN_I;
      loopback_r2    <= loopback_r;
      
      redo_channel_tuning_r   <= REDO_CHANNEL_TUNING_I;
      redo_channel_tuning_r2  <= redo_channel_tuning_r;
      
      drprdy_r <= DRPRDY_I;
   
   end if;
end process;

drprdy_rise <= DRPRDY_I and (not drprdy_r);

-- Implements FSM from http://www.xilinx.com/support/answers/59038.html
TUNING_FSM : process(OCTAL_CTLE_TUNING_RESET_I,DRPCLK_I) 
begin
   if DRPCLK_I'event and DRPCLK_I = '1' then
      case tuning_state is
      
         -- Select a channel to be tunned
         when CH_SEL =>
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            do_tuning <= OCTAL_CTLE_TUNING_ENABLE_I and (not OCTAL_CTLE_INIT_ENABLE_I);
            --
            if (OCTAL_CTLE_INIT_ENABLE_I = '1') or (OCTAL_CTLE_TUNING_ENABLE_I = '1') then
               tuning_state <= INIT_0;
               if       (CHAN_ID_I(0) = '1') then current_chan <= 0; done_i(0) <= '0'; 
                  elsif (CHAN_ID_I(1) = '1') then current_chan <= 1; done_i(1) <= '0'; 
                  elsif (CHAN_ID_I(2) = '1') then current_chan <= 2; done_i(2) <= '0'; 
                  elsif (CHAN_ID_I(3) = '1') then current_chan <= 3; done_i(3) <= '0'; 
                  else  tuning_state <= DONE; -- nothing to do
               end if;
            else
               tuning_state <= CH_SEL;
            end if;
            
         ------------------------------------------------------------------
         -- CTLE tuning bypass : write coefficients only (without tuning)
         ------------------------------------------------------------------  
         when CWRITE => -- Write CTLE coef for current channel 
            DRPDI_O   <= COEFS(current_chan);
            DRPADDR_O <= COEF_REG(current_chan);
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= CWRITE_WAIT;
            
         when CWRITE_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               done_i(current_chan) <= '1';
               tuning_state <= FINISH_0;    
            else
               tuning_state <= CWRITE_WAIT;
            end if;                             
            
         ------------------------------------------------------------------
         -- CTLE tuning initialization
         ------------------------------------------------------------------  
         when INIT_0 => -- 
            DRPDI_O   <= X"00020021";
            DRPADDR_O <= X"C202";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';
            tuning_state <= INIT_0_WAIT;
            
         when INIT_0_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= INIT_1;
            else
               tuning_state <= INIT_0_WAIT;
            end if;
            
         when INIT_1 => -- 
            DRPDI_O   <= X"00000003";
            DRPADDR_O <= X"C207";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= INIT_1_WAIT;
            
         when INIT_1_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= INIT_2;
            else
               tuning_state <= INIT_1_WAIT;
            end if;
            
         when INIT_2 => -- 
            DRPDI_O   <= X"00000002";
            DRPADDR_O <= X"C207";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= INIT_2_WAIT;            
            
         when INIT_2_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= INIT_3;
            else
               tuning_state <= INIT_2_WAIT;
            end if;            
            
         when INIT_3 =>  -- Read address 0xc20c ...
            DRPADDR_O <= X"C20C";
            DRPEN_O   <= '1';
            DRPWE_O   <= '0';         
            tuning_state <= INIT_3_WAIT;

         when INIT_3_WAIT => -- ... till it reads 0x04f
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               if (DRPDO_I(9 downto 0) = "0001001111") then -- 0x04F
                  if (do_tuning = '1') then
                     tuning_state <= TUNE_0;
                  else
                     tuning_state <= CWRITE;
                  end if;
               else
                  tuning_state <= INIT_3;
               end if;  
            else
               tuning_state <= INIT_3_WAIT;
            end if;
            
         ------------------------------------------------------------------
         -- CTLE tuning --------------------------------------------------
         ------------------------------------------------------------------         
         when TUNE_0 => -- Clear DRP
            DRPDI_O   <= X"00000000";
            DRPADDR_O <= CTLE_REG(current_chan);
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= TUNE_0_WAIT;                     
            
         when TUNE_0_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= TUNE_1;
            else
               tuning_state <= TUNE_0_WAIT;
            end if;                     
            
         when TUNE_1 => -- Set tuning mode
            if loopback_r2(current_chan) = '1' then
               DRPDI_O   <= X"00000200";
            else
               DRPDI_O   <= X"00000202";
            end if;
            DRPADDR_O <= CTLE_REG(current_chan) + 1;
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= TUNE_1_WAIT;

         when TUNE_1_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= TUNE_2;
            else
               tuning_state <= TUNE_1_WAIT;
            end if;             
            
         when TUNE_2 => -- Start tuning 
            DRPDI_O   <= X"00000001";
            DRPADDR_O <= CTLE_REG(current_chan);
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= TUNE_2_WAIT;

         when TUNE_2_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= FINISH_0;
            else
               tuning_state <= TUNE_2_WAIT;
            end if;
            
         ------------------------------------------------------------------
         -- CTLE tuning finish
         ------------------------------------------------------------------     
         when FINISH_0 => -- 
            DRPDI_O   <= X"00000021";
            DRPADDR_O <= X"C202";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= FINISH_0_WAIT;
            
         when FINISH_0_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= FINISH_1;
            else
               tuning_state <= FINISH_0_WAIT;
            end if;
            
         when FINISH_1 => -- 
            DRPDI_O   <= X"00000003";
            DRPADDR_O <= X"C207";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= FINISH_1_WAIT;
            
         when FINISH_1_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               tuning_state <= FINISH_2;
            else
               tuning_state <= FINISH_1_WAIT;
            end if;
            
         when FINISH_2 => -- 
            DRPDI_O   <= X"00000002";
            DRPADDR_O <= X"C207";
            DRPEN_O   <= '1';
            DRPWE_O   <= '1';         
            tuning_state <= FINISH_2_WAIT;            
            
         when FINISH_2_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               if (do_tuning = '1') then
                  tuning_state <= DONE_TEST;
               else
                  tuning_state <= CH_DONE;
               end if;
            else
               tuning_state <= FINISH_2_WAIT;
            end if;
            
         when DONE_TEST => -- Monitor tuning done 
            DRPADDR_O <= CTLE_REG(current_chan);
            DRPEN_O   <= '1';
            DRPWE_O   <= '0';         
            tuning_state <= DONE_TEST_WAIT;

         when DONE_TEST_WAIT => -- 
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            if drprdy_rise = '1' then
               if ((DRPDO_I(3 downto 2) = "10") and (loopback_r2(current_chan) = '0')) or 
                  ((DRPDO_I(5 downto 4) = "10") and (loopback_r2(current_chan) = '1')) 
               then
                  done_i(current_chan) <= '1';
                  tuning_state <= CH_DONE;    
               else
                  tuning_state <= DONE_TEST; -- Repeat reading of the status register
               end if;  
            else
               tuning_state <= DONE_TEST_WAIT;
            end if;
            
         when CH_DONE =>
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';
            tuning_state <= CH_SEL; -- Continue with the next channel
            
         when DONE =>
            DRPEN_O   <= '0';
            DRPWE_O   <= '0';         
            if (redo_channel_tuning_r2 = '1') then
               tuning_state <= CH_SEL;
               current_chan <= 0;
               done_i       <= (others => '0');
            elsif (OCTAL_CTLE_TUNING_ENABLE_I = '1') then
               tuning_state <= CH_SEL;            
            else
               tuning_state <= DONE;
            end if;
            
      end case;
      
      if OCTAL_CTLE_TUNING_RESET_I = '1' then
         tuning_state  <= CH_SEL;
         done_i    <= (others => '0');
         DRPEN_O   <= '0';
         DRPWE_O   <= '0';
         current_chan <= 0;         
      end if;
      
   end if;
end process;

CHANNEL_CTLE_TUNING_DONE_O <= done_i;

with tuning_state select
   CTLE_TUNING_STATE_O <= X"00" when INIT_0,
                          X"01" when INIT_0_WAIT,
                          X"02" when INIT_1,
                          X"03" when INIT_1_WAIT,
                          X"04" when INIT_2,
                          X"05" when INIT_2_WAIT,
                          X"06" when INIT_3,
                          X"07" when INIT_3_WAIT,
                          X"08" when TUNE_0,
                          X"09" when TUNE_0_WAIT,                       
                          X"0a" when TUNE_1,
                          X"0b" when TUNE_1_WAIT,
                          X"0c" when TUNE_2,
                          X"0d" when TUNE_2_WAIT,
                          X"10" when FINISH_0,
                          X"11" when FINISH_0_WAIT,
                          X"12" when FINISH_1,
                          X"13" when FINISH_1_WAIT,
                          X"14" when FINISH_2,
                          X"15" when FINISH_2_WAIT,
                          X"0e" when DONE_TEST,
                          X"0f" when DONE_TEST_WAIT,
                          X"16" when DONE,
                          X"17" when CH_DONE,
                          X"18" when CWRITE,
                          X"19" when CWRITE_WAIT,                          
                          X"20" when CH_SEL,
                          X"FF" when others;

end behavioral;