--! count_dsp.vhd
--!
--! \file
--! \brief counter implemented with Virtex-7 DSP slices
--! \author Mario Kuka <xkukam00@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!

library ieee;
use ieee.std_logic_1164.all;
--use ieee.std_logic_arith.all;
--use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;

entity COUNT_DSP is
   generic (
      DATA_WIDTH   : integer;
      --! Input pipeline registers
      REG_IN       : integer := 0;
      --! Reset when MAX == P ( 0 => "NO_RESET", 1 => "RESET_MATCH") 
      AUTO_RESET   : integer := 0
   );
   port (
      --! Clock input
      CLK      : in  std_logic;
      --! Enable input
      ENABLE   : in  std_logic;
      --! Reset input
      RESET    : in  std_logic;
      --! Data input 
      A        : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      --! Data input (must MAX % A = 0), Maximum value 
      MAX      : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      --! Data output
      P        : out std_logic_vector(DATA_WIDTH-1 downto 0)
   );
end COUNT_DSP;

architecture structural of COUNT_DSP is
   signal enable_p         :std_logic_vector(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1))) downto 0);
   signal a_p              :std_logic_vector(DATA_WIDTH-1 downto 0);
   signal max_p            :std_logic_vector(DATA_WIDTH-1 downto 0);

   signal reset_all        : std_logic;
   signal in_out_Vector    : std_logic_vector ((DATA_WIDTH/48) downto 0);
   signal pattern_all      : std_logic_vector (((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)) - 1) downto 0);
   signal temp             : std_logic;

begin  
   --! generate input registers          
   GEN_IN_REG: if(REG_IN = 1 AND DATA_WIDTH > 48) generate
      process(CLK)
      begin
         if(CLK'event) and (CLK='1') then
            if (RESET='1') then
               a_p(DATA_WIDTH-1 downto 48) <= (others => '0');
            elsif (ENABLE = '1') then 
               a_p(DATA_WIDTH-1 downto 48) <= A(DATA_WIDTH-1 downto 48);
            end if;
         end if;
      end process;

      --! when auto reset is ON
      GEN_MAX_RESET: if(AUTO_RESET = 1) generate
         process(CLK)
         begin
            if(CLK'event) and (CLK='1') then
               if (RESET='1') then
                  max_p(DATA_WIDTH-1 downto 48) <= (others => '0');
               elsif (ENABLE = '1') then 
                  max_p(DATA_WIDTH-1 downto 48) <= MAX(DATA_WIDTH-1 downto 48);
               end if;
            end if;
         end process;
      end generate; 

      --! when auto reset is OFF
      GEN_MAX_NO_RESET: if(AUTO_RESET = 0) generate
         max_p <= MAX;
      end generate; 
   end generate;

   GEN_NO_IN_REG: if(REG_IN = 0 OR DATA_WIDTH <= 48) generate
      a_p <= A;
      max_p <= MAX;
   end generate;

   enable_p(0) <= ENABLE;

   GEN_REG_EN: for I in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)))-1 generate 
      process(CLK)
         begin
            if(CLK'event) and (CLK='1') then
               if (RESET='1') then
                  enable_p(I+1) <= '0';
               else 
                  enable_p(I+1) <= enable_p(I);
               end if;
            end if;
         end process;
   end generate;
     
   --! generate pattern logic
   GEN_RST_LOGIC: if (AUTO_RESET = 1) generate
      process(pattern_all)
         variable o : std_logic;
      begin
         o := '1';
         for I in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)) - 1) loop
            o := o and pattern_all(I);
         end loop;
           temp <= o;
      end process;

      GEN_RST_LOGIC: if(DATA_WIDTH > 48) generate
         process(temp, RESET)
         begin 
            if (temp = '1' OR (RESET = '1')) then
               reset_all <= '1';
            else
               reset_all <= '0';
            end if;
         end process;          
      end generate;
   end generate;

   GEN_RTS_NO_LOGIC: if(DATA_WIDTH <= 48 or AUTO_RESET = 0) generate
      reset_all <= RESET;             
   end generate;

   --! first carryin value
   in_out_Vector(0) <= '0';

   GEN_COUNT_DIV: for I in 0 to (DATA_WIDTH/48)-1 generate 
      --! DSP for only 48 bit
      GEN_ONE_DSP_48: if(DATA_WIDTH = 48) generate
         GEN_RESET_OFF: if(AUTO_RESET = 0) generate
            COUNT48_inst: entity work.COUNT48
               generic map(
                  REG_IN  => REG_IN,
                  AUTO_RESET => "NO_RESET"
               )   
               port map (
                  CLK => CLK,
                  ENABLE => enable_p(0), 
                  ENABLE_IN => enable_p(0),
                  RESET => reset_all, 
                  RESET_OUT => reset_all,
                  CAS_CARRY_IN => '0',
                  A => a_p(47+I*48 downto 0+I*48),
                  MAX => max_p(47+I*48 downto 0+I*48),
                  P => P(47+I*48 downto 0+I*48),
                  CAS_CARRY_OUT => open,
                  PATTERN => open         
               );
         end generate;

         GEN_RESET_ON: if(AUTO_RESET = 1) generate
            COUNT48_inst: entity work.COUNT48
               generic map(
                  REG_IN  => REG_IN,
                  AUTO_RESET => "RESET_MATCH"
               )   
               port map (
                  CLK => CLK,
                  ENABLE => enable_p(0), 
                  ENABLE_IN => enable_p(0),
                  RESET => reset_all, 
                  RESET_OUT => reset_all,
                  CAS_CARRY_IN => '0',
                  A => a_p(47+I*48 downto 0+I*48),
                  MAX => max_p(47+I*48 downto 0+I*48),
                  P => P(47+I*48 downto 0+I*48),
                  CAS_CARRY_OUT => open,
                  PATTERN => open         
            );
         end generate;

      end generate;
       
      --! generate first DPS when DATA_WIDTH > 48 
      GEN_FIRST_DSP: if(I = 0 and DATA_WIDTH /= 48) generate
         signal pattern_pom  : std_logic_vector(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1) downto 0);
                   
         type array_pom_out_t is array (0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1)) of std_logic_vector(47 downto 0);
         signal array_pom_out : array_pom_out_t;
      begin
         --! generate output registers
         GEN_REG_OUT: for I in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 2) generate
            process(CLK)
            begin
               if(CLK'event) and (CLK='1') then
                  if (reset_all='1') then
                     array_pom_out(I + 1) <= (others => '0');
                  elsif (enable_p(2+I) = '1') then 
                     array_pom_out(I + 1) <= array_pom_out(I);
                  end if;
               end if;
            end process;
         end generate;

         --! generate registers for DSP pattern output 
         GEN_RST_LOGIC: if (AUTO_RESET = 1) generate
            GEN_REG_OUT: for I in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 2) generate
               process(CLK)
               begin
                  if(CLK'event) and (CLK='1') then
                     if (reset_all='1') then
                        pattern_pom(I + 1) <= '0';
                     elsif (enable_p(2+I) = '1') then 
                        pattern_pom(I + 1) <= pattern_pom(I);
                     end if;
                  end if;
               end process;
            end generate;
         end generate;

         P(47+I*48 downto 0+I*48) <= array_pom_out((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1);
         pattern_all(I) <= pattern_pom(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1)); 
                                    
         COUNT48_inst: entity work.COUNT48
         generic map(
            REG_IN  => REG_IN * 3,
            AUTO_RESET => "NO_RESET"
         )   
         port map (
            CLK => CLK,
            ENABLE => enable_p(1),
            ENABLE_IN => enable_p(0),
            RESET => RESET, 
            RESET_OUT => reset_all,
            CAS_CARRY_IN => in_out_vector(0),
            A => A(47+I*48 downto 0+I*48),
            MAX => MAX(47+I*48 downto 0+I*48),
            P => array_pom_out(0),
            CAS_CARRY_OUT => in_out_vector(1),
            PATTERN => pattern_pom(0)         
         );
      end generate;

      --! generate DSPs between first and last
      GEN_DSP_BETWEEN: if((I > 0) and ((I < ((DATA_WIDTH/48) -1)) OR ((DATA_WIDTH mod 48) > 0))) generate
         signal pattern_pom  : std_logic_vector(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1 - I) downto 0);
              
         type array_pom_out_t is array (0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 1 - I)) of std_logic_vector(47 downto 0);
         signal array_pom_out : array_pom_out_t;
           
         type array_pom_in_t is array (0 to I - 1) of std_logic_vector(47 downto 0);
         signal array_pom_in_A   : array_pom_in_t;
         signal array_pom_in_MAX : array_pom_in_t;
      begin
            
         GEN_REG_OUT: for Y in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 2 - I) generate
            process(CLK)
            begin
               if(CLK'event) and (CLK='1') then
                  if (reset_all='1') then
                     array_pom_out(Y + 1) <= (others => '0');
                  elsif (enable_p(I+2+Y) = '1') then 
                     array_pom_out(Y + 1) <= array_pom_out(Y);
                  end if;
               end if;
            end process;
         end generate;

         GEN_RST_LOGIC: if (AUTO_RESET = 1) generate
            GEN_REG_OUT: for Y in 0 to ((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - 2 - I) generate
               process(CLK)
               begin
                  if(CLK'event) and (CLK='1') then
                     if (reset_all='1') then
                        pattern_pom(Y + 1) <= '0';
                     elsif (enable_p(I+2+Y) = '1') then 
                        pattern_pom(Y + 1) <= pattern_pom(Y);
                     end if;
                  end if;
               end process;
            end generate;
         end generate;
                 
         --! generate input registers for signal A
         GEN_REG_IN_A: for Y in 0 to (I - 2) generate
            process(CLK)
            begin
               if(CLK'event) and (CLK='1') then
                  if (reset_all='1') then
                     array_pom_in_A(Y + 1) <= (others => '0');
                  elsif (enable_p(1+Y) = '1') then 
                     array_pom_in_A(Y + 1) <= array_pom_in_A(Y);
                  end if;
               end if;
            end process;
         end generate;
                   
         --! generate input registers for signal MAX
         GEN_REG_IN_MAX: if (AUTO_RESET = 1) generate
            GEN_REG_IN: for Y in 0 to (I - 2) generate
               process(CLK)
               begin
                  if(CLK'event) and (CLK='1') then
                     if (reset_all='1') then
                        array_pom_in_MAX(Y + 1) <= (others => '0');
                     elsif (enable_p(1+Y) = '1') then 
                        array_pom_in_MAX(Y + 1) <= array_pom_in_MAX(Y);
                     end if;
                  end if;
               end process;
            end generate;
         end generate;

         GEN_CON_MAX: if (AUTO_RESET = 0) generate
            array_pom_in_MAX(I - 1) <= array_pom_in_MAX(0);
         end generate;
            
         array_pom_in_A(0) <= a_p(47+I*48 downto 0+I*48);
         array_pom_in_MAX(0) <= max_p(47+I*48 downto 0+I*48);

         P(47+I*48 downto 0+I*48) <= array_pom_out((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - I - 1);
         pattern_all(I) <= pattern_pom(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) +1)) - I - 1)); 

         COUNT48_inst: entity work.COUNT48
         generic map(
            REG_IN  => 3,
            AUTO_RESET => "NO_RESET"
         )   
         port map (
            CLK => CLK,
            ENABLE => enable_p(I + 1), 
            ENABLE_IN => enable_p(I),
            RESET => reset_all, 
            RESET_OUT => reset_all,
            CAS_CARRY_IN => in_out_vector(I),
            A => array_pom_in_A(I - 1),
            MAX => array_pom_in_MAX(I - 1),
            P => array_pom_out(0),
            CAS_CARRY_OUT => in_out_vector(I + 1),
            PATTERN => pattern_pom(0)            
         );
      end generate;
       
      --! generate last DSP when DATA_WIDTH mod 48 is 0
      GEN_DSP_LAST: if((I > 0) and I = ((DATA_WIDTH/48) - 1) and (DATA_WIDTH mod 48) = 0) generate
         type array_pom_in_t is array (0 to I - 1) of std_logic_vector(47 downto 0);
         signal array_pom_in_A   : array_pom_in_t;
         signal array_pom_in_MAX : array_pom_in_t;
      begin
         GEN_REG_IN_A: for Y in 0 to (I - 2) generate
            process(CLK)
            begin
               if(CLK'event) and (CLK='1') then
                  if (reset_all='1') then
                     array_pom_in_A(Y + 1) <= (others => '0');
                  elsif (enable_p(1+Y) = '1') then 
                     array_pom_in_A(Y + 1) <= array_pom_in_A(Y);
                  end if;
               end if;
            end process;
         end generate;
                   
         GEN_REG_IN_MAX: if (AUTO_RESET = 1) generate
            GEN_REG_IN: for Y in 0 to (I - 2) generate
               process(CLK)
               begin
                  if(CLK'event) and (CLK='1') then
                     if (reset_all='1') then
                        array_pom_in_MAX(Y + 1) <= (others => '0');
                     elsif (enable_p(1+Y) = '1') then 
                        array_pom_in_MAX(Y + 1) <= array_pom_in_MAX(Y);
                     end if;
                  end if;
               end process;
            end generate;
         end generate;

         array_pom_in_A(0) <= a_p(47+I*48 downto 0+I*48);
         array_pom_in_MAX(0) <= max_p(47+I*48 downto 0+I*48);

         COUNT48_inst: entity work.COUNT48
         generic map(
            REG_IN  => 3,
            AUTO_RESET => "NO_RESET"
         )   
         port map (
            CLK => CLK,
            ENABLE => enable_p(I + 1), 
            ENABLE_IN => enable_p(I),
            RESET => reset_all, 
            RESET_OUT => reset_all,
            CAS_CARRY_IN => in_out_vector(I),
            A => array_pom_in_A(I - 1),
            MAX => array_pom_in_MAX(I - 1),
            P => P(47+I*48 downto 0+I*48),
            CAS_CARRY_OUT => open,
            PATTERN    => pattern_all(I)         
         );
      end generate;
   end generate;

   --! generate last DSP when DATA_WIDTH mod 48 /= 0 and generate one DSP when DATA_WIDTH < 48
   GEN_COUNT_MOD: if (DATA_WIDTH mod 48 > 0) generate
      signal Amod : std_logic_vector(47 downto 0);
      signal Bmod : std_logic_vector(47 downto 0);
      signal Pmod : std_logic_vector(47 downto 0);

      type array_pom_in_t is array (0 to (DATA_WIDTH/48) - 1) of std_logic_vector((DATA_WIDTH mod 48)-1 downto 0);
      signal array_pom_in_A   : array_pom_in_t;
      signal array_pom_in_MAX : array_pom_in_t;
   begin   
      GEN_REG_IN_A: for Y in 0 to ((DATA_WIDTH/48) - 2) generate
         process(CLK)
         begin
            if(CLK'event) and (CLK='1') then
               if (reset_all='1') then
                  array_pom_in_A(Y + 1) <= (others => '0');
               elsif (enable_p(1+Y) = '1') then 
                  array_pom_in_A(Y + 1) <= array_pom_in_A(Y);
               end if;
            end if;
         end process;
      end generate;
   
      GEN_REG_IN_MAX: if (AUTO_RESET = 1) generate
         GEN_REG_IN: for Y in 0 to ((DATA_WIDTH/48) - 2) generate
            process(CLK)
            begin
               if(CLK'event) and (CLK='1') then
                  if (reset_all='1') then
                     array_pom_in_MAX(Y + 1) <= (others => '0');
                  elsif (enable_p(1+Y) = '1') then 
                     array_pom_in_MAX(Y + 1) <= array_pom_in_MAX(Y);
                  end if;
               end if;
            end process;
         end generate;
      end generate;
        
      GEN_LAST_DSP: if(DATA_WIDTH > 48) generate
         array_pom_in_A(0) <= a_p(a_p'LENGTH-1 downto a_p'LENGTH-1-(DATA_WIDTH mod 48)+1);
         array_pom_in_MAX(0) <= max_p(max_p'LENGTH-1 downto max_p'LENGTH-1-(DATA_WIDTH mod 48)+1);
 
         Amod((DATA_WIDTH mod 48)-1 downto 0) <= array_pom_in_A(DATA_WIDTH/48 - 1);
         Bmod((DATA_WIDTH mod 48)-1 downto 0) <= array_pom_in_MAX(DATA_WIDTH/48 - 1);
      end generate;

      GEN_ONE_DSP: if(DATA_WIDTH < 48) generate
         Amod((DATA_WIDTH mod 48)-1 downto 0) <= a_p(a_p'LENGTH-1 downto a_p'LENGTH-1-(DATA_WIDTH mod 48)+1);
         Bmod((DATA_WIDTH mod 48)-1 downto 0) <= max_p(max_p'LENGTH-1 downto max_p'LENGTH-1-(DATA_WIDTH mod 48)+1);
      end generate;

      Amod(47 downto (DATA_WIDTH mod 48)) <= (others => '0');
      Bmod(47 downto (DATA_WIDTH mod 48)) <= (others => '0');

      --! when DATA_WIDTH < 48
      GEN_DSP_ONE: if(DATA_WIDTH < 48) generate
         GEN_RESET_OFF: if(AUTO_RESET = 0) generate
            COUNT48_inst: entity work.COUNT48
            generic map(
               REG_IN  => REG_IN,
               AUTO_RESET => "NO_RESET"
            )   
            port map (
               CLK => CLK,
               ENABLE => enable_p(0), 
               ENABLE_IN => enable_p(0),
               RESET => reset_all, 
               RESET_OUT => reset_all,
               CAS_CARRY_IN => '0',
               A =>  Amod,
               MAX => Bmod,
               P => Pmod,
               CAS_CARRY_OUT => open,
               PATTERN => open         
            );
         end generate;

         GEN_RESET_ON: if(AUTO_RESET = 1) generate
            COUNT48_inst: entity work.COUNT48
            generic map(
               REG_IN  => REG_IN,
               AUTO_RESET => "RESET_MATCH"
            )   
            port map (
               CLK => CLK,
               ENABLE => enable_p(0), 
               ENABLE_IN => enable_p(0),
               RESET => reset_all, 
               RESET_OUT => reset_all,
               CAS_CARRY_IN => '0',
               A =>  Amod,
               MAX => Bmod,
               P => Pmod,
               CAS_CARRY_OUT => open,
               PATTERN => open         
            );
         end generate;
      end generate;

      --! last DSP
      GEN_DSP_LAST: if(DATA_WIDTH > 48) generate
         COUNT48_inst: entity work.COUNT48
         generic map(
            REG_IN  => 3,
            AUTO_RESET => "NO_RESET"
         )   
         port map (
            CLK => CLK,
            ENABLE => enable_p(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)))), 
            ENABLE_IN => enable_p(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)))-1),
            RESET => reset_all, 
            RESET_OUT => reset_all,
            CAS_CARRY_IN => in_out_Vector(DATA_WIDTH/48),
            A => Amod,
            MAX => Bmod,
            P => Pmod,
            CAS_CARRY_OUT => open,
            PATTERN => pattern_all(((DATA_WIDTH/48) + (1 mod ((DATA_WIDTH mod 48) + 1)) - 1))         
         );
      end generate;

      P(P'LENGTH-1 downto P'LENGTH-1-(DATA_WIDTH mod 48)+1) <= Pmod((DATA_WIDTH mod 48)-1 downto 0); 

   end generate;
end architecture;
