-- buf.vhd: Buf unit of CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use work.ibuf_pkg.all;
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture cgmii_ibuf_buf_arch of cgmii_ibuf_buf is

   -- Constants declaration ---------------------------------------------------

   constant HFIFO_WIDTH       : integer := HDR_WIDTH + log2(HDR_WIDTH/8) + 2;
   constant HFIFO_DATA_POS_H  : integer := HFIFO_WIDTH - 1;
   constant HFIFO_DATA_POS_L  : integer := HFIFO_WIDTH - HDR_WIDTH;
   constant HFIFO_SOP_POS     : integer := HFIFO_DATA_POS_L - 1;
   constant HFIFO_EOP_POS     : integer := HFIFO_SOP_POS - 1;
   constant HFIFO_REM_POS_H   : integer := HFIFO_EOP_POS - 1;
   constant HFIFO_REM_POS_L   : integer := 0;


   -- Signals declaration -----------------------------------------------------

   -- Data input signals
   signal rx_data_in        : std_logic_vector(511 downto 0);
   signal rx_sop_in         : std_logic;
   signal rx_sop_pos_in     : std_logic_vector(2 downto 0);
   signal rx_eop_in         : std_logic;
   signal rx_eop_pos_in     : std_logic_vector(5 downto 0);
     
   -- Statistics input signals
   signal rx_mac_err_in     : std_logic;
   signal rx_mintu_err_in   : std_logic;
   signal rx_mtu_err_in     : std_logic;
   signal rx_sau_err_in     : std_logic;
   signal rx_crc_err_in     : std_logic;
   signal rx_frame_err_in   : std_logic;
   signal rx_frame_len_in   : std_logic_vector(15 downto 0);

   -- Data and header destination ready signals
   signal tx_dst_rdy_in     : std_logic;  
   signal tx_hdst_rdy_n_in  : std_logic;

   -- PACODAG interface signals   
   signal ctrl_data_in      : std_logic_vector(HDR_WIDTH-1 downto 0);
   signal ctrl_sop_n_in     : std_logic;
   signal ctrl_eop_n_in     : std_logic;
   signal ctrl_rem_in       : std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   signal ctrl_src_rdy_n_in : std_logic;
   signal ctrl_rdy_in       : std_logic;

   -- Registers interface signals   
   signal ibuf_en_in        : std_logic;
   signal error_mask_in     : std_logic_vector(4 downto 0);

   -- Error controls masking and corresponding discarding
   signal err_vector        : std_logic_vector(5 downto 0);
   signal err_mask          : std_logic_vector(5 downto 0);
   signal err_masked        : std_logic_vector(5 downto 0);
   signal err_discard       : std_logic;

   -- 1st pipeline stage registers
   signal reg1_rx_data      : std_logic_vector(511 downto 0);
   signal reg1_rx_sop       : std_logic := '0';
   signal reg1_rx_sop_pos   : std_logic_vector(2 downto 0);
   signal reg1_rx_eop       : std_logic := '0';
   signal reg1_rx_eop_pos   : std_logic_vector(5 downto 0);
   signal reg1_rx_mac_err   : std_logic;
   signal reg1_rx_mintu_err : std_logic;
   signal reg1_rx_mtu_err   : std_logic;
   signal reg1_rx_crc_err   : std_logic;
   signal reg1_rx_frame_err : std_logic;
   signal reg1_rx_frame_len : std_logic_vector(15 downto 0);
   signal reg1_err_discard  : std_logic := '0';

   -- FSM related signals
   signal fsm_dfifo_release           : std_logic;
   signal fsm_dfifo_ovf               : std_logic;
   signal fsm_ctrl_sop                : std_logic;
   signal fsm_ctrl_stat_dv            : std_logic;
   signal fsm_ctrl_ovf                : std_logic;
   signal fsm_status                  : std_logic_vector(1 downto 0);
   signal fsm_frame_received          : std_logic;
   signal fsm_frame_discarded         : std_logic;
   signal fsm_frame_discarded_buf_ovf : std_logic;
   
   -- DFIFO signals
   signal dfifo_rx_dst_rdy       : std_logic;
   signal dfifo_full             : std_logic;
   signal dfifo_tx_data          : std_logic_vector(511 downto 0);
   signal dfifo_tx_sop_pos       : std_logic_vector(2 downto 0);
   signal dfifo_tx_eop_pos       : std_logic_vector(5 downto 0);
   signal dfifo_tx_sop           : std_logic;
   signal dfifo_tx_eop           : std_logic;
   signal dfifo_tx_src_rdy       : std_logic;
   
   -- HFIFO signals
   signal hfifo_wr               : std_logic;
   signal hfifo_data_in          : std_logic_vector(HFIFO_WIDTH-1 downto 0);
   signal hfifo_full             : std_logic;
   signal hfifo_rd               : std_logic;
   signal hfifo_data_out         : std_logic_vector(HFIFO_WIDTH-1 downto 0);
   signal hfifo_data_vld         : std_logic;
   signal hfifo_empty            : std_logic;
   
   -- Output header/footer data
   signal hfifo_tx_data          : std_logic_vector(HDR_WIDTH-1 downto 0);
   signal hfifo_tx_sop_n         : std_logic;
   signal hfifo_tx_eop_n         : std_logic;
   signal hfifo_tx_rem           : std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   signal hfifo_tx_src_rdy_n     : std_logic;
   
   -- Output status information
   signal sync_status_flu        : std_logic_vector(4 downto 0);
   signal sync_status_cgmii      : std_logic_vector(4 downto 0);
   signal status_out             : std_logic_vector(16 downto 0);


begin

   -- -------------------------------------------------------------------------
   --                        Input ports connection
   -- -------------------------------------------------------------------------
   
   -- Data input
   rx_data_in        <= RX_DATA;
   rx_sop_in         <= RX_SOP;
   rx_sop_pos_in     <= RX_SOP_POS;
   rx_eop_in         <= RX_EOP;
   rx_eop_pos_in     <= RX_EOP_POS;
  
   -- Statistics input
   rx_mac_err_in     <= RX_MAC_ERR;
   rx_mintu_err_in   <= RX_MINTU_ERR;
   rx_mtu_err_in     <= RX_MTU_ERR;
   rx_sau_err_in     <= RX_SAU_ERR;
   rx_crc_err_in     <= RX_CRC_ERR;
   rx_frame_err_in   <= RX_FRAME_ERR;
   rx_frame_len_in   <= RX_FRAME_LEN;

   -- Data and header destination ready inputs
   tx_dst_rdy_in     <= TX_DST_RDY;
   tx_hdst_rdy_n_in  <= TX_HDST_RDY_N;
   
   -- PACODAG interface
   ctrl_data_in      <= CTRL_DATA;
   ctrl_sop_n_in     <= CTRL_SOP_N;
   ctrl_eop_n_in     <= CTRL_EOP_N;
   ctrl_rem_in       <= CTRL_REM;
   ctrl_src_rdy_n_in <= CTRL_SRC_RDY_N;
   ctrl_rdy_in       <= CTRL_RDY;
   
   -- Registers interface
   ibuf_en_in        <= IBUF_EN;
   error_mask_in     <= ERROR_MASK;
   

   -- -------------------------------------------------------------------------
   --                        Error controls masking
   -- -------------------------------------------------------------------------
  
   -- Masking of errors -------------------------------------------------------

   -- Compose all error signals into one vector
   err_vector <= rx_sau_err_in & rx_mac_err_in & rx_mtu_err_in &
                 rx_mintu_err_in & rx_crc_err_in & rx_frame_err_in;
                 
   -- Compose error mask vector
   err_mask <= '1' & error_mask_in;
   
   -- Do masking of errors
   err_masked <= err_vector and err_mask;

   
   -- Deriving an information whether to discard or not -----------------------

   -- Active when an error occurs and EOP is active
   err_discard <= '1' when ((err_masked /= 0) and (rx_eop_in = '1')) else '0';
 

   -- -------------------------------------------------------------------------
   --                         1st pipeline stage
   -- -------------------------------------------------------------------------
 
   -- Registers reg1_rx_data, reg1_rx_sop_pos and reg1_rx_eop_pos
   regs1_rx_data_sop_eop_pos_p : process(CGMII_CLK)
   begin
      if (CGMII_CLK'event and CGMII_CLK = '1') then
         reg1_rx_data    <= rx_data_in;
         reg1_rx_sop_pos <= rx_sop_pos_in;
         reg1_rx_eop_pos <= rx_eop_pos_in;
      end if;
   end process regs1_rx_data_sop_eop_pos_p;
   
   -- Registers reg1_rx_sop and reg1_rx_eop
   regs1_rx_sop_eop_p : process(CGMII_CLK)
   begin
      if (CGMII_CLK'event and CGMII_CLK = '1') then
         if (not RESET_BY_INIT and CGMII_RESET = '1') then
            reg1_rx_sop <= '0';
            reg1_rx_eop <= '0';
         else
            reg1_rx_sop <= rx_sop_in;
            reg1_rx_eop <= rx_eop_in;
         end if;
      end if;
   end process regs1_rx_sop_eop_p;
 
   -- Registers for storing input statistic values
   regs1_stats_p : process(CGMII_CLK)
   begin
      if (CGMII_CLK'event and CGMII_CLK = '1') then
         reg1_rx_mac_err    <= rx_mac_err_in;
         reg1_rx_mintu_err  <= rx_mintu_err_in;
         reg1_rx_mtu_err    <= rx_mtu_err_in;
         reg1_rx_crc_err    <= rx_crc_err_in;
         reg1_rx_frame_err  <= rx_frame_err_in;
         reg1_rx_frame_len  <= rx_frame_len_in;
      end if;
   end process regs1_stats_p;

   -- Register for storing information about discarding
   reg1_err_discard_p : process(CGMII_CLK)
   begin
      if (CGMII_CLK'event and CGMII_CLK = '1') then
         if (not RESET_BY_INIT and CGMII_RESET = '1') then
            reg1_err_discard <= '0';
         else
            reg1_err_discard <= err_discard;
         end if;
      end if;
   end process reg1_err_discard_p;
   

   -- -------------------------------------------------------------------------
   --                                FSM
   -- -------------------------------------------------------------------------
  
   -- FSM instantiation
   fsm_i : entity work.fsm
   generic map(
      RESET_BY_INIT           => RESET_BY_INIT
   )
   port map(
      CLK                     => CGMII_CLK,
      RESET                   => CGMII_RESET,

      -- Inputs
      IBUF_EN                 => ibuf_en_in,

      RX_SOP                  => reg1_rx_sop,
      RX_SOP_POS              => reg1_rx_sop_pos,
      RX_EOP                  => reg1_rx_eop,
      RX_EOP_POS              => reg1_rx_eop_pos,

      DISCARD                 => reg1_err_discard,

      DFIFO_FULL              => dfifo_full,
      HFIFO_FULL	            => hfifo_full,

      CTRL_RDY                => ctrl_rdy_in,

      -- Outputs
      DFIFO_RELEASE           => fsm_dfifo_release,
      DFIFO_OVF               => fsm_dfifo_ovf,

      CTRL_SOP                => fsm_ctrl_sop,
      CTRL_STAT_DV            => fsm_ctrl_stat_dv,
      CTRL_OVF                => fsm_ctrl_ovf,
      
      STATUS                  => fsm_status,

      FRAME_RECEIVED          => fsm_frame_received,
      FRAME_DISCARDED         => fsm_frame_discarded,
      FRAME_DISCARDED_BUF_OVF => fsm_frame_discarded_buf_ovf
   );

   
   -- -------------------------------------------------------------------------
   --                                  DFIFO
   -- -------------------------------------------------------------------------
   
   -- DFIFO instantiation
   dfifo_i: entity work.flu_prfifo
   generic map(
      ITEMS             => DFIFO_SIZE,
      DATA_WIDTH        => 512,
      STATUS_WIDTH      => 1,
      SOP_POS_W         => 3,
      HELPER_FIFO_ITEMS => DFIFO_SIZE/2
   )
   port map(
      -- Common ports
      RX_CLK            => CGMII_CLK,
      RX_RESET          => CGMII_RESET,
      TX_CLK            => FLU_CLK,
      TX_RESET          => FLU_RESET,

      -- Status port
      STATUS            => open,

      -- Input interface
      RX_DATA           => reg1_rx_data,
      RX_SOP_POS        => reg1_rx_sop_pos,
      RX_EOP_POS        => reg1_rx_eop_pos,
      RX_SOP            => reg1_rx_sop,
      RX_EOP            => reg1_rx_eop,

      RX_SRC_RDY        => '1',
      RX_DST_RDY        => dfifo_rx_dst_rdy, 
      RX_RELEASE        => fsm_dfifo_release,

      -- Output interface
      TX_DATA           => dfifo_tx_data,
      TX_SOP_POS        => dfifo_tx_sop_pos,
      TX_EOP_POS        => dfifo_tx_eop_pos,
      TX_SOP            => dfifo_tx_sop,
      TX_EOP            => dfifo_tx_eop,
      TX_SRC_RDY        => dfifo_tx_src_rdy,
      TX_DST_RDY        => tx_dst_rdy_in
   );

   -- Deriving dfifo_full signal
   dfifo_full       <= not dfifo_rx_dst_rdy;

   
   -- -------------------------------------------------------------------------
   --                                  HFIFO
   -- -------------------------------------------------------------------------
   
   -- Input data composition
   hfifo_data_in(HFIFO_DATA_POS_H downto HFIFO_DATA_POS_L) <= ctrl_data_in;
   hfifo_data_in(HFIFO_SOP_POS)                            <= ctrl_sop_n_in;
   hfifo_data_in(HFIFO_EOP_POS)                            <= ctrl_eop_n_in;
   hfifo_data_in(HFIFO_REM_POS_H downto HFIFO_REM_POS_L)   <= ctrl_rem_in;

   -- HFIFO write signal
   hfifo_wr <= not (hfifo_full or ctrl_src_rdy_n_in);

   -- HFIFO read signal
   hfifo_rd <= not tx_hdst_rdy_n_in;

   -- HFIFO instantiation
   hfifo_i: entity work.asfifo_bram
   generic map(
      ITEMS          => HFIFO_SIZE,
      DATA_WIDTH     => HFIFO_WIDTH,
      STATUS_WIDTH   => 1,
      AUTO_PIPELINE  => true
   )
   port map(
      -- Common ports
      CLK_WR         => CGMII_CLK,
      RST_WR         => CGMII_RESET,
      CLK_RD         => FLU_CLK,
      RST_RD         => FLU_RESET,

      -- Status port
      STATUS         => open,

      -- Input interface
      WR             => hfifo_wr,
      DI             => hfifo_data_in,
      FULL           => hfifo_full,

      -- Output interface
      RD             => hfifo_rd,
      DO             => hfifo_data_out,
      DO_DV          => hfifo_data_vld,
      EMPTY          => hfifo_empty
   );

   -- Decomposition of output data signal
   hfifo_tx_data  <= hfifo_data_out(HFIFO_DATA_POS_H downto HFIFO_DATA_POS_L);
   hfifo_tx_sop_n <= hfifo_data_out(HFIFO_SOP_POS);
   hfifo_tx_eop_n <= hfifo_data_out(HFIFO_EOP_POS);
   hfifo_tx_rem   <= hfifo_data_out(HFIFO_REM_POS_H downto HFIFO_REM_POS_L);
   hfifo_tx_src_rdy_n  <= not hfifo_data_vld;


   -- -------------------------------------------------------------------------
   --          Status signal composition (from lowest to highest bits)
   -- -------------------------------------------------------------------------

   -- Synchronization of signals from FLU_CLK domain to CGMII_CLK domain ------

   -- composition of vector for synchronization
   sync_status_flu(0) <= dfifo_tx_src_rdy;
   sync_status_flu(1) <= tx_dst_rdy_in;
   sync_status_flu(2) <= hfifo_rd;
   sync_status_flu(3) <= hfifo_data_vld;
   sync_status_flu(4) <= hfifo_empty;

   -- synchronization component instantiation
   sync_status_flu_cgmii : entity work.ASYNC_BUS_HANDSHAKE
   generic map(
      DATA_WIDTH => 5
   )
   port map(
      --! A clock domain
      ACLK       => FLU_CLK,
      ARST       => FLU_RESET,
      ADATAIN    => sync_status_flu,
      ASEND      => '1',
      AREADY     => open,

      --! B clock domain
      BCLK       => CGMII_CLK,
      BRST       => CGMII_RESET,
      BDATAOUT   => sync_status_cgmii,
      BLOAD      => '1',
      BVALID     => open
   );

   -- Composition of final status signal --------------------------------------

   status_out(C_PACODAG_OVF_POS)    <= fsm_ctrl_ovf;
   status_out(C_DFIFO_OVF_POS)      <= fsm_dfifo_ovf;
   status_out(C_FR_DISCARDED_POS)   <= '0';
   status_out(C_BUFFER_OVF_POS)     <= '0';
   status_out(C_FSM_STATUS_DEBUG_H downto C_FSM_STATUS_DEBUG_L) <= fsm_status;

   status_out(C_HFIFO_FULL_POS)     <= hfifo_full;
   status_out(C_HFIFO_EMPTY_POS)    <= sync_status_cgmii(4);
   status_out(C_HFIFO_WR_POS)       <= hfifo_wr;
   status_out(C_HFIFO_RD_POS)       <= sync_status_cgmii(2);
   status_out(C_HFIFO_DO_DV_POS)    <= sync_status_cgmii(3);

   status_out(C_DFIFO_FULL_POS)     <= dfifo_full;
   status_out(C_DFIFO_EMPTY_POS)    <= not sync_status_cgmii(0);
   status_out(C_DFIFO_WR_POS)       <= not fsm_dfifo_release;
   status_out(C_DFIFO_RD_POS)       <= sync_status_cgmii(1);
   status_out(C_DFIFO_DO_DV_POS)    <= sync_status_cgmii(0) and sync_status_cgmii(1);

   status_out(C_CTRL_RDY_POS)       <= ctrl_rdy_in;


   -- -------------------------------------------------------------------------
   --                         Output ports connection
   -- -------------------------------------------------------------------------

   -- Data payload output
   TX_DATA                 <= dfifo_tx_data;
   TX_SOP_POS              <= dfifo_tx_sop_pos;
   TX_EOP_POS              <= dfifo_tx_eop_pos;
   TX_SOP                  <= dfifo_tx_sop;
   TX_EOP                  <= dfifo_tx_eop;
   TX_SRC_RDY              <= dfifo_tx_src_rdy;
   
   -- Data header/footer output
   TX_HDATA                <= hfifo_tx_data;
   TX_HSOP_N               <= hfifo_tx_sop_n;
   TX_HEOP_N               <= hfifo_tx_eop_n;
   TX_HREM                 <= hfifo_tx_rem;
   TX_HSRC_RDY_N           <= hfifo_tx_src_rdy_n;
   
   -- Frame processing signals
   FRAME_RECEIVED          <= fsm_frame_received;
   FRAME_DISCARDED         <= fsm_frame_discarded;
   FRAME_DISCARDED_BUF_OVF <= fsm_frame_discarded_buf_ovf;

   -- Statistics for PACODAG
   CTRL_STAT_MAC_ERR       <= reg1_rx_mac_err;
   CTRL_STAT_MINTU_ERR     <= reg1_rx_mintu_err;
   CTRL_STAT_MTU_ERR       <= reg1_rx_mtu_err;
   CTRL_STAT_CRC_ERR       <= reg1_rx_crc_err;
   CTRL_STAT_FRAME_ERR     <= reg1_rx_frame_err;
   CTRL_STAT_FRAME_LEN     <= reg1_rx_frame_len;
   
   -- PACODAG interface
   CTRL_CLK                <= CGMII_CLK;
   CTRL_RESET              <= CGMII_RESET;
   CTRL_DST_RDY_N          <= hfifo_full;
   CTRL_SOP                <= fsm_ctrl_sop;
   CTRL_STAT_DV            <= fsm_ctrl_stat_dv;
   
   -- Input value for register placed in Mem module
   STATUS                  <= status_out;

end architecture cgmii_ibuf_buf_arch;
