-- cgmii_dec_ctrl_dec.vhd: Control decoder for CGMII_dec unit of CGMII
--                         Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use work.cgmii_pkg.all;

-- ----------------------------------------------------------------------------
--                           Entity declaration
-- ----------------------------------------------------------------------------

entity cgmii_dec_ctrl_dec is
   port(    
      -- Input CGMII interface -----------------------------------------------
      RXD               : in  std_logic_vector(511 downto 0);
      RXC               : in  std_logic_vector(63 downto 0);
      
      -- Output control positions ----------------------------------------------
      ERROR_POS         : out std_logic_vector(63 downto 0);  
      START_POS         : out std_logic_vector(63 downto 0);
      TERMINATE_POS     : out std_logic_vector(63 downto 0);
      PREAMBLE_POS      : out std_logic_vector(7 downto 0);
      LOCFAULT_POS      : out std_logic_vector(7 downto 0)
   );
end entity cgmii_dec_ctrl_dec;


-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture cgmii_dec_ctrl_dec_arch of cgmii_dec_ctrl_dec is
 
begin

   -- -------------------------------------------------------------------------
   --                        CGMII error detection
   -- -------------------------------------------------------------------------

   -- generating detector of Error control characters   
   error_detect : for i in 0 to 63 generate

      -- detect CGMII error on each lane
      error_detect_p : process(RXD((i+1)*8-1 downto i*8), RXC(i))
      begin
         if ((RXD((i+1)*8-1 downto i*8) = C_CGMII_ERROR) and
             (RXC(i) = '1')) then
            ERROR_POS(i) <= '1';
         else
            ERROR_POS(i) <= '0';
         end if;
      end process error_detect_p;
      
   end generate error_detect;   

   -- -------------------------------------------------------------------------
   --              Detecting Start control character positions
   -- -------------------------------------------------------------------------

   -- generating detector of Start control characters
   start_detect : for j in 0 to 7 generate
   
      -- detect Start control characters starting at each eighth lane
      start_detect_p : process(RXD((j*64)+7 downto j*64), RXC(j*8))
      begin
         if ((RXD((j*64)+7 downto j*64) = C_CGMII_START) and
             (RXC(j*8) = '1')) then
            START_POS((j+1)*8-1 downto j*8) <= X"01";
         else
            START_POS((j+1)*8-1 downto j*8) <= X"00";
         end if;
      end process start_detect_p;

   end generate start_detect;
 
   -- -------------------------------------------------------------------------
   --                     Local fault sequence detection
   -- -------------------------------------------------------------------------

   -- generating detector of Local fault control sequence 
   locfault_detect : for j in 0 to 7 generate
   
      -- detect local fault sequence starting at each eighth lane
      locfault_detect_p : process(RXD((j+1)*64-1 downto j*64),
                                  RXC((j+1)*8-1 downto j*8))
      begin
         if ((RXD((j+1)*64-1 downto j*64) = C_CGMII_LOCFAULT_D) and
             (RXC((j+1)*8-1 downto j*8) = C_CGMII_LOCFAULT_C)) then
            LOCFAULT_POS(j) <= '1';
         else
            LOCFAULT_POS(j) <= '0';
         end if;
      end process locfault_detect_p;

   end generate locfault_detect;

   -- -------------------------------------------------------------------------
   --             Detecting Start and Preamble pattern positions
   -- -------------------------------------------------------------------------
 
   -- generating detector of Start control characters and Preamble pattern
   preamble_detect : for j in 0 to 7 generate
   
      -- detect Start control characters and Preamble pattern
      -- starting at each eighth lane
      preamble_detect_p : process(RXD((j+1)*64-1 downto j*64),
                               RXC((j+1)*8-1 downto j*8))
      begin
         if ((RXD((j+1)*64-1 downto j*64) = C_CGMII_PREAMBLE_D) and
             (RXC((j+1)*8-1 downto j*8) = C_CGMII_PREAMBLE_C)) then
            PREAMBLE_POS(j) <= '1';
         else
            PREAMBLE_POS(j) <= '0';
         end if;
      end process preamble_detect_p;

   end generate preamble_detect;
   
   -- -------------------------------------------------------------------------
   --                         Terminate detection
   -- -------------------------------------------------------------------------

   -- generating detector of Terminate control characters 
   terminate_detect : for i in 0 to 63 generate

      -- detect terminate on each lane
      terminate_detect_p : process(RXD((i+1)*8-1 downto i*8), RXC(i))
      begin
         if ((RXD((i+1)*8-1 downto i*8) = C_CGMII_TERMINATE) and
             (RXC(i) = '1')) then
            TERMINATE_POS(i) <= '1';
         else
            TERMINATE_POS(i) <= '0';
         end if;
      end process terminate_detect_p;

   end generate terminate_detect;

end architecture cgmii_dec_ctrl_dec_arch;
