-- asfifo_bram_release_prfifo.vhd: Asynchronous fifo from BlockRAM memories
-- with packet discarding, created as a modification of asfifo_bram_release
-- Copyright (C) 2003, 2012 CESNET
-- Author(s): Martinek Tomas <martinek@liberouter.org>
--            Martin Mikusek <martin.mikusek@liberouter.org>
--            Jan Pazdera    <pazdera@liberouter.org>
--            Viktor Pus     <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- library containing log2 function
use work.math_pack.all;

-- auxilarity function needed to compute address width
use WORK.bmem_func.all;

-- pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
-- pragma translate_on
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity asfifo_bram_release_prfifo is
    generic(
      -- ITEMS = Numer of items in FIFO
      -- !!!!!!!!!!! Must be (2^n)-1, n>=2 !!!!!!!!!!!!!!!!!!!!!!
      ITEMS        : integer;

      -- Data Width
      DATA_WIDTH   : integer;

      -- Width of status information of fifo fullness
      -- Note: 2**STATUS_WIDTH MUST BE!! less or equal
      --       than ITEMS
      STATUS_WIDTH : integer;
      -- Prepare data to the output automatically
      AUTO_PIPELINE : boolean := false
   );
   port (
      -- Write interface
      CLK_WR      : in  std_logic;
      RESET_WR    : in  std_logic;
      WR          : in  std_logic;
      DI          : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      FULL        : out std_logic;
      STATUS      : out std_logic_vector(STATUS_WIDTH-1 downto 0);
      MARK        : in  std_logic; -- mark actual write position
      RELEASE     : in  std_logic; -- release writed data after MARK
      RELEASE_WR  : in  std_logic; -- write the current word even if releasing

      -- Read interface
      CLK_RD      : in  std_logic;
      RESET_RD    : in  std_logic;
      RD          : in  std_logic;
      DO          : out std_logic_vector(DATA_WIDTH-1 downto 0);
      DO_DV       : out std_logic;
      EMPTY       : out std_logic
   );
end asfifo_bram_release_prfifo;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture behavioral of asfifo_bram_release_prfifo is

   -- Number of address bits
   constant FADDR            : integer := LOG2(ITEMS);

   signal BRAM_TYPE_RES      : integer;

   -- FIFO part signals
   signal cnt_read_addr      : std_logic_vector(FADDR downto 0);
   signal sync_read_addr     : std_logic_vector(FADDR downto 0);

   signal cnt_write_addr     : std_logic_vector(FADDR downto 0);
   signal cnt_write_addr_rel : std_logic_vector(FADDR downto 0);
   signal mark_cnt_write_addr: std_logic_vector(FADDR downto 0);
   signal sync_write_addr    : std_logic_vector(FADDR downto 0);
   signal fcomp_write_addr   : std_logic_vector(FADDR downto 0);

   signal read_allow         : std_logic;
   signal write_allow        : std_logic;
   signal reg_empty          : std_logic;
   signal reg_full           : std_logic;

   signal sig_rd             : std_logic;
   signal sig_dv             : std_logic;
   signal sig_status         : std_logic_vector(FADDR downto 0);


begin

cond_ap : if AUTO_PIPELINE = true generate
   sig_rd <= RD or not sig_dv;
end generate;

cond_nap : if AUTO_PIPELINE = false generate
   sig_rd <= RD;
end generate;

--* Main memory 
memory : entity work.dp_bmem(behavioral)
generic map(
   BRAM_TYPE   => GET_BRAM_TYPE(DATA_WIDTH, ITEMS),
   DATA_WIDTH  => DATA_WIDTH,
   ITEMS       => ITEMS,
   OUTPUT_REG  => TRUE
)
port map(
   -- Interface A, will be used for writing only
   CLKA        => CLK_WR,
   RSTA        => RESET_WR,
   PIPE_ENA    => '1',
   REA         => '0',
   WEA         => write_allow,
   ADDRA       => cnt_write_addr_rel(FADDR-1 downto 0),
   DIA         => DI,

   -- Interface B, will be used for reading only
   CLKB        => CLK_RD,
   RSTB        => RESET_RD,
   PIPE_ENB    => sig_rd,
   REB         => read_allow,
   WEB         => '0',
   ADDRB       => cnt_read_addr(FADDR-1 downto 0),
   DIB         => (others => '0'),
   DOB_DV      => sig_dv,
   DOB         => DO
);

cnt_write_addr_rel <= mark_cnt_write_addr 
                         when RELEASE = '1' and RELEASE_WR = '1' else
                      cnt_write_addr;

--* Write address register
--* Incremented when writing a word, loaded when releasing a packet.
--* When releasing and writing, loaded and incremented.
cnt_write_addr_p: process (CLK_WR)
begin
   if CLK_WR'event and CLK_WR = '1' then
      if RESET_WR = '1' then
         cnt_write_addr <= (others => '0');
      else
         if RELEASE='1' then
            if RELEASE_WR = '1' and write_allow = '1' then
               cnt_write_addr <= mark_cnt_write_addr + 1;
            else
               cnt_write_addr <= mark_cnt_write_addr;
            end if;
         elsif write_allow = '1' then
            cnt_write_addr <= cnt_write_addr + 1;
         end if;
      end if;
   end if;
end process;

--* Store the mark
mark_cnt_write_addr_p : process(CLK_WR)
begin
   if CLK_WR'event and CLK_WR = '1' then
      if RESET_WR = '1' then
         mark_cnt_write_addr <= (others => '0');
      else
         if MARK = '1' then
            if RELEASE = '1' then
               if RELEASE_WR = '1' and write_allow = '1' then
                  mark_cnt_write_addr <= mark_cnt_write_addr;-- + 1;
               end if;
            else
               mark_cnt_write_addr <= cnt_write_addr;
            end if;
         end if;
      end if;
   end if;
end process;

--* Synchronize mark to the other clock domain
write_addr_to_clkwr: entity work.ASYNC_BUS_HANDSHAKE
generic map (
   DATA_WIDTH => FADDR + 1
)
port map (
   --! A clock domain
   ACLK       => CLK_WR,
   ARST       => RESET_WR,
   ADATAIN    => mark_cnt_write_addr,
   ASEND      => '1',
   AREADY     => open,

   --! B clock domain
   BCLK       => CLK_RD,
   BRST       => RESET_RD,
   BDATAOUT   => sync_write_addr,
   BLOAD      => '1',
   BVALID     => open
);

--* Read address counter
cnt_read_addr_p : process(CLK_RD)
begin
   if CLK_RD'event and CLK_RD = '1' then
      if RESET_RD = '1' then
         cnt_read_addr <= (others => '0');
      else
         if read_allow = '1' then
            cnt_read_addr <= cnt_read_addr + 1;
         end if;
      end if;
   end if;
end process;

--* Synchronize read address to the other clock domain
read_addr_to_clkwr: entity work.ASYNC_BUS_HANDSHAKE
generic map (
   DATA_WIDTH => FADDR + 1
)
port map (
   --! A clock domain
   ACLK       => CLK_RD,
   ARST       => RESET_RD,
   ADATAIN    => cnt_read_addr,
   ASEND      => '1',
   AREADY     => open,

   --! B clock domain
   BCLK       => CLK_WR,
   BRST       => RESET_WR,
   BDATAOUT   => sync_read_addr,
   BLOAD      => '1',
   BVALID     => open
);

--* Compute and store empty flag
reg_empty_p : process(CLK_RD)
begin
   if CLK_RD'event and CLK_RD = '1' then
      if RESET_RD = '1' then
         reg_empty <= '1';
      else
         if read_allow = '1' then
            if sync_write_addr = (cnt_read_addr+1) then
               reg_empty <= '1';
            else
               reg_empty <= '0';
            end if;
         else
            if sync_write_addr = cnt_read_addr then
               reg_empty <= '1';
            else
               reg_empty <= '0';
            end if;
         end if;
      end if;
   end if;
end process;

read_allow <= (RD or not sig_dv) and not reg_empty;

fcomp_write_addr <= cnt_write_addr when write_allow = '0' else
                    cnt_write_addr+1;

--* Compute and store full flag.
reg_full_p : process(CLK_WR)
begin
   if CLK_WR'event and CLK_WR = '1' then
      if RESET_WR = '1' then
         reg_full <= '1';
      else
         if fcomp_write_addr(FADDR) /= sync_read_addr(FADDR) and
            fcomp_write_addr(FADDR-1 downto 0) = 
               sync_read_addr(FADDR-1 downto 0) then
            reg_full <= '1';
         else
            reg_full <= '0';
         end if;
      end if;
   end if;
end process;

sig_status <= cnt_write_addr - sync_read_addr;
STATUS <= sig_status(FADDR downto FADDR-STATUS_WIDTH+1);

write_allow <= WR and not reg_full;

EMPTY <= reg_empty;
FULL <= reg_full;
DO_DV <= sig_dv;
--DO_DV <= sig_dv and not reg_empty;

end behavioral;
-- ----------------------------------------------------------------------------

