-- addr_dec.vhd: Address decoder for CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture addr_dec_arch of addr_dec is

   -- Constants declaration ---------------------------------------------------

   constant TRFC_L_ADDR          : std_logic_vector(7 downto 2) := "000000"; -- 0x00
   constant CFC_L_ADDR           : std_logic_vector(7 downto 2) := "000001"; -- 0x04
   constant DFC_L_ADDR           : std_logic_vector(7 downto 2) := "000010"; -- 0x08
   constant BODFC_L_ADDR         : std_logic_vector(7 downto 2) := "000011"; -- 0x0C
   constant TRFC_H_ADDR          : std_logic_vector(7 downto 2) := "000100"; -- 0x10
   constant CFC_H_ADDR           : std_logic_vector(7 downto 2) := "000101"; -- 0x14
   constant DFC_H_ADDR           : std_logic_vector(7 downto 2) := "000110"; -- 0x18
   constant BODFC_H_ADDR         : std_logic_vector(7 downto 2) := "000111"; -- 0x1C
   constant IBUF_EN_ADDR         : std_logic_vector(7 downto 2) := "001000"; -- 0x20
   constant ERROR_MASK_ADDR      : std_logic_vector(7 downto 2) := "001001"; -- 0x24
   constant STATUS_ADDR          : std_logic_vector(7 downto 2) := "001010"; -- 0x28
   constant COMMAND_ADDR         : std_logic_vector(7 downto 2) := "001011"; -- 0x2C
   constant MIN_FRAME_LEN_ADDR   : std_logic_vector(7 downto 2) := "001100"; -- 0x30
   constant MAX_FRAME_LEN_ADDR   : std_logic_vector(7 downto 2) := "001101"; -- 0x34
   constant MAC_CHECK_MODE_ADDR  : std_logic_vector(7 downto 2) := "001110"; -- 0x38
   constant OROC_L_ADDR          : std_logic_vector(7 downto 2) := "001111"; -- 0x3C
   constant OROC_H_ADDR          : std_logic_vector(7 downto 2) := "010000"; -- 0x40
   constant CAM_BASE_ADDR        : std_logic_vector(7 downto 2) := "100000"; -- 0x80

   --RFC2819 counter addresses
   constant CRC_ERR_L_ADDR          : std_logic_vector(7 downto 2) := "000000";
   constant OVER_MTU_L_ADDR         : std_logic_vector(7 downto 2) := "000001";
   constant BELOW_MIN_L_ADDR        : std_logic_vector(7 downto 2) := "000010";
   constant BCAST_FRAMES_L_ADDR     : std_logic_vector(7 downto 2) := "000011";
   constant MCAST_FRAMES_L_ADDR     : std_logic_vector(7 downto 2) := "000100";
   constant FRAGMENT_FRAMES_L_ADDR  : std_logic_vector(7 downto 2) := "000101";
   constant JABBER_FRAMES_L_ADDR    : std_logic_vector(7 downto 2) := "000110";
   constant TRANS_OCTETS_L_ADDR     : std_logic_vector(7 downto 2) := "000111";
   constant FRAMES_64_L_ADDR        : std_logic_vector(7 downto 2) := "001000";
   constant FRAMES_65_127_L_ADDR    : std_logic_vector(7 downto 2) := "001001";
   constant FRAMES_128_255_L_ADDR   : std_logic_vector(7 downto 2) := "001010";
   constant FRAMES_256_511_L_ADDR   : std_logic_vector(7 downto 2) := "001011";
   constant FRAMES_512_1023_L_ADDR  : std_logic_vector(7 downto 2) := "001100";
   constant FRAMES_1024_1518_L_ADDR : std_logic_vector(7 downto 2) := "001101";

   constant CRC_ERR_H_ADDR          : std_logic_vector(7 downto 2) := "001110";
   constant OVER_MTU_H_ADDR         : std_logic_vector(7 downto 2) := "001111";
   constant BELOW_MIN_H_ADDR        : std_logic_vector(7 downto 2) := "010000";
   constant BCAST_FRAMES_H_ADDR     : std_logic_vector(7 downto 2) := "010001";
   constant MCAST_FRAMES_H_ADDR     : std_logic_vector(7 downto 2) := "010010";
   constant FRAGMENT_FRAMES_H_ADDR  : std_logic_vector(7 downto 2) := "010011";
   constant JABBER_FRAMES_H_ADDR    : std_logic_vector(7 downto 2) := "010100";
   constant TRANS_OCTETS_H_ADDR     : std_logic_vector(7 downto 2) := "010101";
   constant FRAMES_64_H_ADDR        : std_logic_vector(7 downto 2) := "010110";
   constant FRAMES_65_127_H_ADDR    : std_logic_vector(7 downto 2) := "010111";
   constant FRAMES_128_255_H_ADDR   : std_logic_vector(7 downto 2) := "011000";
   constant FRAMES_256_511_H_ADDR   : std_logic_vector(7 downto 2) := "011001";
   constant FRAMES_512_1023_H_ADDR  : std_logic_vector(7 downto 2) := "011010";
   constant FRAMES_1024_1518_H_ADDR : std_logic_vector(7 downto 2) := "011011";

   -- Command constants
   constant CMD_SAMPLE           : std_logic_vector(2 downto 0) := "001"; -- 0x01
   constant CMD_RESET            : std_logic_vector(2 downto 0) := "010"; -- 0x02
   constant CMD_SW_BASE_REG      : std_logic_vector(2 downto 0) := "011"; -- 0x03

   -- Signals declaration -----------------------------------------------------

   -- General data input signals
   signal data_rd_in                      : std_logic_vector(31 downto 0);
   signal data_vld_in                     : std_logic;

   -- CAM input signals
   signal cam_busy_in                     : std_logic;

   -- Statistics input signals
   signal total_frames_cnt_in             : std_logic_vector(63 downto 0);
   signal correct_frames_cnt_in           : std_logic_vector(63 downto 0);
   signal discarded_frames_cnt_in         : std_logic_vector(63 downto 0);
   signal discarded_buf_ovf_frames_cnt_in : std_logic_vector(63 downto 0);
   signal octets_received_ok_cnt_in       : std_logic_vector(63 downto 0);
   signal crc_error_cnt_in                : std_logic_vector(63 downto 0);
   signal over_mtu_cnt_in                 : std_logic_vector(63 downto 0);
   signal below_min_cnt_in                : std_logic_vector(63 downto 0);
   signal bcast_frames_cnt_in             : std_logic_vector(63 downto 0);
   signal mcast_frames_cnt_in             : std_logic_vector(63 downto 0);
   signal fragment_frames_cnt_in          : std_logic_vector(63 downto 0);
   signal jabber_frames_cnt_in            : std_logic_vector(63 downto 0);
   signal frames_64_cnt_in                : std_logic_vector(63 downto 0);
   signal frames_65_127_cnt_in            : std_logic_vector(63 downto 0);
   signal frames_128_255_cnt_in           : std_logic_vector(63 downto 0);
   signal frames_256_511_cnt_in           : std_logic_vector(63 downto 0);
   signal frames_512_1023_cnt_in          : std_logic_vector(63 downto 0);
   signal frames_1024_1518_cnt_in         : std_logic_vector(63 downto 0);
   signal trans_octets_cnt_in             : std_logic_vector(63 downto 0);
   signal cnt_valid_in                    : std_logic;

   -- MI32 master input interface (input signals)
   signal mi_dwr_in                       : std_logic_vector(31 downto 0);
   signal mi_addr_in                      : std_logic_vector(31 downto 0);
   signal mi_rd_in                        : std_logic;
   signal mi_wr_in                        : std_logic;
   signal mi_be_in                        : std_logic_vector(3 downto 0);

   -- MI32 master output interface (output signals)
   signal mi_drd_out                      : std_logic_vector(31 downto 0);
   signal mi_ardy_out                     : std_logic;
   signal mi_drdy_out                     : std_logic;

   -- MI32 slave interface signals
   signal mi_dwr_s                        : std_logic_vector(31 downto 0);
   signal mi_addr_s                       : std_logic_vector(31 downto 0);
   signal mi_rd_s                         : std_logic;
   signal mi_wr_s                         : std_logic;
   signal mi_be_s                         : std_logic_vector(3 downto 0);
   signal mi_drd_s                        : std_logic_vector(31 downto 0);
   signal mi_ardy_s                       : std_logic;
   signal mi_drdy_s                       : std_logic;

   -- Register selection signals
   signal mx_base_regs_out                : std_logic_vector(31 downto 0);
   signal mx_rfc2819_out                  : std_logic_vector(31 downto 0);
   signal mx_base_drdy_out                : std_logic;
   signal mx_rfc2819_drdy_out             : std_logic;

   -- CS signals
   signal trfc_l_cs                       : std_logic;
   signal cfc_l_cs                        : std_logic;
   signal dfc_l_cs                        : std_logic;
   signal bodfc_l_cs                      : std_logic;
   signal oroc_l_cs                       : std_logic;
   signal trfc_h_cs                       : std_logic;
   signal cfc_h_cs                        : std_logic;
   signal dfc_h_cs                        : std_logic;
   signal bodfc_h_cs                      : std_logic;
   signal oroc_h_cs                       : std_logic;
   signal ibuf_en_cs                      : std_logic;
   signal error_mask_cs                   : std_logic;
   signal status_cs                       : std_logic;
   signal cmd_cs                          : std_logic;
   signal min_frame_len_cs                : std_logic;
   signal max_frame_len_cs                : std_logic;
   signal mac_check_mode_cs               : std_logic;
   signal cam_cs                          : std_logic;
   signal stats_rfc2819_cs                : std_logic;

   signal crc_err_l_cs                    : std_logic;
   signal over_mtu_l_cs                   : std_logic;
   signal below_min_l_cs                  : std_logic;
   signal bcast_frames_l_cs               : std_logic;
   signal mcast_frames_l_cs               : std_logic;
   signal fragment_frames_l_cs            : std_logic;
   signal jabber_frames_l_cs              : std_logic;
   signal frames_64_l_cs                  : std_logic;
   signal frames_65_127_l_cs              : std_logic;
   signal frames_128_255_l_cs             : std_logic;
   signal frames_256_511_l_cs             : std_logic;
   signal frames_512_1023_l_cs            : std_logic;
   signal frames_1024_1518_l_cs           : std_logic;
   signal trans_octets_l_cs               : std_logic;

   signal crc_err_h_cs                    : std_logic;
   signal over_mtu_h_cs                   : std_logic;
   signal below_min_h_cs                  : std_logic;
   signal bcast_frames_h_cs               : std_logic;
   signal mcast_frames_h_cs               : std_logic;
   signal fragment_frames_h_cs            : std_logic;
   signal jabber_frames_h_cs              : std_logic;
   signal frames_64_h_cs                  : std_logic;
   signal frames_65_127_h_cs              : std_logic;
   signal frames_128_255_h_cs             : std_logic;
   signal frames_256_511_h_cs             : std_logic;
   signal frames_512_1023_h_cs            : std_logic;
   signal frames_1024_1518_h_cs           : std_logic;
   signal trans_octets_h_cs               : std_logic;

   -- WE signals
   signal ibuf_en_we_out                  : std_logic;
   signal error_mask_we_out               : std_logic;
   signal status_we_out                   : std_logic;
   signal cmd_we_out                      : std_logic;
   signal min_frame_len_we_out            : std_logic;
   signal max_frame_len_we_out            : std_logic;
   signal mac_check_mode_we_out           : std_logic;
   signal cam_we_out                      : std_logic;

   -- Signals related to imitation of the Command register
   signal reg_cmd_we_out                  : std_logic := '0';
   signal reg_cmd_sample                  : std_logic;
   signal sw_reset_out                    : std_logic;
   signal read_en_out                     : std_logic;
   signal last_addr_en_out                : std_logic;

   -- Output register switching
   signal switch_rfc         : std_logic;
   signal reg_sel_register_out   : std_logic;
   signal sel_register_out       : std_logic;

   -- RD signals
   signal trfc_l_rd_out                   : std_logic;
   signal cfc_l_rd_out                    : std_logic;
   signal dfc_l_rd_out                    : std_logic;
   signal bodfc_l_rd_out                  : std_logic;
   signal oroc_l_rd_out                   : std_logic;
   signal trfc_h_rd_out                   : std_logic;
   signal cfc_h_rd_out                    : std_logic;
   signal dfc_h_rd_out                    : std_logic;
   signal bodfc_h_rd_out                  : std_logic;
   signal oroc_h_rd_out                   : std_logic;
   signal ibuf_en_rd_out                  : std_logic;
   signal error_mask_rd_out               : std_logic;
   signal status_rd_out                   : std_logic;
   signal min_frame_len_rd_out            : std_logic;
   signal max_frame_len_rd_out            : std_logic;
   signal mac_check_mode_rd_out           : std_logic;
   signal cam_rd_out                      : std_logic;

   signal crc_err_l_out                   : std_logic;
   signal over_mtu_l_out                  : std_logic;
   signal below_min_l_out                 : std_logic;
   signal bcast_frames_l_out              : std_logic;
   signal mcast_frames_l_out              : std_logic;
   signal fragment_frames_l_out           : std_logic;
   signal jabber_frames_l_out             : std_logic;
   signal frames_64_l_out                 : std_logic;
   signal frames_65_127_l_out             : std_logic;
   signal frames_128_255_l_out            : std_logic;
   signal frames_256_511_l_out            : std_logic;
   signal frames_512_1023_l_out           : std_logic;
   signal frames_1024_1518_l_out          : std_logic;
   signal trans_octets_l_out              : std_logic;

   signal crc_err_h_out                   : std_logic;
   signal over_mtu_h_out                  : std_logic;
   signal below_min_h_out                 : std_logic;
   signal bcast_frames_h_out              : std_logic;
   signal mcast_frames_h_out              : std_logic;
   signal fragment_frames_h_out           : std_logic;
   signal jabber_frames_h_out             : std_logic;
   signal frames_64_h_out                 : std_logic;
   signal frames_65_127_h_out             : std_logic;
   signal frames_128_255_h_out            : std_logic;
   signal frames_256_511_h_out            : std_logic;
   signal frames_512_1023_h_out           : std_logic;
   signal frames_1024_1518_h_out          : std_logic;
   signal trans_octets_h_out              : std_logic;

   -- Counters data valid signals
   signal trfc_l_vld                      : std_logic;
   signal cfc_l_vld                       : std_logic;
   signal dfc_l_vld                       : std_logic;
   signal bodfc_l_vld                     : std_logic;
   signal oroc_l_vld                      : std_logic;
   signal trfc_h_vld                      : std_logic;
   signal cfc_h_vld                       : std_logic;
   signal dfc_h_vld                       : std_logic;
   signal bodfc_h_vld                     : std_logic;
   signal oroc_h_vld                      : std_logic;

   signal crc_err_l_vld                   : std_logic;
   signal over_mtu_l_vld                  : std_logic;
   signal below_min_l_vld                 : std_logic;
   signal bcast_frames_l_vld              : std_logic;
   signal mcast_frames_l_vld              : std_logic;
   signal fragment_frames_l_vld           : std_logic;
   signal jabber_frames_l_vld             : std_logic;
   signal frames_64_l_vld                 : std_logic;
   signal frames_65_127_l_vld             : std_logic;
   signal frames_128_255_l_vld            : std_logic;
   signal frames_256_511_l_vld            : std_logic;
   signal frames_512_1023_l_vld           : std_logic;
   signal frames_1024_1518_l_vld          : std_logic;
   signal trans_octets_l_vld              : std_logic;

   signal crc_err_h_vld                   : std_logic;
   signal over_mtu_h_vld                  : std_logic;
   signal below_min_h_vld                 : std_logic;
   signal bcast_frames_h_vld              : std_logic;
   signal mcast_frames_h_vld              : std_logic;
   signal fragment_frames_h_vld           : std_logic;
   signal jabber_frames_h_vld             : std_logic;
   signal frames_64_h_vld                 : std_logic;
   signal frames_65_127_h_vld             : std_logic;
   signal frames_128_255_h_vld            : std_logic;
   signal frames_256_511_h_vld            : std_logic;
   signal frames_512_1023_h_vld           : std_logic;
   signal frames_1024_1518_h_vld          : std_logic;
   signal trans_octets_h_vld              : std_logic;

begin

   -- -------------------------------------------------------------------------
   --                       Input ports assignment
   -- -------------------------------------------------------------------------

   -- MI32 master input interface
   mi_dwr_in                        <= MI_DWR;
   mi_addr_in                       <= MI_ADDR;
   mi_rd_in                         <= MI_RD;
   mi_wr_in                         <= MI_WR;
   mi_be_in                         <= MI_BE;

   -- General data input signals (mem -> address decoder)
   data_rd_in                       <= DATA_RD;
   data_vld_in                      <= DATA_VLD;

   -- CAM busy input signal (mem -> address decoder)
   cam_busy_in                      <= CAM_BUSY;

   -- Statistics input (statictical unit -> address decoder)
   total_frames_cnt_in              <= TOTAL_FRAMES_CNT;
   correct_frames_cnt_in            <= CORRECT_FRAMES_CNT;
   discarded_frames_cnt_in          <= DISCARDED_FRAMES_CNT;
   discarded_buf_ovf_frames_cnt_in  <= DISCARDED_BUF_OVF_FRAMES_CNT;
   octets_received_ok_cnt_in        <= OCTETS_RECEIVED_OK_CNT;
   crc_error_cnt_in                 <= CRC_ERROR_CNT;
   over_mtu_cnt_in                  <= OVER_MTU_CNT;
   below_min_cnt_in                 <= BELOW_MIN_CNT;
   bcast_frames_cnt_in              <= BCAST_FRAMES_CNT;
   mcast_frames_cnt_in              <= MCAST_FRAMES_CNT;
   fragment_frames_cnt_in           <= FRAGMENT_FRAMES_CNT;
   jabber_frames_cnt_in             <= JABBER_FRAMES_CNT;
   frames_64_cnt_in                 <= FRAMES_64_CNT;
   frames_65_127_cnt_in             <= FRAMES_65_127_CNT;
   frames_128_255_cnt_in            <= FRAMES_128_255_CNT;
   frames_256_511_cnt_in            <= FRAMES_256_511_CNT;
   frames_512_1023_cnt_in           <= FRAMES_512_1023_CNT;
   frames_1024_1518_cnt_in          <= FRAMES_1024_1518_CNT;
   trans_octets_cnt_in              <= TRANSFERED_OCTETS_CNT;
   cnt_valid_in                     <= CNT_VALID;

   -- -------------------------------------------------------------------------
   --                     Asynchronous MI32 transition
   -- -------------------------------------------------------------------------
   mi32_async_handshake_i : entity work.MI32_ASYNC_HANDSHAKE
   port map(
      -- Master interface
      CLK_M          => MI_CLK,
      RESET_M        => MI_RESET,
      MI_M_DWR      	=> mi_dwr_in,
      MI_M_ADDR     	=> mi_addr_in,
      MI_M_RD       	=> mi_rd_in,
      MI_M_WR       	=> mi_wr_in,
      MI_M_BE       	=> mi_be_in,
      MI_M_DRD       => mi_drd_out,
      MI_M_ARDY     	=> mi_ardy_out,
      MI_M_DRDY     	=> mi_drdy_out,

      -- Slave interface
      CLK_S          => INT_CLK,
      RESET_S        => INT_RESET,
      MI_S_DWR      	=> mi_dwr_s,
      MI_S_ADDR     	=> mi_addr_s,
      MI_S_RD        => mi_rd_s,
      MI_S_WR       	=> mi_wr_s,
      MI_S_BE       	=> mi_be_s,
      MI_S_DRD      	=> mi_drd_s,
      MI_S_ARDY     	=> mi_ardy_s,
      MI_S_DRDY     	=> mi_drdy_s
   );


   -- -------------------------------------------------------------------------
   --                            Address decoder
   -- -------------------------------------------------------------------------

   -- CS signals generation ---------------------------------------------------

   cs_gen_p : process(mi_addr_s(8 downto 2))
   begin
      trfc_l_cs               <= '0';
      cfc_l_cs                <= '0';
      dfc_l_cs                <= '0';
      bodfc_l_cs              <= '0';
      oroc_l_cs               <= '0';
      trfc_h_cs               <= '0';
      cfc_h_cs                <= '0';
      dfc_h_cs                <= '0';
      bodfc_h_cs              <= '0';
      oroc_h_cs               <= '0';
      ibuf_en_cs              <= '0';
      error_mask_cs           <= '0';
      status_cs               <= '0';
      cmd_cs                  <= '0';
      min_frame_len_cs        <= '0';
      max_frame_len_cs        <= '0';
      mac_check_mode_cs       <= '0';
      cam_cs                  <= '0';


      -- CS signals for particular registers
      case (mi_addr_s(7 downto 2)) is
         when TRFC_L_ADDR           => trfc_l_cs               <= '1';
         when CFC_L_ADDR            => cfc_l_cs                <= '1';
         when DFC_L_ADDR            => dfc_l_cs                <= '1';
         when BODFC_L_ADDR          => bodfc_l_cs              <= '1';
         when OROC_L_ADDR           => oroc_l_cs               <= '1';
         when TRFC_H_ADDR           => trfc_h_cs               <= '1';
         when CFC_H_ADDR            => cfc_h_cs                <= '1';
         when DFC_H_ADDR            => dfc_h_cs                <= '1';
         when BODFC_H_ADDR          => bodfc_h_cs              <= '1';
         when OROC_H_ADDR           => oroc_h_cs               <= '1';
         when IBUF_EN_ADDR          => ibuf_en_cs              <= '1';
         when ERROR_MASK_ADDR       => error_mask_cs           <= '1';
         when STATUS_ADDR           => status_cs               <= '1';
         when COMMAND_ADDR          => cmd_cs                  <= '1';
         when MIN_FRAME_LEN_ADDR    => min_frame_len_cs        <= '1';
         when MAX_FRAME_LEN_ADDR    => max_frame_len_cs        <= '1';
         when MAC_CHECK_MODE_ADDR   => mac_check_mode_cs       <= '1';

         when others => null;
      end case;

      -- CS signal for CAM
      if (mi_addr_s(8 downto log2(MAC_COUNT)+3) =
      '0' & CAM_BASE_ADDR(7 downto log2(MAC_COUNT)+3)) then
         cam_cs   <= '1';
      end if;

   end process cs_gen_p;

   --RFC2819 CS signals - first bunch of RFC2819 statistics
   regs_rfc2819_cs_gen_p : process(mi_addr_s(8 downto 2))
   begin
      crc_err_l_cs            <= '0';
      over_mtu_l_cs           <= '0';
      below_min_l_cs          <= '0';
      bcast_frames_l_cs       <= '0';
      mcast_frames_l_cs       <= '0';
      fragment_frames_l_cs    <= '0';
      jabber_frames_l_cs      <= '0';
      trans_octets_l_cs       <= '0';

      crc_err_h_cs            <= '0';
      over_mtu_h_cs           <= '0';
      below_min_h_cs          <= '0';
      bcast_frames_h_cs       <= '0';
      mcast_frames_h_cs       <= '0';
      fragment_frames_h_cs    <= '0';
      jabber_frames_h_cs      <= '0';
      trans_octets_h_cs       <= '0';

      frames_64_l_cs          <= '0';
      frames_65_127_l_cs      <= '0';
      frames_128_255_l_cs     <= '0';
      frames_256_511_l_cs     <= '0';
      frames_512_1023_l_cs    <= '0';
      frames_1024_1518_l_cs   <= '0';

      frames_64_h_cs          <= '0';
      frames_65_127_h_cs      <= '0';
      frames_128_255_h_cs     <= '0';
      frames_256_511_h_cs     <= '0';
      frames_512_1023_h_cs    <= '0';
      frames_1024_1518_h_cs   <= '0';

      stats_rfc2819_cs        <= '0';


      case (mi_addr_s(7 downto 2)) is
         when CRC_ERR_L_ADDR          => crc_err_l_cs          <= '1';
         when OVER_MTU_L_ADDR         => over_mtu_l_cs         <= '1';
         when BELOW_MIN_L_ADDR        => below_min_l_cs        <= '1';
         when BCAST_FRAMES_L_ADDR     => bcast_frames_l_cs     <= '1';
         when MCAST_FRAMES_L_ADDR     => mcast_frames_l_cs     <= '1';
         when FRAGMENT_FRAMES_L_ADDR  => fragment_frames_l_cs  <= '1';
         when JABBER_FRAMES_L_ADDR    => jabber_frames_l_cs    <= '1';
         when TRANS_OCTETS_L_ADDR     => trans_octets_l_cs     <= '1';

         when CRC_ERR_H_ADDR          => crc_err_h_cs          <= '1';
         when OVER_MTU_H_ADDR         => over_mtu_h_cs         <= '1';
         when BELOW_MIN_H_ADDR        => below_min_h_cs        <= '1';
         when BCAST_FRAMES_H_ADDR     => bcast_frames_h_cs     <= '1';
         when MCAST_FRAMES_H_ADDR     => mcast_frames_h_cs     <= '1';
         when FRAGMENT_FRAMES_H_ADDR  => fragment_frames_h_cs  <= '1';
         when JABBER_FRAMES_H_ADDR    => jabber_frames_h_cs    <= '1';
         when TRANS_OCTETS_H_ADDR     => trans_octets_h_cs     <= '1';

         when FRAMES_64_L_ADDR        => frames_64_l_cs        <= '1';
         when FRAMES_65_127_L_ADDR    => frames_65_127_l_cs    <= '1';
         when FRAMES_128_255_L_ADDR   => frames_128_255_l_cs   <= '1';
         when FRAMES_256_511_L_ADDR   => frames_256_511_l_cs   <= '1';
         when FRAMES_512_1023_L_ADDR  => frames_512_1023_l_cs  <= '1';
         when FRAMES_1024_1518_L_ADDR => frames_1024_1518_l_cs <= '1';

         when FRAMES_64_H_ADDR        => frames_64_h_cs        <= '1';
         when FRAMES_65_127_H_ADDR    => frames_65_127_h_cs    <= '1';
         when FRAMES_128_255_H_ADDR   => frames_128_255_h_cs   <= '1';
         when FRAMES_256_511_H_ADDR   => frames_256_511_h_cs   <= '1';
         when FRAMES_512_1023_H_ADDR  => frames_512_1023_h_cs  <= '1';
         when FRAMES_1024_1518_H_ADDR => frames_1024_1518_h_cs <= '1';

         when others => null;
      end case;

      -- CS signal for RFC2819 statistics
      if (mi_addr_s(8) = '1') then
         stats_rfc2819_cs <= '1';
      end if;
   end process regs_rfc2819_cs_gen_p;


   -- Writing part ------------------------------------------------------------

   -- WE signals generation
   ibuf_en_we_out          <= mi_wr_s and ibuf_en_cs;
   error_mask_we_out       <= mi_wr_s and error_mask_cs;
   status_we_out           <= mi_wr_s and status_cs;
   cmd_we_out              <= mi_wr_s and cmd_cs;
   min_frame_len_we_out    <= mi_wr_s and min_frame_len_cs;
   max_frame_len_we_out    <= mi_wr_s and max_frame_len_cs;
   mac_check_mode_we_out   <= mi_wr_s and mac_check_mode_cs;
   cam_we_out              <= mi_wr_s and cam_cs;


   -- Imitation of IBUF command register --------------------------------------

   -- Register for delaying cmd_we_out
   reg_cmd_we_out_p : process(INT_CLK)
   begin
      if (INT_CLK'event and INT_CLK = '1') then
         if (not RESET_BY_INIT and INT_RESET = '1') then
            reg_cmd_we_out <= '0';
         else
            reg_cmd_we_out <= cmd_we_out;
         end if;
      end if;
   end process reg_cmd_we_out_p;

   -- Register for delaying sample frame counters command
   reg_cmd_sample_p : process(INT_CLK)
   begin
      if (INT_CLK'event and INT_CLK = '1') then
         if (mi_dwr_s(2 downto 0) = CMD_SAMPLE) then
            reg_cmd_sample <= '1';
         else
            reg_cmd_sample <= '0';
         end if;
      end if;
   end process reg_cmd_sample_p;

   -- Imitation of IBUF command register
   reg_cmd_p : process(cmd_we_out, mi_dwr_s(1 downto 0),
                       reg_cmd_we_out, reg_cmd_sample)
   begin
      sw_reset_out      <= '0';
      read_en_out       <= '0';
      last_addr_en_out  <= '0';
      switch_rfc    <= '0';

      -- Command register WE
      if (cmd_we_out = '1') then
         if (mi_dwr_s(2 downto 0) = CMD_SAMPLE) then     -- sample frame counters (1st step)
            last_addr_en_out  <= '1';
         elsif (mi_dwr_s(2 downto 0) = CMD_RESET) then   -- reset frame counters (1st step)
            sw_reset_out      <= '1';
         --Switch between output registers
         elsif (mi_dwr_s(2 downto 0) = CMD_SW_BASE_REG) then
            -- Switch to base register field
            switch_rfc    <= '1';
         end if;

      -- Delayed command register WE
      elsif (reg_cmd_we_out = '1') then
         if (reg_cmd_sample = '1') then   -- sample frame counters (2nd step)
            read_en_out       <= '1';
         end if;
      end if;

   end process reg_cmd_p;


   -- Reading part ------------------------------------------------------------
   -- Address space switching
   reg_sel_register_outp: process(INT_CLK)
   begin
      if (INT_CLK'event and INT_CLK = '1')then
         if (INT_RESET = '1' )then
            -- Select base register field by default
            reg_sel_register_out <= '0';
         else
            if(switch_rfc = '1')then
               -- Select base register field
               reg_sel_register_out <= not(reg_sel_register_out);
            end if;
         end if;
      end if;
   end process;


   -- RD signals generation
   trfc_l_rd_out           <= mi_rd_s and trfc_l_cs;
   cfc_l_rd_out            <= mi_rd_s and cfc_l_cs;
   dfc_l_rd_out            <= mi_rd_s and dfc_l_cs;
   bodfc_l_rd_out          <= mi_rd_s and bodfc_l_cs;
   oroc_l_rd_out           <= mi_rd_s and oroc_l_cs;
   trfc_h_rd_out           <= mi_rd_s and trfc_h_cs;
   cfc_h_rd_out            <= mi_rd_s and cfc_h_cs;
   dfc_h_rd_out            <= mi_rd_s and dfc_h_cs;
   bodfc_h_rd_out          <= mi_rd_s and bodfc_h_cs;
   oroc_h_rd_out           <= mi_rd_s and oroc_h_cs;
   ibuf_en_rd_out          <= mi_rd_s and ibuf_en_cs;
   error_mask_rd_out       <= mi_rd_s and error_mask_cs;
   status_rd_out           <= mi_rd_s and status_cs;
   min_frame_len_rd_out    <= mi_rd_s and min_frame_len_cs;
   max_frame_len_rd_out    <= mi_rd_s and max_frame_len_cs;
   mac_check_mode_rd_out   <= mi_rd_s and mac_check_mode_cs;
   cam_rd_out              <= mi_rd_s and cam_cs;

   crc_err_l_out           <= mi_rd_s and crc_err_l_cs;
   over_mtu_l_out          <= mi_rd_s and over_mtu_l_cs;
   below_min_l_out         <= mi_rd_s and below_min_l_cs;
   bcast_frames_l_out      <= mi_rd_s and bcast_frames_l_cs;
   mcast_frames_l_out      <= mi_rd_s and mcast_frames_l_cs;
   fragment_frames_l_out   <= mi_rd_s and fragment_frames_l_cs;
   jabber_frames_l_out     <= mi_rd_s and jabber_frames_l_cs;
   frames_64_l_out         <= mi_rd_s and frames_64_l_cs;
   frames_65_127_l_out     <= mi_rd_s and frames_65_127_l_cs;
   frames_128_255_l_out    <= mi_rd_s and frames_128_255_l_cs;
   frames_256_511_l_out    <= mi_rd_s and frames_256_511_l_cs;
   frames_512_1023_l_out   <= mi_rd_s and frames_512_1023_l_cs;
   frames_1024_1518_l_out  <= mi_rd_s and frames_1024_1518_l_cs;
   trans_octets_l_out      <= mi_rd_s and trans_octets_l_cs;

   crc_err_h_out           <= mi_rd_s and crc_err_h_cs;
   over_mtu_h_out          <= mi_rd_s and over_mtu_h_cs;
   below_min_h_out         <= mi_rd_s and below_min_h_cs;
   bcast_frames_h_out      <= mi_rd_s and bcast_frames_h_cs;
   mcast_frames_h_out      <= mi_rd_s and mcast_frames_h_cs;
   fragment_frames_h_out   <= mi_rd_s and fragment_frames_h_cs;
   jabber_frames_h_out     <= mi_rd_s and jabber_frames_h_cs;
   frames_64_h_out         <= mi_rd_s and frames_64_h_cs;
   frames_65_127_h_out     <= mi_rd_s and frames_65_127_h_cs;
   frames_128_255_h_out    <= mi_rd_s and frames_128_255_h_cs;
   frames_256_511_h_out    <= mi_rd_s and frames_256_511_h_cs;
   frames_512_1023_h_out   <= mi_rd_s and frames_512_1023_h_cs;
   frames_1024_1518_h_out  <= mi_rd_s and frames_1024_1518_h_cs;
   trans_octets_h_out      <= mi_rd_s and trans_octets_h_cs;



   -- Output data multiplexor
   mx_base_mi_drd_s_p : process(mi_addr_s(7 downto 2), total_frames_cnt_in,
                           correct_frames_cnt_in, discarded_frames_cnt_in,
                           discarded_buf_ovf_frames_cnt_in,
                           octets_received_ok_cnt_in, data_rd_in)
   begin
      case (mi_addr_s(7 downto 2)) is
         when TRFC_L_ADDR  => mx_base_regs_out <= total_frames_cnt_in(31 downto 0);
         when CFC_L_ADDR   => mx_base_regs_out <= correct_frames_cnt_in(31 downto 0);
         when DFC_L_ADDR   => mx_base_regs_out <= discarded_frames_cnt_in(31 downto 0);
         when BODFC_L_ADDR => mx_base_regs_out <= discarded_buf_ovf_frames_cnt_in(31 downto 0);
         when OROC_L_ADDR  => mx_base_regs_out <= octets_received_ok_cnt_in(31 downto 0);
         when TRFC_H_ADDR  => mx_base_regs_out <= total_frames_cnt_in(63 downto 32);
         when CFC_H_ADDR   => mx_base_regs_out <= correct_frames_cnt_in(63 downto 32);
         when DFC_H_ADDR   => mx_base_regs_out <= discarded_frames_cnt_in(63 downto 32);
         when BODFC_H_ADDR => mx_base_regs_out <= discarded_buf_ovf_frames_cnt_in(63 downto 32);
         when OROC_H_ADDR  => mx_base_regs_out <= octets_received_ok_cnt_in(63 downto 32);

         when others       => mx_base_regs_out <= data_rd_in;
      end case;
   end process mx_base_mi_drd_s_p;

   -- Output data multiplexor - first bunch of RFC2819 statistics
   mx_rfc2819_out_p : process(mi_addr_s(7 downto 2), data_rd_in,
                           crc_error_cnt_in,over_mtu_cnt_in,below_min_cnt_in,
                           bcast_frames_cnt_in,mcast_frames_cnt_in,
                           fragment_frames_cnt_in,jabber_frames_cnt_in,
                           frames_64_cnt_in,frames_65_127_cnt_in,
                           frames_128_255_cnt_in,frames_256_511_cnt_in,
                           frames_512_1023_cnt_in,frames_1024_1518_cnt_in,
                           trans_octets_cnt_in)
   begin
      case (mi_addr_s(7 downto 2)) is
         when CRC_ERR_L_ADDR          => mx_rfc2819_out <= crc_error_cnt_in(31 downto 0);
         when OVER_MTU_L_ADDR         => mx_rfc2819_out <= over_mtu_cnt_in(31 downto 0);
         when BELOW_MIN_L_ADDR        => mx_rfc2819_out <= below_min_cnt_in(31 downto 0);
         when BCAST_FRAMES_L_ADDR     => mx_rfc2819_out <= bcast_frames_cnt_in(31 downto 0);
         when MCAST_FRAMES_L_ADDR     => mx_rfc2819_out <= mcast_frames_cnt_in(31 downto 0);
         when FRAGMENT_FRAMES_L_ADDR  => mx_rfc2819_out <= fragment_frames_cnt_in(31 downto 0);
         when JABBER_FRAMES_L_ADDR    => mx_rfc2819_out <= jabber_frames_cnt_in(31 downto 0);
         when TRANS_OCTETS_L_ADDR     => mx_rfc2819_out <= trans_octets_cnt_in(31 downto 0);

         when CRC_ERR_H_ADDR          => mx_rfc2819_out <= crc_error_cnt_in(63 downto 32);
         when OVER_MTU_H_ADDR         => mx_rfc2819_out <= over_mtu_cnt_in(63 downto 32);
         when BELOW_MIN_H_ADDR        => mx_rfc2819_out <= below_min_cnt_in(63 downto 32);
         when BCAST_FRAMES_H_ADDR     => mx_rfc2819_out <= bcast_frames_cnt_in(63 downto 32);
         when MCAST_FRAMES_H_ADDR     => mx_rfc2819_out <= mcast_frames_cnt_in(63 downto 32);
         when FRAGMENT_FRAMES_H_ADDR  => mx_rfc2819_out <= fragment_frames_cnt_in(63 downto 32);
         when JABBER_FRAMES_H_ADDR    => mx_rfc2819_out <= jabber_frames_cnt_in(63 downto 32);
         when TRANS_OCTETS_H_ADDR     => mx_rfc2819_out <= trans_octets_cnt_in(63 downto 32);

         when FRAMES_64_L_ADDR        => mx_rfc2819_out <= frames_64_cnt_in(31 downto 0);
         when FRAMES_65_127_L_ADDR    => mx_rfc2819_out <= frames_65_127_cnt_in(31 downto 0);
         when FRAMES_128_255_L_ADDR   => mx_rfc2819_out <= frames_128_255_cnt_in(31 downto 0);
         when FRAMES_256_511_L_ADDR   => mx_rfc2819_out <= frames_256_511_cnt_in(31 downto 0);
         when FRAMES_512_1023_L_ADDR  => mx_rfc2819_out <= frames_512_1023_cnt_in(31 downto 0);
         when FRAMES_1024_1518_L_ADDR => mx_rfc2819_out <= frames_1024_1518_cnt_in(31 downto 0);

         when FRAMES_64_H_ADDR        => mx_rfc2819_out <= frames_64_cnt_in(63 downto 32);
         when FRAMES_65_127_H_ADDR    => mx_rfc2819_out <= frames_65_127_cnt_in(63 downto 32);
         when FRAMES_128_255_H_ADDR   => mx_rfc2819_out <= frames_128_255_cnt_in(63 downto 32);
         when FRAMES_256_511_H_ADDR   => mx_rfc2819_out <= frames_256_511_cnt_in(63 downto 32);
         when FRAMES_512_1023_H_ADDR  => mx_rfc2819_out <= frames_512_1023_cnt_in(63 downto 32);
         when FRAMES_1024_1518_H_ADDR => mx_rfc2819_out <= frames_1024_1518_cnt_in(63 downto 32);

         when others       => mx_rfc2819_out <= data_rd_in;
      end case;
   end process mx_rfc2819_out_p;


   -- Counters data valid signals
   trfc_l_vld  <= cnt_valid_in and trfc_l_rd_out;
   cfc_l_vld   <= cnt_valid_in and cfc_l_rd_out;
   dfc_l_vld   <= cnt_valid_in and dfc_l_rd_out;
   bodfc_l_vld <= cnt_valid_in and bodfc_l_rd_out;
   oroc_l_vld  <= cnt_valid_in and oroc_l_rd_out;
   trfc_h_vld  <= cnt_valid_in and trfc_h_rd_out;
   cfc_h_vld   <= cnt_valid_in and cfc_h_rd_out;
   dfc_h_vld   <= cnt_valid_in and dfc_h_rd_out;
   bodfc_h_vld <= cnt_valid_in and bodfc_h_rd_out;
   oroc_h_vld  <= cnt_valid_in and oroc_h_rd_out;

   crc_err_l_vld           <= cnt_valid_in and crc_err_l_out;
   over_mtu_l_vld          <= cnt_valid_in and over_mtu_l_out;
   below_min_l_vld         <= cnt_valid_in and below_min_l_out;
   bcast_frames_l_vld      <= cnt_valid_in and bcast_frames_l_out;
   mcast_frames_l_vld      <= cnt_valid_in and mcast_frames_l_out;
   fragment_frames_l_vld   <= cnt_valid_in and fragment_frames_l_out;
   jabber_frames_l_vld     <= cnt_valid_in and jabber_frames_l_out;
   frames_64_l_vld         <= cnt_valid_in and frames_64_l_out;
   frames_65_127_l_vld     <= cnt_valid_in and frames_65_127_l_out;
   frames_128_255_l_vld    <= cnt_valid_in and frames_128_255_l_out;
   frames_256_511_l_vld    <= cnt_valid_in and frames_256_511_l_out;
   frames_512_1023_l_vld   <= cnt_valid_in and frames_512_1023_l_out;
   frames_1024_1518_l_vld  <= cnt_valid_in and frames_1024_1518_l_out;
   trans_octets_l_vld      <= cnt_valid_in and trans_octets_l_out;

   crc_err_h_vld           <= cnt_valid_in and crc_err_h_out;
   over_mtu_h_vld          <= cnt_valid_in and over_mtu_h_out;
   below_min_h_vld         <= cnt_valid_in and below_min_h_out;
   bcast_frames_h_vld      <= cnt_valid_in and bcast_frames_h_out;
   mcast_frames_h_vld      <= cnt_valid_in and mcast_frames_h_out;
   fragment_frames_h_vld   <= cnt_valid_in and fragment_frames_h_out;
   jabber_frames_h_vld     <= cnt_valid_in and jabber_frames_h_out;
   frames_64_h_vld         <= cnt_valid_in and frames_64_h_out;
   frames_65_127_h_vld     <= cnt_valid_in and frames_65_127_h_out;
   frames_128_255_h_vld    <= cnt_valid_in and frames_128_255_h_out;
   frames_256_511_h_vld    <= cnt_valid_in and frames_256_511_h_out;
   frames_512_1023_h_vld   <= cnt_valid_in and frames_512_1023_h_out;
   frames_1024_1518_h_vld  <= cnt_valid_in and frames_1024_1518_h_out;
   trans_octets_h_vld      <= cnt_valid_in and trans_octets_h_out;

   -- Output data valid multiplexor
   mx_base_mi_drdy_s_p : process(mi_addr_s(7 downto 2),
                            trfc_l_vld, cfc_l_vld, dfc_l_vld, bodfc_l_vld,
                            trfc_h_vld, cfc_h_vld, dfc_h_vld, bodfc_h_vld,
                            oroc_l_vld, oroc_h_vld, data_vld_in)
   begin
      case (mi_addr_s(7 downto 2)) is
         when TRFC_L_ADDR  => mx_base_drdy_out <= trfc_l_vld;
         when CFC_L_ADDR   => mx_base_drdy_out <= cfc_l_vld;
         when DFC_L_ADDR   => mx_base_drdy_out <= dfc_l_vld;
         when BODFC_L_ADDR => mx_base_drdy_out <= bodfc_l_vld;
         when OROC_L_ADDR  => mx_base_drdy_out <= oroc_l_vld;
         when TRFC_H_ADDR  => mx_base_drdy_out <= trfc_h_vld;
         when CFC_H_ADDR   => mx_base_drdy_out <= cfc_h_vld;
         when DFC_H_ADDR   => mx_base_drdy_out <= dfc_h_vld;
         when BODFC_H_ADDR => mx_base_drdy_out <= bodfc_h_vld;
         when OROC_H_ADDR  => mx_base_drdy_out <= oroc_h_vld;

         when others       => mx_base_drdy_out <= data_vld_in;
      end case;
   end process mx_base_mi_drdy_s_p;

   -- Output data valid multiplexor
   mx_rfc2819_drdy_out_p : process(mi_addr_s(7 downto 2),

                            crc_err_l_vld, over_mtu_l_vld, bcast_frames_l_vld,below_min_l_vld,
                            mcast_frames_l_vld,fragment_frames_l_vld,jabber_frames_l_vld,
                            trans_octets_l_vld,

                            crc_err_h_vld, over_mtu_h_vld, bcast_frames_h_vld,below_min_h_vld,
                            mcast_frames_h_vld,fragment_frames_h_vld,jabber_frames_h_vld,
                            trans_octets_h_vld,

                            frames_64_l_vld,frames_65_127_l_vld,frames_128_255_l_vld,
                            frames_256_511_l_vld,frames_512_1023_l_vld,frames_1024_1518_l_vld,

                            frames_64_h_vld,frames_65_127_h_vld,frames_128_255_h_vld,
                            frames_256_511_h_vld,frames_512_1023_h_vld,frames_1024_1518_h_vld,

                            data_vld_in)
   begin
      case (mi_addr_s(7 downto 2)) is
         when CRC_ERR_L_ADDR          => mx_rfc2819_drdy_out <= crc_err_l_vld;
         when OVER_MTU_L_ADDR         => mx_rfc2819_drdy_out <= over_mtu_l_vld;
         when BELOW_MIN_L_ADDR        => mx_rfc2819_drdy_out <= below_min_l_vld;
         when BCAST_FRAMES_L_ADDR     => mx_rfc2819_drdy_out <= bcast_frames_l_vld;
         when MCAST_FRAMES_L_ADDR     => mx_rfc2819_drdy_out <= mcast_frames_l_vld;
         when FRAGMENT_FRAMES_L_ADDR  => mx_rfc2819_drdy_out <= fragment_frames_l_vld;
         when JABBER_FRAMES_L_ADDR    => mx_rfc2819_drdy_out <= jabber_frames_l_vld;
         when TRANS_OCTETS_L_ADDR     => mx_rfc2819_drdy_out <= trans_octets_l_vld;

         when CRC_ERR_H_ADDR          => mx_rfc2819_drdy_out <= crc_err_h_vld;
         when OVER_MTU_H_ADDR         => mx_rfc2819_drdy_out <= over_mtu_h_vld;
         when BELOW_MIN_H_ADDR        => mx_rfc2819_drdy_out <= below_min_h_vld;
         when BCAST_FRAMES_H_ADDR     => mx_rfc2819_drdy_out <= bcast_frames_h_vld;
         when MCAST_FRAMES_H_ADDR     => mx_rfc2819_drdy_out <= mcast_frames_h_vld;
         when FRAGMENT_FRAMES_H_ADDR  => mx_rfc2819_drdy_out <= fragment_frames_h_vld;
         when JABBER_FRAMES_H_ADDR    => mx_rfc2819_drdy_out <= jabber_frames_h_vld;
         when TRANS_OCTETS_H_ADDR     => mx_rfc2819_drdy_out <= trans_octets_h_vld;

         when FRAMES_64_L_ADDR        => mx_rfc2819_drdy_out <= frames_64_l_vld;
         when FRAMES_65_127_L_ADDR    => mx_rfc2819_drdy_out <= frames_65_127_l_vld;
         when FRAMES_128_255_L_ADDR   => mx_rfc2819_drdy_out <= frames_128_255_l_vld;
         when FRAMES_256_511_L_ADDR   => mx_rfc2819_drdy_out <= frames_256_511_l_vld;
         when FRAMES_512_1023_L_ADDR  => mx_rfc2819_drdy_out <= frames_512_1023_l_vld;
         when FRAMES_1024_1518_L_ADDR => mx_rfc2819_drdy_out <= frames_1024_1518_l_vld;

         when FRAMES_64_H_ADDR        => mx_rfc2819_drdy_out <= frames_64_h_vld;
         when FRAMES_65_127_H_ADDR    => mx_rfc2819_drdy_out <= frames_65_127_h_vld;
         when FRAMES_128_255_H_ADDR   => mx_rfc2819_drdy_out <= frames_128_255_h_vld;
         when FRAMES_256_511_H_ADDR   => mx_rfc2819_drdy_out <= frames_256_511_h_vld;
         when FRAMES_512_1023_H_ADDR  => mx_rfc2819_drdy_out <= frames_512_1023_h_vld;
         when FRAMES_1024_1518_H_ADDR => mx_rfc2819_drdy_out <= frames_1024_1518_h_vld;

         when others       => mx_rfc2819_drdy_out <= data_vld_in;
      end case;
   end process mx_rfc2819_drdy_out_p;

   -- Generation of selection signal
   sel_register_out <= '1' when ((stats_rfc2819_cs = '1') or
                                 (stats_rfc2819_cs = '0' and reg_sel_register_out = '1'))
                       else '0';

   -- Output selection from register fields
   out_reg_field_selp: process(sel_register_out,
                              mx_base_regs_out,mx_rfc2819_out,
                              mx_base_drdy_out,mx_rfc2819_drdy_out)
   begin
      case sel_register_out is
            -- Select base registers (no change in the address space)
         when '0' => mi_drd_s   <= mx_base_regs_out;
                     mi_drdy_s  <= mx_base_drdy_out;
            -- Select first portion of extended RFC2819
         when '1' => mi_drd_s   <= mx_rfc2819_out;
                     mi_drdy_s  <= mx_rfc2819_drdy_out;
            -- Send vector of "zeros"
         when others => mi_drd_s <= (others=>'0');
                        mi_drdy_s <= '1';
      end case;
   end process;

   -- ARDY setting ------------------------------------------------------------

   -- Writing is done immediately, reading waits for valid output data
   mi_ardy_s <= (mi_wr_s or mi_rd_s) and not cam_busy_in;


   -- -------------------------------------------------------------------------
   --                       Output ports assignment
   -- -------------------------------------------------------------------------

   -- MI32 master output interface
   MI_DRD                        <= mi_drd_out;
   MI_ARDY                       <= mi_ardy_out;
   MI_DRDY                       <= mi_drdy_out;

   -- General data signals (address decoder -> mem)
   DATA_WR                       <= mi_dwr_s;

   -- Registers write interface (address decoder -> mem)
   IBUF_EN_WE                    <= ibuf_en_we_out;
   ERROR_MASK_WE                 <= error_mask_we_out;
   STATUS_WE                     <= status_we_out;
   MIN_FRAME_LEN_WE              <= min_frame_len_we_out;
   MAX_FRAME_LEN_WE              <= max_frame_len_we_out;
   MAC_CHECK_MODE_WE             <= mac_check_mode_we_out;

   -- Registers read interface (address decoder -> mem)
   IBUF_EN_RD                    <= ibuf_en_rd_out;
   ERROR_MASK_RD                 <= error_mask_rd_out;
   STATUS_RD                     <= status_rd_out;
   MIN_FRAME_LEN_RD              <= min_frame_len_rd_out;
   MAX_FRAME_LEN_RD              <= max_frame_len_rd_out;
   MAC_CHECK_MODE_RD             <= mac_check_mode_rd_out;

   -- CAM read/write interface (address decoder -> mem)
   CAM_ADDR                      <= mi_addr_s(log2(MAC_COUNT)+2 downto 2);
   CAM_WE                        <= cam_we_out;
   CAM_RE                        <= cam_rd_out;

   -- Statistical unit control interface (address decoder -> statistical unit)
   SW_RESET                      <= sw_reset_out;
   READ_EN                       <= read_en_out;
   LAST_ADDR_EN                  <= last_addr_en_out;

end architecture addr_dec_arch;
