-- network_mod_ent.vhd : entity of network module for one 100 Gbps port of
--                       Fiberblaze FB1CG card
--!
--! \file
--! \brief Network Module for one 100 Gbps port of Fiberblaze FB1CG card
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;

--! Package with log2 function
use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                             Entity Declaration
-- ----------------------------------------------------------------------------

entity network_mod is
generic (
   --! \brief Size of packet data FIFO inside IBUF in bytes
   --! \details Must be at least as long as the longest possible packet.
   --!          Default value is 4096 without 16KB support or 16384 with
   --!          16KB support.
   IBUF_DFIFO_BYTES      : integer := 4096;
   --! \brief Size of control data FIFO inside IBUF in bytes
   IBUF_HFIFO_BYTES      : integer := 1024;
   --! \brief Data width of IBUF's FL interface for SZE headers
   --! \details In bits, must be power of 2 and greater than 8.
   HDR_WIDTH             : integer := 128;
   --! \brief Size of CAM for MAC addresses within IBUF
   --! \details Must be lower or equal to 16.
   IBUF_MAC_COUNT        : integer := 16;
   --! \brief Keep FCS (CRC) for further processing of a packet
   --! \details True = keep, False = remove.
   INBANDFCS             : boolean := false;
   --! \brief Length of link error timeout counter (number of counter bits)
   --! \details If error don't occur for 2^IBUF_CNT_ERROR_LENGTH cycles, the
   --!          link is declared to be up.
   IBUF_CNT_ERROR_LENGTH : integer := 5;

   --! \brief Size of packet data fifo inside OBUFs in bytes
   --! \details Must be at least as long as the longest possible packet.
   --!          Default value is 4096 without 16KB support or 16384 with
   --!          16KB support.
   OBUF_DFIFO_BYTES      : integer := 4096;
   --! \brief Size of helper FIFO inside OBUF in bytes
   OBUF_HFIFO_BYTES      : integer := 1024;

   --! \brief Length of the internal LED controller counter which determines
   --!        period of LED blinks
   LED_CTRL_CNTR_SIZE    : integer := 27
);
port (
   --! \name IBUF's CGMII interface
   -- -------------------------------------------------------------------------
   --! \brief CGMII receive clock
   CGMII_RXCLK           : in  std_logic;
   --! \brief CGMII receive reset
   CGMII_RXRESET         : in  std_logic;
   --! \brief CGMII receive data
   CGMII_RXD             : in  std_logic_vector(511 downto 0);
   --! \brief CGMII receive control
   CGMII_RXC             : in  std_logic_vector( 63 downto 0);

   --! \name OBUF's CGMII interface
   -- -------------------------------------------------------------------------
   --! \brief CGMII transmit clock
   CGMII_TXCLK           : in  std_logic;
   --! \brief CGMII transmit reset
   CGMII_TXRESET         : in  std_logic;
   --! \brief CGMII transmit data
   CGMII_TXD             : out std_logic_vector(511 downto 0);
   --! \brief CGMII transmit control
   CGMII_TXC             : out std_logic_vector( 63 downto 0);

   --! \name Clock and reset for IBUF's FLU and FL interfaces
   -- -------------------------------------------------------------------------
   --! \brief Output clock
   TX_CLK                : in  std_logic;
   --! \brief Output reset
   TX_RESET              : in  std_logic;

   --! \name IBUF's FrameLink Unaligned interface (data payload output)
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   TX_DATA               : out std_logic_vector(511 downto 0);
   --! \brief Position of the start of the payload
   --! \details Valid only if TX_SOP is set to '1'.
   TX_SOP_POS            : out std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the payload
   --! \details Valid only if TX_EOP is set to '1'.
   TX_EOP_POS            : out std_logic_vector(  5 downto 0);
   --! \brief Start of the payload
   TX_SOP                : out std_logic;
   --! \brief End of the payload
   TX_EOP                : out std_logic;
   --! \brief Source is ready
   TX_SRC_RDY            : out std_logic;
   --! \brief Destination is ready
   TX_DST_RDY            : in  std_logic;

   --! \name IBUF's FrameLink interface (data header output)
   -- -------------------------------------------------------------------------
   --! \brief Header data
   TX_HDATA              : out std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of the header, active in '0'
   TX_HSOP_N             : out std_logic;
   --! \brief End of the header, active in '0'
   TX_HEOP_N             : out std_logic;
   --! \brief Position of the end of the header
   --! \details Valid only if TX_HEOP_N is set to '0'.
   TX_HREM               : out std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   TX_HSRC_RDY_N         : out std_logic;
   --! \brief Destination is ready, active in '0'
   TX_HDST_RDY_N         : in  std_logic;

   --! OBUF's FrameLink Unaligned interface (data input)
   -- -------------------------------------------------------------------------
   --! \brief Input clock
   RX_CLK                : in  std_logic;
   --! \brief Input reset
   RX_RESET              : in  std_logic;
   --! \brief Packet data
   RX_DATA               : in  std_logic_vector(511 downto 0);
   --! \brief Position of the start of the packet
   --! \details Valid only if RX_SOP is set to '1'.
   RX_SOP_POS            : in  std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the packet
   --! \details Valid only if RX_EOP is set to '1'.
   RX_EOP_POS            : in  std_logic_vector(  5 downto 0);
   --! \brief Start of the packet
   RX_SOP                : in  std_logic;
   --! \brief End of the packet
   RX_EOP                : in  std_logic;
   --! \brief Source is ready
   RX_SRC_RDY            : in  std_logic;
   --! \brief Destination is ready
   RX_DST_RDY            : out std_logic;

   --! \name Sampling unit interface
   -- -------------------------------------------------------------------------
   --! \brief Clock for SAU component
   SAU_CLK               : out std_logic;
   --! \brief Reset for SAU component
   SAU_RESET             : out std_logic;
   --! \brief Request for sampling information
   SAU_REQ               : out std_logic;
   --! \brief Accept incoming frame
   SAU_ACCEPT            : in  std_logic;
   --! \brief Sampling information valid
   SAU_DV                : in  std_logic;

   --! \name Control data generator clock and reset
   -- -------------------------------------------------------------------------
   --! \brief Control data generator clock
   CTRL_CLK              : out std_logic;
   --! \brief Control data generator reset
   CTRL_RESET            : out std_logic;

   --! \name Statistics interface of control data generator
   -- -------------------------------------------------------------------------
   --! \brief New frame is being received; prepare to generate new control data
   CTRL_SOP              : out std_logic;
   --! \brief MAC address is not accepted
   CTRL_STAT_MAC_ERR     : out std_logic;
   --! \brief Frame does not have minimal length
   CTRL_STAT_MINTU_ERR   : out std_logic;
   --! \brief Frame is longer than maximal length
   CTRL_STAT_MTU_ERR     : out std_logic;
   --! \brief Frame has bad CRC
   CTRL_STAT_CRC_ERR     : out std_logic;
   --! \brief Incoming protocol error
   CTRL_STAT_FRAME_ERR   : out std_logic;
   --! \brief Frame length
   CTRL_STAT_FRAME_LEN   : out std_logic_vector(15 downto 0);
   --! \brief Statistics for control data generator are valid
   CTRL_STAT_DV          : out std_logic;
   --! \brief Control data generator is ready to receive new request
   CTRL_RDY              : in  std_logic;

   --! \name Data interface of control data generator
   -- -------------------------------------------------------------------------
   --! \brief Control data
   CTRL_DATA             : in  std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of control data, active in '0'
   CTRL_SOP_N            : in  std_logic;
   --! \brief End of control data, active in '0'
   CTRL_EOP_N            : in  std_logic;
   --! \brief Position of the end of control data
   --! \details Valid only if CTRL_EOP_N is set to '0'.
   CTRL_REM              : in  std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Control data generator is ready to send data, active in '0'
   CTRL_SRC_RDY_N        : in  std_logic;
   --! \brief Network module is ready to accept data, active in '0'
   CTRL_DST_RDY_N        : out std_logic;

   --! \name MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   MI_CLK                : in  std_logic;
   --! \brief Reset signal
   MI_RESET              : in  std_logic;
   --! \brief Input Data
   MI_DWR      	         : in  std_logic_vector(31 downto 0);
   --! \brief Address
   MI_ADDR     	         : in  std_logic_vector(31 downto 0);
   --! \brief Read Request
   MI_RD       	         : in  std_logic;
   --! \brief Write Request
   MI_WR                 : in  std_logic;
   --! \brief Byte Enable
   MI_BE                 : in  std_logic_vector( 3 downto 0);
   --! \brief Output Data
   MI_DRD      	         : out std_logic_vector(31 downto 0);
   --! \brief Address Ready
   MI_ARDY     	         : out std_logic;
   --! \brief Data Ready
   MI_DRDY     	         : out std_logic;

   --! \name Repeater control interface
   -- -------------------------------------------------------------------------
   --! \brief 2 bits to control the value of repeater's output
   --! \details "00" ... copy the output of OBUF
   --!          "01" ... send Idle control characters
   --!          "10" ... copy the output of OBUF
   --!          "11" ... copy the input of IBUF (i.e. reapeat input data)
   REPEATER_CTRL         : in  std_logic_vector(1 downto 0);

   --! \name LED interface
   -- -------------------------------------------------------------------------
   --! \brief Link LED driver, active in '0'
   LINK_LED              : out std_logic;
   --! \brief Activity LED driver, active in '0'
   ACTIVITY_LED          : out std_logic;

   --! \name Link presence interface
   -- -------------------------------------------------------------------------
   --! \brief Link presence signal ('1' = link is up, '0' = link is down)
   LINK            	 : out std_logic
);
end entity network_mod;