-- gtz_caui4_tx_startup_fsm.vhd : GTZ TX startup
--               
-- Copyright (C) 2014 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:
--   http://www.xilinx.com/support/answers/59038.html
-- 

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;


entity gtz_tx_startup_fsm is
   port (
        CLK            : in std_logic;
        RESET          : in std_logic;
        --
        MMCM_LOCKED_I  : in std_logic;
        TXRDY_I        : in std_logic;
        RXRDY_I        : in std_logic;
        TXRESETDONE    : in std_logic;
        PLLRECALEN_O   : out std_logic;
        MMCM_RESET_O   : out std_logic;
        TXEN_O         : out std_logic;
        RXEN_O         : out std_logic;
        TXRESET_O      : out std_logic;
        --
        RESET_DONE     : out std_logic;
        STATE_O        : out std_logic_vector(7 downto 0) -- Debug only
   );
end gtz_tx_startup_fsm;

architecture behavioral of gtz_tx_startup_fsm is

type t_ctle_reg_addr is array (3 downto 0) of std_logic_vector(15 downto 0);
constant CTLE_REG : t_ctle_reg_addr := (X"0681", X"0481", X"0281", X"0081");

type T_STARTUP_STATE is (PLLCAL_0,
                         PLLCAL_WAIT,
                         PLLCAL_WAIT1,
                         PLLCAL_WAIT2,
                         ASSERT_TXRESET,
                         TXRSTDONE_WAIT,
                         DONE_WAIT,
                         DONE
                          );
                         
signal startup_state : T_STARTUP_STATE;
signal timer    : std_logic_vector(14 downto 0);
signal timer_en : std_logic;
signal mmcm_locked_si  : std_logic := '0';
signal mmcm_locked_s   : std_logic := '0';
signal txrdy_si        : std_logic := '0';
signal txrdy_s         : std_logic := '0';
signal rxrdy_si        : std_logic := '0';
signal rxrdy_s         : std_logic := '0';
signal txresetdone_si  : std_logic := '0';
signal txresetdone_s   : std_logic := '0';

attribute SHREG_EXTRACT : string;
attribute ASYNC_REG     : string;
attribute ASYNC_REG of mmcm_locked_si : signal is "true";
attribute SHREG_EXTRACT of mmcm_locked_si : signal is "no";
attribute ASYNC_REG of mmcm_locked_s : signal is "true";
attribute SHREG_EXTRACT of mmcm_locked_s : signal is "no";
attribute ASYNC_REG of txrdy_si : signal is "true";
attribute SHREG_EXTRACT of txrdy_si : signal is "no";
attribute ASYNC_REG of txrdy_s : signal is "true";
attribute SHREG_EXTRACT of txrdy_s : signal is "no";
attribute ASYNC_REG of rxrdy_si : signal is "true";
attribute SHREG_EXTRACT of rxrdy_si : signal is "no";
attribute ASYNC_REG of rxrdy_s : signal is "true";
attribute SHREG_EXTRACT of rxrdy_s : signal is "no";
attribute ASYNC_REG of txresetdone_si : signal is "true";
attribute SHREG_EXTRACT of txresetdone_si : signal is "no";
attribute ASYNC_REG of txresetdone_s : signal is "true";
attribute SHREG_EXTRACT of txresetdone_s : signal is "no";
     
begin

-- Synchronize inputs to the CLK domain
RESYNC: process(CLK)
begin
   if CLK'event and CLK = '1' then
      mmcm_locked_si <= MMCM_LOCKED_I;
      mmcm_locked_s  <= mmcm_locked_si;
      
      txrdy_si <= TXRDY_I;
      txrdy_s  <= txrdy_si;
      
      rxrdy_si <= RXRDY_I;
      rxrdy_s  <= rxrdy_si;
      
      txresetdone_si <= TXRESETDONE;
      txresetdone_s  <= txresetdone_si;
   end if;
end process;

process(CLK)
begin
   if CLK'event and CLK = '1' then
      if timer_en = '1' then
         timer <= timer + 1;
      else
         timer <= (others => '0');
      end if;
   end if;
end process;

-- Implements FSM from http://www.xilinx.com/support/answers/59038.html
FSM : process(RESET,CLK) 
begin

   if CLK'event and CLK = '1' then
   
      if RESET = '1' then
         startup_state  <= PLLCAL_0;
         PLLRECALEN_O   <= '0';
         RXEN_O         <= '1';
         TXEN_O         <= '1';      
         TXRESET_O      <= '0';
         RESET_DONE     <= '0';
      else   
   
         TXRESET_O      <= '0';
         PLLRECALEN_O   <= '0';
         timer_en       <= '0';
         RESET_DONE     <= '0';
         MMCM_RESET_O   <= '0';
         
         case startup_state is
       
            when PLLCAL_0 => -- Start PLL calibration
               PLLRECALEN_O   <= '1';
               RXEN_O         <= '0'; -- Power off serdes
               TXEN_O         <= '0';
               TXRESET_O      <= '0';
               timer_en       <= '1';
               startup_state  <= PLLCAL_WAIT;
               
            when PLLCAL_WAIT => -- Pulse TXEN and RXEN low for at least 100 REFCLK cycles
               PLLRECALEN_O   <= '1';
               RXEN_O         <= '0';
               TXEN_O         <= '0';
               TXRESET_O      <= '0';
               timer_en       <= '1';
               if timer(10)= '1' then   
                  startup_state  <= PLLCAL_WAIT1;
               else
                  startup_state  <= PLLCAL_WAIT;
               end if;
               
            when PLLCAL_WAIT1 => -- Wait for TXRDY and RXRDY
               PLLRECALEN_O   <= '1';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '0';
               if (txrdy_s = '1') and (rxrdy_s = '1') then
                  startup_state  <= PLLCAL_WAIT2;
                  MMCM_RESET_O   <= '1';
               else
                  startup_state  <= PLLCAL_WAIT1;
               end if;
            
           when PLLCAL_WAIT2 => -- Wait for 50us as specified in AR#59038
               PLLRECALEN_O   <= '0';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '0';
               timer_en       <= '1';
               if (timer(14)= '1') and (mmcm_locked_s = '1') then 
                  startup_state  <= ASSERT_TXRESET;
                  TXRESET_O   <= '1';
               else
                  startup_state  <= PLLCAL_WAIT2;
               end if;
               
            when ASSERT_TXRESET => -- 
               PLLRECALEN_O   <= '0';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '1';
               startup_state  <= TXRSTDONE_WAIT;
            
            when TXRSTDONE_WAIT => -- 
               PLLRECALEN_O   <= '0';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '0';
               if (txresetdone_s = '1') then
                  startup_state  <= DONE_WAIT;
               else
                  startup_state  <= TXRSTDONE_WAIT;
               end if;
               
            when DONE_WAIT => -- Wait for 50us as specified in AR#59038
               PLLRECALEN_O   <= '0';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '0';            
               timer_en       <= '1';
               if timer(14)= '1' then  
                  startup_state  <= DONE;
               else
                  startup_state  <= DONE_WAIT;
               end if;
               
            when DONE => 
               PLLRECALEN_O   <= '0';
               RXEN_O         <= '1';
               TXEN_O         <= '1';
               TXRESET_O      <= '0';
               RESET_DONE     <= '1';
               startup_state  <= DONE;
               
         end case;
      end if;
   end if;
end process;

with startup_state select
   STATE_O <= X"00" when PLLCAL_0,
              X"01" when PLLCAL_WAIT,
              X"02" when PLLCAL_WAIT1,
              X"03" when PLLCAL_WAIT2,
              X"04" when ASSERT_TXRESET,
              X"05" when TXRSTDONE_WAIT,
              X"06" when DONE_WAIT,
              X"07" when DONE,
              X"FF" when others;

end behavioral;