-- flu_block_reord_control.vhd: FLU_BLOCK_REORD control component.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
--! Reordering control component
entity FLU_BLOCK_REORD_CONTROL is
   port (
      -----------------------------------------------------
      --! \name Common interface
      -----------------------------------------------------
      --! Clock signal
      CLK         : in std_logic;
      --! Synchronous reset
      RESET       : in std_logic;

      -----------------------------------------------------
      --! \name Input Intefrace 
      -----------------------------------------------------
      --! Measured FLU inter-frame gap
      FLU_GAP     : in std_logic_vector(4 downto 0);
      --! SOP block control information (one-hot) in selected reading window
      SOP_V_IN    : in std_logic_vector(7 downto 0);
      --! EOP block control information (one-hot) in selected reading window
      EOP_V_IN    : in std_logic_vector(7 downto 0);

      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! Data selection signal - lane 0
      DSEL_ADDR0 : out std_logic_vector(3 downto 0);
      --! Data selection signal - lane 1
      DSEL_ADDR1 : out std_logic_vector(7 downto 4);
      --! Data selection signal - lane 2
      DSEL_ADDR2 : out std_logic_vector(11 downto 8);
      --! Data selection signal - lane 3
      DSEL_ADDR3 : out std_logic_vector(15 downto 12);
      --! Data selection signal - lane 4
      DSEL_ADDR4 : out std_logic_vector(19 downto 16);
      --! Data selection signal - lane 5
      DSEL_ADDR5 : out std_logic_vector(23 downto 20);
      --! Data selection signal - lane 6
      DSEL_ADDR6 : out std_logic_vector(27 downto 24);
      --! Data selection signal - lane 7
      DSEL_ADDR7 : out std_logic_vector(31 downto 28);

      --! Control selection signal - lane 0
      CSEL_ADDR0 : out std_logic_vector(3 downto 0);
      --! Control selection signal - lane 1
      CSEL_ADDR1 : out std_logic_vector(7 downto 4);
      --! Control selection signal - lane 2
      CSEL_ADDR2 : out std_logic_vector(11 downto 8);
      --! Control selection signal - lane 3
      CSEL_ADDR3 : out std_logic_vector(15 downto 12);
      --! Control selection signal - lane 4
      CSEL_ADDR4 : out std_logic_vector(19 downto 16);
      --! Control selection signal - lane 5
      CSEL_ADDR5 : out std_logic_vector(23 downto 20);
      --! Control selection signal - lane 6
      CSEL_ADDR6 : out std_logic_vector(27 downto 24);
      --! Control selection signal - lane 7
      CSEL_ADDR7 : out std_logic_vector(31 downto 28);
   
      --! Delayed/actual EOP_POS in block selection
      EOP_POS_ADDR : out std_logic;

      --! Output SOP vector (one-hot coding)
      SOP_V_OUT   : out std_logic_vector(7 downto 0);
      --! Output EOP vector (one-hot coding)
      EOP_V_OUT   : out std_logic_vector(7 downto 0);

      --! Destination ready
      DST_RDY     : out std_logic;
      --! FLU gap acknowledge signal driven by this component
      FLU_GAP_ACK : out std_logic
   );
end entity FLU_BLOCK_REORD_CONTROL;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of flu_block_reord_control is
   -- Constants -------------------------------------------
   --! Base address increment (window skip constant)
   constant BASE_ADDR_INCREMENT : std_logic_vector := "1001"; -- dec(1001) = 9

   -- FSM -------------------------------------------------
   --! State type definition
   type T_FSM_STATE is (DATA,SOP_MOVED);
   --! Actual state signal 
   signal actual_state  : T_FSM_STATE;
   --! Next state signal
   signal next_state    : T_FSM_STATE;

   --! Signal for detection that SOP needs to be shifted to next FLU word
   signal sop_to_next   : std_logic;

   --Address part signals ---------------------------------
   --! Base address register
   signal base_addr     : std_logic_vector(3 downto 0);
   --! Last used block selection
   signal base_addr_sel : std_logic_vector(2 downto 0);

   --Base address constants
   signal base_addr_0   : std_logic_vector(3 downto 0) := "0000";
   signal base_addr_1   : std_logic_vector(3 downto 0) := "0001";
   signal base_addr_2   : std_logic_vector(3 downto 0) := "0010";
   signal base_addr_3   : std_logic_vector(3 downto 0) := "0011";
   signal base_addr_4   : std_logic_vector(3 downto 0) := "0100";
   signal base_addr_5   : std_logic_vector(3 downto 0) := "0101";
   signal base_addr_6   : std_logic_vector(3 downto 0) := "0110";
   signal base_addr_7   : std_logic_vector(3 downto 0) := "0111";

   --Output address selection signals
   signal port0_addr_sel   : std_logic_vector(2 downto 0);
   signal port1_addr_sel   : std_logic_vector(2 downto 0);
   signal port2_addr_sel   : std_logic_vector(2 downto 0);
   signal port3_addr_sel   : std_logic_vector(2 downto 0);
   signal port4_addr_sel   : std_logic_vector(2 downto 0);
   signal port5_addr_sel   : std_logic_vector(2 downto 0);
   signal port6_addr_sel   : std_logic_vector(2 downto 0);
   signal port7_addr_sel   : std_logic_vector(2 downto 0);

   --DIC & Block shift lookup table
   --! DIC register
   signal dic_reg          : std_logic_vector(2 downto 0) := (others => '0');
   --! DIC register enable signal
   signal dic_reg_en       : std_logic;
   --! New DIC value from DIC lookup table
   signal new_dic_val      : std_logic_vector(2 downto 0);
   
   --! SOP block shift value from lookup table
   signal sop_block_shift  : std_logic_vector(1 downto 0);

   --! Register which holds a number of remaining blocks to insert
   signal rem_blocks_reg   : std_logic_vector(1 downto 0) := (others => '0');
   --! Remaining blocks value
   signal rem_blocks_val   : std_logic_vector(1 downto 0);
   --! Register enable signals
   signal rem_blocks_en    : std_logic;

   --! Wait enable -> detect condition when base addr is in register address place AND some data are moved to next word
   signal wait_en             : std_logic;
   --! Wait enable for SOP_MOVED state
   signal wait_sop_moved_en   : std_logic;
   --! Base repair signal 
   signal base_rep_en      : std_logic;
   --! EOP detecting signal
   signal eop_det_en       : std_logic := '0';
   --! Destination ready signal
   signal dst_rdy_out      : std_logic;

begin

   --! This signal detects whether the flu gap was used in this clock cycle or not
   FLU_GAP_ACK <= '1' when((SOP_V_IN /= "00000000" and next_state /= SOP_MOVED) or actual_state = SOP_MOVED)else
                  '0';

   --------------------------------------------------------
   --DIC & Block shift lookup table
   --------------------------------------------------------
   --! \brief DIC register
   --! \details This process implements DIC register with synchronous reset

   dic_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            dic_reg <= "000";
         else
            if(dic_reg_en = '1')then
               dic_reg <= new_dic_val;
            end if;
         end if;
      end if;
   end process;

   --! Block shift lookup table
   DIC_SHIFT_LOOKUP_I:entity work.DIC_BLOCK_TABLE
   port map(
      -----------------------------------------------------
      --! \name Input interface
      -----------------------------------------------------
      --! Actual DIC value
      ACT_DIC_VAL       => dic_reg,
      --! Inter-frame gap between packets
      FLU_GAP           => FLU_GAP,

      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! SOP block shift with respect to ACT_DIC_VAL, SOP_VECT and FLU_GAP
      SOP_BLOCK_SHIFT   => sop_block_shift,
      --! New DIC value with respect to ACT_DIC_VAL, SOP_VECT and FLU_GAP
      NEW_DIC_VAL       => new_dic_val
   );

   --------------------------------------------------------
   -- FSM
   --------------------------------------------------------
   --! \brief State register
   --! \details This process stores actual FSM state

   state_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            actual_state <= DATA;
         else
            actual_state <= next_state;
         end if;
      end if;
   end process;

   --! \brief Remaining blocks register

   rem_blocks_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            rem_blocks_reg <= "00";
         else
            if(rem_blocks_en = '1')then
               rem_blocks_reg <= rem_blocks_val;
            end if;
         end if;
      end if;
   end process;

   --! \brief End of the frame detection.
   --! \details This process set EOP_DET_EN signal when the EOP was detected
   --! and no SOP has been detected.

   eop_det_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            eop_det_en <= '0';
         else
            if(eop_det_en = '1')then
               --END of the transfer has been detected
               if(SOP_V_IN /= "00000000" and EOP_V_IN = "00000000")then
                  eop_det_en <= '0';
               end if; --SOP_V_IN ...
            else
               --Transfer is running
               if((EOP_V_IN /= "00000000" and SOP_V_IN = "00000000") or ((EOP_V_IN /= "00000000" and SOP_V_IN /= "00000000") and (EOP_V_IN > SOP_V_IN)  ))then
                  --No transfer is continuing. Enable eop_det_en signal
                  eop_det_en <= '1';
               end if; --EOP_V_IN ...
            end if; --eop_det_en ...
         end if;
      end if;
   end process;

   --Helping signals implementation
   sop_to_next <= '1' when ((SOP_V_IN(5) = '1' and sop_block_shift = 3) or
                            (SOP_V_IN(6) = '1' and sop_block_shift > 1) or
                            (SOP_V_IN(7) = '1' and sop_block_shift /= 0))
                  else '0';

   --Helping signal which detects situation when all data reg blocks are not transfered because of SOP block 
   --shifting in register data address block. This detection signal is used in DATA state.
      wait_en <= '1' when((base_addr = 8 and SOP_V_IN(7 downto 0) /= 0 and sop_block_shift > 0) or
                          (base_addr = 9 and SOP_V_IN(6 downto 0) /= 0 and sop_block_shift > 1 ) or
                          (base_addr = 10 and SOP_V_IN(5 downto 0) /= 0 and sop_block_shift > 2)) 
                  else '0';


   --Helping signal which detects situation when all data reg blocks are not transfered because of SOP block 
   --shifting in register data address block. This detection signal is used in SOP_MOVED state.
      wait_sop_moved_en <= '1' when((base_addr = 8 and rem_blocks_reg > 0) or
                                    (base_addr = 9 and rem_blocks_reg > 1))
                  else '0';

   --! EOP pos signal map
   --! One special situation can appear when sop < eop in one flu word
   --! In this case when eop overflows to the next flu word I have to
   --! mask EOP_V_IN

  EOP_V_OUT <= (others => '0') when (EOP_V_IN(7) = '1' and base_addr_sel /= "111")else
               EOP_V_IN;

   --! EOP_POS block selection signal impelemtantion
   EOP_POS_ADDR <= '1' when ((EOP_V_IN(0) = '1' and base_addr_0 > 7) or
                             (EOP_V_IN(1) = '1' and base_addr_1 > 7) or
                             (EOP_V_IN(2) = '1' and base_addr_2 > 7) or
                             (EOP_V_IN(3) = '1' and base_addr_3 > 7) or
                             (EOP_V_IN(4) = '1' and base_addr_4 > 7) or
                             (EOP_V_IN(5) = '1' and base_addr_5 > 7) or
                             (EOP_V_IN(6) = '1' and base_addr_6 > 7) or
                             (EOP_V_IN(7) = '1' and base_addr_7 > 7))
                  else '0';

   --! Destionation ready signal 
   DST_RDY <= dst_rdy_out;

   --! \brief FSM transition process
   fsm_transp:process(actual_state,sop_to_next,wait_en)
   begin
      --Default state is loop
      next_state <= actual_state;

      case actual_state is
         when DATA =>
            if(sop_to_next = '1')then
               next_state <= SOP_MOVED;
            end if;

         when SOP_MOVED =>
            next_state <= DATA; 

         when others => null;
      end case;
   end process;

   --! \name FSM output process
   fsm_outputp:process(actual_state,sop_block_shift,SOP_V_IN,EOP_V_IN,rem_blocks_reg,wait_en,wait_sop_moved_en,eop_det_en)
   begin
      -------------------------------------------
      --Defatult values ---------------
      --Address part         
      base_addr_sel  <= "111";
      port0_addr_sel <= "000";
      port1_addr_sel <= "001";
      port2_addr_sel <= "010";
      port3_addr_sel <= "011";
      port4_addr_sel <= "100";
      port5_addr_sel <= "101";
      port6_addr_sel <= "110";
      port7_addr_sel <= "111";
      
      --DIC control
      dic_reg_en <= '0';

      --Rem. blocks control
      rem_blocks_en  <= '0';
      rem_blocks_val <= "00";

      --SOP_V_OUT value (copy)
      SOP_V_OUT <= SOP_V_IN;

      --Destination ready
      dst_rdy_out <= '1';

      --Base repair enable signal
      base_rep_en <= '0';
      -------------------------------------------

      case actual_state is
         when DATA =>
            --DIC register enable signal
            if(SOP_V_IN /= "00000000")then
               dic_reg_en <= '1';

               --Detect if we need to stop transfer
               if(wait_en = '1')then
                  dst_rdy_out <= '0';
                  base_rep_en <= '1';
               end if;
            end if;

            --Output block selection
            for i in 0 to 7 loop
            case i is
            when 0 =>
            if(SOP_V_IN(i) = '1')then
               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00000010";
                  port1_addr_sel <= "000";
                  port2_addr_sel <= "001";
                  port3_addr_sel <= "010";
                  port4_addr_sel <= "011";
                  port5_addr_sel <= "100";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "00000100";
                  port2_addr_sel <= "000";
                  port3_addr_sel <= "001";
                  port4_addr_sel <= "010";
                  port5_addr_sel <= "011";
                  port6_addr_sel <= "100";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "00001000";
                  port3_addr_sel <= "000";
                  port4_addr_sel <= "001";
                  port5_addr_sel <= "010";
                  port6_addr_sel <= "011";
                  port7_addr_sel <= "100";
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 1 =>
            if(SOP_V_IN(i) = '1')then
               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00000100";
                  port2_addr_sel <= "001";
                  port3_addr_sel <= "010";
                  port4_addr_sel <= "011";
                  port5_addr_sel <= "100";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "00001000";
                  port3_addr_sel <= "001";
                  port4_addr_sel <= "010";
                  port5_addr_sel <= "011";
                  port6_addr_sel <= "100";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "00010000";
                  port4_addr_sel <= "001";
                  port5_addr_sel <= "010";
                  port6_addr_sel <= "011";
                  port7_addr_sel <= "100";
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 2 =>
            if(SOP_V_IN(i) = '1')then
               
               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00001000";
                  port3_addr_sel <= "010";
                  port4_addr_sel <= "011";
                  port5_addr_sel <= "100";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "00010000";
                  port4_addr_sel <= "010";
                  port5_addr_sel <= "011";
                  port6_addr_sel <= "100";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "00100000";
                  port5_addr_sel <= "010";
                  port6_addr_sel <= "011";
                  port7_addr_sel <= "100";
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 3 =>
            if(SOP_V_IN(i) = '1')then

               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00010000";
                  port4_addr_sel <= "011";
                  port5_addr_sel <= "100";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "00100000";
                  port5_addr_sel <= "011";
                  port6_addr_sel <= "100";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "01000000";
                  port6_addr_sel <= "011";
                  port7_addr_sel <= "100";
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 4 =>
            if(SOP_V_IN(i) = '1')then

               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00100000";
                  port5_addr_sel <= "100";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "01000000";
                  port6_addr_sel <= "100";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "10000000";
                  port7_addr_sel <= "100";
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 5 =>
            if(SOP_V_IN(i) = '1')then

               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "01000000";
                  port6_addr_sel <= "101";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "10000000";
                  port7_addr_sel <= "101";
                  base_addr_sel <= "101";
               when "11" =>
                  SOP_V_OUT <= "00000000";
                  rem_blocks_en <= '1';
                  base_addr_sel <= "100";
               when others => null;
               end case;
            end if;

            when 6 =>
            if(SOP_V_IN(i) = '1')then

               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "10000000";
                  port7_addr_sel <= "110";
                  base_addr_sel <= "110";
               when "10" =>
                  SOP_V_OUT <= "00000000";
                  base_addr_sel <= "101";
                  rem_blocks_en <= '1';
               when "11" =>
                  SOP_V_OUT <= "00000000";
                  base_addr_sel <= "101";
                  rem_blocks_en <= '1';
                  rem_blocks_val <= "01";
               when others => null;
               end case;
            end if;

            when 7 =>
            if (SOP_V_IN(i) = '1')then

               case sop_block_shift is
               when "01" =>
                  SOP_V_OUT <= "00000000";
                  base_addr_sel <= "110";
                  rem_blocks_en <= '1';
               when "10" =>
                  SOP_V_OUT <= "00000000";
                  base_addr_sel <= "110";
                  rem_blocks_en <= '1';
                  rem_blocks_val <= "01";
               when "11" =>
                  SOP_V_OUT <= "00000000";
                  base_addr_sel <= "110";
                  rem_blocks_en <= '1';
                  rem_blocks_val <= "10";
               when others => null;
               end case;
            end if;
            
            when others => null;
            end case;
            end loop;
         
         when SOP_MOVED =>
            --If you need to wait, disable destination ready and enable repair signal
            if(wait_sop_moved_en = '1')then
               dst_rdy_out <= '0';
               base_rep_en <= '1'; 
            end if;

            --Expected SOP_V indication is 0001. Make some data block changes
            if(rem_blocks_reg = "00")then
               SOP_V_OUT <= "00000001";
            elsif(rem_blocks_reg = "01")then
               SOP_V_OUT <= "00000010";
               port1_addr_sel <= "000";
               port2_addr_sel <= "001";
               port3_addr_sel <= "010";
               port4_addr_sel <= "011";
               port5_addr_sel <= "100";
               port6_addr_sel <= "101";
               port7_addr_sel <= "110";
               base_addr_sel <= "110";
            elsif(rem_blocks_reg = "10")then
               SOP_V_OUT <= "00000100";
               port2_addr_sel <= "000";
               port3_addr_sel <= "001";
               port4_addr_sel <= "010";
               port5_addr_sel <= "011";
               port6_addr_sel <= "100";
               port7_addr_sel <= "101";
               base_addr_sel <= "101";
            end if;

         when others => null;

      end case;
   end process;
   
   --------------------------------------------------------
   --Address part of the control
   --------------------------------------------------------

   --! Base addr0 connection
   base_addr <= base_addr_0;

   --! \brief Base address register
   --! \details This process implements "reading window" movement.

   base_addr_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            base_addr_0 <= "0000";
            base_addr_1 <= "0001";
            base_addr_2 <= "0010";
            base_addr_3 <= "0011";
            base_addr_4 <= "0100";
            base_addr_5 <= "0101";
            base_addr_6 <= "0110";
            base_addr_7 <= "0111";
         else
            if(base_rep_en = '0')then
               --We don't need to repair base address
               if(base_addr_sel = "000")then
                  base_addr_0 <= base_addr_0 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_0 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_0 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_0 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_0 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_0 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_0 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_0 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "001")then
                  base_addr_0 <= base_addr_1 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_1 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_1 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_1 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_1 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_1 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_1 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_1 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "010")then
                  base_addr_0 <= base_addr_2 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_2 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_2 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_2 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_2 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_2 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_2 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_2 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "011")then
                  base_addr_0 <= base_addr_3 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_3 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_3 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_3 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_3 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_3 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_3 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_3 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "100")then
                  base_addr_0 <= base_addr_4 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_4 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_4 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_4 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_4 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_4 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_4 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_4 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "101")then
                  base_addr_0 <= base_addr_5 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_5 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_5 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_5 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_5 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_5 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_5 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_5 + BASE_ADDR_INCREMENT + 7;
               elsif(base_addr_sel = "110")then
                  base_addr_0 <= base_addr_6 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_6 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_6 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_6 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_6 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_6 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_6 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_6 + BASE_ADDR_INCREMENT + 7;
               else
                  base_addr_0 <= base_addr_7 + BASE_ADDR_INCREMENT;
                  base_addr_1 <= base_addr_7 + BASE_ADDR_INCREMENT + 1;
                  base_addr_2 <= base_addr_7 + BASE_ADDR_INCREMENT + 2;
                  base_addr_3 <= base_addr_7 + BASE_ADDR_INCREMENT + 3;
                  base_addr_4 <= base_addr_7 + BASE_ADDR_INCREMENT + 4;
                  base_addr_5 <= base_addr_7 + BASE_ADDR_INCREMENT + 5;
                  base_addr_6 <= base_addr_7 + BASE_ADDR_INCREMENT + 6;
                  base_addr_7 <= base_addr_7 + BASE_ADDR_INCREMENT + 7;
               end if;
            else
               --We need to repair base address register
               if(base_addr_sel = "000")then
                  base_addr_0 <= base_addr_1;
                  base_addr_1 <= base_addr_1 + 1;
                  base_addr_2 <= base_addr_1 + 2;
                  base_addr_3 <= base_addr_1 + 3;
                  base_addr_4 <= base_addr_1 + 4;
                  base_addr_5 <= base_addr_1 + 5;
                  base_addr_6 <= base_addr_1 + 6;
                  base_addr_7 <= base_addr_1 + 7;
               elsif(base_addr_sel = "001")then
                  base_addr_0 <= base_addr_2;
                  base_addr_1 <= base_addr_2 + 1;
                  base_addr_2 <= base_addr_2 + 2;
                  base_addr_3 <= base_addr_2 + 3;
                  base_addr_4 <= base_addr_2 + 4;
                  base_addr_5 <= base_addr_2 + 5;
                  base_addr_6 <= base_addr_2 + 6;
                  base_addr_7 <= base_addr_2 + 7;
               elsif(base_addr_sel = "010")then
                  base_addr_0 <= base_addr_3;
                  base_addr_1 <= base_addr_3 + 1;
                  base_addr_2 <= base_addr_3 + 2;
                  base_addr_3 <= base_addr_3 + 3;
                  base_addr_4 <= base_addr_3 + 4;
                  base_addr_5 <= base_addr_3 + 5;
                  base_addr_6 <= base_addr_3 + 6;
                  base_addr_7 <= base_addr_3 + 7;
               elsif(base_addr_sel = "011")then
                  base_addr_0 <= base_addr_4;
                  base_addr_1 <= base_addr_4 + 1;
                  base_addr_2 <= base_addr_4 + 2;
                  base_addr_3 <= base_addr_4 + 3;
                  base_addr_4 <= base_addr_4 + 4;
                  base_addr_5 <= base_addr_4 + 5;
                  base_addr_6 <= base_addr_4 + 6;
                  base_addr_7 <= base_addr_4 + 7;
               elsif(base_addr_sel = "100")then
                  base_addr_0 <= base_addr_5;
                  base_addr_1 <= base_addr_5 + 1;
                  base_addr_2 <= base_addr_5 + 2;
                  base_addr_3 <= base_addr_5 + 3;
                  base_addr_4 <= base_addr_5 + 4;
                  base_addr_5 <= base_addr_5 + 5;
                  base_addr_6 <= base_addr_5 + 6;
                  base_addr_7 <= base_addr_5 + 7;
               elsif(base_addr_sel = "101")then
                  base_addr_0 <= base_addr_6;
                  base_addr_1 <= base_addr_6 + 1;
                  base_addr_2 <= base_addr_6 + 2;
                  base_addr_3 <= base_addr_6 + 3;
                  base_addr_4 <= base_addr_6 + 4;
                  base_addr_5 <= base_addr_6 + 5;
                  base_addr_6 <= base_addr_6 + 6;
                  base_addr_7 <= base_addr_6 + 7;
               else
                  base_addr_0 <= base_addr_7;
                  base_addr_1 <= base_addr_7 + 1;
                  base_addr_2 <= base_addr_7 + 2;
                  base_addr_3 <= base_addr_7 + 3;
                  base_addr_4 <= base_addr_7 + 4;
                  base_addr_5 <= base_addr_7 + 5;
                  base_addr_6 <= base_addr_7 + 6;
                  base_addr_7 <= base_addr_7 + 7;
               end if;
            end if;
         end if;
      end if;
   end process;

   --Output port address multiplexers
   --Port 0
   port0_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port0_addr_sel)
   begin
      case port0_addr_sel is
         when "000"   => DSEL_ADDR0 <= base_addr_0;
         when "001"   => DSEL_ADDR0 <= base_addr_1;
         when "010"   => DSEL_ADDR0 <= base_addr_2;
         when "011"   => DSEL_ADDR0 <= base_addr_3;
         when "100"   => DSEL_ADDR0 <= base_addr_4;
         when "101"   => DSEL_ADDR0 <= base_addr_5;
         when "110"   => DSEL_ADDR0 <= base_addr_6;
         when "111"   => DSEL_ADDR0 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 1
   port1_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port1_addr_sel)
   begin
      case port1_addr_sel is
         when "000"   => DSEL_ADDR1 <= base_addr_0;
         when "001"   => DSEL_ADDR1 <= base_addr_1;
         when "010"   => DSEL_ADDR1 <= base_addr_2;
         when "011"   => DSEL_ADDR1 <= base_addr_3;
         when "100"   => DSEL_ADDR1 <= base_addr_4;
         when "101"   => DSEL_ADDR1 <= base_addr_5;
         when "110"   => DSEL_ADDR1 <= base_addr_6;
         when "111"   => DSEL_ADDR1 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 2
   port2_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port2_addr_sel)
   begin
      case port2_addr_sel is
         when "000"   => DSEL_ADDR2 <= base_addr_0;
         when "001"   => DSEL_ADDR2 <= base_addr_1;
         when "010"   => DSEL_ADDR2 <= base_addr_2;
         when "011"   => DSEL_ADDR2 <= base_addr_3;
         when "100"   => DSEL_ADDR2 <= base_addr_4;
         when "101"   => DSEL_ADDR2 <= base_addr_5;
         when "110"   => DSEL_ADDR2 <= base_addr_6;
         when "111"   => DSEL_ADDR2 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 3
   port3_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port3_addr_sel)
   begin
      case port3_addr_sel is
         when "000"   => DSEL_ADDR3 <= base_addr_0;
         when "001"   => DSEL_ADDR3 <= base_addr_1;
         when "010"   => DSEL_ADDR3 <= base_addr_2;
         when "011"   => DSEL_ADDR3 <= base_addr_3;
         when "100"   => DSEL_ADDR3 <= base_addr_4;
         when "101"   => DSEL_ADDR3 <= base_addr_5;
         when "110"   => DSEL_ADDR3 <= base_addr_6;
         when "111"   => DSEL_ADDR3 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 4
   port4_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port4_addr_sel)
   begin
      case port4_addr_sel is
         when "000"   => DSEL_ADDR4 <= base_addr_0;
         when "001"   => DSEL_ADDR4 <= base_addr_1;
         when "010"   => DSEL_ADDR4 <= base_addr_2;
         when "011"   => DSEL_ADDR4 <= base_addr_3;
         when "100"   => DSEL_ADDR4 <= base_addr_4;
         when "101"   => DSEL_ADDR4 <= base_addr_5;
         when "110"   => DSEL_ADDR4 <= base_addr_6;
         when "111"   => DSEL_ADDR4 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 5
   port5_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port5_addr_sel)
   begin
      case port5_addr_sel is
         when "000"   => DSEL_ADDR5 <= base_addr_0;
         when "001"   => DSEL_ADDR5 <= base_addr_1;
         when "010"   => DSEL_ADDR5 <= base_addr_2;
         when "011"   => DSEL_ADDR5 <= base_addr_3;
         when "100"   => DSEL_ADDR5 <= base_addr_4;
         when "101"   => DSEL_ADDR5 <= base_addr_5;
         when "110"   => DSEL_ADDR5 <= base_addr_6;
         when "111"   => DSEL_ADDR5 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 6
   port6_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port6_addr_sel)
   begin
      case port6_addr_sel is
         when "000"   => DSEL_ADDR6 <= base_addr_0;
         when "001"   => DSEL_ADDR6 <= base_addr_1;
         when "010"   => DSEL_ADDR6 <= base_addr_2;
         when "011"   => DSEL_ADDR6 <= base_addr_3;
         when "100"   => DSEL_ADDR6 <= base_addr_4;
         when "101"   => DSEL_ADDR6 <= base_addr_5;
         when "110"   => DSEL_ADDR6 <= base_addr_6;
         when "111"   => DSEL_ADDR6 <= base_addr_7;
         when others => null;
      end case;
   end process;

   --Port 7
   port7_addr_muxp:process(base_addr_0,base_addr_1,base_addr_2,base_addr_3,base_addr_4,base_addr_5,base_addr_6,base_addr_7,port7_addr_sel)
   begin
      case port7_addr_sel is
         when "000"   => DSEL_ADDR7 <= base_addr_0;
         when "001"   => DSEL_ADDR7 <= base_addr_1;
         when "010"   => DSEL_ADDR7 <= base_addr_2;
         when "011"   => DSEL_ADDR7 <= base_addr_3;
         when "100"   => DSEL_ADDR7 <= base_addr_4;
         when "101"   => DSEL_ADDR7 <= base_addr_5;
         when "110"   => DSEL_ADDR7 <= base_addr_6;
         when "111"   => DSEL_ADDR7 <= base_addr_7;
         when others => null;
      end case;
   end process;


   --! Control block selection addresses
   CSEL_ADDR0 <= base_addr_0;
   CSEL_ADDR1 <= base_addr_1;
   CSEL_ADDR2 <= base_addr_2;
   CSEL_ADDR3 <= base_addr_3;
   CSEL_ADDR4 <= base_addr_4;
   CSEL_ADDR5 <= base_addr_5;
   CSEL_ADDR6 <= base_addr_6;
   CSEL_ADDR7 <= base_addr_7;

end architecture full;
