-- am_check.vhd : Alignment marker checker for 20-lane PCS
--                     
-- Copyright (C) 2012 CESNET
-- Author(s): Stepan Friedl <friedl@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: $
--
-- NOTES:
-- Marker encoding 
-- Lane  Encoding
-- 0     0xC1, 0x68, 0x21, BIP3, 0x3E, 0x97, 0xDE, BIP7 
-- 1     0x9D, 0x71, 0x8E, BIP3, 0x62, 0x8E, 0x71, BIP7 
-- 2     0x59, 0x4B, 0xE8, BIP3, 0xA6, 0xB4, 0x17, BIP7 
-- 3     0x4D, 0x95, 0x7B, BIP3, 0xB2, 0x6A, 0x84, BIP7 
-- 4     0xF5, 0x07, 0x09, BIP3, 0x0A, 0xF8, 0xF6, BIP7 
-- 5     0xDD, 0x14, 0xC2, BIP3, 0x22, 0xEB, 0x3D, BIP7 
-- 6     0x9A, 0x4A, 0x26, BIP3, 0x65, 0xB5, 0xD9, BIP7 
-- 7     0x7B, 0x45, 0x66, BIP3, 0x84, 0xBA, 0x99, BIP7 
-- 8     0xA0, 0x24, 0x76, BIP3, 0x5F, 0xDB, 0x89, BIP7 
-- 9     0x68, 0xC9, 0xFB, BIP3, 0x97, 0x36, 0x04, BIP7 
-- 10    0xFD, 0x6C, 0x99, BIP3, 0x02, 0x93, 0x66, BIP7
-- 11    0xB9, 0x91, 0x55, BIP3, 0x46, 0x6E, 0xAA, BIP7
-- 12    0x5C, 0xB9, 0xB2, BIP3, 0xA3, 0x46, 0x4D, BIP7
-- 13    0x1A, 0xF8, 0xBD, BIP3, 0xE5, 0x07, 0x42, BIP7
-- 14    0x83, 0xC7, 0xCA, BIP3, 0x7C, 0x38, 0x35, BIP7
-- 15    0x35, 0x36, 0xCD, BIP3, 0xCA, 0xC9, 0x32, BIP7
-- 16    0xC4, 0x31, 0x4C, BIP3, 0x3B, 0xCE, 0xB3, BIP7
-- 17    0xAD, 0xD6, 0xB7, BIP3, 0x52, 0x29, 0x48, BIP7
-- 18    0x5F, 0x66, 0x2A, BIP3, 0xA0, 0x99, 0xD5, BIP7
-- 19    0xC0, 0xF0, 0xE5, BIP3, 0x3F, 0x0F, 0x1A, BIP7
--
-- BIP calculation table
-- 0 2, 10, 18, 26, 34, 42, 50, 58
-- 1 3, 11, 19, 27, 35, 43, 51, 59
-- 2 4, 12, 20, 28, 36, 44, 52, 60
-- 3 0, 5, 13, 21, 29, 37, 45, 53, 61
-- 4 1, 6, 14, 22, 30, 38, 46, 54, 62
-- 5 7, 15, 23, 31, 39, 47, 55, 63
-- 6 8, 16, 24, 32, 40, 48, 56, 64
-- 7 9, 17, 25, 33, 41, 49, 57, 65

library ieee;
use ieee.std_logic_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

entity am_check_20 is
   port (
      RESET     : in std_logic; -- Reset current computation and start a new one
      CLK       : in std_logic; -- 
      EN        : in std_logic; 
      D         : in std_logic_vector(65 downto 0);   -- Input data
      MATCH     : out std_logic_vector(19 downto 0);  -- Marker found, matching lane
      BIP_MATCH : out std_logic                       -- Bit parity computation OK
   );
end am_check_20;

architecture twenty_lane of am_check_20 is

type t_lane_markers is array (0 to 19) of std_logic_vector(23 downto 0);
-- Marker table
constant M_TABLE : t_lane_markers := (X"21" & X"68" & X"C1",
                                      X"8E" & X"71" & X"9D",
                                      X"E8" & X"4B" & X"59",
                                      X"7B" & X"95" & X"4D",
                                      X"09" & X"07" & X"F5",
                                      X"C2" & X"14" & X"DD",
                                      X"26" & X"4A" & X"9A",
                                      X"66" & X"45" & X"7B",
                                      X"76" & X"24" & X"A0",
                                      X"FB" & X"C9" & X"68",
                                      X"99" & X"6C" & X"FD",
                                      X"55" & X"91" & X"B9",
                                      X"B2" & X"B9" & X"5C",
                                      X"BD" & X"F8" & X"1A",
                                      X"CA" & X"C7" & X"83",
                                      X"CD" & X"36" & X"35",
                                      X"4C" & X"31" & X"C4",
                                      X"B7" & X"D6" & X"AD",
                                      X"2A" & X"66" & X"5F",
                                      X"E5" & X"F0" & X"C0");

signal bip_prev : std_logic_vector( 7 downto 0);
signal bip      : std_logic_vector( 7 downto 0);
signal bip_r    : std_logic_vector( 7 downto 0) := X"00";
signal m_match  : std_logic_vector(19 downto 0);
signal sync_hdr : std_logic;

begin

sync_hdr <= '1' when D(1 downto 0) = "01" else '0';

MATCH_GEN: for i in 0 to 19 generate
   m_match(i) <= '1' when (D(25 downto 2) = M_TABLE(i)) and (D(57 downto 34) = (not M_TABLE(i))) 
                 else '0';
   MATCH(i)  <= m_match(i) and sync_hdr;
end generate;

BIP_MATCH <= '1' when (bip_r = D(33 downto 26)) else '0';

bip_prev <= X"00" when RESET = '1' else bip_r;

bip(0) <= bip_prev(0) xor D(2) xor D(10) xor D(18) xor D(26) xor D(34) xor D(42) xor D(50) xor D(58);
bip(1) <= bip_prev(1) xor D(3) xor D(11) xor D(19) xor D(27) xor D(35) xor D(43) xor D(51) xor D(59);
bip(2) <= bip_prev(2) xor D(4) xor D(12) xor D(20) xor D(28) xor D(36) xor D(44) xor D(52) xor D(60);
bip(3) <= bip_prev(3) xor D(0) xor D( 5) xor D(13) xor D(21) xor D(29) xor D(37) xor D(45) xor D(53) xor D(61);
bip(4) <= bip_prev(4) xor D(1) xor D( 6) xor D(14) xor D(22) xor D(30) xor D(38) xor D(46) xor D(54) xor D(62);
bip(5) <= bip_prev(5) xor D(7) xor D(15) xor D(23) xor D(31) xor D(39) xor D(47) xor D(55) xor D(63);
bip(6) <= bip_prev(6) xor D(8) xor D(16) xor D(24) xor D(32) xor D(40) xor D(48) xor D(56) xor D(64);
bip(7) <= bip_prev(7) xor D(9) xor D(17) xor D(25) xor D(33) xor D(41) xor D(49) xor D(57) xor D(65);

BIP_SEQ: process(CLK,RESET)
begin
   if CLK'event and CLK = '1' then
      if EN = '1' then
         bip_r <= bip;
      end if;
   end if;
end process;
-- 

end twenty_lane;
