-- dma_ctrl_rx_newdata.vhd: New Data Process of Generic RX DMA Controller
-- Copyright (C) 2013 CESNET
-- Author(s): Martin Spinler <spinler@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--

library ieee;
library work;
use ieee.std_logic_1164.all;
use ieee.std_logic_arith.all;
use ieee.std_logic_unsigned.all;
use ieee.numeric_std.all;
use work.math_pack.all;
use work.dma_pkg.all;

entity dma_ctrl_rx_newdata is
   generic (
      CHANNELS          : integer;
      BUFFER_SIZE       : integer;
      DATA_ALIGN        : integer
   );
  -- interface description
   port (
      CLK               : in  std_logic;
      RESET             : in  std_logic;

      --! Interface from DMA Buffer
      PACKET_NEW        : in  std_logic;
      PACKET_DISC       : in  std_logic;
      PACKET_CHANNEL    : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      PACKET_LENGTH     : in  std_logic_vector(log2(BUFFER_SIZE) downto 0);

      --! Global signals of NewData process
      CHANNEL           : out std_logic_vector(log2(CHANNELS)-1 downto 0);
      GEN_INTERRUPT     : out std_logic;
      SW_STATUS         : out std_logic_vector(7 downto 0);
      --! Interface for monitoring changes
      SW_CONTROL        : in  std_logic_vector(1 downto 0);
      SW_CHANNEL        : in  std_logic_vector(log2(CHANNELS)-1 downto 0);
      SW_CONTROL_WR     : in  std_logic;
      SW_PTR_WR         : in  std_logic;

      --! Controller state
      ENABLE            : out std_logic_vector(CHANNELS-1 downto 0);
      ENABLE_BUFFER     : out std_logic_vector(CHANNELS-1 downto 0);
      DISCARD           : out std_logic_vector(CHANNELS-1 downto 0);

      --! Interface from Descriptor Manager
      DESC_DATA         : in  std_logic_vector(DMA_DESC_WIDTH-1 downto 0);
      DESC_RDY          : in  std_logic;

      --! Interface to Request unit
      DMA_RDY           : in  std_logic;
      DMA_REQ           : out std_logic;
      DMA_LENGTH        : out std_logic_vector(PAGE_WIDTH - log2(DATA_ALIGN) downto 0);
      DMA_CHANNEL       : out std_logic_vector(log2(CHANNELS) - 1 downto 0);
      --! Whole packet was transfered. Update SWENDPTR
      DMA_COMPLETE      : out std_logic;

      MAXREQUEST        : in  std_logic_vector(PAGE_WIDTH downto 0);
      INTERRUPT_EN      : in  std_logic;
      INTERRUPT_RST     : out std_logic;
      INTERRUPT         : in  std_logic_vector(31 downto 0);
      SWBUFFER_MASK     : in  std_logic_vector(31 downto 0);
      SWSTRPTR          : in  std_logic_vector(31 downto 0);
      --! Valid only when DMA_COMPLETE = '1'
      SWENDPTR          : out std_logic_vector(31 downto 0)
    );
end;

architecture behavioral of dma_ctrl_rx_newdata is

   type t_state is (S_INIT, S_START, S_STOP, S_RUN, S_LOAD, S_LOAD_NEW, S_DMA_REQ, S_NEXT, S_RESET);
   signal present_state       : t_state := S_INIT;
   signal next_state          : t_state;

   --! Constants
   constant HWPTR_WIDTH       : integer := log2(BUFFER_SIZE) + 1;
   constant SWPTR_WIDTH       : integer := 32;

   constant CR_RUN            : integer := 0;
   constant CR_DISCARD        : integer := 1;
   constant zeros             : std_logic_vector(127 downto 0) := (others => '0');

   --! General signals
   signal enable_set          : std_logic := '0';
   signal enable_rst          : std_logic := '0';

   signal enable_buffer_set   : std_logic;
   signal enable_buffer_rst   : std_logic;

   signal discard_set         : std_logic;
   signal discard_rst         : std_logic;

   signal reg_enable          : std_logic_vector(CHANNELS-1 downto 0) := (others => '0');
   signal reg_enable_buffer   : std_logic_vector(CHANNELS-1 downto 0) := (others => '0');
   signal reg_discard         : std_logic_vector(CHANNELS-1 downto 0) := (others => '1');

   signal reg_channel         : std_logic_vector(log2(CHANNELS)-1 downto 0) := (others => '0');
   signal iReg_channel        : integer := 0;

   signal dma_exec            : std_logic;

   --! Next channel signals
   signal controlFlagRst      : std_logic;
   signal reg_newdataFlag     : std_logic_vector(CHANNELS-1 downto 0);
   signal reg_controlFlag     : std_logic_vector(CHANNELS-1 downto 0);
   signal channel_enable      : std_logic_vector(CHANNELS-1 downto 0);
   signal channel_enable_rotated: std_logic_vector(CHANNELS-1 downto 0);
   signal next_channel_out    : std_logic_vector(log2(CHANNELS)-1 downto 0);

   --! Interrupt signals
   signal reg_interrupt       : std_logic_vector(31 downto 0);
   signal reg_interrupt_w     : std_logic_vector(31 downto 0);
   signal reg_interrupt_we    : std_logic;

   --! HW pointers
   signal hwStrPtr            : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwStrPtr_w          : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal hwEndPtr            : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtr_r          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtr_w          : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal hwEndPtrLock_we     : std_logic;
   signal hwEndPtrLock_w      : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtrLock        : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   --! SW pointers
   signal swEndPtr_we         : std_logic;
   signal swEndPtr_w          : std_logic_vector(31 downto 0);
   signal reg_swEndPtr        : std_logic_vector(31 downto 0);

   signal swBuffer_free       : std_logic;
   signal swFreeSpace         : std_logic_vector(31 downto 0);
   signal swFreeSpace_w       : std_logic_vector(31 downto 0);
   signal reg_swFreeSpace     : std_logic_vector(31 downto 0);
   signal swFreeSpace_we      : std_logic;

   --! Descriptor signals
   signal descriptorReady     : std_logic;
   signal descriptorReady_W   : std_logic;
   signal descriptorReady_WE  : std_logic;

   --! Control signals
   signal dataLength          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal hwEndPtrSel         : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal reg_dataLength      : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal transLength         : std_logic_vector(PAGE_WIDTH downto 0);
   signal reg_transLength     : std_logic_vector(PAGE_WIDTH downto 0);
   signal maxLength           : std_logic_vector(PAGE_WIDTH downto 0);

   signal pageSpace           : std_logic_vector(PAGE_WIDTH downto 0);
   signal reg_pageSpace       : std_logic_vector(PAGE_WIDTH downto 0);
   signal use_pageSpace       : std_logic;
   signal reg_use_pageSpace   : std_logic;

   signal dataLock_we         : std_logic;
   signal dataLock_w          : std_logic_vector(HWPTR_WIDTH-1 downto 0);
   signal reg_dataLock        : std_logic_vector(HWPTR_WIDTH-1 downto 0);

   signal use_lock            : std_logic;
   signal have_lock           : std_logic;
   signal complete            : std_logic;

begin

   iReg_channel      <= conv_integer(reg_channel);

   pageSpace         <= PAGE_SIZE - ('0' & DESC_DATA(PAGE_WIDTH-1 downto 0));
   maxLength         <= reg_pageSpace when reg_use_pageSpace = '1' else MAXREQUEST;
   hwEndPtrSel       <= hwEndPtrLock when use_lock = '1' else hwEndPtr;
   dataLength        <= hwEndPtrSel - hwStrPtr;
   transLength       <= maxLength when reg_dataLength > maxLength else reg_dataLength(PAGE_WIDTH downto 0);
   use_pageSpace     <= '1' when pageSpace < MAXREQUEST else '0';
   -- When a channel stops, controller must not react to new packets. use_lock flag will lock the pointer and newer data will not be tranfered.
   use_lock          <= '1' when SW_CONTROL(CR_RUN) = '0' else '0';
   -- have_lock flag has nothing to do with use_lock. See SWENDPTR below.
   have_lock         <= '1' when conv_integer(reg_dataLock) /= 0 and reg_dataLock <= BUFFER_SIZE else '0';
   -- if current transaction crossing reg_dataLock boundary, we have completing an splited packet.
   complete          <= '1' when reg_dataLock <= reg_transLength else '0';

   hwEndPtrLock_we   <= not use_lock;
   hwStrPtr_w        <= hwStrPtr + (zeros(HWPTR_WIDTH-1 downto PAGE_WIDTH+1) & reg_transLength);
   hwEndPtrLock_w    <= hwEndPtr;
   hwEndPtr_w        <= hwEndPtr_r + PACKET_LENGTH;

   swFreeSpace_w     <= swFreeSpace and SWBUFFER_MASK;
   swBuffer_free     <= '1' when reg_swFreeSpace(31 downto log2(BUFFER_SIZE)) > 0 else '0';
   swEndPtr_w        <= (others => '0') when enable_set = '1' else (reg_swEndPtr + reg_transLength) and SWBUFFER_MASK;
   swEndPtr_we       <= dma_exec or enable_set;

   reg_interrupt_we  <= '1' when (dma_exec ='1' and reg_interrupt(0) = '1') or (INTERRUPT_EN = '1' and present_state = S_RUN) else '0';
   reg_interrupt_w   <= INTERRUPT when reg_interrupt(0) = '1' and (reg_interrupt(31 downto 1) & '0') <= reg_transLength else
                        (((reg_interrupt(31 downto 1) & '0') - reg_transLength) and X"FFFFFFFE") or (X"0000000" & "000" & reg_interrupt(0));

   CHANNEL           <= reg_channel;
   ENABLE            <= reg_enable;
   ENABLE_BUFFER     <= reg_enable_buffer;
   DISCARD           <= reg_discard;
   GEN_INTERRUPT     <= dma_exec and INTERRUPT(0);
   INTERRUPT_RST     <= INTERRUPT(0) and not reg_interrupt(0) and dma_exec;
   -- SWENDPTR must always point at EOP only (except starting value 0). This is performance helper for software.
   -- So we must use reg_dataLock instead of reg_transLength.
   SWENDPTR          <= (reg_swEndPtr + reg_dataLock) and SWBUFFER_MASK;
   SW_STATUS         <= "00" & DMA_RDY & "0" & swBuffer_free & reg_newdataFlag(iReg_channel) & DESC_RDY & reg_enable(iReg_channel);

   DMA_REQ           <= dma_exec;
   DMA_CHANNEL       <= reg_channel;
   DMA_LENGTH        <= reg_transLength(PAGE_WIDTH downto log2(DATA_ALIGN));
   DMA_COMPLETE      <= dma_exec and complete;


   -- --------------- Sync logic -------------------------------------------
   sync_logic : process(CLK, RESET, next_state)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state  <= S_RESET;
         else
            present_state  <= next_state;
         end if;
      end if;
   end process sync_logic;

   -- ------------------ Next state logic -------------------------------------
   next_state_logic : process(present_state, SW_CONTROL, reg_newdataFlag, swBuffer_free, DESC_RDY, DMA_RDY, iReg_channel, hwEndPtr, hwEndPtrLock, reg_enable, hwStrPtr)
   begin
      next_state <= present_state;

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
            if (SW_CONTROL(CR_RUN) = '1' and reg_enable(iReg_channel) = '0') then
               next_state  <= S_START;
            elsif (SW_CONTROL(CR_RUN) = '0' and hwStrPtr = hwEndPtrLock) then
               next_state  <= S_STOP;
            elsif(reg_enable(iReg_channel) = '1' and reg_newdataFlag(iReg_channel) = '1' and swBuffer_free = '1' and DESC_RDY = '1') then
               next_state  <= S_RUN;
            else
               next_state  <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when S_START =>
            next_state     <= S_INIT;
         -- ---------------------------------------------
         when S_STOP =>
            next_state     <= S_NEXT;
         -- ---------------------------------------------
         when S_RUN =>
            next_state     <= S_DMA_REQ;
         -- ---------------------------------------------
         when S_DMA_REQ =>
            if(DMA_RDY = '1') then
               next_state  <= S_NEXT;
            end if;
         -- ---------------------------------------------
         when others =>
            next_state     <= S_INIT;
         -- ---------------------------------------------
      end case;
   end process;

   -- ------------------ Output logic -----------------------------------------
   output_logic: process(present_state, reg_swEndPtr, SWBUFFER_MASK, SW_CONTROL, DMA_RDY, reg_dataLock, reg_dataLength, reg_transLength, have_lock, complete)
   begin
      controlFlagRst       <= '0';
      dma_exec             <= '0';

      enable_rst           <= '0';
      enable_set           <= '0';
      enable_buffer_rst    <= '0';
      enable_buffer_set    <= '0';
      discard_rst          <= '0';
      discard_set          <= '0';

      dataLock_w           <= (others => '0');
      dataLock_we          <= '0';
      swFreeSpace_we       <= '0';

      case (present_state) is
         -- ---------------------------------------------
         when S_INIT =>
         -- ---------------------------------------------
         when S_RUN =>
            -- If we don't have any lock (to determine end of packet), lock all new data.
            dataLock_w           <= reg_dataLength;
            dataLock_we          <= not have_lock;

            enable_buffer_rst    <= not SW_CONTROL(CR_RUN);

            if(SW_CONTROL(CR_DISCARD) = '1') then
               discard_set       <= '1';
            else
               discard_rst       <= '1';
            end if;
         -- ---------------------------------------------
         when S_STOP =>
            controlFlagRst       <= '1';

            enable_rst           <= '1';
            discard_set          <= '1';
         -- ---------------------------------------------
         when S_START =>
            controlFlagRst       <= '1';
            enable_set           <= '1';
            enable_buffer_set    <= '1';
         -- ---------------------------------------------
         when S_DMA_REQ =>
            -- We can invalidate dataLock (set it to 0) only when is fully transfered.
            -- Otherwise we must subtract from dataLock counter transfered part.
            if(complete = '0') then
               dataLock_w        <= reg_dataLock - reg_transLength;
            end if;

            if(DMA_RDY = '1') then
               dataLock_we       <= '1';
               dma_exec          <= '1';
            end if;
         -- ---------------------------------------------
         when S_NEXT =>
            swFreeSpace_we       <= '1';
         -- ---------------------------------------------
         when others =>
      end case;
   end process;

   -- ------------- FSM registers ------------------------------
   reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if(present_state = S_INIT) then
            reg_dataLength    <= dataLength;
            reg_pageSpace     <= pageSpace;
            reg_use_pageSpace <= use_pageSpace;

            swFreeSpace       <= (SWSTRPTR - reg_swEndPtr - 1);
         end if;

         if(present_state = S_START) then
            swFreeSpace       <= SWBUFFER_MASK;
         end if;

         if(present_state = S_STOP) then
            swFreeSpace       <= (others => '0');
         end if;

         if(present_state = S_RUN) then
            reg_transLength   <= transLength;
         end if;

         if(present_state = S_NEXT) then
            reg_channel       <= next_channel_out + reg_channel + 1;
         end if;
      end if;
   end process;

   -- ----------- Next channel process ------------------------
   next_channel_fod : entity work.first_one_detector
   generic map (
      DATA_WIDTH        => CHANNELS
   )
   port map(
      MASK              => channel_enable_rotated,
      FIRST_ONE_BINARY  => next_channel_out
   );

   g: for i in 0 to CHANNELS-1 generate
      channel_enable_rotated(i)  <= channel_enable((i + iReg_channel + 1) mod CHANNELS);
      channel_enable(i)          <= reg_newdataFlag(i) or reg_controlFlag(i);
   end generate;

   -- -------- Change flags for next channel select  -----------
   gen_newdataFlag: for i in 0 to CHANNELS-1 generate
      reg_newdataFlagp : process (CLK)
      begin
         if(CLK = '1' and CLK'event) then
            if(i = PACKET_CHANNEL and PACKET_NEW = '1') then
               reg_newdataFlag(i) <= '1';
            elsif((i = reg_channel and hwEndPtr = hwStrPtr) or present_state = S_RESET) then
               reg_newdataFlag(i) <= '0';
            end if;
         end if;

         if(CLK = '1' and CLK'event) then
            if(i = SW_CHANNEL and SW_CONTROL_WR = '1') then
               reg_controlFlag(i) <= '1';
            elsif((i = reg_channel and controlFlagRst = '1') or present_state = S_RESET) then
               reg_controlFlag(i) <= '0';
            end if;
         end if;
      end process;
   end generate;

   -- ---------- Enable / Discard reg process ------------------
   reg_enablep : process(CLK)
   begin
      if(CLK'event AND CLK = '1') then
         if(present_state = S_RESET) then
            reg_enable   <= (others => '0');
         elsif(enable_set = '1') then
            reg_enable(iReg_channel) <= '1';
         elsif(enable_rst = '1') then
            reg_enable(ireg_channel) <= '0';
         end if;

         if(present_state = S_RESET) then
            reg_enable_buffer  <= (others => '0');
         elsif(enable_buffer_set = '1') then
            reg_enable_buffer(iReg_channel) <= '1';
         elsif(enable_buffer_rst = '1') then
            reg_enable_buffer(ireg_channel) <= '0';
         end if;

         if(present_state = S_RESET) then
            reg_discard  <= (others => '1');
         elsif(discard_set = '1') then
            reg_discard(ireg_channel) <= '1';
         elsif(discard_rst = '1') then
            reg_discard(ireg_channel) <= '0';
         end if;
      end if;
   end process reg_enablep;

   -- ---------------   Registers   -----------------------------
   swFreeSpace_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => 32,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => swFreeSpace_w,
      WE             => swFreeSpace_we,
      ADDR           => reg_channel,
      DO             => reg_swFreeSpace
   );

   hwStrPtr_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => hwStrPtr_w,
      WE             => dma_exec,
      ADDR           => reg_channel,
      DO             => hwStrPtr
   );

   hwEndPtr_i: entity work.DP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => hwEndPtr_w,
      WE             => PACKET_NEW,
      ADDRA          => PACKET_CHANNEL,
      DOA            => hwEndPtr_r,

      ADDRB          => reg_channel,
      DOB            => hwEndPtr
   );

   hwEndPtrLock_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => hwEndPtrLock_w,
      WE             => hwEndPtrLock_we,
      ADDR           => reg_channel,
      DO             => hwEndPtrLock
   );

   dataLock_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => HWPTR_WIDTH,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => dataLock_w,
      WE             => dataLock_we,
      ADDR           => reg_channel,
      DO             => reg_dataLock
   );

   swEndPtr_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => 32,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => swEndPtr_w,
      WE             => swEndPtr_we,
      ADDR           => reg_channel,
      DO             => reg_swEndPtr
   );

   interrupt_reg_i: entity work.SP_DISTMEM
   generic map(
      DATA_WIDTH     => 32,
      ITEMS          => CHANNELS,
      DISTMEM_TYPE   => 16
   )
   port map(
      WCLK           => CLK,
      RESET          => RESET,
      DI             => reg_interrupt_w,
      WE             => reg_interrupt_we,
      ADDR           => reg_channel,
      DO             => reg_interrupt
   );

end architecture;
