-- mem.vhd: Mem unit of CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use work.ibuf_pkg.all;

-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture mem_arch of mem is

   -- Constants declaration ---------------------------------------------------

   constant C_IBUF_EN            : std_logic := '0';
   constant C_IBUF_SPEED         : std_logic_vector(2 downto 0) := "101"; -- 100 Gb/s
   constant C_ERROR_MASK         : std_logic_vector(4 downto 0) := (others => '1');
   constant C_MIN_FRAME_LEN      : std_logic_vector(15 downto 0) := X"0040"; -- 64;
   constant C_MAX_FRAME_LEN      : std_logic_vector(15 downto 0) := X"05F6"; -- 1526;
   constant C_MAC_CHECK_MODE     : std_logic_vector(1 downto 0) := (others => '0');


   -- Signals declaration -----------------------------------------------------

   -- General and read/write interface input signals (address decoder -> mem)
   signal data_wr_in             : std_logic_vector(31 downto 0);
   signal sw_reset_in            : std_logic;

   signal reg_ibuf_en_we         : std_logic;
   signal reg_error_mask_we      : std_logic;
   signal reg_status_we          : std_logic;
   signal reg_min_frame_len_we   : std_logic;
   signal reg_max_frame_len_we   : std_logic;
   signal reg_mac_check_mode_we  : std_logic;
   
   signal reg_ibuf_en_rd         : std_logic;
   signal reg_error_mask_rd      : std_logic;
   signal reg_status_rd          : std_logic;
   signal reg_min_frame_len_rd   : std_logic;
   signal reg_max_frame_len_rd   : std_logic;
   signal reg_mac_check_mode_rd  : std_logic;
   
   signal cam_addr_in            : std_logic_vector(log2(MAC_COUNT) downto 0);
   signal cam_we_in              : std_logic;
   signal cam_re_in              : std_logic;

   -- CAM match interface input signals (check unit -> mem)
   signal cam_di_in              : std_logic_vector(63 downto 0);
   signal cam_match_en_in        : std_logic;
   signal cam_match_rst_in       : std_logic;

   -- IBUF status register interface input signal (buf -> mem)
   signal status_in              : std_logic_vector(16 downto 0);

   -- CGMII decoder interface input signal (cgmii_dec -> mem)
   signal link_status_in         : std_logic;
   
   -- Internal registers
   signal reg_ibuf_en            : std_logic := C_IBUF_EN;
   signal reg_error_mask         : std_logic_vector(4 downto 0) := C_ERROR_MASK;
   signal reg_status             : std_logic_vector(3 downto 0) := (others => '0');
   signal reg_min_frame_len      : std_logic_vector(15 downto 0) := C_MIN_FRAME_LEN;
   signal reg_max_frame_len      : std_logic_vector(15 downto 0) := C_MAX_FRAME_LEN;
   signal reg_mac_check_mode     : std_logic_vector(1 downto 0) := C_MAC_CHECK_MODE;
   signal reg_mac_count          : std_logic_vector(4 downto 0);
   signal reg_inbandfcs          : std_logic;
   
   -- Registers reading signals
   signal regs_sel               : std_logic_vector(5 downto 0);
   signal mx_regs_out            : std_logic_vector(31 downto 0);
   signal regs_rd                : std_logic;
   signal regs_data_vld          : std_logic;
   
   -- CAM writing logic
   signal reg_data_wr_l          : std_logic_vector(31 downto 0);
   signal reg_data_wr_l_we       : std_logic;
   signal reg_data_wr_h          : std_logic_vector(16 downto 0);
   signal reg_data_wr_h_we       : std_logic;
   signal cam_data_wr            : std_logic_vector(63 downto 0);
   signal reg_cam_addr           : std_logic_vector(log2(MAC_COUNT)-1 downto 0);
   signal reg_cam_we_in          : std_logic;
   signal reg_cam_we             : std_logic;
   
   -- CAM reading logic
   signal reg_mac_part           : std_logic;
   signal data_out               : std_logic_vector(63 downto 0);
   signal data_out_l             : std_logic_vector(31 downto 0);
   signal data_out_h             : std_logic_vector(31 downto 0);
   signal mx_cam_out             : std_logic_vector(31 downto 0);
   signal data_out_vld           : std_logic;
   
   -- CAM multiplexing reading/writing address
   signal addr                   : std_logic_vector(log2(MAC_COUNT)-1 downto 0);

   -- CAM multiplexing writing/matching logic
   signal match_en               : std_logic;
   signal write_en               : std_logic;
   signal data_in                : std_logic_vector(63 downto 0);  
  
   -- General output data signals (mem -> address decoder)
   signal rd                     : std_logic;
   signal reg_cam_re_last        : std_logic := '0';
   signal mx_data_sel            : std_logic;
   signal mx_data_rd             : std_logic_vector(31 downto 0);
   signal mx_data_vld            : std_logic;

   -- CAM matching output ports signals (mem -> check unit)
   signal cam_match_bus_out      : std_logic_vector(MAC_COUNT-1 downto 0);
   signal cam_match_bus_vld_out  : std_logic;

   -- CAM read/write interface output signals (mem -> address decoder)
   signal cam_busy_out           : std_logic;


begin

   -- -------------------------------------------------------------------------
   --                        Input ports connection
   -- -------------------------------------------------------------------------

   -- Data from MI32 interface (address decoder -> mem)
   data_wr_in              <= DATA_WR;
   
   -- Registers write interface (address decoder -> mem)
   reg_ibuf_en_we          <= IBUF_EN_WE;
   reg_error_mask_we       <= ERROR_MASK_WE;
   reg_status_we           <= STATUS_WE;
   reg_min_frame_len_we    <= MIN_FRAME_LEN_WE;
   reg_max_frame_len_we    <= MAX_FRAME_LEN_WE;
   reg_mac_check_mode_we   <= MAC_CHECK_MODE_WE;

   -- Registers read interface (address decoder -> mem)
   reg_ibuf_en_rd          <= IBUF_EN_RD;
   reg_error_mask_rd       <= ERROR_MASK_RD;
   reg_status_rd           <= STATUS_RD;
   reg_min_frame_len_rd    <= MIN_FRAME_LEN_RD;
   reg_max_frame_len_rd    <= MAX_FRAME_LEN_RD;
   reg_mac_check_mode_rd   <= MAC_CHECK_MODE_RD;
   
   -- CAM read/write interface (address decoder -> mem)
   cam_addr_in             <= CAM_ADDR;
   cam_we_in               <= CAM_WE;
   cam_re_in               <= CAM_RE;
   
   -- CAM match interface (check unit -> mem)
   cam_di_in               <= CAM_DI;
   cam_match_en_in         <= CAM_MATCH_EN;
   cam_match_rst_in        <= CAM_MATCH_RST;

   -- IBUF status register interface (buf -> mem; cgmii_dec -> mem)
   status_in               <= STATUS;
   link_status_in          <= LINK_STATUS;
   
   -- Software reset input
   sw_reset_in             <= SW_RESET;

   -- -------------------------------------------------------------------------
   --                               Registers
   -- -------------------------------------------------------------------------
   
   -- Register reg_ibuf_en
   reg_ibuf_en_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_ibuf_en <= C_IBUF_EN;
         elsif (reg_ibuf_en_we = '1') then
            reg_ibuf_en <= data_wr_in(0);
         end if;
      end if;
   end process reg_ibuf_en_p;

   -- Register reg_error_mask
   reg_error_mask_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_error_mask <= C_ERROR_MASK;
         elsif (reg_error_mask_we = '1') then
            reg_error_mask <= data_wr_in(4 downto 0);
         end if;
      end if;
   end process reg_error_mask_p;

   -- Register reg_status
   reg_status_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_status <= (others => '0');
         elsif (reg_status_we = '1') then
            reg_status <= data_wr_in(3 downto 0);
         else
            reg_status <= reg_status or
                          status_in(3 downto 0);

            -- Mask PACODAG and DFIFO ovf
            if (sw_reset_in = '1') then
               reg_status(C_PACODAG_OVF_POS) <= '0';
               reg_status(C_DFIFO_OVF_POS)   <= '0';
            end if;
         end if;
      end if;
   end process reg_status_p;

   -- Register reg_min_frame_len
   reg_min_frame_len_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_min_frame_len <= C_MIN_FRAME_LEN;
         elsif (reg_min_frame_len_we = '1') then
            reg_min_frame_len <= data_wr_in(15 downto 0);
         end if;
      end if;
   end process reg_min_frame_len_p;

   -- Register reg_max_frame_len
   reg_max_frame_len_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_max_frame_len <= C_MAX_FRAME_LEN;
         elsif (reg_max_frame_len_we = '1') then
            reg_max_frame_len <= data_wr_in(15 downto 0);
         end if;
      end if;
   end process reg_max_frame_len_p;

   -- Register reg_mac_check_mode
   reg_mac_check_mode_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_mac_check_mode <= C_MAC_CHECK_MODE;
         elsif (reg_mac_check_mode_we = '1') then
            reg_mac_check_mode <= data_wr_in(1 downto 0);
         end if;
      end if;
   end process reg_mac_check_mode_p;

   -- Register reg_mac_count
   reg_mac_count <= conv_std_logic_vector(MAC_COUNT, 5);

   -- Register reg_inbandfcs
   reg_inbandfcs <= '1' when INBANDFCS else '0';

   
   -- Registers reading logic -------------------------------------------------

   -- Information about which register is to be read (encoded in one-hot style)
   regs_sel <= reg_ibuf_en_rd &
               reg_error_mask_rd &
               reg_status_rd &
               reg_min_frame_len_rd &
               reg_max_frame_len_rd &
               reg_mac_check_mode_rd;
   
   -- Multiplexor selecting correct output value
   mx_regs_out_p : process(regs_sel, reg_ibuf_en, reg_error_mask, status_in, 
                           reg_status, reg_min_frame_len, reg_max_frame_len,
                           reg_mac_check_mode, reg_mac_count, reg_inbandfcs,
                           link_status_in)
   begin
      case (regs_sel) is
         when "100000" =>
            mx_regs_out <= (31 downto 1 => '0') & reg_ibuf_en;
         when "010000" =>
            mx_regs_out <= (31 downto 5 => '0') & reg_error_mask;
         when "001000" =>
            mx_regs_out <= (31 downto 28 => '0') &
                           reg_mac_count &
                           reg_inbandfcs &
                           '0' &  -- STAT_OVF reserved
                           status_in(16 downto 4) &
                           link_status_in &
                           C_IBUF_SPEED &
                           reg_status;                
         when "000100" =>
            mx_regs_out <= (31 downto 16 => '0') & reg_min_frame_len;
         when "000010" =>
            mx_regs_out <= (31 downto 16 => '0') & reg_max_frame_len;
         when "000001" =>
            mx_regs_out <= (31 downto 2 => '0') & reg_mac_check_mode;
         when others =>
            mx_regs_out <= (others => '0');
      end case;
   end process mx_regs_out_p;
   
   -- Registers are read when regs_rd is active
   regs_rd <=  reg_ibuf_en_rd or
               reg_error_mask_rd or
               reg_status_rd or
               reg_min_frame_len_rd or
               reg_max_frame_len_rd or
               reg_mac_check_mode_rd;
   
   -- Output of mx_regs_out multiplexer contains valid data
   regs_data_vld <= regs_rd;
   
   
   -- -------------------------------------------------------------------------
   --                                  CAM
   -- -------------------------------------------------------------------------
   
   -- CAM instantiation -------------------------------------------------------

   cam_i: entity work.ibuf_cam
   generic map (
      CAM_ROW_WIDTH     => 64,
      CAM_ROW_COUNT     => MAC_COUNT
   )
   port map (
      CLK               => CLK,
      RESET             => RESET,
      ADDR              => addr,
      CAM_BUSY          => cam_busy_out,
      DATA_IN           => data_in,
      WRITE_EN          => write_en,
      READ_EN           => cam_re_in,
      DATA_OUT          => data_out,
      DATA_OUT_VLD      => data_out_vld,
      MATCH_EN          => match_en,
      MATCH_RST         => cam_match_rst_in,
      MATCH_BUS         => cam_match_bus_out,
      MATCH_BUS_VLD     => cam_match_bus_vld_out
   );

   
   -- CAM writing logic -------------------------------------------------------

   -- Write enable for lower part of MAC address
   reg_data_wr_l_we <= cam_we_in when (cam_addr_in(0) = '0') else '0';

   -- Write enable for higher part of MAC address
   reg_data_wr_h_we <= cam_we_in when (cam_addr_in(0) = '1') else '0';
   
   -- Register storing lower part of MAC address
   reg_data_wr_l_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (reg_data_wr_l_we = '1') then
            reg_data_wr_l <= data_wr_in;
         end if;
      end if;
   end process reg_data_wr_l_p;

   -- Register storing higher part of MAC address
   reg_data_wr_h_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (reg_data_wr_h_we = '1') then
            reg_data_wr_h <= data_wr_in(16 downto 0);
         end if;
      end if;
   end process reg_data_wr_h_p;

   -- Register for storing address where actual MAC is going to be written
   reg_cam_addr_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (reg_data_wr_l_we = '1') then
            reg_cam_addr <= cam_addr_in(log2(MAC_COUNT) downto 1);
         end if;
      end if;
   end process reg_cam_addr_p;
   
   -- Reordering from SW format (little endian) to network format (big endian)
   cam_data_wr(63 downto 49) <= (others => '0');               -- not used
   cam_data_wr(48)           <= reg_data_wr_h(16);             -- validity bit
   cam_data_wr(47 downto 40) <= reg_data_wr_l(7  downto 0);    -- LSB
   cam_data_wr(39 downto 32) <= reg_data_wr_l(15 downto 8);
   cam_data_wr(31 downto 24) <= reg_data_wr_l(23 downto 16);
   cam_data_wr(23 downto 16) <= reg_data_wr_l(31 downto 24);
   cam_data_wr(15 downto 8)  <= reg_data_wr_h(7  downto 0);
   cam_data_wr(7  downto 0)  <= reg_data_wr_h(15 downto 8);    -- MSB

   -- Input logic of CAM write_en register
   reg_cam_we_in <=  reg_data_wr_h_we when
                    (reg_cam_addr = cam_addr_in(log2(MAC_COUNT) downto 1)) else
                     '0';
   
   -- Register storing cam_we value
   reg_cam_we_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         reg_cam_we <= reg_cam_we_in;
      end if;
   end process reg_cam_we_p;


   -- Multiplexing of matching/writing operations -----------------------------
   mx_cam_write_match_p : process (reg_ibuf_en, cam_match_en_in, cam_di_in, 
                                   reg_cam_we, cam_data_wr)
   begin
      if (reg_ibuf_en = '1') then   -- matching enabled, writing disabled
         match_en    <= cam_match_en_in;
         write_en    <= '0';
         data_in     <= cam_di_in;
      else                          -- matching disabled, writing enabled
         match_en    <= '0';
         write_en    <= reg_cam_we;
         data_in     <= cam_data_wr;
      end if;
   end process mx_cam_write_match_p;

   
   -- CAM reading logic -------------------------------------------------------

   -- Register with information which part of MAC address is read
   reg_mac_part_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (cam_re_in = '1') then
            reg_mac_part <= cam_addr_in(0);
         end if;
      end if;
   end process reg_mac_part_p;
   
   -- Reordering from network format (big endian) to SW format (little endian)
   data_out_h(31 downto 17) <= (others => '0');          -- not used
   data_out_h(16)           <= data_out(48);             -- validity bit
   data_out_h(15 downto 8)  <= data_out(7 downto 0);     -- MSB
   data_out_h(7 downto 0)   <= data_out(15 downto 8);
   data_out_l(31 downto 24) <= data_out(23 downto 16);
   data_out_l(23 downto 16) <= data_out(31 downto 24);
   data_out_l(15 downto 8)  <= data_out(39 downto 32);
   data_out_l(7 downto 0)   <= data_out(47 downto 40);   -- LSB
   
   -- Multiplexer selecting correct part of MAC address
   mx_cam_out_p : process(reg_mac_part, data_out_l, data_out_h)
   begin
      case (reg_mac_part) is
         when '0' => mx_cam_out <= data_out_l;
         when '1' => mx_cam_out <= data_out_h;
         when others => null;
      end case;
   end process mx_cam_out_p;

   -- Multiplexing of reading/writing address ---------------------------------
   mx_addr_p : process(cam_re_in, reg_cam_addr, cam_addr_in)
   begin
      case (cam_re_in) is
         when '0' => addr <= reg_cam_addr;
         when '1' => addr <= cam_addr_in(log2(MAC_COUNT) downto 1);
         when others => null;
      end case;
   end process mx_addr_p;
   
   
   -- -------------------------------------------------------------------------
   --                 Selecting correct data output values
   -- -------------------------------------------------------------------------
   
   -- Active when any read operation is performed
   rd <= cam_re_in or regs_rd;
   
   -- Register storing cam_re value from time of last performed read operation
   reg_cam_re_last_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_cam_re_last <= '0';
         elsif (rd = '1') then
            reg_cam_re_last <= cam_re_in;
         end if;
      end if;
   end process reg_cam_re_last_p;
   
   -- Multiplexor choosing correct output data selector value
   mx_data_sel_p : process(rd, cam_re_in, reg_cam_re_last)
   begin
      case (rd) is
         when '0' => mx_data_sel <= reg_cam_re_last;
         when '1' => mx_data_sel <= cam_re_in;
         when others => null;
      end case;
   end process mx_data_sel_p;
   
   -- Multiplexor selecting correct data output value
   mx_data_rd_p : process(mx_data_sel, mx_regs_out, mx_cam_out)
   begin
      case (mx_data_sel) is
         when '0' => mx_data_rd <= mx_regs_out;
         when '1' => mx_data_rd <= mx_cam_out;
         when others => null;
      end case;
   end process mx_data_rd_p;
   
   -- Multiplexor selecting correct data valid output value
   mx_data_vld_p : process(mx_data_sel, regs_data_vld, data_out_vld)
   begin
      case (mx_data_sel) is
         when '0' => mx_data_vld <= regs_data_vld;
         when '1' => mx_data_vld <= data_out_vld;
         when others => null;
      end case;
   end process mx_data_vld_p;


   -- -------------------------------------------------------------------------
   --                        Output ports connection
   -- -------------------------------------------------------------------------
   
   -- Data to MI32 interface (mem -> address decoder)
   DATA_RD              <= mx_data_rd;
   DATA_VLD             <= mx_data_vld;
   
   -- CAM match interface (mem -> check unit)
   CAM_MATCH_BUS        <= cam_match_bus_out;
   CAM_MATCH_BUS_VLD    <= cam_match_bus_vld_out;

   -- CAM read/write interface (mem -> address decoder)
   CAM_BUSY             <= cam_busy_out;
   
   -- Check registers interface (mem -> check unit)
   MIN_FRAME_LEN        <= reg_min_frame_len;
   MAX_FRAME_LEN        <= reg_max_frame_len;
   MAC_CHECK_MODE       <= reg_mac_check_mode;

   -- Buf registers interface (mem -> buf)
   IBUF_EN              <= reg_ibuf_en;
   ERROR_MASK           <= reg_error_mask;
   
   
end architecture mem_arch;
