-- math_pack.vhd
--!
--! \file
--! \brief Math package
--! \author Zdenek Salvet <salvet@ics.muni.cz>
--! \author Tomas Martinek <martinek@liberouter.org>
--! \author Viktor Pus <pus@liberouter.org>
--! \author Petr Kobiersky <kobiersky@liberouter.org>
--! \date 2003
--!
--! \section License
--!
--! Copyright (C) 2003 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

--! \brief Package with basic math functions 
--! \details This package contains math functions: log2, max, min and div_roundup.
package math_pack is

   --! \brief Logarithm with base 2.
   --! \details Founds first integer larger or equal to real base 2 logarithm of given number.
   --! @param n Given number.
   --! @return Logarithm value of given number.
   function log2 (n : integer) return integer;
   --! \brief Logarithm with base 2.
   --! \details Founds first integer larger or equal to real base 2 logarithm of given number.
   --! @param n Given number.
   --! @return Logarithm value of given number.
   function log2 (n : std_logic_vector(31 downto 0)) return integer;
   --! \brief Selects higher number from two given.
   --! @param l First given number.
   --! @param r Second given number.
   --! @return Higher number from l r.
   function max(l, r: integer) return integer;
   --! \brief Selects lower number from two given.
   --! @param l First given number.
   --! @param r Second given number.
   --! @return Lower number from l r.
   function min(l, r: integer) return integer;
   --! \brief Selects lower number from two given.
   --! @param l First given number.
   --! @param r Second given number.
   --! @return Lower number from l r.
   function minimum(l, r: integer) return integer;
   --! \brief Integer roundup division.
   --! \details Founds first integer larger or equal to real result of division.
   --! @param dividend A number that is going to be devided.
   --! @param divisor The number by which dividend is divided.
   --! @return Result of division rounded up to nearest integer.
   function div_roundup(dividend, divisor: integer) return integer;

end math_pack;

--! \brief Body of package with basic math functions 
package body math_pack is

   --! \brief Logarithm with base 2.
   function log2 (n : integer) return integer is
      variable a, m : integer;
   begin
      if (n = 1) then
         return 0;
      end if;
      a := 0;
      m := 1;
      while m < n loop
         a := a + 1;
         m := m * 2;
      end loop;
      return a;
   end log2;

   --! \brief Logarithm with base 2.
   function log2 (n : std_logic_vector(31 downto 0)) return integer is
      variable m     : std_logic_vector(32 downto 0);
      variable aux_n : std_logic_vector(32 downto 0);
      variable a     : integer;
   begin
      if (n = X"00000001") then
         return 0;
      end if;
      a     := 0;
      m     := '0' & X"00000001";
      aux_n := '0' & n;
      while m < aux_n loop
         a := a + 1;
         m := m(31 downto 0) & '0';
      end loop;
      return a;
   end log2;

   --! \brief Selects higher number from two given.
   function max(L, R: integer) return integer is
   begin
      if L > R then
         return L;
      else
         return R;
      end if;
   end;
   
   --! \brief Selects lower number from two given.
   function min(L, R: integer) return integer is
   begin
      if L < R then
         return L;
      else
         return R;
      end if;
   end;
   
   --! \brief Selects lower number from two given.
   function minimum(L, R: integer) return integer is
   begin
      if L < R then
         return L;
      else
         return R;
      end if;
   end;

   --! \brief Integer roundup division.
   function div_roundup(dividend, divisor: integer) return integer is
   begin
      if dividend mod divisor = 0 then
         return dividend / divisor;
      else
         return (dividend / divisor) + 1;
      end if;
   end;

   -- ----------------------------------------------------------------------
end math_pack;

