-- flu_prfifo_ent.vhd: Asynchronous packet buffer
-- from BlockRAM memories with discarding
-- Both sides addressed as FIFO, supports unaligned SOPs
-- Copyright (C) 2012 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- library containing log2 function
use work.math_pack.all;

-- auxilarity function needed to compute address width
use WORK.bmem_func.all;

-- pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
-- pragma translate_on
-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity flu_prfifo is
    generic(
      --* Numer of items (words) in buffer
      ITEMS          : integer := 1024;
      --* Data Width in bits.
      --* Must hold: (DATA_WIDTH/(2**SOP_POS_W)) mod 8 = 0
      DATA_WIDTH     : integer := 512;
      --* Width of status information of fifo fullness
      --* Note: 2**STATUS_WIDTH MUST BE!! less or equal to ITEMS
      STATUS_WIDTH   : integer := 4;
      --* Width of SOP_POS signal
      SOP_POS_W      : integer := 3;
      --* Use BRAM output register
      USE_OUTREG     : boolean := true;
      --* Number of items in helper fifo
      HELPER_FIFO_ITEMS:integer := 64
   );
   port (
      -- Write interface (all sync to CLK_WR)
      RX_CLK      : in  std_logic;
      RX_RESET    : in  std_logic;
      --* Write data
      RX_DATA     : in  std_logic_vector(DATA_WIDTH-1 downto 0);
      --* Mark position of RX_SOP
      RX_SOP_POS  : in  std_logic_vector(SOP_POS_W-1 downto 0);
      --* Mark position of RX_EOP
      RX_EOP_POS  : in  std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      --* Mark start of packet
      RX_SOP      : in  std_logic;
      --* Mark end of packet
      RX_EOP      : in  std_logic;
      --* Write request
      RX_SRC_RDY  : in  std_logic;
      --* Fifo not FULL
      RX_DST_RDY  : out std_logic;
      --* Release written data up to last RX_SOP
      RX_RELEASE  : in  std_logic;
      --* Fifo fullness status. 000=(almost)empty, 100=full. May be imprecise!
      STATUS      : out std_logic_vector(STATUS_WIDTH-1 downto 0);

      -- Read interface - Unaligned FrameLink (all sync to CLK_RD)
      TX_CLK      : in  std_logic;
      TX_RESET    : in  std_logic;
      --* Read data
      TX_DATA     : out std_logic_vector(DATA_WIDTH-1 downto 0);
      --* Position of TX_SOP
      TX_SOP_POS  : out std_logic_vector(SOP_POS_W-1 downto 0);
      --* Position of TX_EOP
      TX_EOP_POS  : out std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      --* Start of packet
      TX_SOP      : out std_logic;
      --* End of packet
      TX_EOP      : out std_logic;
      --* Read request
      TX_SRC_RDY  : out std_logic;
      --* Read data valid
      TX_DST_RDY  : in  std_logic
   );
end flu_prfifo;
