-- check.vhd: Check unit of CGMII Input buffer
-- Copyright (C) 2012 CESNET
-- Author(s): Jan Kucera <xkucer73@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;

-- ----------------------------------------------------------------------------
--                        Architecture declaration
-- ----------------------------------------------------------------------------

architecture check_arch of check is

   -- Constants declaration ---------------------------------------------------
   
   -- FIFO storing SAU information params
   constant C_SAU_FIFO_DEPTH        : integer := 6;
   -- FIFO storing MAC information params
   constant C_MAC_FIFO_DEPTH        : integer := 5;


   -- Signals declaration -----------------------------------------------------
   
   -- Datapath input signals
   signal rx_data_in                : std_logic_vector(511 downto 0);
   signal rx_sop_in                 : std_logic;
   signal rx_sop_pos_in             : std_logic_vector(2 downto 0);
   signal rx_eop_in                 : std_logic;
   signal rx_eop_pos_in             : std_logic_vector(5 downto 0);
   signal rx_err_in                 : std_logic;
  
   -- CAM memory inputs
   signal cam_match_bus_in          : std_logic_vector(MAC_COUNT-1 downto 0);
   signal cam_match_bus_vld_in      : std_logic;

   -- Registers
   signal min_frame_len_in          : std_logic_vector(15 downto 0);
   signal max_frame_len_in          : std_logic_vector(15 downto 0);
   signal mac_check_mode_in         : std_logic_vector(1 downto 0);

   -- Sampling unit inputs
   signal sau_accept_in             : std_logic_vector(0 downto 0);
   signal sau_dv_in                 : std_logic;

   -- Input registers for datapath signals
   signal reg_rx_data               : std_logic_vector(511 downto 0);
   signal reg_rx_sop                : std_logic := '0';
   signal reg_rx_sop_pos            : std_logic_vector(2 downto 0);
   signal reg_rx_eop                : std_logic := '0';
   signal reg_rx_eop_pos            : std_logic_vector(5 downto 0);
   signal reg_rx_err                : std_logic := '0';

   -- Output registers for datapath signals
   signal reg6_data                 : std_logic_vector(511 downto 0);
   signal reg6_sop                  : std_logic := '0';
   signal reg6_sop_pos              : std_logic_vector(2 downto 0);
   signal reg6_eop                  : std_logic := '0';
   signal reg6_eop_pos              : std_logic_vector(5 downto 0);
   signal reg6_err                  : std_logic := '0';
    
   -- Shift register generated when INBANDFCS = false
   signal sh_reg_sop_data_in        : std_logic_vector(515 downto 0);
   signal sh_reg_sop_data_out       : std_logic_vector(515 downto 0);

   -- Shift register generated when INBANDFCS = true
   signal sh_reg_datapath_in        : std_logic_vector(523 downto 0);
   signal sh_reg_datapath_out       : std_logic_vector(523 downto 0);
   
   -- CRC removing
   signal reg1_eop                  : std_logic := '0';
   signal reg1_eop_pos              : std_logic_vector(5 downto 0);
   signal reg1_err                  : std_logic := '0';
   signal reg2_eop_in               : std_logic;
   signal reg2_eop_pos_in           : std_logic_vector(5 downto 0);
   signal reg2_err_in               : std_logic;
   signal reg2_eop                  : std_logic := '0';
   signal reg2_eop_pos              : std_logic_vector(5 downto 0);
   signal reg2_err                  : std_logic := '0';
   signal reg3_eop                  : std_logic := '0';
   signal reg3_eop_pos              : std_logic_vector(5 downto 0);
   signal reg3_err                  : std_logic := '0';
   signal reg4_eop                  : std_logic := '0';
   signal reg4_eop_pos              : std_logic_vector(5 downto 0);
   signal reg4_err                  : std_logic := '0';
   signal reg5_eop                  : std_logic := '0';
   signal reg5_eop_pos              : std_logic_vector(5 downto 0);
   signal reg5_err                  : std_logic := '0';

   -- MAC address statistics
   signal mx_dst_mac_extract        : std_logic_vector(47 downto 0);
   signal mem_err                   : std_logic;
   signal broadcast_err             : std_logic;
   signal reg1_broadcast_err        : std_logic := '0';
   signal reg2_broadcast_err        : std_logic := '0';
   signal multicast_err             : std_logic;
   signal reg1_multicast_err        : std_logic := '0';
   signal reg2_multicast_err        : std_logic := '0';
   signal mx_mac_err                : std_logic_vector(0 downto 0);
   signal mac_fifo_in               : std_logic_vector(2 downto 0);
   signal mac_fifo_dout             : std_logic_vector(2 downto 0);
   signal mac_fifo_empty            : std_logic;
   signal mac_err_out               : std_logic;
   signal mac_broadcast_out         : std_logic;
   signal mac_multicast_out         : std_logic;

   -- Sampling unit FIFO
   signal sau_fifo_dout             : std_logic_vector(0 downto 0);
   signal sau_fifo_empty            : std_logic;
   signal sau_err_out               : std_logic; 
   
   -- Frame length statistics
   signal frame_len_add             : std_logic_vector(7 downto 0);
   signal frame_len_add_sop         : std_logic_vector(7 downto 0);
   signal frame_len_sub_sop         : std_logic_vector(7 downto 0);
   signal reg1_frame_len_add        : std_logic_vector(7 downto 0) := (others => '0');
   signal reg1_frame_len_add_sop    : std_logic_vector(7 downto 0) := (others => '0');
   signal reg1_frame_len_sub_sop    : std_logic_vector(7 downto 0) := (others => '0');
   signal reg2_frame_len_add_sop    : std_logic_vector(7 downto 0) := (others => '0');
   signal reg2_frame_len_add_in     : std_logic_vector(7 downto 0);
   signal reg2_frame_len_add        : std_logic_vector(7 downto 0) := (others => '0');
   signal reg1_sop                  : std_logic := '0';
   signal reg2_sop                  : std_logic := '0';
   signal reg3_sop                  : std_logic := '0';
   signal cnt_frame_len             : std_logic_vector(15 downto 0) := (others => '0');
   signal reg4_cnt_frame_len_in     : std_logic_vector(15 downto 0);
   signal reg5_cnt_frame_len_in     : std_logic_vector(15 downto 0);
   signal reg4_cnt_frame_len        : std_logic_vector(15 downto 0) := (others => '0');
   signal reg5_cnt_frame_len        : std_logic_vector(15 downto 0) := (others => '0');
   signal reg6_cnt_frame_len        : std_logic_vector(15 downto 0) := (others => '0');
   signal frame_len_out             : std_logic_vector(15 downto 0);
   signal reg4_mintu_err_in         : std_logic;
   signal reg5_mintu_err_in         : std_logic;
   signal reg4_mtu_err_in           : std_logic;
   signal reg5_mtu_err_in           : std_logic;
   signal reg4_mintu_err            : std_logic := '0';
   signal reg5_mintu_err            : std_logic := '0';
   signal reg6_mintu_err            : std_logic := '0';
   signal reg4_mtu_err              : std_logic := '0';
   signal reg5_mtu_err              : std_logic := '0';
   signal reg6_mtu_err              : std_logic := '0';
   signal mintu_err_out             : std_logic;
   signal mtu_err_out               : std_logic;
   
   -- CRC checking
   signal whole_packet              : std_logic;
   signal end_val_sel               : std_logic;
   signal end_val_err               : std_logic;
   signal no_crc_err                : std_logic;
   signal crc_received              : std_logic_vector(31 downto 0);
   signal reg_whole_packet          : std_logic := '0';
   signal reg_end_val_sel           : std_logic := '0';
   signal reg_end_val_err           : std_logic := '0';
   signal reg_no_crc_err            : std_logic := '0';
   signal reg_crc_received          : std_logic_vector(31 downto 0);
   signal reg1_whole_packet         : std_logic := '0';
   signal reg2_whole_packet         : std_logic := '0';
   signal reg1_end_val_sel          : std_logic := '0';
   signal reg2_end_val_sel          : std_logic := '0';
   signal reg3_end_val_sel          : std_logic := '0';
   signal reg4_end_val_sel          : std_logic := '0';
   signal reg5_end_val_sel          : std_logic := '0';
   signal reg6_end_val_sel          : std_logic := '0';
   signal reg1_end_val_err          : std_logic := '0';
   signal reg2_end_val_err          : std_logic := '0';
   signal reg3_end_val_err          : std_logic := '0';
   signal reg4_end_val_err          : std_logic := '0';
   signal reg5_end_val_err          : std_logic := '0';
   signal reg6_end_val_err          : std_logic := '0';
   signal reg1_no_crc_err           : std_logic := '0';
   signal reg2_no_crc_err           : std_logic := '0';
   signal reg3_no_crc_err           : std_logic := '0';
   signal reg1_crc_received_in      : std_logic_vector(31 downto 0);
   signal reg1_crc_received         : std_logic_vector(31 downto 0);
   signal reg2_crc_received         : std_logic_vector(31 downto 0);
   signal reg3_crc_received         : std_logic_vector(31 downto 0);
   signal reg4_crc_received         : std_logic_vector(31 downto 0);
   signal reg5_crc_received         : std_logic_vector(31 downto 0);
   signal reg6_crc_received         : std_logic_vector(31 downto 0);
   signal eop_crc_in                : std_logic;
   signal eop_pos_crc_in            : std_logic_vector(5 downto 0);
   signal start_crc_in              : std_logic_vector(5 downto 0);
   signal mask_crc_in               : std_logic_vector(5 downto 0);
   signal dv_crc_in                 : std_logic := '0';
   signal crc_out                   : std_logic_vector(31 downto 0);
   signal crc_dv_out                : std_logic;
   signal crc_computed              : std_logic_vector(31 downto 0);
   signal crc_err_active            : std_logic;
   signal reg7_crc_err              : std_logic := '0';
   signal crc_err_out               : std_logic;
   signal crc_rem_err_out           : std_logic;

    
begin


   -- -------------------------------------------------------------------------
   --             Assignment of input port values to signals
   -- -------------------------------------------------------------------------

   -- Datapath inputs
   rx_data_in           <= RX_DATA;
   rx_sop_in            <= RX_SOP;
   rx_sop_pos_in        <= RX_SOP_POS;
   rx_eop_in            <= RX_EOP;
   rx_eop_pos_in        <= RX_EOP_POS;
   rx_err_in            <= RX_ERR;
   
   -- CAM memory inputs
   cam_match_bus_in     <= CAM_MATCH_BUS;
   cam_match_bus_vld_in <= CAM_MATCH_BUS_VLD;
   
   -- Registers
   min_frame_len_in     <= MIN_FRAME_LEN;
   max_frame_len_in     <= MAX_FRAME_LEN;
   mac_check_mode_in    <= MAC_CHECK_MODE;
   
   -- Sampling unit inputs
   sau_accept_in(0)     <= SAU_ACCEPT;
   sau_dv_in            <= SAU_DV;
   
  
   -- -------------------------------------------------------------------------
   --                Input registers for datapath inputs
   -- -------------------------------------------------------------------------

   -- Input registers with reset  
   regs_input_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_rx_sop <= '0';
            reg_rx_eop <= '0';
            reg_rx_err <= '0';
         else
            reg_rx_sop <= rx_sop_in;
            reg_rx_eop <= rx_eop_in;
            reg_rx_err <= rx_err_in;
         end if;
      end if;
   end process regs_input_p;      

   -- Input registers without reset
   regs_no_reset_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         reg_rx_data    <= rx_data_in;
         reg_rx_sop_pos <= rx_sop_pos_in;
         reg_rx_eop_pos <= rx_eop_pos_in;
      end if;
   end process regs_no_reset_p;

  
   -- -------------------------------------------------------------------------
   --                       Optional CRC part removing
   -- -------------------------------------------------------------------------
   
   crc_remove_gen : if (INBANDFCS = false) generate
   begin
    
      -- Register for EOP signal
      reg1_eop_p : process (CLK)
      begin
         if (CLK'event and CLK = '1') then
            if (not RESET_BY_INIT and RESET = '1') then
               reg1_eop <= '0';
            -- Erase original EOP value
            elsif (reg_end_val_sel = '1' and reg_end_val_err = '0') then 
               reg1_eop <= '0';
            else
               reg1_eop <= reg_rx_eop;
            end if;
         end if;
      end process reg1_eop_p;

      -- Register for EOP_POS signal
      reg1_eop_pos_p : process (CLK)
      begin
         if (CLK'event and CLK = '1') then
            reg1_eop_pos <= reg_rx_eop_pos;
         end if;
      end process reg1_eop_pos_p;

      -- Register for ERR signal
      reg1_err_p : process (CLK)
      begin
         if (CLK'event and CLK = '1') then
            if (not RESET_BY_INIT and RESET = '1') then
               reg1_err <= '0';
            -- Erase original ERR value
            elsif (reg_end_val_sel = '1' and reg_end_val_err = '0') then
               reg1_err <= '0';
            else
               reg1_err <= reg_rx_err or reg_end_val_err or reg_no_crc_err;
            end if;
         end if;
      end process reg1_err_p;
      
      -- Multiplexer for choosing correct value
      mx_crc_remove : process (reg_end_val_sel, reg_end_val_err, reg1_no_crc_err,
                               reg1_end_val_err, reg1_eop, reg1_eop_pos, reg1_err,
                               reg_rx_eop, reg_rx_eop_pos, reg_rx_err)
      begin
         if (reg_end_val_sel = '1' and reg_end_val_err = '0') then
            reg2_eop_in     <= reg_rx_eop;
            reg2_eop_pos_in <= reg_rx_eop_pos - 4;
            reg2_err_in     <= reg_rx_err;
         elsif (reg1_no_crc_err = '1' or reg1_end_val_err = '1') then
            reg2_eop_in     <= reg1_eop;
            reg2_eop_pos_in <= reg1_eop_pos;
            reg2_err_in     <= reg1_err;
         else
            reg2_eop_in     <= reg1_eop;
            reg2_eop_pos_in <= reg1_eop_pos - 4;
            reg2_err_in     <= reg1_err;
         end if;
      end process mx_crc_remove;
      
      -- Control registers within datapath (need to be reseted)
      regs_control_p : process (CLK)
      begin
         if (CLK'event and CLK = '1') then
            if (not RESET_BY_INIT and RESET = '1') then
               reg2_eop <= '0';
               reg2_err <= '0';
               reg3_eop <= '0';
               reg3_err <= '0';
               reg4_eop <= '0';
               reg4_err <= '0';
               reg5_eop <= '0';
               reg5_err <= '0';
               reg6_eop <= '0';
               reg6_err <= '0';
            else
               reg2_eop <= reg2_eop_in;
               reg2_err <= reg2_err_in;
               reg3_eop <= reg2_eop;
               reg3_err <= reg2_err;
               reg4_eop <= reg3_eop;
               reg4_err <= reg3_err;
               reg5_eop <= reg4_eop;
               reg5_err <= reg4_err;
               reg6_eop <= reg5_eop;
               reg6_err <= reg5_err;
            end if;
         end if;
      end process regs_control_p;

      -- Data registers within datapath
      regs_data_p : process (CLK)
      begin
         if (CLK'event and CLK = '1') then
            reg2_eop_pos   <= reg2_eop_pos_in;
            reg3_eop_pos   <= reg2_eop_pos;
            reg4_eop_pos   <= reg3_eop_pos;
            reg5_eop_pos   <= reg4_eop_pos;
            reg6_eop_pos   <= reg5_eop_pos;
         end if;
      end process regs_data_p;

      
      -- Shift register for not changed signals (sop, data) -------------------

      -- Composition of shift register input signal
      sh_reg_sop_data_in <= reg_rx_sop_pos & reg_rx_sop & reg_rx_data;
      
      -- Shift register instantiation
      sh_reg_sop_data_i: entity work.sh_reg_bus
      generic map(
         NUM_BITS       => 6,
         DATA_WIDTH     => 516
      )
      port map(
         CLK            => CLK,
         DIN            => sh_reg_sop_data_in,
         CE             => '1',
         DOUT           => sh_reg_sop_data_out
      );
      
      -- Decomposition of shift register output signal
      reg6_data      <= sh_reg_sop_data_out(511 downto 0);
      reg6_sop       <= sh_reg_sop_data_out(512);
      reg6_sop_pos   <= sh_reg_sop_data_out(515 downto 513);

   end generate crc_remove_gen;
   
   
   -- -------------------------------------------------------------------------
   --           Datapath signals shifting (when CRC is not removed)
   -- -------------------------------------------------------------------------

   crc_keep_gen: if (INBANDFCS = true) generate

      -- Composition of shift register input signal
      sh_reg_datapath_in <= (reg_rx_err or reg_end_val_err or reg_no_crc_err) &
                            reg_rx_eop_pos & reg_rx_eop & reg_rx_sop_pos &
                            reg_rx_sop & reg_rx_data;

      -- Shift register instantiation
      sh_reg_datapath_i: entity work.sh_reg_bus
      generic map(
         NUM_BITS       => 6,
         DATA_WIDTH     => 524
      )
      port map(
         CLK            => CLK,
         DIN            => sh_reg_datapath_in,
         CE             => '1',
         DOUT           => sh_reg_datapath_out
      );
      
      -- Decomposition of shift register output signal
      reg6_data      <= sh_reg_datapath_out(511 downto 0);
      reg6_sop       <= sh_reg_datapath_out(512);
      reg6_sop_pos   <= sh_reg_datapath_out(515 downto 513);
      reg6_eop       <= sh_reg_datapath_out(516);
      reg6_eop_pos   <= sh_reg_datapath_out(522 downto 517);
      reg6_err       <= sh_reg_datapath_out(523);
      
   end generate crc_keep_gen;


   -- -------------------------------------------------------------------------
   --             Destination MAC address errors + MAC statistics
   -- -------------------------------------------------------------------------

   -- Extraction of destination MAC address -----------------------------------
   mx_dst_mac_extract_p : process(reg_rx_sop_pos, reg_rx_data)
   begin
      case (reg_rx_sop_pos) is
         when "000"  => mx_dst_mac_extract <= reg_rx_data(47 downto 0);
         when "001"  => mx_dst_mac_extract <= reg_rx_data(111 downto 64);
         when "010"  => mx_dst_mac_extract <= reg_rx_data(175 downto 128);
         when "011"  => mx_dst_mac_extract <= reg_rx_data(239 downto 192);
         when "100"  => mx_dst_mac_extract <= reg_rx_data(303 downto 256);
         when "101"  => mx_dst_mac_extract <= reg_rx_data(367 downto 320);
         when "110"  => mx_dst_mac_extract <= reg_rx_data(431 downto 384);
         when "111"  => mx_dst_mac_extract <= reg_rx_data(495 downto 448);
         when others => mx_dst_mac_extract <= (others => '0');
      end case;
   end process mx_dst_mac_extract_p;   


   -- Not in CAM memory error -------------------------------------------------

   -- Deriving of mem_err signal
   mem_err <= '1' when (cam_match_bus_in = 0) else '0';


   -- Broadcast error ---------------------------------------------------------

   -- Possible broadcast error occurence
   broadcast_err <=  '1' when (mx_dst_mac_extract /= X"FFFFFFFFFFFF") else '0';

   -- Storing broadcast error into register
   reg1_broadcast_err_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_broadcast_err <= '0';
         elsif (reg_rx_sop = '1') then
            reg1_broadcast_err <= broadcast_err;
         end if;
      end if;
   end process reg1_broadcast_err_p;

   -- Registers holding broadcast_err value
   regs_broadcast_err_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg2_broadcast_err <= '0';
         else
            reg2_broadcast_err <= reg1_broadcast_err;
         end if;
      end if;
   end process regs_broadcast_err_p;
   

   -- Multicast error ---------------------------------------------------------

   -- Possible multicast error occurence
   multicast_err <= not mx_dst_mac_extract(0);

   -- Storing multicast error into register
   reg1_multicast_err_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_multicast_err <= '0';
         elsif (reg_rx_sop = '1') then
            reg1_multicast_err <= multicast_err;
         end if;
      end if;
   end process reg1_multicast_err_p;
   
   -- Registers holding multicast_err value
   regs_multicast_err_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg2_multicast_err <= '0';
         else
            reg2_multicast_err <= reg1_multicast_err;
         end if;
      end if;
   end process regs_multicast_err_p;


   -- MAC errors dealing ------------------------------------------------------

   -- Multiplexor between different MAC check modes
   mx_mac_err_p : process (mac_check_mode_in, mem_err, reg2_broadcast_err,
                           reg2_multicast_err)
   begin
      case (mac_check_mode_in) is
         when "00"   => mx_mac_err(0) <= '0';
         when "01"   => mx_mac_err(0) <= mem_err;
         when "10"   => mx_mac_err(0) <= mem_err and reg2_broadcast_err;
         when "11"   => mx_mac_err(0) <= mem_err and reg2_broadcast_err and
                                         reg2_multicast_err;
      	when others => null;
      end case;
   end process mx_mac_err_p;

   -- Map input for sh_fifo with MAC information
   mac_fifo_in(0) <= mx_mac_err(0);
   mac_fifo_in(1) <= not(reg2_broadcast_err);
   mac_fifo_in(2) <= not(reg2_multicast_err);

   -- FIFO for storing MAC info and error information
   sh_fifo_mac_i: entity work.sh_fifo
   generic map(
      FIFO_WIDTH        => 3,
      FIFO_DEPTH        => C_MAC_FIFO_DEPTH,
      USE_INREG         => true,
      USE_OUTREG        => false
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,

      DIN               => mac_fifo_in,
      WE                => cam_match_bus_vld_in,

      FULL              => open,
      DOUT              => mac_fifo_dout,
      RE                => reg6_eop,
      EMPTY             => mac_fifo_empty,
      STATUS		      => open
   );
   
   -- MAC error signal derivation
   mac_err_out <= mac_fifo_dout(0) or mac_fifo_empty;

   -- Output information about MAC address
   mac_broadcast_out <= mac_fifo_dout(1);
   mac_multicast_out <= mac_fifo_dout(2);

   -- -------------------------------------------------------------------------
   --                                 SAU FIFO
   -- -------------------------------------------------------------------------
   
   -- FIFO for storing sampling information
   sh_fifo_sau_i: entity work.sh_fifo
   generic map(
      FIFO_WIDTH        => 1,
      FIFO_DEPTH        => C_SAU_FIFO_DEPTH,
      USE_INREG         => true,
      USE_OUTREG        => false
   )
   port map(
      CLK               => CLK,
      RESET             => RESET,
      DIN               => sau_accept_in,
      WE                => sau_dv_in,
      FULL              => open,
      DOUT              => sau_fifo_dout,
      RE                => reg6_eop,
      EMPTY             => sau_fifo_empty,
      STATUS		      => open
   );
   
   -- Sampling error signal derivation
   sau_err_out <= not sau_fifo_dout(0) and not sau_fifo_empty;
   
   
   -- -------------------------------------------------------------------------
   --                 Frame length counter + length statistics
   -- -------------------------------------------------------------------------
   
   -- Frame length counter ----------------------------------------------------

   -- Counter's increment in the middle and at the end of the frame
   frame_len_add <=
      -- Correct size of the last word
      ("00" & reg_rx_eop_pos) + 1 when (reg_rx_eop = '1') else
      -- Full word length in the middle of the frame
      "01000000";

   -- Counter's increment at the beginning of the frame
   frame_len_add_sop <=
      -- Different size of the first word
      "01000000" when ((reg_rx_sop = '1') and 
                      (reg_rx_sop_pos = "000")) else
      "00111000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "001")) else
      "00110000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "010")) else
      "00101000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "011")) else
      "00100000" when ((reg_rx_sop = '1') and 
                      (reg_rx_sop_pos = "100")) else
      "00011000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "101")) else
      "00010000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "110")) else
      "00001000" when ((reg_rx_sop = '1') and
                      (reg_rx_sop_pos = "111")) else
      -- Zero value when reg_rx_sop is not active
      "00000000";

   -- Counter's decrement at the beginning of the frame
   frame_len_sub_sop <= 
      -- Correct size of the first word
      ("00" & reg_rx_sop_pos & "000") when (reg_rx_sop = '1') else
      -- Zero value when reg_rx_sop is not active
      "00000000";      

   -- Storing frame_len_add_sop, frame_len_sub_sop and frame_len_add values
   reg1_frame_len_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_frame_len_add <= (others => '0');
            reg1_frame_len_add_sop <= (others => '0');
            reg1_frame_len_sub_sop <= (others => '0');
         else
            reg1_frame_len_add <= frame_len_add;
            reg1_frame_len_add_sop <= frame_len_add_sop;
            reg1_frame_len_sub_sop <= frame_len_sub_sop;
         end if;
      end if;
   end process reg1_frame_len_p;

   -- Storing and holding reg_rx_sop value
   regs_sop_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_sop <= '0';
            reg2_sop <= '0';
            reg3_sop <= '0';
         else
            reg1_sop <= reg_rx_sop;
            reg2_sop <= reg1_sop;
            reg3_sop <= reg2_sop;
         end if;
      end if;
   end process regs_sop_p;

   -- Delaying reg1_frame_len_add_sop value by 1 clock cycle
   reg2_frame_len_add_sop_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg2_frame_len_add_sop <= (others => '0');
         else
            reg2_frame_len_add_sop <= reg1_frame_len_add_sop;
         end if;
      end if;
   end process reg2_frame_len_add_sop_p;

   -- Multiplexer for choosing correct situation
   mx_frame_len_add : process (reg1_frame_len_add, reg2_frame_len_add_sop,
                               reg1_frame_len_sub_sop, reg1_whole_packet)
   begin
      if (reg1_whole_packet = '0') then
         reg2_frame_len_add_in <= reg1_frame_len_add + reg2_frame_len_add_sop;
      else
         reg2_frame_len_add_in <= reg1_frame_len_add - reg1_frame_len_sub_sop;            
      end if;
   end process mx_frame_len_add;

   -- Storing of increment value 
   reg2_frame_len_add_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
	      if (not RESET_BY_INIT and RESET = '1') then
            reg2_frame_len_add <= (others => '0');
         else
            reg2_frame_len_add <= reg2_frame_len_add_in;
         end if;
      end if;      
   end process reg2_frame_len_add_p;

   -- Frame length counter
   cnt_frame_len_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            cnt_frame_len <= (others => '0');
         elsif (reg2_whole_packet = '1' or reg3_sop = '1') then
            cnt_frame_len <= (15 downto 8 => '0') & reg2_frame_len_add;
         else
            cnt_frame_len <= cnt_frame_len + reg2_frame_len_add;
         end if;
      end if;
   end process cnt_frame_len_p;

   -- Interconnection when CRC is not removed
   cnt_keep_crc : if (INBANDFCS = true) generate
      reg4_cnt_frame_len_in <= cnt_frame_len;
      reg5_cnt_frame_len_in <= reg4_cnt_frame_len; 
   end generate cnt_keep_crc;
   
   -- Interconnection when CRC is removed
   cnt_remove_crc : if (INBANDFCS = false) generate
      reg4_cnt_frame_len_in <=
         cnt_frame_len when (reg3_no_crc_err = '1' or reg3_end_val_err = '1') else
         cnt_frame_len - 4;
      reg5_cnt_frame_len_in <=
         reg4_cnt_frame_len_in when (reg3_end_val_sel = '1' and reg3_end_val_err = '0') else
         reg4_cnt_frame_len; 
   end generate cnt_remove_crc;

   -- Register storing frame_len counter value
   regs_cnt_frame_len_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg4_cnt_frame_len <= (others => '0');
            reg5_cnt_frame_len <= (others => '0');
            reg6_cnt_frame_len <= (others => '0');
         else
            reg4_cnt_frame_len <= reg4_cnt_frame_len_in;
            reg5_cnt_frame_len <= reg5_cnt_frame_len_in;
            reg6_cnt_frame_len <= reg5_cnt_frame_len;
         end if;
      end if;
   end process regs_cnt_frame_len_p;

   -- Output frame length counter
   frame_len_out <= reg6_cnt_frame_len; 

   
   -- Length statistics -------------------------------------------------------

   -- Active, when frame does not have minimal or maximallength
   reg4_mintu_err_in <= '1' when cnt_frame_len < min_frame_len_in else '0';
   reg4_mtu_err_in <= '1' when cnt_frame_len > max_frame_len_in else '0';

   -- Interconnection when CRC is not removed
   minmax_keep_crc : if (INBANDFCS = true) generate
      reg5_mintu_err_in <= reg4_mintu_err;
      reg5_mtu_err_in <= reg4_mtu_err;  
   end generate minmax_keep_crc;

   -- Interconnection when CRC is removed
   minmax_remove_crc : if (INBANDFCS = false) generate
      reg5_mintu_err_in <=
         reg4_mintu_err_in when (reg3_end_val_sel = '1' and reg3_end_val_err = '0') else
         reg4_mintu_err;
      reg5_mtu_err_in <=
         reg4_mtu_err_in when (reg3_end_val_sel = '1' and reg3_end_val_err = '0') else
         reg4_mtu_err;
   end generate minmax_remove_crc;
     
   -- Registers for storing mintu_err and mtu_err values
   regs_mintu_mtu_err_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg4_mintu_err <= '0';
            reg5_mintu_err <= '0';
            reg6_mintu_err <= '0';
            reg4_mtu_err <= '0';
            reg5_mtu_err <= '0';
            reg6_mtu_err <= '0';
         else
            reg4_mintu_err <= reg4_mintu_err_in;
            reg5_mintu_err <= reg5_mintu_err_in;
            reg6_mintu_err <= reg5_mintu_err;
            reg4_mtu_err <= reg4_mtu_err_in;
            reg5_mtu_err <= reg5_mtu_err_in;
            reg6_mtu_err <= reg5_mtu_err;
         end if;
      end if;
   end process regs_mintu_mtu_err_p;

   -- Output frame length statistics
   mintu_err_out <= reg6_mintu_err;
   mtu_err_out <= reg6_mtu_err; 


   -- -------------------------------------------------------------------------
   --                    CRC computation and checking              
   -- -------------------------------------------------------------------------

   -- Detections and extractions from original data ---------------------------

   -- Detection of case when there is the whole packet in the word
   whole_packet <= '1' when ((rx_sop_in = '1') and (rx_eop_in = '1') and 
      ((rx_sop_pos_in & "000") < rx_eop_pos_in)) else '0';

   -- Detection of the case when the whole last word removing is required and
   -- the situation when the whole last word removing could cause an error
   end_val_p : process (rx_eop_in, rx_eop_pos_in, reg_rx_eop)
      begin
         if (rx_eop_in = '1') then
            if (rx_eop_pos_in < "000100") then
               end_val_sel <= '1';
               end_val_err <= reg_rx_eop;
            else
               end_val_sel <= '0';
               end_val_err <= '0';
            end if;
         else
            end_val_sel <= '0';
            end_val_err <= '0';
         end if;
      end process end_val_p;

   -- Detection of the case when the packet is too short and there is no crc in
   no_crc_err_p : process (rx_sop_in, rx_eop_in, rx_sop_pos_in, rx_eop_pos_in, whole_packet)
      begin
         if (rx_sop_in = '1' and rx_eop_in = '1' and whole_packet = '1') then
            if (((rx_sop_pos_in & "000") + 4) > rx_eop_pos_in) then
               no_crc_err <= '1';
            else
               no_crc_err <= '0';
            end if;
         else
            no_crc_err <= '0';
         end if;
   end process no_crc_err_p;

   -- Multiplexer for CRC extraction
   mx_crc_extract_p : process(rx_eop_pos_in, rx_data_in, reg_rx_data)
   begin
      case (rx_eop_pos_in) is
         when "000000" => crc_received <= rx_data_in(7 downto 0) &
                                          reg_rx_data(511 downto 488);
         when "000001" => crc_received <= rx_data_in(15 downto 0) &
                                          reg_rx_data(511 downto 496);
         when "000010" => crc_received <= rx_data_in(23 downto 0) &
                                          reg_rx_data(511 downto 504);
         when "000011" => crc_received <= rx_data_in(31 downto 0);
         when "000100" => crc_received <= rx_data_in(39 downto 8);
         when "000101" => crc_received <= rx_data_in(47 downto 16);
         when "000110" => crc_received <= rx_data_in(55 downto 24);
         when "000111" => crc_received <= rx_data_in(63 downto 32);
         when "001000" => crc_received <= rx_data_in(71 downto 40);
         when "001001" => crc_received <= rx_data_in(79 downto 48);
         when "001010" => crc_received <= rx_data_in(87 downto 56);
         when "001011" => crc_received <= rx_data_in(95 downto 64);
         when "001100" => crc_received <= rx_data_in(103 downto 72);
         when "001101" => crc_received <= rx_data_in(111 downto 80);
         when "001110" => crc_received <= rx_data_in(119 downto 88);
         when "001111" => crc_received <= rx_data_in(127 downto 96);
         when "010000" => crc_received <= rx_data_in(135 downto 104);
         when "010001" => crc_received <= rx_data_in(143 downto 112);
         when "010010" => crc_received <= rx_data_in(151 downto 120);
         when "010011" => crc_received <= rx_data_in(159 downto 128);
         when "010100" => crc_received <= rx_data_in(167 downto 136);
         when "010101" => crc_received <= rx_data_in(175 downto 144);
         when "010110" => crc_received <= rx_data_in(183 downto 152);
         when "010111" => crc_received <= rx_data_in(191 downto 160);
         when "011000" => crc_received <= rx_data_in(199 downto 168);
         when "011001" => crc_received <= rx_data_in(207 downto 176);
         when "011010" => crc_received <= rx_data_in(215 downto 184);
         when "011011" => crc_received <= rx_data_in(223 downto 192);
         when "011100" => crc_received <= rx_data_in(231 downto 200);
         when "011101" => crc_received <= rx_data_in(239 downto 208);
         when "011110" => crc_received <= rx_data_in(247 downto 216);
         when "011111" => crc_received <= rx_data_in(255 downto 224);
         when "100000" => crc_received <= rx_data_in(263 downto 232);
         when "100001" => crc_received <= rx_data_in(271 downto 240);
         when "100010" => crc_received <= rx_data_in(279 downto 248);
         when "100011" => crc_received <= rx_data_in(287 downto 256);
         when "100100" => crc_received <= rx_data_in(295 downto 264);
         when "100101" => crc_received <= rx_data_in(303 downto 272);
         when "100110" => crc_received <= rx_data_in(311 downto 280);
         when "100111" => crc_received <= rx_data_in(319 downto 288);
         when "101000" => crc_received <= rx_data_in(327 downto 296);
         when "101001" => crc_received <= rx_data_in(335 downto 304);
         when "101010" => crc_received <= rx_data_in(343 downto 312);
         when "101011" => crc_received <= rx_data_in(351 downto 320);
         when "101100" => crc_received <= rx_data_in(359 downto 328);
         when "101101" => crc_received <= rx_data_in(367 downto 336);
         when "101110" => crc_received <= rx_data_in(375 downto 344);
         when "101111" => crc_received <= rx_data_in(383 downto 352);
         when "110000" => crc_received <= rx_data_in(391 downto 360);
         when "110001" => crc_received <= rx_data_in(399 downto 368);
         when "110010" => crc_received <= rx_data_in(407 downto 376);
         when "110011" => crc_received <= rx_data_in(415 downto 384);
         when "110100" => crc_received <= rx_data_in(423 downto 392);
         when "110101" => crc_received <= rx_data_in(431 downto 400);
         when "110110" => crc_received <= rx_data_in(439 downto 408);
         when "110111" => crc_received <= rx_data_in(447 downto 416);
         when "111000" => crc_received <= rx_data_in(455 downto 424);
         when "111001" => crc_received <= rx_data_in(463 downto 432);
         when "111010" => crc_received <= rx_data_in(471 downto 440);
         when "111011" => crc_received <= rx_data_in(479 downto 448);
         when "111100" => crc_received <= rx_data_in(487 downto 456);
         when "111101" => crc_received <= rx_data_in(495 downto 464);
         when "111110" => crc_received <= rx_data_in(503 downto 472);
         when "111111" => crc_received <= rx_data_in(511 downto 480);
         when others   => crc_received <= (others => '0');
      end case; 
   end process mx_crc_extract_p;

   -- Registers storing end_val_sel, end_val_err, no_crc_err,
   -- whole_packet and crc_received values
   reg_crc_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg_whole_packet <= '0';
            reg_end_val_sel  <= '0';
            reg_end_val_err  <= '0';
            reg_no_crc_err   <= '0';
         else
            reg_whole_packet <= whole_packet;
            reg_end_val_sel  <= end_val_sel;
            reg_end_val_err  <= end_val_err;
            reg_no_crc_err   <= no_crc_err;
            reg_crc_received <= crc_received;
         end if;
      end if;
   end process reg_crc_p;


   -- Preparing inputs for CRC module and delaying extracted signals ----------

   -- Start of packet position in correct format (multiplied by 8 and negated)
   start_crc_in <= not (reg_rx_sop_pos & "000");

   -- Multiplexer for choosing correct value (based on end_val_sel signal)
   mx_crc : process (end_val_sel, end_val_err, rx_eop_in, rx_eop_pos_in, crc_received, 
                     reg_end_val_sel, reg_rx_eop, reg_rx_eop_pos, reg_crc_received)
   begin
      -- Choose correct value
      if (end_val_sel = '1' and end_val_err = '0') then
         eop_crc_in           <= rx_eop_in;
         eop_pos_crc_in       <= rx_eop_pos_in - 4;
         reg1_crc_received_in <= crc_received;
      else
         eop_crc_in           <= reg_rx_eop;
         eop_pos_crc_in       <= reg_rx_eop_pos - 4;
         reg1_crc_received_in <= reg_crc_received;
      end if;

      -- Erase original EOP value
      if (reg_end_val_sel = '1') then
         eop_crc_in <= '0'; 
      end if;
   end process mx_crc;

   -- Mask for CRC module
   mask_crc_in <= not eop_pos_crc_in;

   -- Data valid signal generation
   dv_crc_in_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            dv_crc_in <= '0';
         elsif (reg_no_crc_err = '1') then
            dv_crc_in <= '0';
         elsif (rx_sop_in = '1') then
            dv_crc_in <= '1';
         elsif ((eop_crc_in = '1') and (reg_rx_sop /= '1')) then
            dv_crc_in <= '0';
         end if;
      end if;
   end process dv_crc_in_p;

   -- Registers holding and delaying end_val_sel, no_crc_err,
   -- whole_packet and crc_received values
   regs_crc_p : process(CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (not RESET_BY_INIT and RESET = '1') then
            reg1_whole_packet <= '0';
            reg2_whole_packet <= '0';
            reg1_end_val_sel  <= '0';
            reg2_end_val_sel  <= '0';
            reg3_end_val_sel  <= '0';
            reg4_end_val_sel  <= '0';
            reg5_end_val_sel  <= '0';
            reg6_end_val_sel  <= '0';
            reg1_end_val_err  <= '0';
            reg2_end_val_err  <= '0';
            reg3_end_val_err  <= '0';
            reg4_end_val_err  <= '0';
            reg5_end_val_err  <= '0';
            reg6_end_val_err  <= '0';
            reg1_no_crc_err   <= '0';
            reg2_no_crc_err   <= '0';
            reg3_no_crc_err   <= '0';
         else
            reg1_whole_packet <= reg_whole_packet;
            reg2_whole_packet <= reg1_whole_packet;
            reg1_end_val_sel  <= reg_end_val_sel;
            reg2_end_val_sel  <= reg1_end_val_sel;
            reg3_end_val_sel  <= reg2_end_val_sel;
            reg4_end_val_sel  <= reg3_end_val_sel;
            reg5_end_val_sel  <= reg4_end_val_sel;
            reg6_end_val_sel  <= reg5_end_val_sel;
            reg1_end_val_err  <= reg_end_val_err;
            reg2_end_val_err  <= reg1_end_val_err;
            reg3_end_val_err  <= reg2_end_val_err;
            reg4_end_val_err  <= reg3_end_val_err;
            reg5_end_val_err  <= reg4_end_val_err;
            reg6_end_val_err  <= reg5_end_val_err;
            reg1_no_crc_err   <= reg_no_crc_err;
            reg2_no_crc_err   <= reg1_no_crc_err;
            reg3_no_crc_err   <= reg2_no_crc_err;
            reg1_crc_received <= reg1_crc_received_in;
            reg2_crc_received <= reg1_crc_received;
            reg3_crc_received <= reg2_crc_received;
            reg4_crc_received <= reg3_crc_received;
            reg5_crc_received <= reg4_crc_received;
            reg6_crc_received <= reg5_crc_received;
         end if;
      end if;
   end process regs_crc_p;


   -- CRC module instantiation, CRC module output reordering ------------------

   -- CRC module instantiation
   crc_i : entity work.crc32_gen_start  
   generic map(
      DATA_WIDTH => 512,
      REG_BITMAP => 31  -- 11111
   )
   port map(
      CLK   => CLK,
      RESET => RESET,
      DI    => reg_rx_data,
      DI_DV => dv_crc_in,
      SOP   => reg_rx_sop,
      EOP   => eop_crc_in,
      START => start_crc_in,
      MASK  => mask_crc_in,
      CRC   => crc_out,
      DO_DV => crc_dv_out
   );

   -- Computed CRC reordering
   crc_computed(7 downto 0)   <= crc_out(31 downto 24);
   crc_computed(15 downto 8)  <= crc_out(23 downto 16);
   crc_computed(23 downto 16) <= crc_out(15 downto 8);
   crc_computed(31 downto 24) <= crc_out(7 downto 0);


   -- Comparison of computed and extracted CRC values -------------------------

   -- Detection of CRC error
   crc_compare : process(CLK, crc_computed, reg6_crc_received, crc_dv_out)
   begin
      if (crc_dv_out = '1') then
         if (crc_computed = reg6_crc_received) then
            crc_err_active <= '0';
         else
            crc_err_active <= '1';
         end if;
      else
         crc_err_active <= '1';
      end if;
   end process crc_compare;


   -- Variable interconnection according to INBANDFCS value -------------------

   -- Interconnection when CRC is not removed
   crc_err_keep_crc : if (INBANDFCS = true) generate
      -- Register reg6_crc_err
      reg7_crc_err_p : process(CLK)
      begin
         if (CLK'event and CLK = '1') then
            if (not RESET_BY_INIT and RESET = '1') then
	            reg7_crc_err <= '0';
            else
               reg7_crc_err <= crc_err_active;
            end if;
         end if;
      end process reg7_crc_err_p;

      -- Selection of crc_rem_err value
      crc_rem_err_out <= '0';

      -- Selection of correct crc_err value
      crc_err_out <= reg7_crc_err when (reg6_end_val_sel = '1' and reg6_end_val_err = '0') else
                     crc_err_active;
   end generate crc_err_keep_crc;
   
   -- Interconnection when CRC is removed
   crc_err_remove_crc : if (INBANDFCS = false) generate
      -- Selection of crc_rem_err value
      crc_rem_err_out <= reg6_end_val_err;

      -- Direct assignment of crc_err_active value to crc_err
      crc_err_out <= crc_err_active;
   end generate crc_err_remove_crc;

 
   -- -------------------------------------------------------------------------
   --                    Dealing with output ports
   -- -------------------------------------------------------------------------

   -- CAM memory outputs
   CAM_DI         <= X"0001" & mx_dst_mac_extract;
   CAM_MATCH_EN   <= reg_rx_sop;
   CAM_MATCH_RST  <= '0';
 
   -- Sampling unit output
   SAU_CLK     <= CLK;
   SAU_RESET   <= RESET;
   SAU_REQ     <= reg_rx_sop;
   
   -- Datapath outputs
   TX_DATA     <= reg6_data;
   TX_SOP      <= reg6_sop;
   TX_SOP_POS  <= reg6_sop_pos;
   TX_EOP      <= reg6_eop;
   TX_EOP_POS  <= reg6_eop_pos;
   
   -- Statistics
   MAC_ERR     <= mac_err_out;
   MINTU_ERR   <= mintu_err_out;
   MTU_ERR     <= mtu_err_out;
   SAU_ERR     <= sau_err_out;
   CRC_ERR     <= crc_err_out;
   CRC_REM_ERR <= crc_rem_err_out;
   FRAME_ERR   <= reg6_err;
   FRAME_LEN   <= frame_len_out; 
   MAC_MCAST   <= mac_multicast_out;
   MAC_BCAST   <= mac_broadcast_out;

end architecture check_arch;
