-- tree_func.vhd: Auxilarity function and constant neeeded by dp_bmem
-- Copyright (C) 2013 CESNET
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_func.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                        Package declaration
-- ----------------------------------------------------------------------------
package tree_func is    
  -- implementation memory type
  type MEMORY_TYPE is (mem_regarray, mem_distmem, mem_bmem);
    
  -- used implementation type of tree stages
  type STAGE_TYPE is range 0 to 4; 
  constant STAGE_TYPE_NOTHING  : STAGE_TYPE := 0;
  constant STAGE_TYPE_REGARRAY : STAGE_TYPE := 1;
  constant STAGE_TYPE_DISTMEM  : STAGE_TYPE := 2;
  constant STAGE_TYPE_BMEM     : STAGE_TYPE := 3;
  constant STAGE_TYPE_DIST3MEM : STAGE_TYPE := 4;
  type STAGE_TYPE_ARRAY is array (integer range <>) of STAGE_TYPE; 
  
  -- internal fuction for bram type
  function GET_BEST_BRAM_TYPE (items : integer; size : integer) return integer;
  -- internal function for distmem type
  function GET_DISTMEM_TYPE (items : integer) return integer;
  -- internal function for stage type to memory type conversion
  function STAGE2MEMORY (stg : STAGE_TYPE) return MEMORY_TYPE;
  -- internal function for std_logic to boolean conversion
  function LOGIC2BOOLEAN (logic : std_logic) return boolean;  

end tree_func;


-- ----------------------------------------------------------------------------
--                      Package body declaration
-- ----------------------------------------------------------------------------
package body tree_func is

  function GET_BEST_BRAM_TYPE(items : integer; size : integer) return integer is
  variable s36, s18, s9 : integer;
  begin
    s36 := (((size-1)/36)+1)*(((items-1)/512)+1);
    s18 := (((size-1)/18)+1)*(((items-1)/1024)+1);
    s9  := (((size-1)/9)+1)*(((items-1)/2048)+1);
    if s36<=s18 and s36<=s9 then
      return 36;
    elsif s18<=s9 then
      return 18;
    else
      return 9;
    end if;
  end function;
  
  function GET_DISTMEM_TYPE(items : integer) return integer is
  begin
    if items>32 then
      return 64;
    elsif items>16 then
      return 32;
    else
      return 16;
    end if;
  end function;
   
  function STAGE2MEMORY (stg : STAGE_TYPE) return MEMORY_TYPE is
  begin
    if stg=STAGE_TYPE_REGARRAY then
      return mem_regarray;
    elsif stg=STAGE_TYPE_DISTMEM then
      return mem_distmem;
    else
      return mem_bmem;
    end if; 
  end function;
    
  function LOGIC2BOOLEAN (logic : std_logic) return boolean is
  begin
    if logic='1' then
      return true;
    else
      return false;
    end if;
  end function;
   
end tree_func;
