-- tree_config_memory_adc.vhd: Memory access simplifier for binary search tree over prefixes configurator 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_config_memory_adc.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree_config_memory_adc is
  signal cfg_di_record       : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal cfg_di_result       : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal cfg_di_record_reg   : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal cfg_di_result_reg   : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal cfg_addr_reg        : std_logic_vector(max(0,TREE_STAGES-1) downto 0);
  signal cfg_we_key_reg      : std_logic;
  signal cfg_we_data_reg     : std_logic;
  signal cfg_en_reg          : std_logic;
  signal mem_cs              : std_logic_vector(TREE_STAGES-1 downto 0);
  signal mem_cs_flip         : std_logic_vector(TREE_STAGES-1 downto 0);
  signal mem_cs0             : std_logic;
  signal mem_di_record_reg   : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal mem_di_result_reg   : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal mem_addr_reg        : std_logic_vector(max(0,TREE_STAGES-1) downto 0);
  signal mem_we_key_reg      : std_logic;
  signal mem_we_data_reg     : std_logic;
  signal mem_en_reg          : std_logic_vector(TREE_STAGES downto 0);
  signal mem_en0_reg         : std_logic;
  
  signal cfg_do_record_reg   : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal cfg_do_result_reg   : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal mem_sel_do          : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal mem_sel_do_fix      : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  type record_array_t is array (integer range <>) of std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  signal mem_do_records      : record_array_t(TREE_STAGES-1 downto 0);
  signal mem_do_result       : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal mem_do_records_reg  : record_array_t(TREE_STAGES-1 downto 0);
  signal mem_do_result_reg   : std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
  signal mem_drdy            : std_logic_vector(TREE_STAGES downto 0);
  signal mem0_drdy           : std_logic;
  signal mem_drdy_reg        : std_logic_vector(TREE_STAGES downto 0);
  signal mem0_drdy_reg       : std_logic; 
  
  signal we_key_sync         : std_logic_vector(4*TREE_STAGES+2 downto 0) := (others => '0');
  
begin
-- INPUT from Configurator -----------------------------------------------------
  -- record and result aggregation
  cfg_di_record <= CFG_DI_KEY & CFG_DI_KEYLEN & CFG_DI_ISEND & CFG_DI_KEYVLD;
  cfg_di_result <= CFG_DI_DATALEN & CFG_DI_DATA & CFG_DI_DATAVLD;
  
  -- input registering
  cfg_in_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      cfg_di_record_reg <= cfg_di_record;
      cfg_di_result_reg <= cfg_di_result;
      cfg_addr_reg      <= CFG_ADDR;
      cfg_we_key_reg    <= CFG_WE_KEY;
      cfg_we_data_reg   <= CFG_WE_DATA;
    end if;
  end process;
  cfg_in_vld_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        cfg_en_reg <= '0';
      else
        cfg_en_reg <= CFG_EN;
      end if;
    end if;
  end process;
  
  -- ADC: compute tree level from address
  mem_cs_flip <= cfg_addr_reg and ((not cfg_addr_reg)+1) and (cfg_addr_reg'length-1 downto 0 => cfg_en_reg);
  mem_cs0     <= '1' when cfg_addr_reg=(cfg_addr_reg'length-1 downto 0 => '0') and cfg_en_reg='1' else '0';
  mem_cs_flip_fix_gen : for i in 0 to TREE_STAGES-1 generate
    mem_cs(i) <= mem_cs_flip(TREE_STAGES-1-i); 
  end generate;
  
  -- output registering
  mem_in_reg_gen : if MEM_REG generate 
    mem_in_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        mem_di_record_reg <= cfg_di_record_reg;
        mem_di_result_reg <= cfg_di_result_reg;
        mem_addr_reg      <= cfg_addr_reg;
        mem_we_key_reg    <= cfg_we_key_reg;
        mem_we_data_reg   <= cfg_we_data_reg;
      end if;
    end process;
    mem_in_vld_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        if RESET='1' then
          mem_en_reg  <= (others => '0');
          mem_en0_reg <= '0';
        else
          mem_en_reg  <= cfg_en_reg & mem_cs;
          mem_en0_reg <= mem_cs0;
        end if;
      end if;
    end process;
  end generate;
  mem_in_noreg_gen : if not MEM_REG generate
    mem_di_record_reg <= cfg_di_record_reg;
    mem_di_result_reg <= cfg_di_result_reg;
    mem_addr_reg      <= cfg_addr_reg;
    mem_we_key_reg    <= cfg_we_key_reg;
    mem_we_data_reg   <= cfg_we_data_reg;
    mem_en_reg        <= cfg_en_reg & mem_cs;
    mem_en0_reg       <= mem_cs0;
  end generate;
  
  -- output connection
  mem_in_connect_gen : for i in 0 to TREE_STAGES-1 generate
    NODE_DI(mem_di_record_reg'length*(i+1)-1 downto mem_di_record_reg'length*i) <= mem_di_record_reg;
    NODE_ADDR(mem_addr_reg'length*(i+1)-1 downto mem_addr_reg'length*i)         <= mem_addr_reg;
  end generate;
  NODE_WE   <= (others => mem_we_key_reg);
  NODE_EN   <= mem_en_reg(TREE_STAGES-1 downto 0) and (TREE_STAGES-1 downto 0 => (not mem_we_data_reg or mem_we_key_reg));
  LEAF_DI   <= mem_di_result_reg;
  LEAF_ADDR <= mem_addr_reg;
  LEAF_WE   <= mem_we_data_reg; 
  LEAF_EN   <= mem_en_reg(TREE_STAGES) and (mem_we_data_reg or not mem_we_key_reg);


-- OUTPUT into Configurator ----------------------------------------------------
  -- record and result parsing  
  CFG_DO_KEYVLD   <= cfg_do_record_reg(0);       
  CFG_DO_ISEND    <= cfg_do_record_reg(1);  
  CFG_DO_KEYLEN   <= cfg_do_record_reg(log2(KEY_WIDTH)+1 downto 2);
  CFG_DO_KEY      <= cfg_do_record_reg(cfg_do_record_reg'length-1 downto cfg_do_record_reg'length-KEY_WIDTH);
  CFG_DO_DATALEN  <= cfg_do_result_reg(cfg_do_result_reg'length-1 downto DATA_WIDTH+1);
  CFG_DO_DATA     <= cfg_do_result_reg(DATA_WIDTH downto 1);
  CFG_DO_DATAVLD  <= cfg_do_result_reg(0); 
  
  -- output registering
  cfg_out_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if mem_drdy_reg(TREE_STAGES)='1' then
        cfg_do_record_reg <= mem_sel_do_fix;
        cfg_do_result_reg <= mem_do_result_reg;
      end if;
    end if;
  end process;
  cfg_out_vld_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        CFG_DRDY <= '0';
      else
        CFG_DRDY <= mem_drdy_reg(TREE_STAGES);
      end if;
    end if;
  end process;
  
  -- ADC: selection of correct output data
  adc_do_sel : process(mem_drdy_reg,mem_do_records_reg)
    variable res : std_logic_vector(KEY_WIDTH+log2(KEY_WIDTH)+2-1 downto 0);
  begin
    res := mem_do_records_reg(0);
    for i in 0 to TREE_STAGES-1 loop
      if mem_drdy_reg(i)='1' then
        res := mem_do_records_reg(i);
      end if;
    end loop;
    mem_sel_do <= res;
  end process; 
  mem_sel_do_fix(mem_sel_do'length-1 downto 1) <= mem_sel_do(mem_sel_do'length-1 downto 1); -- invalidate keyvld for read from address 0
  mem_sel_do_fix(0)                            <= mem_sel_do(0) and not mem0_drdy_reg;
  
  -- input registering
  mem_out_reg_gen : if MEM_REG generate
    mem_out_reg : process(CLK)
    begin
      if CLK'event and CLK='1' then
        mem_do_records_reg <= mem_do_records;
        mem_do_result_reg  <= mem_do_result;
        mem_drdy_reg  <= mem_drdy;
        mem0_drdy_reg <= mem0_drdy;
      end if;
    end process; 
  end generate;
  mem_out_noreg_gen : if not MEM_REG generate
    mem_do_records_reg <= mem_do_records;
    mem_do_result_reg  <= mem_do_result;
    mem_drdy_reg       <= mem_drdy;
    mem0_drdy_reg      <= mem0_drdy; 
  end generate;
  
  -- input connection
  mem_out_connect_gen : for i in 0 to TREE_STAGES-1 generate
    mem_do_records(i) <= NODE_DO(mem_do_records(i)'length*(i+1)-1 downto mem_do_records(i)'length*i);
  end generate;
  mem_do_result <= LEAF_DO;
  mem_drdy      <= LEAF_DRDY & NODE_DRDY;
  

-- Synchronization of signals around tree memory -------------------------------
  sync_reg_around_mem : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        mem0_drdy  <= '0';
      else
        mem0_drdy  <= mem_en0_reg;
      end if;
    end if;
  end process;   
  
  
-- Synchronization of key write with search engine -----------------------------
  we_key_sync(0) <= '0';
  we_key_sync_gen : for i in 0 to TREE_STAGES-1 generate
    reg0_gen : if USE_REGS(3*i)='1' generate
      reg0 : process (CLK)
      begin
        if CLK'event and CLK='1' then
          we_key_sync(4*i+1) <= we_key_sync(4*i+0);
        end if;
      end process;
    end generate;
    noreg0_gen : if USE_REGS(3*i+0)='0' generate
      we_key_sync(4*i+1) <= we_key_sync(4*i+0);
    end generate;
    reg1_gen : if USE_STAGES(i)=STAGE_TYPE_BMEM generate
      reg1 : process (CLK)
      begin
        if CLK'event and CLK='1' then
          we_key_sync(4*i+2) <= we_key_sync(4*i+1);
        end if;
      end process;
    end generate;
    noreg1_gen : if not(USE_STAGES(i)=STAGE_TYPE_BMEM) generate
      we_key_sync(4*i+2) <= we_key_sync(4*i+1);
    end generate;
    reg2_gen : if USE_REGS(3*i+1)='1' generate
      reg2 : process (CLK)
      begin
        if CLK'event and CLK='1' then
          we_key_sync(4*i+3) <= we_key_sync(4*i+2) or (mem_cs(i) and cfg_en_reg and cfg_we_key_reg);
        end if;
      end process;
    end generate;
    noreg2_gen : if USE_REGS(3*i+1)='0' generate
      we_key_sync(4*i+3) <= we_key_sync(4*i+2) or (mem_cs(i) and cfg_en_reg and cfg_we_key_reg);
    end generate;
    reg3_gen : if USE_REGS(3*i+2)='1' generate
      reg3 : process (CLK)
      begin
        if CLK'event and CLK='1' then
          we_key_sync(4*i+4) <= we_key_sync(4*i+3);
        end if;
      end process;
    end generate;
    noreg3_gen : if USE_REGS(3*i+2)='0' generate
      we_key_sync(4*i+4) <= we_key_sync(4*i+3);
    end generate;
  end generate;
  end_reg : process (CLK)
  begin
    if CLK'event and CLK='1' then
      we_key_sync(4*TREE_STAGES+1) <= we_key_sync(4*TREE_STAGES);
      we_key_sync(4*TREE_STAGES+2) <= we_key_sync(4*TREE_STAGES+1);
    end if;
  end process;
  CFG_WE_KEY_SYNC <= we_key_sync(4*TREE_STAGES+2);
end architecture;

