-- header_insert.vhd: Header insert architecture
-- Copyright (c) 2012 Brno University of Technology 
-- Author: Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: header_insert.vhd 2542 2012-10-06 09:21:59Z xkekel00 $
--
-- TODO:
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--  Architecture: HEADER_INSERT
-- ----------------------------------------------------------------------------

architecture HEADER_INSERT_ARCH of HEADER_INSERT is
  constant HEADER_LENGTH              : integer := 16+((RESULT_WIDTH-1)/8)+1;
  constant DREM                       : std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0) := conv_std_logic_vector(HEADER_LENGTH-1,log2(DATA_WIDTH/8));
  signal segment_size                 : std_logic_vector(15 downto 0);
  signal hardware_size                : std_logic_vector(15 downto 0);
  signal reserved                     : std_logic_vector(127 downto 0);
  signal hw_header                    : std_logic_vector(HEADER_LENGTH*8+DATA_WIDTH-1 downto 0);
  signal hw_header_mx                 : std_logic_vector(DATA_WIDTH-1 downto 0);
  signal hw_header_vld                : std_logic_vector(HEADER_LENGTH+DATA_WIDTH/8-1 downto 0);
  signal hw_header_vld_mx             : std_logic_vector(DATA_WIDTH/8-1 downto 0);
  signal hw_header_vld_act            : std_logic;
  signal hw_header_rx_next            : std_logic_vector(HEADER_LENGTH+DATA_WIDTH/8-1 downto 0);
  signal hw_header_rx_next_mx         : std_logic_vector(DATA_WIDTH/8-1 downto 0);
  signal hw_header_rx_next_act        : std_logic;
  signal word_cnt                     : std_logic_vector(log2(((HEADER_LENGTH*8-1)/DATA_WIDTH)+1+1)-1 downto 0);
  signal word_cnt_max                 : std_logic;
  signal word_cnt_last                : std_logic;
  signal word_cnt_fast                : std_logic;
  signal rx_ready                     : std_logic;
  signal sig_tx_src_rdy_n             : std_logic; 
begin
  rx_ready          <= not RX_SRC_RDY_N;
  hardware_size     <= conv_std_logic_vector(HEADER_LENGTH-4,16);
--  segment_size      <= conv_std_logic_vector(((HEADER_LENGTH-16-1)/8+1)*8,16) + RX_DATA(15 downto 0);
  segment_size      <= RX_DATA(15 downto 0);
  hw_header         <= (DATA_WIDTH-1 downto 0 => '0') & RESULT_ID & reserved(127 downto 40) & (3 downto 0 => DENY) & reserved(35 downto 32) & hardware_size & segment_size;
  hw_header_vld     <= (DATA_WIDTH/8-1 downto 0 => '1') & (HEADER_LENGTH-16-1 downto 0 => INSERT_READY) & (10 downto 0 => rx_ready) & (rx_ready and INSERT_READY) & rx_ready & rx_ready & rx_ready & rx_ready;
  hw_header_rx_next <= (DATA_WIDTH/8-1 downto 0 => '0') & (HEADER_LENGTH-16-1 downto 0 => '0') & (15 downto 0 => '1'); 
  reserved_gen : for i in 0 to 127 generate
    reserved(i) <= RX_DATA(i mod DATA_WIDTH);
  end generate;
  
  -- SZE header word counter
  process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        word_cnt <= (others => '0');
      elsif TX_DST_RDY_N='0' and sig_tx_src_rdy_n='0' then
        if RX_EOF_N='0' and word_cnt_max='1' then
          word_cnt <= (others => '0');
        elsif word_cnt_fast='1' and DENY='1' then
          word_cnt <=conv_std_logic_vector((HEADER_LENGTH*8-1)/DATA_WIDTH+1,word_cnt'length); 
        elsif word_cnt_max='0' then
          word_cnt <= word_cnt+1;
        end if;
      end if;
    end if;
  end process;
  word_cnt_fast <= '1' when word_cnt=conv_std_logic_vector(127/DATA_WIDTH,word_cnt'length) else '0';
  word_cnt_max  <= '1' when word_cnt=conv_std_logic_vector((HEADER_LENGTH*8-1)/DATA_WIDTH+1,word_cnt'length) else '0';
  word_cnt_last <= '1' when word_cnt=conv_std_logic_vector((HEADER_LENGTH*8-1)/DATA_WIDTH,word_cnt'length) else '0';
  
  -- SZE header data selector
  process(word_cnt, word_cnt_max, hw_header, hw_header_vld, hw_header_rx_next)
  begin
    hw_header_mx         <= (others => '0');
    hw_header_vld_mx     <= (others => '0');
    hw_header_rx_next_mx <= (others => '0');
    for i in 0 to ((HEADER_LENGTH*8-1)/DATA_WIDTH) loop
      if(conv_std_logic_vector(i,word_cnt'length)=word_cnt) then
        hw_header_mx         <= hw_header((i+1)*DATA_WIDTH-1 downto i*DATA_WIDTH);
        hw_header_vld_mx     <= hw_header_vld((i+1)*DATA_WIDTH/8-1 downto i*DATA_WIDTH/8);
        hw_header_rx_next_mx <= hw_header_rx_next((i+1)*DATA_WIDTH/8-1 downto i*DATA_WIDTH/8);
      end if;
    end loop; 
  end process;
  hw_header_vld_act     <= '1' when hw_header_vld_mx=(hw_header_vld_mx'length-1 downto 0 => '1') else '0';
  hw_header_rx_next_act <= '0' when hw_header_rx_next_mx=(hw_header_rx_next_mx'length-1 downto 0 => '0') else '1';

  -- output signals mapping
  TX_DATA  <= RX_DATA when word_cnt_max='1' else hw_header_mx;  
  TX_REM   <= RX_REM when word_cnt_max='1' else DREM;
  TX_SOF_N <= RX_SOF_N or RX_SRC_RDY_N;
  TX_EOF_N <= RX_EOF_N or not word_cnt_max;
  TX_SOP_N <= (RX_SOP_N or not word_cnt_max) and (RX_SOF_N or RX_SRC_RDY_N);
  TX_EOP_N <= not word_cnt_last and (RX_EOP_N or not word_cnt_max);
  TX_SRC_RDY_N <= sig_tx_src_rdy_n;
  sig_tx_src_rdy_n <= RX_SRC_RDY_N when word_cnt_max='1' else not hw_header_vld_act;
  RX_DST_RDY_N <= TX_DST_RDY_N or (not word_cnt_max and (not hw_header_rx_next_act or not hw_header_vld_act)); 
  INSERT_NEXT <= not RX_EOF_N and not RX_SRC_RDY_N and not TX_DST_RDY_N and word_cnt_max;
end architecture HEADER_INSERT_ARCH;
