--!
--! bus_handshake_fsm.vhd: FSM for BUS Handshake
--! Copyright (C) 2014 CESNET
--! Author(s): Jakub Cabal <jakubcabal@gmail.com>
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!
--! $Id$
--!
--! TODO:

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

--! pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
--! pragma translate_on

   --! -------------------------------------------------------------------------
   --!                      Entity declaration
   --! -------------------------------------------------------------------------

entity BUS_HANDSHAKE_FSM is
   Generic (
      TRANSMIT_FSM: boolean := true --! For TRANSMIT_FSM = true,
                                    --! for RECEIVE_FSM = false
   );
   Port ( 
      CLK       : in  STD_LOGIC;    --! Clock
      RST       : in  STD_LOGIC;    --! Reset
      ACK       : in  STD_LOGIC;    --! Signal ACK
      EVENT     : in  STD_LOGIC;    --! Signal EVENT
      READY     : out STD_LOGIC     --! Signal READY
   );
end BUS_HANDSHAKE_FSM;

   --! -------------------------------------------------------------------------
   --!                      Architecture declaration
   --! -------------------------------------------------------------------------

architecture FULL of BUS_HANDSHAKE_FSM is

   --! -------------------------------------------------------------------------
   --!                      SIGNALS
   --! -------------------------------------------------------------------------

   type state is (st0,st1);
   signal present_st : state;
   signal next_st    : state;
   signal default_st : state;

--! -------------------------------------------------------------------------
begin
--! -------------------------------------------------------------------------

   default_st0 : if TRANSMIT_FSM generate
      default_st <= st0;
   end generate;

   default_st1 : if NOT TRANSMIT_FSM generate
      default_st <= st1;
   end generate;

   --! Present State register
   present_state_reg: process(CLK)
      begin
      if (rising_edge(CLK)) then
         if (RST='1') then
            present_st <= default_st;
         else
            present_st <= next_st;
         end if;
      end if;
   end process;

   --! Next State logic
   next_state_logic: process (present_st, ACK, EVENT)
   begin
      case present_st is

         --! STATE st0
         when st0 =>

            if (EVENT = '1') then
               next_st <= st1;
            else
               next_st <= st0;
            end if;

         --! STATE st1
         when st1 =>

            if (ACK = '1') then
               next_st <= st0;
            else
               next_st <= st1;
            end if;

         --! Others STATE
         when others => null;

      end case;
   end process;

   --! Output logic
   output_logic: process (present_st, ACK, EVENT)
   begin
      case present_st is

         --! STATE st0
         when st0 =>

            READY <= '1';

         --! STATE st1
         when st1 =>

            READY <= '0';

         --! Others STATE
         when others => null;

      end case;
   end process;

end architecture FULL;
