-- netcope_adc.vhd
--!
--! \file
--! \brief NetCOPE MI32 addressing infrastructure.
--! \author Lukas Kekely <kekely@cesnet.cz>
--! \date 2013
--!
--! \section License
--!
--! Copyright (C) 2013 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
--! Package containing log2 function.
use work.math_pack.all;

--! \brief Implementation of NetCOPE MI32 addressing infrastructure.
architecture full of NETCOPE_ADC is
  --! Array of 32-bit signals
  type logic32_array_t is array (integer range <>) of std_logic_vector(31 downto 0);
  --! MI 32 interface
  type mi32_t is record
    DWR      : std_logic_vector(31 downto 0);
    ADDR     : std_logic_vector(31 downto 0);
    RD       : std_logic;
    WR       : std_logic;
    BE       : std_logic_vector(3  downto 0);
    DRD      : std_logic_vector(31 downto 0);
    ARDY     : std_logic;
    DRDY     : std_logic;
  end record;

  signal in_piped : mi32_t;
  signal is_user_space     : std_logic;
  signal user_rd, user_wr  : std_logic;
  signal dec1_en           : std_logic;
  signal dec1_cs           : std_logic_vector(1 downto 0);
  signal dec1_rd, dec1_wr  : std_logic_vector(1 downto 0);
  signal dec2_en           : std_logic;
  signal dec2_cs           : std_logic_vector(3 downto 0);
  signal dec2_rd, dec2_wr  : std_logic_vector(3 downto 0);
  signal dec3_en           : std_logic;
  signal dec3_cs           : std_logic_vector(1 downto 0);
  signal dec3_rd, dec3_wr  : std_logic_vector(1 downto 0);

  signal drd               : logic32_array_t(0 to 6);
  signal drd_sel           : std_logic_vector(log2(7)-1 downto 0);
  signal drdy              : std_logic_vector(6 downto 0);
  signal ardy              : std_logic_vector(6 downto 0);

begin
  --! \brief Pipe on input MI interface
  IN_MI_PIPE : entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 26,
    DATA_WIDTH => 32,
    USE_OUTREG => IN_PIPE,
    FAKE_PIPE  => not IN_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => IN_MI_DWR,
    IN_ADDR   => IN_MI_ADDR(25 downto 0),
    IN_RD     => IN_MI_RD,
    IN_WR     => IN_MI_WR,
    IN_ARDY   => IN_MI_ARDY,
    IN_BE     => IN_MI_BE,
    IN_DRD    => IN_MI_DRD,
    IN_DRDY   => IN_MI_DRDY,
    OUT_DWR   => in_piped.DWR,
    OUT_ADDR  => in_piped.ADDR(25 downto 0),
    OUT_RD    => in_piped.RD,
    OUT_WR    => in_piped.WR,
    OUT_ARDY  => in_piped.ARDY,
    OUT_BE    => in_piped.BE,
    OUT_DRD   => in_piped.DRD,
    OUT_DRDY  => in_piped.DRDY
  );
  in_piped.addr(31 downto 26) <= (others => '0');

  -- Address decoder for the highest level (NetCOPE vs. User)
  is_user_space <= in_piped.addr(25);
  --! \brief Address decoder of the highest level inside NetCOPE (ETH vs. rest of NetCOPE)
  decoder1 : entity work.dec1fn_enable
    generic map(2)
    port map (in_piped.addr(23 downto 23),dec1_en,dec1_cs);
  --! \brief Address decoder of the second level inside NetCOPE
  decoder2 : entity work.dec1fn_enable
    generic map(4)
    port map (in_piped.addr(15 downto 14),dec2_en,dec2_cs);
  --! \brief Address decoder of the third level inside NetCOPE
  decoder3 : entity work.dec1fn_enable
    generic map(2)
    port map (in_piped.addr(13 downto 13),dec3_en,dec3_cs);

  -- Read and write signals from address decoders
  dec1_en    <= not is_user_space;
  dec2_en    <= dec1_en and dec1_cs(0);
  dec3_en    <= dec2_en and dec2_cs(0);
  user_rd    <= is_user_space and in_piped.rd;
  user_wr    <= is_user_space and in_piped.wr;
  dec1_rd(1) <= in_piped.rd and dec1_cs(1);
  dec1_wr(1) <= in_piped.wr and dec1_cs(1);
  dec2_rd    <= (3 downto 0 => in_piped.rd) and dec2_cs;
  dec2_wr    <= (3 downto 0 => in_piped.wr) and dec2_cs;
  dec3_rd    <= (1 downto 0 => in_piped.rd) and dec3_cs;
  dec3_wr    <= (1 downto 0 => in_piped.wr) and dec3_cs;

  --! Agregation of ARDY signals from output MI interfaces
  in_piped.ardy <= (ardy(0) and dec3_cs(0))    or -- ID
                   (ardy(1) and dec2_cs(1))    or -- TSU
                   (ardy(2) and dec2_cs(2))    or -- Network
                   (ardy(3) and dec2_cs(3))    or -- DMA
                   (ardy(4) and dec1_cs(1))    or -- ETH
                   (ardy(5) and is_user_space) or -- USER
                   (ardy(6) and dec3_cs(1));      -- BOOT
  --! Agregation of DRDY signals from output MI interfaces
  drdy_or : entity work.GEN_OR
    generic map (7)
    port map (drdy, in_piped.drdy);
  --! Selection of DRD signal from output MI interfaces
  drd_enc : entity work.GEN_ENC
    generic map (7)
    port map(drdy,drd_sel);
  in_piped.drd <= drd(conv_integer(drd_sel));

  --! \brief Pipe on output interface into ID component
  ID_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 13,
    DATA_WIDTH => 32,
    USE_OUTREG => ID_PIPE,
    FAKE_PIPE  => not ID_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(12 downto 0),
    IN_RD     => dec3_rd(0),
    IN_WR     => dec3_wr(0),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(0),
    IN_DRDY   => drdy(0),
    IN_ARDY   => ardy(0),
    OUT_DWR   => ID_MI_DWR,
    OUT_ADDR  => ID_MI_ADDR(12 downto 0),
    OUT_RD    => ID_MI_RD,
    OUT_WR    => ID_MI_WR,
    OUT_ARDY  => ID_MI_ARDY,
    OUT_BE    => ID_MI_BE,
    OUT_DRD   => ID_MI_DRD,
    OUT_DRDY  => ID_MI_DRDY
  );
  ID_MI_ADDR(31 downto 13) <= (others => '0');

  --! \brief Pipe on output interface into BOOT component
  BOOT_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 13,
    DATA_WIDTH => 32,
    USE_OUTREG => BOOT_PIPE,
    FAKE_PIPE  => not BOOT_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(12 downto 0),
    IN_RD     => dec3_rd(1),
    IN_WR     => dec3_wr(1),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(6),
    IN_DRDY   => drdy(6),
    IN_ARDY   => ardy(6),
    OUT_DWR   => BOOT_MI_DWR,
    OUT_ADDR  => BOOT_MI_ADDR(12 downto 0),
    OUT_RD    => BOOT_MI_RD,
    OUT_WR    => BOOT_MI_WR,
    OUT_ARDY  => BOOT_MI_ARDY,
    OUT_BE    => BOOT_MI_BE,
    OUT_DRD   => BOOT_MI_DRD,
    OUT_DRDY  => BOOT_MI_DRDY
  );
  BOOT_MI_ADDR(31 downto 13) <= (others => '0');

  --! \brief Pipe on output interface into TSU
  TSU_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 14,
    DATA_WIDTH => 32,
    USE_OUTREG => TSU_PIPE,
    FAKE_PIPE  => not TSU_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(13 downto 0),
    IN_RD     => dec2_rd(1),
    IN_WR     => dec2_wr(1),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(1),
    IN_DRDY   => drdy(1),
    IN_ARDY   => ardy(1),
    OUT_DWR   => TSU_MI_DWR,
    OUT_ADDR  => TSU_MI_ADDR(13 downto 0),
    OUT_RD    => TSU_MI_RD,
    OUT_WR    => TSU_MI_WR,
    OUT_ARDY  => TSU_MI_ARDY,
    OUT_BE    => TSU_MI_BE,
    OUT_DRD   => TSU_MI_DRD,
    OUT_DRDY  => TSU_MI_DRDY
  );
  TSU_MI_ADDR(31 downto 14) <= (others => '0');

  --! \brief Pipe on output interface into Network Module
  NET_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 14,
    DATA_WIDTH => 32,
    USE_OUTREG => NET_PIPE,
    FAKE_PIPE  => not NET_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(13 downto 0),
    IN_RD     => dec2_rd(2),
    IN_WR     => dec2_wr(2),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(2),
    IN_DRDY   => drdy(2),
    IN_ARDY   => ardy(2),
    OUT_DWR   => NET_MI_DWR,
    OUT_ADDR  => NET_MI_ADDR(13 downto 0),
    OUT_RD    => NET_MI_RD,
    OUT_WR    => NET_MI_WR,
    OUT_ARDY  => NET_MI_ARDY,
    OUT_BE    => NET_MI_BE,
    OUT_DRD   => NET_MI_DRD,
    OUT_DRDY  => NET_MI_DRDY
  );
  NET_MI_ADDR(31 downto 14) <= (others => '0');

  --! \brief Pipe on output interface into DMA
  DMA_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 14,
    DATA_WIDTH => 32,
    USE_OUTREG => DMA_PIPE,
    FAKE_PIPE  => not DMA_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(13 downto 0),
    IN_RD     => dec2_rd(3),
    IN_WR     => dec2_wr(3),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(3),
    IN_DRDY   => drdy(3),
    IN_ARDY   => ardy(3),
    OUT_DWR   => DMA_MI_DWR,
    OUT_ADDR  => DMA_MI_ADDR(13 downto 0),
    OUT_RD    => DMA_MI_RD,
    OUT_WR    => DMA_MI_WR,
    OUT_ARDY  => DMA_MI_ARDY,
    OUT_BE    => DMA_MI_BE,
    OUT_DRD   => DMA_MI_DRD,
    OUT_DRDY  => DMA_MI_DRDY
  );
  DMA_MI_ADDR(31 downto 14) <= (others => '0');

  --! \brief Pipe on output interface for Physical Ethernet Layer
  ETH_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 24,
    DATA_WIDTH => 32,
    USE_OUTREG => ETH_PIPE,
    FAKE_PIPE  => not ETH_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(23 downto 0),
    IN_RD     => dec1_rd(1),
    IN_WR     => dec1_wr(1),
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(4),
    IN_DRDY   => drdy(4),
    IN_ARDY   => ardy(4),
    OUT_DWR   => ETH_MI_DWR,
    OUT_ADDR  => ETH_MI_ADDR(23 downto 0),
    OUT_RD    => ETH_MI_RD,
    OUT_WR    => ETH_MI_WR,
    OUT_ARDY  => ETH_MI_ARDY,
    OUT_BE    => ETH_MI_BE,
    OUT_DRD   => ETH_MI_DRD,
    OUT_DRDY  => ETH_MI_DRDY
  );
  ETH_MI_ADDR(31 downto 24) <= (others => '0');

  --! \brief Pipe on output interface to User Space
  USER_MI_PIPE: entity work.MI_PIPE
  generic map(
    ADDR_WIDTH => 25,
    DATA_WIDTH => 32,
    USE_OUTREG => USER_PIPE,
    FAKE_PIPE  => not USER_PIPE
  ) port map(
    CLK       => CLK,
    RESET     => RESET,
    IN_DWR    => in_piped.DWR,
    IN_ADDR   => in_piped.ADDR(24 downto 0),
    IN_RD     => user_rd,
    IN_WR     => user_wr,
    IN_BE     => in_piped.BE,
    IN_DRD    => drd(5),
    IN_DRDY   => drdy(5),
    IN_ARDY   => ardy(5),
    OUT_DWR   => USER_MI_DWR,
    OUT_ADDR  => USER_MI_ADDR(24 downto 0),
    OUT_RD    => USER_MI_RD,
    OUT_WR    => USER_MI_WR,
    OUT_ARDY  => USER_MI_ARDY,
    OUT_BE    => USER_MI_BE,
    OUT_DRD   => USER_MI_DRD,
    OUT_DRDY  => USER_MI_DRDY
  );
  USER_MI_ADDR(31 downto 25) <= (others => '0');
end architecture;
