-- mi_int.vhd: MI32 interface connection - architecture declaration
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture mi_int_arch of pci_mi_int is

   -- Types declaration -------------------------------------------------------
   type state_type is (S_IDLE, 
                       S_WRITE_0, S_WRITE_1, S_WRITE_2, S_WRITE_3,
                                  S_WRITE_4, S_WRITE_5, S_WRITE_6, S_WRITE_7,
                       S_WAIT_FOR_DATA,
                       S_READ,
                       S_WAIT_FOR_COMPLETION_SENT); 
 
   -- Signals declaration -----------------------------------------------------
   -- counter of Dwords to process
   signal dwords_to_process          : std_logic_vector(10 downto 0);
   signal dwords_to_process_inactive : std_logic;

   -- read addressing counter
   signal read_addr_cnt              : std_logic_vector(10 downto 0);

   -- address translation
   signal addr_mask                  : std_logic_vector(63 downto 0);
   signal bar_base_addr              : std_logic_vector(31 downto 0);
   signal translated_addr            : std_logic_vector(63 downto 0);
   signal reg_translated_addr        : std_logic_vector(31 downto 0);

   -- address offset counter
   signal cnt_addr_offset_dwords     : std_logic_vector(10 downto 0);
   signal cnt_addr_offset_bytes      : std_logic_vector(12 downto 0);

   -- final mi_addr composition
   signal mi_addr_sig                : std_logic_vector(31 downto 0);

   -- SOP register
   signal reg_sop                    : std_logic;
   
   -- FSM signals
   signal present_state              : state_type;
   signal next_state                 : state_type;

   -- masking data read from MI32
   signal rd_byte_en                 : std_logic_vector(3 downto 0);
   signal rd_byte_en_mask            : std_logic_vector(31 downto 0);
   signal mi_drd_masked              : std_logic_vector(31 downto 0);

   -- values for output ports used also within this component
   signal ready_sig                  : std_logic;
   signal mi_wr_sig                  : std_logic;
   signal mi_rd_sig                  : std_logic;

begin


   -- -------------------------------------------------------------------------
   --  Counter of Dwords to process
   -- -------------------------------------------------------------------------

   dwords_to_process_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            dwords_to_process <= (others => '0');
         else
            if (VALID = '1' and ready_sig = '1' AND SOP = '1') then
               dwords_to_process          <= DWORD_COUNT;
               dwords_to_process_inactive <= '1';
            elsif ((mi_wr_sig = '1' AND MI_ARDY = '1') OR (MI_DRDY = '1')) then
               dwords_to_process          <= dwords_to_process - 1;
               dwords_to_process_inactive <= '0';
            end if;
         end if;
      end if;
   end process dwords_to_process_p;

 
   -- -------------------------------------------------------------------------
   --  Read addressing counter
   -- -------------------------------------------------------------------------

   read_addr_cnt_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            read_addr_cnt <= (others => '0');
         else
            if (VALID = '1' and ready_sig = '1' AND SOP = '1') then
               read_addr_cnt <= DWORD_COUNT;
            elsif (mi_rd_sig = '1' AND MI_ARDY = '1') then
               read_addr_cnt <= read_addr_cnt - 1;
            end if;
         end if;
      end if;
   end process read_addr_cnt_p;


   -- -------------------------------------------------------------------------
   --  Address translation and addressing register for MI32
   -- -------------------------------------------------------------------------

   -- Address translation -----------------------------------------------------
   -- computation of address mask
   addr_mask_p : process(BAR_APERTURE)
      variable length_var : integer;
      variable mask_var   : std_logic_vector(63 downto 0);
   begin
      length_var := CONV_INTEGER(BAR_APERTURE);
      mask_var := (others => '0');
      for i in 0 to 63 loop
         if (i < length_var) then
            mask_var(i) := '1';
         else
            exit;
         end if; 
      end loop;
      addr_mask <= mask_var;
   end process addr_mask_p;

   -- selection of correct BAR base address
   bar_base_addr <= BAR0_BASE_ADDR    when (BAR_ID = "000") else
                    BAR1_BASE_ADDR    when (BAR_ID = "001") else
                    BAR2_BASE_ADDR    when (BAR_ID = "010") else
                    BAR3_BASE_ADDR    when (BAR_ID = "011") else
                    BAR4_BASE_ADDR    when (BAR_ID = "100") else
                    BAR5_BASE_ADDR    when (BAR_ID = "101") else
                    EXP_ROM_BASE_ADDR when (BAR_ID = "110") else
                    (others => '0');
 
   -- address translation
   translated_addr <= (ADDR AND addr_mask) + bar_base_addr;

   -- Translated address register ---------------------------------------------
   reg_translated_addr_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (VALID = '1' and ready_sig = '1' AND SOP = '1') then
            reg_translated_addr <= translated_addr(31 downto 0);
         end if;
      end if;
   end process reg_translated_addr_p;

   -- Address offset counter (counts in multiples of 4 bytes) -----------------
   cnt_addr_offset_dwords_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (VALID = '1' and ready_sig = '1' AND SOP = '1') then
            cnt_addr_offset_dwords <= (others => '0');
         elsif ((mi_wr_sig = '1' OR mi_rd_sig = '1')  AND MI_ARDY = '1') then
            cnt_addr_offset_dwords <= cnt_addr_offset_dwords + 1;
         end if;
      end if;
   end process cnt_addr_offset_dwords_p;

   cnt_addr_offset_bytes <= cnt_addr_offset_dwords & "00";

   -- Composition of final value for MI_ADDR ----------------------------------
   mi_addr_sig <= reg_translated_addr + cnt_addr_offset_bytes;


   -- -------------------------------------------------------------------------
   --  SOP register
   -- -------------------------------------------------------------------------

   reg_sop_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            reg_sop <= '0';
         else
            reg_sop <= SOP;
         end if;
      end if;
   end process reg_sop_p;


   -- -------------------------------------------------------------------------
   --  FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process (CLK)
   begin
      if (CLK'event and CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process (present_state, SOP, VALID, REQ_TYPE, MI_ARDY,
                                 MI_DRDY, FULL, COMPLETION_SENT,
                                 dwords_to_process)
   begin
      case (present_state) is

         when S_IDLE =>
            if (VALID = '1' AND SOP = '1') then
               if (REQ_TYPE = "0000") then
                  next_state <= S_READ;
               elsif (REQ_TYPE = "0001") then
                  next_state <= S_WRITE_4;
               else
                  next_state <= S_WAIT_FOR_COMPLETION_SENT;
               end if;
            else
               next_state <= S_IDLE;
            end if;

         when S_WRITE_0 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_1;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_0;
            end if;

         when S_WRITE_1 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_2;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_1;
            end if;

         when S_WRITE_2 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_3;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_2;
            end if;

         when S_WRITE_3 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_4;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_3;
            end if;

         when S_WRITE_4 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_5;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_4;
            end if;

         when S_WRITE_5 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_6;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_5;
            end if;

         when S_WRITE_6 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  next_state <= S_WRITE_7;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_6;
            end if;

         when S_WRITE_7 =>
            if (MI_ARDY = '1') then
               if (dwords_to_process > 1) then
                  if (VALID = '1') then
                     next_state <= S_WRITE_0;
                  else
                     next_state <= S_WAIT_FOR_DATA;
                  end if;
               else
                  next_state <= S_IDLE;
               end if;
            else
               next_state <= S_WRITE_7;
            end if;

         when S_WAIT_FOR_DATA =>
            if (VALID = '1') then
               next_state <= S_WRITE_0;
            else
               next_state <= S_WAIT_FOR_DATA;
            end if;

         when S_READ =>
            if (MI_DRDY = '1' AND FULL = '0') then
               if (dwords_to_process > 1) then
                  next_state <= S_READ;
               else
                  next_state <= S_WAIT_FOR_COMPLETION_SENT;
               end if;
            else
               next_state <= S_READ;
            end if;

         when S_WAIT_FOR_COMPLETION_SENT =>
            if (COMPLETION_SENT = '1') then
               next_state <= S_IDLE;
            else
               next_state <= S_WAIT_FOR_COMPLETION_SENT;
            end if;

--         when others =>
--            next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process (present_state, SOP, VALID, REQ_TYPE, REG_CQ_DATA,
                             REG_CQ_USER_FIRST_BE, REG_CQ_USER_LAST_BE,
                             reg_sop, MI_DRDY, FULL, dwords_to_process,
                             read_addr_cnt, dwords_to_process_inactive)
   begin
      -- default values
      ready_sig       <= '0';
      MI_DWR          <= (others => '0');
      MI_BE           <= (others => '0');
      mi_rd_sig       <= '0';
      mi_wr_sig       <= '0';
      READ_REQ        <= '0';
      UNSUPPORTED_REQ <= '0';
      EOP             <= '0';
      WR              <= '0';
      rd_byte_en      <= (others => '0');

      case (present_state) is

         when S_IDLE =>
            ready_sig <= '1';
            if (VALID = '1' AND SOP = '1') then
               if (REQ_TYPE = "0000") then
                  READ_REQ  <= '1';
               elsif (REQ_TYPE /= "0001") then
                  UNSUPPORTED_REQ <= '1';
               end if;
            end if;

         when S_WRITE_0 =>
            MI_DWR    <= REG_CQ_DATA(31 downto 0);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_1 =>
            MI_DWR    <= REG_CQ_DATA(63 downto 32);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_2 =>
            MI_DWR    <= REG_CQ_DATA(95 downto 64);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_3 =>
            MI_DWR    <= REG_CQ_DATA(127 downto 96);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_4 =>
            MI_DWR    <= REG_CQ_DATA(159 downto 128);
            mi_wr_sig <= '1';
            if (reg_sop = '1') then
               MI_BE <= REG_CQ_USER_FIRST_BE;
            else
               if (dwords_to_process <= 1) then
                  MI_BE <= REG_CQ_USER_LAST_BE;
               else
                  MI_BE <= (others => '1');
               end if;
            end if;

         when S_WRITE_5 =>
            MI_DWR    <= REG_CQ_DATA(191 downto 160);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_6 =>
            MI_DWR    <= REG_CQ_DATA(223 downto 192);
            mi_wr_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WRITE_7 =>
            MI_DWR    <= REG_CQ_DATA(255 downto 224);
            mi_wr_sig <= '1';
            ready_sig <= '1';
            if (dwords_to_process > 1) then
               MI_BE <= (others => '1');
            else
               MI_BE <= REG_CQ_USER_LAST_BE;
            end if;

         when S_WAIT_FOR_DATA =>
               ready_sig <= '1';

         when S_READ =>
            if (FULL = '0') then
               -- read requests on MI32 interface
               if (read_addr_cnt /= 0) then
                  mi_rd_sig <= '1';
                  MI_BE <= (others => '1');
               end if;
               -- masking data read from MI32 interface
               if (dwords_to_process_inactive = '1') then
                  rd_byte_en <= REG_CQ_USER_FIRST_BE;
               elsif (dwords_to_process <= 1) then
                  rd_byte_en <= REG_CQ_USER_LAST_BE;
               else
                  rd_byte_en <= (others => '1');
               end if;
               -- writing read data to the cc_int component
               if (MI_DRDY = '1') then
                  WR <= '1';
                  if (dwords_to_process <= 1) then
                     EOP <= '1';
                  end if;
               end if;
            end if;

         when others =>
            null;

      end case;      
      
   end process output_logic_p;


   -- -------------------------------------------------------------------------
   --  Masking data read from MI32 according to byte enable signals
   -- -------------------------------------------------------------------------

   -- creation of mask for data read from MI32
   rd_byte_en_mask <= X"00000000" when (rd_byte_en = "0000") else
                      X"000000FF" when (rd_byte_en = "0001") else 
                      X"0000FF00" when (rd_byte_en = "0010") else 
                      X"0000FFFF" when (rd_byte_en = "0011") else 
                      X"00FF0000" when (rd_byte_en = "0100") else 
                      X"00FF00FF" when (rd_byte_en = "0101") else 
                      X"00FFFF00" when (rd_byte_en = "0110") else 
                      X"00FFFFFF" when (rd_byte_en = "0111") else 
                      X"FF000000" when (rd_byte_en = "1000") else 
                      X"FF0000FF" when (rd_byte_en = "1001") else 
                      X"FF00FF00" when (rd_byte_en = "1010") else 
                      X"FF00FFFF" when (rd_byte_en = "1011") else 
                      X"FFFF0000" when (rd_byte_en = "1100") else 
                      X"FFFF00FF" when (rd_byte_en = "1101") else 
                      X"FFFFFF00" when (rd_byte_en = "1110") else 
                      X"FFFFFFFF";

   -- masking data read rom MI32
   mi_drd_masked <= MI_DRD AND rd_byte_en_mask;

   -- -------------------------------------------------------------------------
   --  Assigning values to output ports
   -- -------------------------------------------------------------------------
   
   -- interface to cq_int
   READY   <= ready_sig;

   -- MI32 interface
   MI_ADDR <= mi_addr_sig;
   MI_WR   <= mi_wr_sig;
   MI_RD   <= mi_rd_sig;

   -- interface to cc_int
   DATA    <= mi_drd_masked;

end architecture mi_int_arch;
