--!
--! general_fsm.vhd: Closed-loop solution CDC
--! Copyright (C) 2014 CESNET
--! Author(s): Jakub Cabal <jakubcabal@gmail.com>
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!
--! $Id$
--!
--! TODO:

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

--! pragma translate_off
library UNISIM;
use UNISIM.vcomponents.all;
--! pragma translate_on

   --! -------------------------------------------------------------------------
   --!                      Entity declaration
   --! -------------------------------------------------------------------------

entity ASYNC_GENERAL_FSM is
   Generic (
      DETECT_RISING_EDGE  : BOOLEAN := false;
      DETECT_FALLING_EDGE : BOOLEAN := false
   ); 
   Port (
      ACLK       : in  STD_LOGIC;   --! Clock
      ARST       : in  STD_LOGIC;   --! Reset
      ADATAIN    : in  STD_LOGIC;   --! Signal ADATAIN
      ADATA      : out STD_LOGIC;   --! Signal ADATA
      SIG_AREADY : in  STD_LOGIC;   --! Signal SIG_AREADY
      AREADY     : out STD_LOGIC    --! Signal AREADY
   );
end ASYNC_GENERAL_FSM;

   --! -------------------------------------------------------------------------
   --!                      Architecture declaration
   --! -------------------------------------------------------------------------

architecture FULL of ASYNC_GENERAL_FSM is

   signal sig_adata     : std_logic := '0';
   signal adata_next    : std_logic := '0';
   signal last_adatain  : std_logic := '0';

   signal comp_last_adatain   : std_logic := '0';
   signal comp_last_adatain2  : std_logic := '0';
   signal comp_adatain        : std_logic := '0';
   signal comp_adatain2       : std_logic := '0';

   type state is (st0,st1);
   signal present_st : state := st0;
   signal next_st    : state := st0;

--! -------------------------------------------------------------------------
begin
--! -------------------------------------------------------------------------

   ADATA <= sig_adata;

   --! sig_adata and last_adatain register
   process(ACLK)
   begin
      if (rising_edge(ACLK)) then
         if (ARST='1') then
            sig_adata <= '0';
            last_adatain <= '0';
         else
            last_adatain <= ADATAIN;
            sig_adata <= adata_next;
         end if;
      end if;
   end process;

   --! -------------------------------------------------------------------------

   double_mode : if (DETECT_RISING_EDGE AND DETECT_FALLING_EDGE) generate
      comp_last_adatain  <= '0';
      comp_adatain       <= '1';
      comp_last_adatain2 <= '1';
      comp_adatain2      <= '0';
   end generate;

   rising_mode : if (DETECT_RISING_EDGE AND NOT DETECT_FALLING_EDGE) generate
      comp_last_adatain  <= '0';
      comp_adatain       <= '1';
      comp_last_adatain2 <= '0';
      comp_adatain2      <= '1';
   end generate;

   falling_mode : if (NOT DETECT_RISING_EDGE AND DETECT_FALLING_EDGE) generate
      comp_last_adatain  <= '1';
      comp_adatain       <= '0';
      comp_last_adatain2 <= '1';
      comp_adatain2      <= '0';
   end generate;
 
   --! -------------------------------------------------------------------------
   --!                      FSM
   --! -------------------------------------------------------------------------

   --! Present State register
   present_state_reg: process(ACLK)
   begin
      if (rising_edge(ACLK)) then
         if (ARST='1') then
            present_st <= st0;
         else
            present_st <= next_st;
         end if;
      end if;
   end process;

   --! Next State logic
   next_state_logic: process (present_st, ADATAIN, SIG_AREADY, last_adatain, comp_last_adatain, comp_adatain, comp_last_adatain2, comp_adatain2)
   begin
      case present_st is

         --! STATE st0
         when st0 =>
            if ((last_adatain = comp_last_adatain AND ADATAIN = comp_adatain) OR (last_adatain = comp_last_adatain2 AND ADATAIN = comp_adatain2)) then
               next_st <= st1;
            else
               next_st <= st0;
            end if;

         --! STATE st1
         when st1 =>
            if (SIG_AREADY = '1') then
               next_st <= st0;
            else
               next_st <= st1;
            end if;

         --! Others STATE
         when others => null;

      end case;
   end process;

   --Output logic
   output_logic: process (present_st, ADATAIN, sig_adata, last_adatain, comp_last_adatain, comp_adatain, comp_last_adatain2, comp_adatain2)
   begin
      case present_st is

         --! STATE st0
         when st0 =>
            AREADY <= '1';

            if ((last_adatain = comp_last_adatain AND ADATAIN = comp_adatain) OR (last_adatain = comp_last_adatain2 AND ADATAIN = comp_adatain2)) then
               if (sig_adata = '1') then
                  adata_next <= '0';
               else
                  adata_next <= '1';
               end if;
            else
               if (sig_adata = '1') then
                  adata_next <= '1';
               else
                  adata_next <= '0';
               end if;
            end if;

         --! STATE st1
         when st1 =>
            AREADY <= '0';

            if (sig_adata = '1') then
               adata_next <= '1';
            else
               adata_next <= '0';
            end if;

         --! Others STATE
         when others => null;

      end case;
   end process;

end architecture FULL;
