-- dma_down_int.vhd: DMA DOWN interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture dma_down_int_arch of dma_down_int is

   -- Types declaration -------------------------------------------------------

   type state_type is (S_IDLE, S_READ_COMPLETION); 

   -- Signals declaration -----------------------------------------------------

   -- storing dma_down_hdr value
   signal dma_down_hdr_sig        : std_logic_vector(31 downto 0);
   signal reg_dma_down_hdr_sig    : std_logic_vector(31 downto 0) := (others => '0');
   signal reg_dma_down_hdr_sig_we : std_logic;

   -- FSM signals
   signal present_state : state_type;
   signal next_state    : state_type;


begin

   -- -------------------------------------------------------------------------
   --                            Auxiliary logic
   -- -------------------------------------------------------------------------

   -- register for storing dma_down_hdr value
   reg_dma_down_hdr_sig_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (reg_dma_down_hdr_sig_we = '1') then
            reg_dma_down_hdr_sig <= dma_down_hdr_sig;
         end if;
      end if;
   end process reg_dma_down_hdr_sig_p;


   -- -------------------------------------------------------------------------
   --                                 FSM
   -- -------------------------------------------------------------------------

   -- present state register --------------------------------------------------
   present_state_reg_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (RESET = '1') then
            present_state <= S_IDLE;
         else
            present_state <= next_state;
         end if;        
      end if;
   end process present_state_reg_p;
 
   -- next state logic --------------------------------------------------------
   next_state_logic_p : process(present_state, DMA_SRC_RDY, DMA_DOWN_DST_RDY,
                                DMA_SOP, DMA_EOP)
   begin
      case (present_state) is

         when S_IDLE =>
            if (DMA_SRC_RDY = '1' AND DMA_DOWN_DST_RDY = '1' AND
                DMA_SOP = '1') then
               if (DMA_EOP = '1') then
                  next_state <= S_IDLE;
               else
                  next_state <= S_READ_COMPLETION;
               end if;
            else
               next_state <= S_IDLE;
            end if;            

         when S_READ_COMPLETION =>
            if (DMA_SRC_RDY = '1' AND DMA_DOWN_DST_RDY = '1') then
               if (DMA_EOP = '1') then
                  next_state <= S_IDLE;
               else
                  next_state <= S_READ_COMPLETION;
               end if; 
            else
               next_state <= S_READ_COMPLETION;
            end if;

--         when others =>
--            next_state <= S_IDLE;

      end case;      
   end process next_state_logic_p;
 
   -- output logic ------------------------------------------------------------
   output_logic_p : process(present_state, DMA_SRC_RDY, DMA_DOWN_DST_RDY,
                            DMA_SOP, DMA_EOP, DMA_DOWN_HDR_ID,
                            DMA_DOWN_HDR_TAG, DMA_HDR_DWORDS,
                            PCIE_TAG_RELEASE, reg_dma_down_hdr_sig)
   begin
      -- default values
      dma_down_hdr_sig        <= reg_dma_down_hdr_sig;
      reg_dma_down_hdr_sig_we <= '0';
      DMA_DOWN_SOP            <= '0';
      DMA_DOWN_EOP            <= '0';
      DMA_DOWN_SRC_RDY        <= '0';
      DMA_DST_RDY             <= '0';
      TAG_RELEASE             <= '0';

      case (present_state) is

         when S_IDLE =>
            if (DMA_SRC_RDY = '1' AND DMA_DOWN_DST_RDY = '1' AND
                DMA_SOP = '1') then
               if (DMA_EOP = '1') then
                  dma_down_hdr_sig        <=
                     DMA_DOWN_HDR_ID  & -- component ID
                     DMA_DOWN_HDR_TAG & -- transaction tag
                     "00000"          & -- transaction type (memory read)
                     DMA_HDR_DWORDS;    -- Dwords count
                  reg_dma_down_hdr_sig_we <= '1';
                  DMA_DOWN_SOP            <= '1';
                  DMA_DOWN_EOP            <= '1';
                  DMA_DOWN_SRC_RDY        <= '1';
                  DMA_DST_RDY             <= '1';
                  TAG_RELEASE             <= PCIE_TAG_RELEASE;
               else
                  dma_down_hdr_sig        <=
                     DMA_DOWN_HDR_ID  & -- component ID
                     DMA_DOWN_HDR_TAG & -- transaction tag
                     "00000"          & -- transaction type (memory read)
                     DMA_HDR_DWORDS;    -- Dwords count
                  reg_dma_down_hdr_sig_we <= '1';
                  DMA_DOWN_SOP            <= '1';
                  DMA_DOWN_SRC_RDY        <= '1';
                  DMA_DST_RDY             <= '1';
                  TAG_RELEASE             <= PCIE_TAG_RELEASE;
               end if;
            end if;            

         when S_READ_COMPLETION =>
            if (DMA_SRC_RDY = '1' AND DMA_DOWN_DST_RDY = '1') then
               if (DMA_EOP = '1') then
                  DMA_DOWN_EOP     <= '1';
                  DMA_DOWN_SRC_RDY <= '1';
                  DMA_DST_RDY      <= '1';
               else
                  DMA_DOWN_SRC_RDY <= '1';
                  DMA_DST_RDY      <= '1';
               end if; 
            end if;

--         when others =>
--            null;

      end case;      
   end process output_logic_p;


   -- -------------------------------------------------------------------------
   --                    Assigning values to output ports
   -- -------------------------------------------------------------------------

   -- DMA DOWN interface 
   DMA_DOWN_DATA <= DMA_DATA;
   DMA_DOWN_HDR  <= dma_down_hdr_sig;

   -- interface to tag_manager
   TAG           <= PCIE_TAG;

end architecture dma_down_int_arch;
