-- dma_bus.vhd : Fiberblaze FB1CG card DMA bus infrastructure
--!
--! \file
--! \brief Fiberblaze FB1CG card DMA bus infrastructure
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                             Entity declaration
-- ----------------------------------------------------------------------------

entity dma_bus is
generic (
   --! \brief Select whether single or double PCIe cores are used
   --! \details single PCIe = false; double PCIe = true
   DOUBLE_PCIE            : boolean := true;
   --! \brief Insert pipelines between DMA splitter/switch and DMA tranfromers
   --! \details It is meaningful only if DOUBLE_PCIE = true.
   PIPES                  : boolean := true;
   --! \brief Enable debug statistics and control
   DEBUG_ENABLED          : boolean := false
);
port (
   --! \name User side clock and reset
   -- -------------------------------------------------------------------------
   --! \brief User clock
   USER_CLK               : in  std_logic;
   --! \brief User reset
   USER_RESET             : in  std_logic;

   --! \name DMA UP - user side interface (clocked at USER_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   USER_DMA_UP_DATA       : in  std_logic_vector(511 downto 0);
   --! \brief Transaction header
   --! \details Valid when USER_DMA_UP_SOP is valid.
   USER_DMA_UP_HDR        : in  std_logic_vector(95 downto 0);
   --! \brief Start of transaction
   --! \details Valid when USER_DMA_UP_SRC_RDY = USER_DMA_UP_DST_RDY = '1'.
   USER_DMA_UP_SOP        : in  std_logic;
   --! \brief End of transaction
   --! \details Valid when USER_DMA_UP_SRC_RDY = USER_DMA_UP_DST_RDY = '1'.
   USER_DMA_UP_EOP        : in  std_logic;
   --! \brief Source is ready to transmit data
   USER_DMA_UP_SRC_RDY    : in  std_logic;
   --! \brief Destination is ready to receive data
   USER_DMA_UP_DST_RDY    : out std_logic;

   --! \name DMA DOWN - user side interface (clocked at USER_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   USER_DMA_DOWN_DATA     : out std_logic_vector(511 downto 0);
   --! \brief Transaction header
   --! \details Valid when USER_DMA_DOWN_SOP is valid.
   USER_DMA_DOWN_HDR      : out std_logic_vector(31 downto 0);
   --! \brief Start of transaction
   --! \details Valid when USER_DMA_DOWN_SRC_RDY = USER_DMA_DOWN_DST_RDY = '1'.
   USER_DMA_DOWN_SOP      : out std_logic;
   --! \brief End of transaction
   --! \details Valid when USER_DMA_DOWN_SRC_RDY = USER_DMA_DOWN_DST_RDY = '1'.
   USER_DMA_DOWN_EOP      : out std_logic;
   --! \brief Source is ready to transmit data
   USER_DMA_DOWN_SRC_RDY  : out std_logic;
   --! \brief Destination is ready to receive data
   USER_DMA_DOWN_DST_RDY  : in  std_logic;

   --! \name Debug signals for DMA bus - (clocked at DMA_CLK)
   -- --------------------------------------------------------------------------
   --! \brief Enable path to PCIe 1 when using double PCIe
   USER_DMA_DEBUG_PCIE0_BUS_ENABLED : in  std_logic;
   --! \brief Request counters sampling
   USER_DMA_DEBUG_PCIE_BUS_SAMPLE   : in  std_logic;
   --! \brief Counter of transmitted bytes over PCIe 0
   USER_DMA_DEBUG_PCIE0_BUS_BYTES   : out std_logic_vector(63 downto 0);
   --! \brief Counter of transmitted bytes over PCIe 1
   USER_DMA_DEBUG_PCIE1_BUS_BYTES   : out std_logic_vector(63 downto 0);

   --! \name Clock and reset for DMA 0 bus - PCIe side interface
   -- -------------------------------------------------------------------------
   --! \brief DMA 0 bus clock
   PCIE_DMA0_CLK          : in  std_logic;
   --! \brief DMA 0 bus reset
   PCIE_DMA0_RESET        : in  std_logic;

   --! \name DMA UP 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   PCIE_DMA0_UP_DATA      : out std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when PCIE_DMA0_UP_SOP is valid.
   PCIE_DMA0_UP_HDR       : out std_logic_vector(95 downto 0);
   --! \brief Start of transaction
   --! \details Valid when PCIE_DMA0_UP_SRC_RDY = PCIE_DMA0_UP_DST_RDY = '1'.
   PCIE_DMA0_UP_SOP       : out std_logic;
   --! \brief End of transaction
   --! \details Valid when PCIE_DMA0_UP_SRC_RDY = PCIE_DMA0_UP_DST_RDY = '1'.
   PCIE_DMA0_UP_EOP       : out std_logic;
   --! \brief Source is ready to transmit data
   PCIE_DMA0_UP_SRC_RDY   : out std_logic;
   --! \brief Destination is ready to receive data
   PCIE_DMA0_UP_DST_RDY   : in  std_logic;

   --! \name DMA DOWN 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   PCIE_DMA0_DOWN_DATA    : in  std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when PCIE_DMA0_DOWN_SOP is valid.
   PCIE_DMA0_DOWN_HDR     : in  std_logic_vector(31 downto 0);
   --! \brief Start of transaction
   --! \details Valid when
   --!          PCIE_DMA0_DOWN_SRC_RDY = PCIE_DMA0_DOWN_DST_RDY = '1'.
   PCIE_DMA0_DOWN_SOP     : in  std_logic;
   --! \brief End of transaction
   --! \details Valid when
   --!          PCIE_DMA0_DOWN_SRC_RDY = PCIE_DMA0_DOWN_DST_RDY = '1'.
   PCIE_DMA0_DOWN_EOP     : in  std_logic;
   --! \brief Source is ready to transmit data
   PCIE_DMA0_DOWN_SRC_RDY : in  std_logic;
   --! \brief Destination is ready to receive data
   PCIE_DMA0_DOWN_DST_RDY : out std_logic;

   --! \name Clock and reset for DMA 1 bus - PCIe side interface
   -- -------------------------------------------------------------------------
   --! \brief DMA 1 bus clock
   PCIE_DMA1_CLK          : in  std_logic;
   --! \brief DMA 1 bus reset
   PCIE_DMA1_RESET        : in  std_logic;

   --! \name DMA UP 1 - PCIe side interface (clocked at PCIE_DMA1_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   PCIE_DMA1_UP_DATA      : out std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when PCIE_DMA1_UP_SOP is valid.
   PCIE_DMA1_UP_HDR       : out std_logic_vector(95 downto 0);
   --! \brief Start of transaction
   --! \details Valid when PCIE_DMA1_UP_SRC_RDY = PCIE_DMA1_UP_DST_RDY = '1'.
   PCIE_DMA1_UP_SOP       : out std_logic;
   --! \brief End of transaction
   --! \details Valid when PCIE_DMA1_UP_SRC_RDY = PCIE_DMA1_UP_DST_RDY = '1'.
   PCIE_DMA1_UP_EOP       : out std_logic;
   --! \brief Source is ready to transmit data
   PCIE_DMA1_UP_SRC_RDY   : out std_logic;
   --! \brief Destination is ready to receive data
   PCIE_DMA1_UP_DST_RDY   : in  std_logic;

   --! \name DMA DOWN 1 - PCIe side interface (clocked at PCIE_DMA1_CLK)
   -- -------------------------------------------------------------------------
   --! \brief Transaction data
   PCIE_DMA1_DOWN_DATA    : in  std_logic_vector(255 downto 0);
   --! \brief Transaction header
   --! \details Valid when PCIE_DMA1_DOWN_SOP is valid.
   PCIE_DMA1_DOWN_HDR     : in  std_logic_vector(31 downto 0);
   --! \brief Start of transaction
   --! \details Valid when
   --!          PCIE_DMA1_DOWN_SRC_RDY = PCIE_DMA1_DOWN_DST_RDY = '1'.
   PCIE_DMA1_DOWN_SOP     : in  std_logic;
   --! \brief End of transaction
   --! \details Valid when
   --!          PCIE_DMA1_DOWN_SRC_RDY = PCIE_DMA1_DOWN_DST_RDY = '1'.
   PCIE_DMA1_DOWN_EOP     : in  std_logic;
   --! \brief Source is ready to transmit data
   PCIE_DMA1_DOWN_SRC_RDY : in  std_logic;
   --! \brief Destination is ready to receive data
   PCIE_DMA1_DOWN_DST_RDY : out std_logic
);
end entity dma_bus;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of dma_bus is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! Outputs from ASFIFO on DMA UP interface
   signal asfifo_dma_up_data      : std_logic_vector(511 downto 0);
   signal asfifo_dma_up_hdr       : std_logic_vector(95 downto 0);
   signal asfifo_dma_up_sop       : std_logic;
   signal asfifo_dma_up_eop       : std_logic;
   signal asfifo_dma_up_src_rdy   : std_logic;
   signal asfifo_dma_up_dst_rdy   : std_logic;

   --! Registered outputs from ASFIFO on DMA UP interface
   signal reg_asfifo_dma_up_data  : std_logic_vector(511 downto 0);
   signal reg_asfifo_dma_up_hdr   : std_logic_vector(95 downto 0);
   signal reg_asfifo_dma_up_sop   : std_logic;
   signal reg_asfifo_dma_up_eop   : std_logic;
   signal reg_asfifo_dma_up_src_rdy : std_logic;
   signal reg_asfifo_dma_up_dst_rdy : std_logic;

   --! Inputs to ASFIFO on DMA DOWN interface
   signal asfifo_dma_down_data    : std_logic_vector(511 downto 0);
   signal asfifo_dma_down_hdr     : std_logic_vector(31 downto 0);
   signal asfifo_dma_down_sop     : std_logic;
   signal asfifo_dma_down_eop     : std_logic;
   signal asfifo_dma_down_src_rdy : std_logic;
   signal asfifo_dma_down_dst_rdy : std_logic;

   --! Splitter outputs to PCIe 0
   signal split0_dma_up_data      : std_logic_vector(511 downto 0);
   signal split0_dma_up_hdr       : std_logic_vector(95 downto 0);
   signal split0_dma_up_sop       : std_logic;
   signal split0_dma_up_eop       : std_logic;
   signal split0_dma_up_src_rdy   : std_logic;
   signal split0_dma_up_dst_rdy   : std_logic;

   --! Splitter outputs to PCIe 1
   signal split1_dma_up_data      : std_logic_vector(511 downto 0);
   signal split1_dma_up_hdr       : std_logic_vector(95 downto 0);
   signal split1_dma_up_sop       : std_logic;
   signal split1_dma_up_eop       : std_logic;
   signal split1_dma_up_src_rdy   : std_logic;
   signal split1_dma_up_dst_rdy   : std_logic;

   --! Switch inputs from PCIe 0
   signal switch0_dma_down_data   : std_logic_vector(511 downto 0);
   signal switch0_dma_down_hdr    : std_logic_vector(31 downto 0);
   signal switch0_dma_down_sop    : std_logic;
   signal switch0_dma_down_eop    : std_logic;
   signal switch0_dma_down_src_rdy: std_logic;
   signal switch0_dma_down_dst_rdy: std_logic;

   --! Switch inputs from PCIe 1
   signal switch1_dma_down_data   : std_logic_vector(511 downto 0);
   signal switch1_dma_down_hdr    : std_logic_vector(31 downto 0);
   signal switch1_dma_down_sop    : std_logic;
   signal switch1_dma_down_eop    : std_logic;
   signal switch1_dma_down_src_rdy: std_logic;
   signal switch1_dma_down_dst_rdy: std_logic;

   --! Pipe outputs to PCIe 0 DMA UP interface
   signal pipe0_dma_up_data       : std_logic_vector(511 downto 0);
   signal pipe0_dma_up_hdr        : std_logic_vector(95 downto 0);
   signal pipe0_dma_up_sop        : std_logic;
   signal pipe0_dma_up_eop        : std_logic;
   signal pipe0_dma_up_src_rdy    : std_logic;
   signal pipe0_dma_up_dst_rdy    : std_logic;

   --! Pipe inputs from PCIe 0 DMA DOWN interface
   signal pipe0_dma_down_data     : std_logic_vector(511 downto 0);
   signal pipe0_dma_down_hdr      : std_logic_vector(31 downto 0);
   signal pipe0_dma_down_sop      : std_logic;
   signal pipe0_dma_down_eop      : std_logic;
   signal pipe0_dma_down_src_rdy  : std_logic;
   signal pipe0_dma_down_dst_rdy  : std_logic;

   --! Pipe outputs to PCIe 1 DMA UP interface
   signal pipe1_dma_up_data       : std_logic_vector(511 downto 0);
   signal pipe1_dma_up_hdr        : std_logic_vector(95 downto 0);
   signal pipe1_dma_up_sop        : std_logic;
   signal pipe1_dma_up_eop        : std_logic;
   signal pipe1_dma_up_src_rdy    : std_logic;
   signal pipe1_dma_up_dst_rdy    : std_logic;

   --! Pipe inputs from PCIe 1 DMA DOWN interface
   signal pipe1_dma_down_data     : std_logic_vector(511 downto 0);
   signal pipe1_dma_down_hdr      : std_logic_vector(31 downto 0);
   signal pipe1_dma_down_sop      : std_logic;
   signal pipe1_dma_down_eop      : std_logic;
   signal pipe1_dma_down_src_rdy  : std_logic;
   signal pipe1_dma_down_dst_rdy  : std_logic;

   --! Outputs from transformer on PCIe 0 UP interface
   signal transform0_dma_up_data      : std_logic_vector(255 downto 0);
   signal transform0_dma_up_hdr       : std_logic_vector(95 downto 0);
   signal transform0_dma_up_sop       : std_logic;
   signal transform0_dma_up_eop       : std_logic;
   signal transform0_dma_up_src_rdy   : std_logic;
   signal transform0_dma_up_dst_rdy   : std_logic;

   --! Inputs to transformer on PCIe 0 DOWN interface
   signal transform0_dma_down_data    : std_logic_vector(255 downto 0);
   signal transform0_dma_down_hdr     : std_logic_vector(31 downto 0);
   signal transform0_dma_down_sop     : std_logic;
   signal transform0_dma_down_eop     : std_logic;
   signal transform0_dma_down_src_rdy : std_logic;
   signal transform0_dma_down_dst_rdy : std_logic;

   --! Helper signals
   signal split_balance          : std_logic := '1';
   signal reg_split_balance      : std_logic := '1';

   signal split_dma_up_in        : std_logic_vector(512+96+1 downto 0);

   signal split0_dma_up_wr       : std_logic;
   signal split0_dma_up_full     : std_logic;
   signal split0_dma_up_last     : std_logic;
   signal split0_dma_up_status   : std_logic_vector( 4 downto 0);
   signal split0_dma_up_out      : std_logic_vector(512+96+1 downto 0);
   signal split0_dma_up_read     : std_logic;
   signal split0_dma_up_empty    : std_logic;

   signal split1_dma_up_wr       : std_logic;
   signal split1_dma_up_full     : std_logic;
   signal split1_dma_up_last     : std_logic;
   signal split1_dma_up_status   : std_logic_vector( 4 downto 0);
   signal split1_dma_up_out      : std_logic_vector(512+96+1 downto 0);
   signal split1_dma_up_read     : std_logic;
   signal split1_dma_up_empty    : std_logic;

   signal async_dma_debug_pcie0_bus_enabled : std_logic;
   signal async_dma_debug_pcie_bus_sample   : std_logic;
   signal debug_bytes_cnt0   : std_logic_vector(63 downto 0);
   signal debug_bytes_cnt1   : std_logic_vector(63 downto 0);

-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --     Asynchronous FIFOs between PCIE_CLK domain and USER_CLK domain
   -- -------------------------------------------------------------------------

   --! Asynchronous FIFO on DMA UP interface
   -- -------------------------------------------------------------------------

   asfifo_dma_up_i : entity work.asfifo_dma_bus
   generic map (
      DATA_WIDTH     => 512,
      HDR_WIDTH      =>  96
   )
   port map (
      --! Write interface
      WR_CLK         => USER_CLK,
      WR_RESET       => USER_RESET,
      WR_DMA_DATA    => USER_DMA_UP_DATA,
      WR_DMA_HDR     => USER_DMA_UP_HDR,
      WR_DMA_SOP     => USER_DMA_UP_SOP,
      WR_DMA_EOP     => USER_DMA_UP_EOP,
      WR_DMA_SRC_RDY => USER_DMA_UP_SRC_RDY,
      WR_DMA_DST_RDY => USER_DMA_UP_DST_RDY,

      --! Read interface
      RD_CLK         => PCIE_DMA1_CLK,
      RD_RESET       => PCIE_DMA1_RESET,
      RD_DMA_DATA    => asfifo_dma_up_data,
      RD_DMA_HDR     => asfifo_dma_up_hdr,
      RD_DMA_SOP     => asfifo_dma_up_sop,
      RD_DMA_EOP     => asfifo_dma_up_eop,
      RD_DMA_SRC_RDY => asfifo_dma_up_src_rdy,
      RD_DMA_DST_RDY => asfifo_dma_up_dst_rdy
   );

   --! Asynchronous FIFO on DMA DOWN interface
   -- -------------------------------------------------------------------------

   asfifo_dma_down_i : entity work.asfifo_dma_bus
   generic map (
      DATA_WIDTH     => 512,
      HDR_WIDTH      =>  32
   )
   port map (
      --! Write interface
      WR_CLK         => PCIE_DMA1_CLK,
      WR_RESET       => PCIE_DMA1_RESET,
      WR_DMA_DATA    => asfifo_dma_down_data,
      WR_DMA_HDR     => asfifo_dma_down_hdr,
      WR_DMA_SOP     => asfifo_dma_down_sop,
      WR_DMA_EOP     => asfifo_dma_down_eop,
      WR_DMA_SRC_RDY => asfifo_dma_down_src_rdy,
      WR_DMA_DST_RDY => asfifo_dma_down_dst_rdy,

      --! Read interface
      RD_CLK         => USER_CLK,
      RD_RESET       => USER_RESET,
      RD_DMA_DATA    => USER_DMA_DOWN_DATA,
      RD_DMA_HDR     => USER_DMA_DOWN_HDR,
      RD_DMA_SOP     => USER_DMA_DOWN_SOP,
      RD_DMA_EOP     => USER_DMA_DOWN_EOP,
      RD_DMA_SRC_RDY => USER_DMA_DOWN_SRC_RDY,
      RD_DMA_DST_RDY => USER_DMA_DOWN_DST_RDY
   );

   -- Asynchronous DEBUG interface
   -- --------------------------------------------------------------------------
   debug_async_gen : if DEBUG_ENABLED generate
      bus0_enable_async : entity work.ASYNC_OPEN_LOOP
      port map (
         ACLK     => USER_CLK,
         ARST     => USER_RESET,
         ADATAIN  => USER_DMA_DEBUG_PCIE0_BUS_ENABLED,
         BCLK     => PCIE_DMA1_CLK,
         BRST     => PCIE_DMA1_RESET,
         BDATAOUT => async_dma_debug_pcie0_bus_enabled
      );
      sample_counters_async : entity work.ASYNC_GENERAL
      port map (
         ADATAIN   => USER_DMA_DEBUG_PCIE_BUS_SAMPLE,
         AREADY    => open,
         ACLK      => USER_CLK,
         ARST      => USER_RESET,
         BDATAOUT  => async_dma_debug_pcie_bus_sample,
         BCLK      => PCIE_DMA1_CLK,
         BRST      => PCIE_DMA1_RESET
      );
   end generate;
   no_debug_async_gen : if not DEBUG_ENABLED generate
      async_dma_debug_pcie0_bus_enabled <= '1';
      async_dma_debug_pcie_bus_sample   <= '0';
   end generate;


   -- -------------------------------------------------------------------------
   --             Generate when single PCIe core is instantiated
   -- -------------------------------------------------------------------------

   single_pcie_gen : if (NOT DOUBLE_PCIE) generate


      -- ----------------------------------------------------------------------
      --                  Unconnected PCIe DMA 0 interface
      -- ----------------------------------------------------------------------

      --! When single PCIe core is utilized, only the PCIe core 1 (PCIE_X0Y1)
      --! is instantiated.

      --! DMA UP 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
      PCIE_DMA0_UP_DATA      <= (others => '0');
      PCIE_DMA0_UP_HDR       <= (others => '0');
      PCIE_DMA0_UP_SOP       <= '0';
      PCIE_DMA0_UP_EOP       <= '0';
      PCIE_DMA0_UP_SRC_RDY   <= '0';

      --! DMA DOWN 0 - PCIe side interface (clocked at PCIE_DMA0_CLK)
      PCIE_DMA0_DOWN_DST_RDY <= '0';


      -- ----------------------------------------------------------------------
      --                Transformers on PCIe DMA 1 interface
      -- ----------------------------------------------------------------------

      --! Transform upstream DMA bus width from 512 to 256 bits
      -- ----------------------------------------------------------------------

      transformer_dma_up : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 512,
         TX_DATA_WIDTH  => 256,
         HDR_WIDTH      =>  96
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => asfifo_dma_up_data,
         RX_HDR         => asfifo_dma_up_hdr,
         RX_SOP         => asfifo_dma_up_sop,
         RX_EOP         => asfifo_dma_up_eop,
         RX_SRC_RDY     => asfifo_dma_up_src_rdy,
         RX_DST_RDY     => asfifo_dma_up_dst_rdy,

         --! TX interface
         TX_DATA        => PCIE_DMA1_UP_DATA,
         TX_HDR         => PCIE_DMA1_UP_HDR,
         TX_SOP         => PCIE_DMA1_UP_SOP,
         TX_EOP         => PCIE_DMA1_UP_EOP,
         TX_SRC_RDY     => PCIE_DMA1_UP_SRC_RDY,
         TX_DST_RDY     => PCIE_DMA1_UP_DST_RDY
      );

      --! Transform downstream DMA bus width from 256 to 512 bits
      -- ----------------------------------------------------------------------

      transformer_dma_down : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 256,
         TX_DATA_WIDTH  => 512,
         HDR_WIDTH      =>  32
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => PCIE_DMA1_DOWN_DATA,
         RX_HDR         => PCIE_DMA1_DOWN_HDR,
         RX_SOP         => PCIE_DMA1_DOWN_SOP,
         RX_EOP         => PCIE_DMA1_DOWN_EOP,
         RX_SRC_RDY     => PCIE_DMA1_DOWN_SRC_RDY,
         RX_DST_RDY     => PCIE_DMA1_DOWN_DST_RDY,

         --! TX interface
         TX_DATA        => asfifo_dma_down_data,
         TX_HDR         => asfifo_dma_down_hdr,
         TX_SOP         => asfifo_dma_down_sop,
         TX_EOP         => asfifo_dma_down_eop,
         TX_SRC_RDY     => asfifo_dma_down_src_rdy,
         TX_DST_RDY     => asfifo_dma_down_dst_rdy
      );

   end generate single_pcie_gen;


   -- -------------------------------------------------------------------------
   --             Generate when double PCIe cores are instantiated
   -- -------------------------------------------------------------------------

   double_pcie_gen : if (DOUBLE_PCIE) generate

      --! WATCH OUT! PCIe core 1 is the primary one (i.e. when only one PCIe
      --!            core is used, it has to be PCIe core 1)


      -- -------------------------------------------------------------------------
      --         Split DMA packets to PCIe 0 and PCIe 1 DMA UP interfaces
      -- -------------------------------------------------------------------------

      split0_dma_up_i : entity work.FIFO_STATUS
      generic map(
         DATA_WIDTH     => 96+512+2,
         ITEMS          => 16,
         BLOCK_SIZE     => 4
      )
      port map(
         RESET          => PCIE_DMA1_RESET,
         CLK            => PCIE_DMA1_CLK,

         -- Write interface
         DATA_IN        => split_dma_up_in,
         WRITE_REQ      => split0_dma_up_wr,
         FULL           => split0_dma_up_full,
         LSTBLK         => split0_dma_up_last,
         STATUS         => split0_dma_up_status,

         -- Read interface
         DATA_OUT       => split0_dma_up_out,
         READ_REQ       => split0_dma_up_read,
         EMPTY          => split0_dma_up_empty
      );

      split1_dma_up_i : entity work.FIFO_STATUS
      generic map(
         DATA_WIDTH     => 96+512+2,
         ITEMS          => 16,
         BLOCK_SIZE     => 4
      )
      port map(
         RESET          => PCIE_DMA1_RESET,
         CLK            => PCIE_DMA1_CLK,

         -- Write interface
         DATA_IN        => split_dma_up_in,
         WRITE_REQ      => split1_dma_up_wr,
         FULL           => split1_dma_up_full,
         LSTBLK         => split1_dma_up_last,
         STATUS         => split1_dma_up_status,

         -- Read interface
         DATA_OUT       => split1_dma_up_out,
         READ_REQ       => split1_dma_up_read,
         EMPTY          => split1_dma_up_empty
      );

      process (PCIE_DMA1_CLK)
      begin
         if(PCIE_DMA1_CLK'event and PCIE_DMA1_CLK = '1') then
            -- Data Valid signal
            reg_asfifo_dma_up_src_rdy <= asfifo_dma_up_src_rdy and asfifo_dma_up_dst_rdy;

            reg_asfifo_dma_up_data  <= asfifo_dma_up_data;
            reg_asfifo_dma_up_hdr   <= asfifo_dma_up_hdr;
            reg_asfifo_dma_up_sop   <= asfifo_dma_up_sop;
            reg_asfifo_dma_up_eop   <= asfifo_dma_up_eop;
         end if;
      end process;

      asfifo_dma_up_dst_rdy <= reg_asfifo_dma_up_dst_rdy;

      split_dma_up_in       <= reg_asfifo_dma_up_data & (reg_asfifo_dma_up_hdr and X"FFFFFFFFFFFFFFFFFFFF7FFF") & reg_asfifo_dma_up_eop & reg_asfifo_dma_up_sop;

      split0_dma_up_data    <= split0_dma_up_out(512+96+1 downto 96+2);
      split0_dma_up_hdr     <= split0_dma_up_out(96+1 downto 2);
      split0_dma_up_eop     <= split0_dma_up_out(1);
      split0_dma_up_sop     <= split0_dma_up_out(0);
      split0_dma_up_read    <= split0_dma_up_dst_rdy;
      split0_dma_up_src_rdy <= not split0_dma_up_empty;

      split1_dma_up_data    <= split1_dma_up_out(512+96+1 downto 96+2);
      split1_dma_up_hdr     <= split1_dma_up_out(96+1 downto 2);
      split1_dma_up_eop     <= split1_dma_up_out(1);
      split1_dma_up_sop     <= split1_dma_up_out(0);
      split1_dma_up_read    <= split1_dma_up_dst_rdy;
      split1_dma_up_src_rdy <= not split1_dma_up_empty;

      process (PCIE_DMA1_CLK, asfifo_dma_up_sop, asfifo_dma_up_hdr, split_balance, reg_split_balance, split1_dma_up_status, split0_dma_up_status, asfifo_dma_up_src_rdy, split0_dma_up_last, split1_dma_up_last, async_dma_debug_pcie0_bus_enabled, reg_asfifo_dma_up_src_rdy)
      begin
         if(PCIE_DMA1_CLK'event and PCIE_DMA1_CLK = '1') then
            if(asfifo_dma_up_src_rdy = '1' and asfifo_dma_up_dst_rdy = '1' and asfifo_dma_up_sop = '1') then
               reg_split_balance  <= split_balance;
            end if;
         end if;

         if(reg_split_balance = '1') then
            reg_asfifo_dma_up_dst_rdy <= not split1_dma_up_last;
            split1_dma_up_wr  <= reg_asfifo_dma_up_src_rdy;
            split0_dma_up_wr  <= '0';
         else
            reg_asfifo_dma_up_dst_rdy <= not split0_dma_up_last;
            split0_dma_up_wr  <= reg_asfifo_dma_up_src_rdy;
            split1_dma_up_wr  <= '0';
         end if;

         -- DMA_REQUEST_UNIT_ID check: packet from DMA_CTRL_[R|T]X_REQUEST can be balanced
         if(asfifo_dma_up_hdr(31 downto 25) = "0000000") then -- bit 24 is RX/TX
            if(split1_dma_up_status > split0_dma_up_status or async_dma_debug_pcie0_bus_enabled = '0') then
               split_balance  <= '1';
            else
               split_balance  <= '0';
            end if;
         else
            split_balance     <= asfifo_dma_up_hdr(15) or not async_dma_debug_pcie0_bus_enabled;
         end if;
      end process;

      -- -------------------------------------------------------------------------
      --        Join DMA packets from PCIe 0 and PCIe 1 DMA DOWN interfaces
      -- -------------------------------------------------------------------------

      dma_switch_i : entity work.DMA_SWITCH
      generic map (
         DATA_WIDTH        => 512,
         HDR_WIDTH         =>  32
      )
      port map (
         CLK               => PCIE_DMA1_CLK,
         RESET             => PCIE_DMA1_RESET,

         DMA_RX0_DATA      => switch0_dma_down_data,
         DMA_RX0_HDR       => switch0_dma_down_hdr,
         DMA_RX0_SOP       => switch0_dma_down_sop,
         DMA_RX0_EOP       => switch0_dma_down_eop,
         DMA_RX0_SRC_RDY   => switch0_dma_down_src_rdy,
         DMA_RX0_DST_RDY   => switch0_dma_down_dst_rdy,

         DMA_RX1_DATA      => switch1_dma_down_data,
         DMA_RX1_HDR       => switch1_dma_down_hdr,
         DMA_RX1_SOP       => switch1_dma_down_sop,
         DMA_RX1_EOP       => switch1_dma_down_eop,
         DMA_RX1_SRC_RDY   => switch1_dma_down_src_rdy,
         DMA_RX1_DST_RDY   => switch1_dma_down_dst_rdy,

         DMA_TX_DATA       => asfifo_dma_down_data,
         DMA_TX_HDR        => asfifo_dma_down_hdr,
         DMA_TX_SOP        => asfifo_dma_down_sop,
         DMA_TX_EOP        => asfifo_dma_down_eop,
         DMA_TX_SRC_RDY    => asfifo_dma_down_src_rdy,
         DMA_TX_DST_RDY    => asfifo_dma_down_dst_rdy
      );


      -- ----------------------------------------------------------------------
      --                 Generate when DMA pipes are enabled
      -- ----------------------------------------------------------------------

      pipes_gen : if (PIPES) generate

         --! PIPE on PCIe 0 UP interface
         -- -------------------------------------------------------------------

         pipe0_dma_up_i : entity work.DMA_PIPE
         generic map (
            DATA_WIDTH => 512,
            HDR_WIDTH  =>  96,

            USE_OUTREG => true,
            FAKE_PIPE  => false
         )
         port map (
            CLK        => PCIE_DMA1_CLK,
            RESET      => PCIE_DMA1_RESET,

            RX_DATA    => split0_dma_up_data,
            RX_HDR     => split0_dma_up_hdr,
            RX_SOP     => split0_dma_up_sop,
            RX_EOP     => split0_dma_up_eop,
            RX_SRC_RDY => split0_dma_up_src_rdy,
            RX_DST_RDY => split0_dma_up_dst_rdy,

            TX_DATA    => pipe0_dma_up_data,
            TX_HDR     => pipe0_dma_up_hdr,
            TX_SOP     => pipe0_dma_up_sop,
            TX_EOP     => pipe0_dma_up_eop,
            TX_SRC_RDY => pipe0_dma_up_src_rdy,
            TX_DST_RDY => pipe0_dma_up_dst_rdy
         );

         --! PIPE on PCIe 0 DOWN interface
         -- -------------------------------------------------------------------

         pipe0_dma_down_i : entity work.DMA_PIPE
         generic map (
            DATA_WIDTH => 512,
            HDR_WIDTH  =>  32,

            USE_OUTREG => true,
            FAKE_PIPE  => false
         )
         port map (
            CLK        => PCIE_DMA1_CLK,
            RESET      => PCIE_DMA1_RESET,

            RX_DATA    => pipe0_dma_down_data,
            RX_HDR     => pipe0_dma_down_hdr,
            RX_SOP     => pipe0_dma_down_sop,
            RX_EOP     => pipe0_dma_down_eop,
            RX_SRC_RDY => pipe0_dma_down_src_rdy,
            RX_DST_RDY => pipe0_dma_down_dst_rdy,

            TX_DATA    => switch0_dma_down_data,
            TX_HDR     => switch0_dma_down_hdr,
            TX_SOP     => switch0_dma_down_sop,
            TX_EOP     => switch0_dma_down_eop,
            TX_SRC_RDY => switch0_dma_down_src_rdy,
            TX_DST_RDY => switch0_dma_down_dst_rdy
         );

         --! PIPE on PCIe 1 UP interface
         -- -------------------------------------------------------------------

         pipe1_dma_up_i : entity work.DMA_PIPE
         generic map (
            DATA_WIDTH => 512,
            HDR_WIDTH  =>  96,

            USE_OUTREG => true,
            FAKE_PIPE  => false
         )
         port map (
            CLK        => PCIE_DMA1_CLK,
            RESET      => PCIE_DMA1_RESET,

            RX_DATA    => split1_dma_up_data,
            RX_HDR     => split1_dma_up_hdr,
            RX_SOP     => split1_dma_up_sop,
            RX_EOP     => split1_dma_up_eop,
            RX_SRC_RDY => split1_dma_up_src_rdy,
            RX_DST_RDY => split1_dma_up_dst_rdy,

            TX_DATA    => pipe1_dma_up_data,
            TX_HDR     => pipe1_dma_up_hdr,
            TX_SOP     => pipe1_dma_up_sop,
            TX_EOP     => pipe1_dma_up_eop,
            TX_SRC_RDY => pipe1_dma_up_src_rdy,
            TX_DST_RDY => pipe1_dma_up_dst_rdy
         );

         --! PIPE on PCIe 1 DOWN interface
         -- -------------------------------------------------------------------

         pipe1_dma_down_i : entity work.DMA_PIPE
         generic map (
            DATA_WIDTH => 512,
            HDR_WIDTH  =>  32,

            USE_OUTREG => true,
            FAKE_PIPE  => false
         )
         port map (
            CLK        => PCIE_DMA1_CLK,
            RESET      => PCIE_DMA1_RESET,

            RX_DATA    => pipe1_dma_down_data,
            RX_HDR     => pipe1_dma_down_hdr,
            RX_SOP     => pipe1_dma_down_sop,
            RX_EOP     => pipe1_dma_down_eop,
            RX_SRC_RDY => pipe1_dma_down_src_rdy,
            RX_DST_RDY => pipe1_dma_down_dst_rdy,

            TX_DATA    => switch1_dma_down_data,
            TX_HDR     => switch1_dma_down_hdr,
            TX_SOP     => switch1_dma_down_sop,
            TX_EOP     => switch1_dma_down_eop,
            TX_SRC_RDY => switch1_dma_down_src_rdy,
            TX_DST_RDY => switch1_dma_down_dst_rdy
         );

      end generate pipes_gen;


      -- ----------------------------------------------------------------------
      --                Generate when DMA pipes are disabled
      -- ----------------------------------------------------------------------

      not_pipes_gen : if (NOT PIPES) generate

         --! Direct connection instead of PIPE on PCIe 0 UP interface
         -- -------------------------------------------------------------------

         pipe0_dma_up_data        <= split0_dma_up_data;
         pipe0_dma_up_hdr         <= split0_dma_up_hdr;
         pipe0_dma_up_sop         <= split0_dma_up_sop;
         pipe0_dma_up_eop         <= split0_dma_up_eop;
         pipe0_dma_up_src_rdy     <= split0_dma_up_src_rdy;
         pipe0_dma_up_dst_rdy     <= split0_dma_up_dst_rdy;

         --! Direct connection instead of PIPE on PCIe 0 DOWN interface
         -- -------------------------------------------------------------------

         switch0_dma_down_data    <= pipe0_dma_down_data;
         switch0_dma_down_hdr     <= pipe0_dma_down_hdr;
         switch0_dma_down_sop     <= pipe0_dma_down_sop;
         switch0_dma_down_eop     <= pipe0_dma_down_eop;
         switch0_dma_down_src_rdy <= pipe0_dma_down_src_rdy;
         switch0_dma_down_dst_rdy <= pipe0_dma_down_dst_rdy;

         --! Direct connection instead of PIPE on PCIe 1 UP interface
         -- -------------------------------------------------------------------

         pipe1_dma_up_data        <= split1_dma_up_data;
         pipe1_dma_up_hdr         <= split1_dma_up_hdr;
         pipe1_dma_up_sop         <= split1_dma_up_sop;
         pipe1_dma_up_eop         <= split1_dma_up_eop;
         pipe1_dma_up_src_rdy     <= split1_dma_up_src_rdy;
         pipe1_dma_up_dst_rdy     <= split1_dma_up_dst_rdy;

         --! Direct connection instead of PIPE on PCIe 1 DOWN interface
         -- -------------------------------------------------------------------

         switch0_dma_down_data    <= pipe0_dma_down_data;
         switch0_dma_down_hdr     <= pipe0_dma_down_hdr;
         switch0_dma_down_sop     <= pipe0_dma_down_sop;
         switch0_dma_down_eop     <= pipe0_dma_down_eop;
         switch0_dma_down_src_rdy <= pipe0_dma_down_src_rdy;
         switch0_dma_down_dst_rdy <= pipe0_dma_down_dst_rdy;

      end generate not_pipes_gen;


      -- ----------------------------------------------------------------------
      --                Transformers on PCIe DMA 0 interface
      -- ----------------------------------------------------------------------

      --! Transform upstream DMA bus width from 512 to 256 bits
      -- ----------------------------------------------------------------------

      transformer0_dma_up : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 512,
         TX_DATA_WIDTH  => 256,
         HDR_WIDTH      => 96
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => pipe0_dma_up_data,
         RX_HDR         => pipe0_dma_up_hdr,
         RX_SOP         => pipe0_dma_up_sop,
         RX_EOP         => pipe0_dma_up_eop,
         RX_SRC_RDY     => pipe0_dma_up_src_rdy,
         RX_DST_RDY     => pipe0_dma_up_dst_rdy,

         --! TX interface
         TX_DATA        => transform0_dma_up_data,
         TX_HDR         => transform0_dma_up_hdr,
         TX_SOP         => transform0_dma_up_sop,
         TX_EOP         => transform0_dma_up_eop,
         TX_SRC_RDY     => transform0_dma_up_src_rdy,
         TX_DST_RDY     => transform0_dma_up_dst_rdy
      );

      --! Transform downstream DMA bus width from 256 to 512 bits
      -- ----------------------------------------------------------------------

      transformer0_dma_down : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 256,
         TX_DATA_WIDTH  => 512,
         HDR_WIDTH      => 32
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => transform0_dma_down_data,
         RX_HDR         => transform0_dma_down_hdr,
         RX_SOP         => transform0_dma_down_sop,
         RX_EOP         => transform0_dma_down_eop,
         RX_SRC_RDY     => transform0_dma_down_src_rdy,
         RX_DST_RDY     => transform0_dma_down_dst_rdy,

         --! TX interface
         TX_DATA        => pipe0_dma_down_data,
         TX_HDR         => pipe0_dma_down_hdr,
         TX_SOP         => pipe0_dma_down_sop,
         TX_EOP         => pipe0_dma_down_eop,
         TX_SRC_RDY     => pipe0_dma_down_src_rdy,
         TX_DST_RDY     => pipe0_dma_down_dst_rdy
      );


      -- ----------------------------------------------------------------------
      --                Transformers on PCIe DMA 1 interface
      -- ----------------------------------------------------------------------

      --! Transform upstream DMA bus width from 512 to 256 bits
      -- ----------------------------------------------------------------------

      transformer1_dma_up : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 512,
         TX_DATA_WIDTH  => 256,
         HDR_WIDTH      => 96
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => pipe1_dma_up_data,
         RX_HDR         => pipe1_dma_up_hdr,
         RX_SOP         => pipe1_dma_up_sop,
         RX_EOP         => pipe1_dma_up_eop,
         RX_SRC_RDY     => pipe1_dma_up_src_rdy,
         RX_DST_RDY     => pipe1_dma_up_dst_rdy,

         --! TX interface
         TX_DATA        => PCIE_DMA1_UP_DATA,
         TX_HDR         => PCIE_DMA1_UP_HDR,
         TX_SOP         => PCIE_DMA1_UP_SOP,
         TX_EOP         => PCIE_DMA1_UP_EOP,
         TX_SRC_RDY     => PCIE_DMA1_UP_SRC_RDY,
         TX_DST_RDY     => PCIE_DMA1_UP_DST_RDY
      );

      --! Transform downstream DMA bus width from 256 to 512 bits
      -- ----------------------------------------------------------------------

      transformer1_dma_down : entity work.DMA_TRANSFORMER
      generic map (
         RX_DATA_WIDTH  => 256,
         TX_DATA_WIDTH  => 512,
         HDR_WIDTH      => 32
      )
      port map (
         CLK            => PCIE_DMA1_CLK,
         RESET          => PCIE_DMA1_RESET,

         --! RX interface
         RX_DATA        => PCIE_DMA1_DOWN_DATA,
         RX_HDR         => PCIE_DMA1_DOWN_HDR,
         RX_SOP         => PCIE_DMA1_DOWN_SOP,
         RX_EOP         => PCIE_DMA1_DOWN_EOP,
         RX_SRC_RDY     => PCIE_DMA1_DOWN_SRC_RDY,
         RX_DST_RDY     => PCIE_DMA1_DOWN_DST_RDY,

         --! TX interface
         TX_DATA        => pipe1_dma_down_data,
         TX_HDR         => pipe1_dma_down_hdr,
         TX_SOP         => pipe1_dma_down_sop,
         TX_EOP         => pipe1_dma_down_eop,
         TX_SRC_RDY     => pipe1_dma_down_src_rdy,
         TX_DST_RDY     => pipe1_dma_down_dst_rdy
      );


      -- ----------------------------------------------------------------------
      --   Asynchronous FIFOs between PCIE_DMA0_CLK and PCIE_DMA1_CLK domains
      -- ----------------------------------------------------------------------

      --! Asynchronous FIFO on PCIe 0 UP interface
      -- ----------------------------------------------------------------------

      asfifo_pci_up_i : entity work.asfifo_dma_bus
      generic map (
         DATA_WIDTH     => 256,
         HDR_WIDTH      =>  96
      )
      port map (
         --! Write interface
         WR_CLK         => PCIE_DMA1_CLK,
         WR_RESET       => PCIE_DMA1_RESET,
         WR_DMA_DATA    => transform0_dma_up_data,
         WR_DMA_HDR     => transform0_dma_up_hdr,
         WR_DMA_SOP     => transform0_dma_up_sop,
         WR_DMA_EOP     => transform0_dma_up_eop,
         WR_DMA_SRC_RDY => transform0_dma_up_src_rdy,
         WR_DMA_DST_RDY => transform0_dma_up_dst_rdy,

         --! Read interface
         RD_CLK         => PCIE_DMA0_CLK,
         RD_RESET       => PCIE_DMA0_RESET,
         RD_DMA_DATA    => PCIE_DMA0_UP_DATA,
         RD_DMA_HDR     => PCIE_DMA0_UP_HDR,
         RD_DMA_SOP     => PCIE_DMA0_UP_SOP,
         RD_DMA_EOP     => PCIE_DMA0_UP_EOP,
         RD_DMA_SRC_RDY => PCIE_DMA0_UP_SRC_RDY,
         RD_DMA_DST_RDY => PCIE_DMA0_UP_DST_RDY
      );

      --! Asynchronous FIFO on PCIe 0 DOWN interface
      -- ----------------------------------------------------------------------

      asfifo_pci_down_i : entity work.asfifo_dma_bus
      generic map (
         DATA_WIDTH     => 256,
         HDR_WIDTH      =>  32
      )
      port map (
         --! Write interface
         WR_CLK         => PCIE_DMA0_CLK,
         WR_RESET       => PCIE_DMA0_RESET,
         WR_DMA_DATA    => PCIE_DMA0_DOWN_DATA,
         WR_DMA_HDR     => PCIE_DMA0_DOWN_HDR,
         WR_DMA_SOP     => PCIE_DMA0_DOWN_SOP,
         WR_DMA_EOP     => PCIE_DMA0_DOWN_EOP,
         WR_DMA_SRC_RDY => PCIE_DMA0_DOWN_SRC_RDY,
         WR_DMA_DST_RDY => PCIE_DMA0_DOWN_DST_RDY,

         --! Read interface
         RD_CLK         => PCIE_DMA1_CLK,
         RD_RESET       => PCIE_DMA1_RESET,
         RD_DMA_DATA    => transform0_dma_down_data,
         RD_DMA_HDR     => transform0_dma_down_hdr,
         RD_DMA_SOP     => transform0_dma_down_sop,
         RD_DMA_EOP     => transform0_dma_down_eop,
         RD_DMA_SRC_RDY => transform0_dma_down_src_rdy,
         RD_DMA_DST_RDY => transform0_dma_down_dst_rdy
      );

   end generate double_pcie_gen;

   debug_byte_counters : if DEBUG_ENABLED generate
      sampled_cnts_reg : process(PCIE_DMA1_CLK)
      begin
         if PCIE_DMA1_CLK'event and PCIE_DMA1_CLK = '1' then
            if async_dma_debug_pcie_bus_sample = '1' then
               USER_DMA_DEBUG_PCIE0_BUS_BYTES   <= debug_bytes_cnt0;
               USER_DMA_DEBUG_PCIE1_BUS_BYTES   <= debug_bytes_cnt1;
            end if;
         end if;
      end process;
      bus0_bytes_cnt : process(PCIE_DMA1_CLK)
      begin
         if PCIE_DMA1_CLK'event and PCIE_DMA1_CLK = '1' then
            if PCIE_DMA1_RESET = '1' then
               debug_bytes_cnt0 <= (others => '0');
            elsif pipe0_dma_up_sop = '1' and pipe0_dma_up_src_rdy = '1' and pipe0_dma_up_dst_rdy = '1' then
               debug_bytes_cnt0 <= debug_bytes_cnt0 + (X"000000000000" & "000" & pipe0_dma_up_hdr(10 downto 0) & "00");
            end if;
         end if;
      end process;
      bus1_bytes_cnt : process(PCIE_DMA1_CLK)
      begin
         if PCIE_DMA1_CLK'event and PCIE_DMA1_CLK = '1' then
            if PCIE_DMA1_RESET = '1' then
               debug_bytes_cnt1 <= (others => '0');
            elsif pipe1_dma_up_sop = '1' and pipe1_dma_up_src_rdy = '1' and pipe1_dma_up_dst_rdy = '1' then
               debug_bytes_cnt1 <= debug_bytes_cnt1 + (X"000000000000" & "000" & pipe1_dma_up_hdr(10 downto 0) & "00");
            end if;
         end if;
      end process;
   end generate;
   no_debug_byte_counters : if not DEBUG_ENABLED generate
      USER_DMA_DEBUG_PCIE0_BUS_BYTES <= (others => '0');
      USER_DMA_DEBUG_PCIE1_BUS_BYTES <= (others => '0');
   end generate;

end architecture full;
