-- cq_int.vhd: Completer Request interface connection - architecture
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;


-- ----------------------------------------------------------------------------
--                       Architecture declaration
-- ----------------------------------------------------------------------------

architecture cq_int_arch of cq_int is

   -- Signals for input ports of CQ interface 
   signal cq_data_in             : std_logic_vector(255 downto 0);
   signal cq_user_first_be_in    : std_logic_vector(3 downto 0);
   signal cq_user_last_be_in     : std_logic_vector(3 downto 0);
   signal cq_user_sop_in         : std_logic; 
   signal cq_user_tph_present_in : std_logic;
   signal cq_user_tph_type_in    : std_logic_vector(1 downto 0);
   signal cq_user_tph_st_tag_in  : std_logic_vector(7 downto 0);
   signal cq_valid_in            : std_logic;

   -- Signals for input ports of interface with mi_int component
   signal ready_in               : std_logic;

begin


   -- -------------------------------------------------------------------------
   --  Assignment of input ports to internal signals
   -- -------------------------------------------------------------------------
  
   -- CQ interface
   cq_valid_in <= CQ_VALID;

   cq_int_connection_p : process(cq_valid_in, ready_in, CQ_DATA,
                                 CQ_USER_FIRST_BE, CQ_USER_LAST_BE,
                                 CQ_USER_SOP, CQ_USER_TPH_PRESENT,
                                 CQ_USER_TPH_TYPE, CQ_USER_TPH_ST_TAG)
   begin
      if (cq_valid_in = '1' AND ready_in = '1') then
         cq_data_in             <= CQ_DATA;
         cq_user_first_be_in    <= CQ_USER_FIRST_BE;
         cq_user_last_be_in     <= CQ_USER_LAST_BE;
         cq_user_sop_in         <= CQ_USER_SOP;
         cq_user_tph_present_in <= CQ_USER_TPH_PRESENT;
         cq_user_tph_type_in    <= CQ_USER_TPH_TYPE;
         cq_user_tph_st_tag_in  <= CQ_USER_TPH_ST_TAG;
      else
         cq_data_in             <= (others => '0');
         cq_user_first_be_in    <= (others => '0');
         cq_user_last_be_in     <= (others => '0');
         cq_user_sop_in         <= '0';
         cq_user_tph_present_in <= '0';
         cq_user_tph_type_in    <= (others => '0');
         cq_user_tph_st_tag_in  <= (others => '0');
      end if;
   end process cq_int_connection_p;

   -- interface with mi_int component
   ready_in               <= READY;


   -- -------------------------------------------------------------------------
   --  Driving outputs of Completer Request (CQ) Interface
   -- -------------------------------------------------------------------------

   CQ_READY <= ready_in;


   -- -------------------------------------------------------------------------
   --  Driving outputs of register interface
   -- -------------------------------------------------------------------------

   -- registers (without reset)
   regs_cq_no_reset_p : process(CLK)
   begin
      if (CLK'event AND CLK = '1') then
         if (cq_valid_in = '1' AND ready_in = '1') then
            REG_CQ_DATA             <= cq_data_in;
            if (cq_user_sop_in = '1') then
               REG_CQ_USER_FIRST_BE    <= cq_user_first_be_in;
               REG_CQ_USER_LAST_BE     <= cq_user_last_be_in;
            end if;
            REG_CQ_USER_TPH_PRESENT <= cq_user_tph_present_in;
            REG_CQ_USER_TPH_TYPE    <= cq_user_tph_type_in;
            REG_CQ_USER_TPH_ST_TAG  <= cq_user_tph_st_tag_in;
         end if;
      end if;
   end process regs_cq_no_reset_p;


   -- -------------------------------------------------------------------------
   --  Driving outputs of interface with mi_int component
   -- -------------------------------------------------------------------------

   SOP          <= cq_user_sop_in;
   VALID        <= cq_valid_in;
   ADDR         <= cq_data_in(63 downto 2) & "00";
   DWORD_COUNT  <= cq_data_in(74 downto 64);
   REQ_TYPE     <= cq_data_in(78 downto 75);
   BAR_ID       <= cq_data_in(114 downto 112);
   BAR_APERTURE <= cq_data_in(120 downto 115);


   -- -------------------------------------------------------------------------
   --  Driving outputs of interface with cc_int component
   -- -------------------------------------------------------------------------

   DATA_LOWER_ADDR  <= cq_data_in(6 downto 2);
   DATA_DWORD_COUNT <= cq_data_in(74 downto 64);
   USER_FIRST_BE    <= cq_user_first_be_in;
   USER_LAST_BE     <= cq_user_last_be_in;

end architecture cq_int_arch;
