-- axi2dma_ent.vhd: AXI2DMA - entity declaration
-- Copyright (C) 2013 CESNET
-- Author(s): Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id$
--
-- TODO:
--
--


library IEEE;
use IEEE.std_logic_1164.all;


-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------

entity axi2dma is
   generic(
      -- Pipe component on AXI interfaces (RQ, RC)
      AXI_PIPE                 : boolean := true;
      -- Use output registers in AXI pipe components
      -- (applied only when AXI_PIPE = true)
      AXI_PIPE_USE_OUTREG      : boolean := true;
      -- Pipe component between DMA UP/DOWN and RQ/RC_INT components
      INTERNAL_PIPE            : boolean := true;
      -- Use output registers in internal pipe components
      -- (applied only when INTERNAL_PIPE = true)
      INTERNAL_PIPE_USE_OUTREG : boolean := true;
      -- Pipe component on DMA interfaces (UP, DOWN)
      DMA_PIPE                 : boolean := true;
      -- Use output registers in DMA pipe components
      -- (applied only when DMA_PIPE = true)
      DMA_PIPE_USE_OUTREG      : boolean := true
   );
   port(
      -- Common signals -------------------------------------------------------
      -- Clock signal
      CLK              : in std_logic;
      -- Synchronous reset
      RESET            : in  std_logic;

      -- Requester Request Interface (RQ) -------------------------------------
      -- Data bus
      RQ_DATA          : out std_logic_vector(255 downto 0);
      -- Set of signals with sideband information about trasferred transaction
      RQ_USER          : out std_logic_vector(59 downto 0);
      -- Indication of the last word of a transaction
      RQ_LAST          : out std_logic;
      -- Indication of valid data
      -- each bit determines validity of different Dword (1 Dword = 4 Bytes)
      RQ_KEEP          : out std_logic_vector(7 downto 0);
      -- PCIe core is ready to receive a transaction
      RQ_READY         : in  std_logic;
      -- User application sends valid data
      RQ_VALID         : out std_logic;
      -- Sequence number of the request in PCIe core's pipeline
      RQ_SEQ_NUM       : in  std_logic_vector(3 downto 0);
      -- Valid signal for RQ_SEQ_NUM
      RQ_SEQ_NUM_VLD   : in  std_logic;
      -- Tag assigned to non-posted request by PCIe core    
      RQ_TAG           : in  std_logic_vector(5 downto 0);
      -- Valid signal for RQ_TAG
      RQ_TAG_VLD       : in  std_logic;

      -- Requester Completion Interface (RC) ----------------------------------
      -- Data bus
      RC_DATA          : in  std_logic_vector(255 downto 0);
      -- Set of signals with sideband information about trasferred transaction
      RC_USER          : in  std_logic_vector(74 downto 0);
      -- Indication of the last word of a transaction
      RC_LAST          : in  std_logic;
      -- Indication of valid data
      -- each bit determines validity of different Dword (1 Dword = 4 Bytes)
      RC_KEEP          : in  std_logic_vector(7 downto 0);
      -- PCIe core sends valid transaction data
      RC_VALID         : in std_logic;
      -- User application is ready to receive a transaction
      RC_READY         : out std_logic;

      -- DMA UP Interface (from user application to host system) --------------
      -- Input data
      DMA_UP_DATA      : in  std_logic_vector(255 downto 0);
      -- Header for input data (valid when SOP is valid)
      DMA_UP_HDR       : in  std_logic_vector(95 downto 0);
      -- Start of transaction (valid when DMA_UP_SRC_RDY=DMA_UP_DST_RDY='1')
      DMA_UP_SOP       : in  std_logic;
      -- End of transaction (valid when DMA_UP_SRC_RDY = DMA_UP_DST_RDY = '1')
      DMA_UP_EOP       : in  std_logic;
      -- Source is ready to transmit data
      DMA_UP_SRC_RDY   : in  std_logic;
      -- Destination is ready to receive data
      DMA_UP_DST_RDY   : out std_logic;

      -- DMA DOWN Interface (from host system to user application) ------------
      -- Output data
      DMA_DOWN_DATA    : out std_logic_vector(255 downto 0);
      -- Header for output data (valid when SOP is valid)
      DMA_DOWN_HDR     : out std_logic_vector(31 downto 0);
      -- Start of transaction
      -- (valid when DMA_DOWN_SRC_RDY=DMA_DOWN_DST_RDY='1')
      DMA_DOWN_SOP     : out std_logic;
      -- End of transaction (valid when DMA_DOWN_SRC_RDY=DMA_DOWN_DST_RDY='1')
      DMA_DOWN_EOP     : out std_logic;
      -- Source is ready to transmit data
      DMA_DOWN_SRC_RDY : out std_logic;
      -- Destination is ready to receive data
      DMA_DOWN_DST_RDY : in  std_logic
   );
end entity axi2dma;

