-- tree_config_memory_adc_ent.vhd: Memory access simplifier for binary search tree over prefixes configurator 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_config_memory_adc_ent.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity tree_config_memory_adc is
  generic(
    -- number of non-leaf tree stages (levels) (capacity is 2**(TS-1) prefixes)
    TREE_STAGES    : integer := 10;
    -- width of keys inside tree nodes
    KEY_WIDTH      : integer := 128;
    -- width of data inside leaf nodes
    DATA_WIDTH     : integer := 32;
    -- registering of interfaces from memory side
    MEM_REG        : boolean := false;
    -- types of memories used for stages (from root)
    USE_STAGES     : STAGE_TYPE_ARRAY := (STAGE_TYPE_REGARRAY, STAGE_TYPE_REGARRAY, STAGE_TYPE_REGARRAY, STAGE_TYPE_DIST3MEM, STAGE_TYPE_NOTHING, 
                                          STAGE_TYPE_NOTHING,  STAGE_TYPE_DISTMEM,  STAGE_TYPE_DISTMEM,  STAGE_TYPE_BMEM,     STAGE_TYPE_BMEM);
    -- memory type used in leaf stage
    USE_STAGE_LEAF : MEMORY_TYPE := mem_bmem;
    -- bitmap of register usage inside processing pipeline (width is TS*3+1)
    USE_REGS       : std_logic_vector := "0010010010010010010010010010011"
  );
  port(
    CLK           : in  std_logic;
    RESET         : in  std_logic;
    
    -- memory configuration interfaces for non-leaf stages of the tree
    NODE_DI       : out std_logic_vector(TREE_STAGES*(KEY_WIDTH+log2(KEY_WIDTH)+2)-1 downto 0);
    NODE_DO       : in  std_logic_vector(TREE_STAGES*(KEY_WIDTH+log2(KEY_WIDTH)+2)-1 downto 0);
    NODE_DRDY     : in  std_logic_vector(TREE_STAGES-1 downto 0);
    NODE_ADDR     : out std_logic_vector(TREE_STAGES*TREE_STAGES-1 downto 0);
    NODE_WE       : out std_logic_vector(TREE_STAGES-1 downto 0); 
    NODE_EN       : out std_logic_vector(TREE_STAGES-1 downto 0);
        
    -- memory configuration interface for leaf stage of the tree
    LEAF_DI       : out std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
    LEAF_DO       : in  std_logic_vector(DATA_WIDTH+log2(KEY_WIDTH) downto 0);
    LEAF_DRDY     : in  std_logic;
    LEAF_ADDR     : out std_logic_vector(max(0,TREE_STAGES-1) downto 0);
    LEAF_WE       : out std_logic; 
    LEAF_EN       : out std_logic;
    
    -- configurator interface for tree memory access (INPUT)
    CFG_DI_KEY     : in  std_logic_vector(KEY_WIDTH-1 downto 0);
    CFG_DI_KEYLEN  : in  std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    CFG_DI_ISEND   : in  std_logic;
    CFG_DI_KEYVLD  : in  std_logic;
    CFG_DI_DATA    : in  std_logic_vector(DATA_WIDTH-1 downto 0);
    CFG_DI_DATALEN : in  std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    CFG_DI_DATAVLD : in  std_logic;
    CFG_ADDR       : in  std_logic_vector(max(0,TREE_STAGES-1) downto 0);
    CFG_WE_KEY     : in  std_logic;
    CFG_WE_DATA    : in  std_logic;
    CFG_EN         : in  std_logic;
    
    -- configurator interface for tree memory access (OUTPUT)
    CFG_DO_KEY     : out std_logic_vector(KEY_WIDTH-1 downto 0);
    CFG_DO_KEYLEN  : out std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    CFG_DO_ISEND   : out std_logic;
    CFG_DO_KEYVLD  : out std_logic;
    CFG_DO_DATA    : out std_logic_vector(DATA_WIDTH-1 downto 0);
    CFG_DO_DATALEN : out std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
    CFG_DO_DATAVLD : out std_logic;
    CFG_DRDY       : out std_logic;
    CFG_WE_KEY_SYNC: out std_logic
  );
end entity;