-- tree_3stages_distmem.vhd: Three stages (levels) of binary search tree implemented using distmems 
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_3stages_distmem.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture arch of tree_3stages_distmem is
  constant ITEMS0               : integer := 2**TREE_LEVEL;
  constant ITEMS1               : integer := ITEMS0*2;
  constant ITEMS2               : integer := ITEMS1*2;
  constant ITEM_WIDTH           : integer := 1+KEY_WIDTH+1+log2(KEY_WIDTH);
  
  signal mem0_actual_addr        : std_logic_vector(TREE_LEVEL+2 downto 0);
  signal reg_in0_addr            : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal reg_in0_key             : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_in0_vld             : std_logic := '0';
  signal reg_in0_foundable       : std_logic := '0';
  signal reg_in0_chen            : std_logic := '0';
  
  signal mem1_actual_addr        : std_logic_vector(TREE_LEVEL+2 downto 0);
  signal reg_in1_addr            : std_logic_vector(TREE_LEVEL downto 0);
  signal reg_in1_key             : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_in1_vld             : std_logic := '0';
  signal reg_in1_foundable       : std_logic := '0';
  signal reg_in1_chen            : std_logic := '0';
  
  signal mem2_actual_addr        : std_logic_vector(TREE_LEVEL+2 downto 0);
  signal reg_in2_addr            : std_logic_vector(TREE_LEVEL+1 downto 0);
  signal reg_in2_key             : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_in2_vld             : std_logic := '0';
  signal reg_in2_foundable       : std_logic := '0';
  signal reg_in2_chen            : std_logic := '0';
  
  signal mem0_addr               : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal mem0_key                : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal mem0_vld                : std_logic := '0';
  signal mem0_foundable          : std_logic := '0';
  signal mem0_chen               : std_logic := '0';
  signal mem0_item               : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal mem1_addr               : std_logic_vector(TREE_LEVEL downto 0);
  signal mem1_key                : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal mem1_vld                : std_logic := '0';
  signal mem1_foundable          : std_logic := '0';
  signal mem1_chen               : std_logic := '0';
  signal mem1_item               : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal mem2_addr               : std_logic_vector(TREE_LEVEL+1 downto 0);
  signal mem2_key                : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal mem2_vld                : std_logic := '0';
  signal mem2_foundable          : std_logic := '0';
  signal mem2_chen               : std_logic := '0';
  signal mem2_item               : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal reg_mem0_addr           : std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
  signal reg_mem0_key            : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_mem0_vld            : std_logic := '0';
  signal reg_mem0_foundable      : std_logic := '0';
  signal reg_mem0_chen           : std_logic := '0';
  signal reg_mem0_item           : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal reg_mem1_addr           : std_logic_vector(TREE_LEVEL downto 0);
  signal reg_mem1_key            : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_mem1_vld            : std_logic := '0';
  signal reg_mem1_foundable      : std_logic := '0';
  signal reg_mem1_chen           : std_logic := '0';
  signal reg_mem1_item           : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal reg_mem2_addr           : std_logic_vector(TREE_LEVEL+1 downto 0);
  signal reg_mem2_key            : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_mem2_vld            : std_logic := '0';
  signal reg_mem2_foundable      : std_logic := '0';
  signal reg_mem2_chen           : std_logic := '0';
  signal reg_mem2_item           : std_logic_vector(ITEM_WIDTH-1 downto 0);
  
  signal cmp0_item_vld       : std_logic;
  signal cmp0_item_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cmp0_item_isend     : std_logic;
  signal cmp0_item_length    : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal cmp0                : std_logic;
  signal cmp0_res            : std_logic;

  signal cmp1_item_vld       : std_logic;
  signal cmp1_item_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cmp1_item_isend     : std_logic;
  signal cmp1_item_length    : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal cmp1                : std_logic;
  signal cmp1_res            : std_logic;
  
  signal cmp2_item_vld       : std_logic;
  signal cmp2_item_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal cmp2_item_isend     : std_logic;
  signal cmp2_item_length    : std_logic_vector(log2(KEY_WIDTH)-1 downto 0);
  signal cmp2                : std_logic;
  signal cmp2_res            : std_logic;  
  
  signal reg_out0_addr       : std_logic_vector(TREE_LEVEL downto 0);
  signal reg_out0_key        : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_out0_vld        : std_logic := '0';
  signal reg_out0_foundable  : std_logic := '0';
  signal reg_out0_chen       : std_logic := '0';
  
  signal reg_out1_addr       : std_logic_vector(TREE_LEVEL+1 downto 0);
  signal reg_out1_key        : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_out1_vld        : std_logic := '0';
  signal reg_out1_foundable  : std_logic := '0';
  signal reg_out1_chen       : std_logic := '0';
  
  signal reg_out2_addr       : std_logic_vector(TREE_LEVEL+2 downto 0);
  signal reg_out2_key        : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal reg_out2_vld        : std_logic := '0';
  signal reg_out2_foundable  : std_logic := '0';
  signal reg_out2_chen       : std_logic := '0';
  
  signal connect01_addr      : std_logic_vector(TREE_LEVEL downto 0);
  signal connect01_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal connect01_vld       : std_logic := '0';
  signal connect01_foundable : std_logic := '0';
  signal connect01_chen      : std_logic := '0';
  
  signal connect12_addr      : std_logic_vector(TREE_LEVEL+1 downto 0);
  signal connect12_key       : std_logic_vector(KEY_WIDTH-1 downto 0);
  signal connect12_vld       : std_logic := '0';
  signal connect12_foundable : std_logic := '0';
  signal connect12_chen      : std_logic := '0';
  
  signal cfg_addr        : std_logic_vector(TREE_LEVEL+2 downto 0);
  signal cfg_di          : std_logic_vector(ITEM_WIDTH-1 downto 0);
  signal cfg_do          : std_logic_vector(ITEM_WIDTH-1 downto 0);
  signal cfg_do_reg      : std_logic_vector(ITEM_WIDTH-1 downto 0);
  signal cfg_we          : std_logic;
begin    
-- memory stage ----------------------------------------------------------------
  -- computing of write and read enable for memory
  cfg_di   <= CFG0_DI when CFG0_EN='1' else
              CFG1_DI when CFG1_EN='1' else
              CFG2_DI;
  cfg_addr <= "001"&CFG0_ADDR when CFG0_EN='1' else
              "01" &CFG1_ADDR when CFG1_EN='1' else
              "1"  &CFG2_ADDR;
  cfg_we   <= (CFG0_EN and CFG0_WE) or
              (CFG1_EN and CFG1_WE) or
              (CFG2_EN and CFG2_WE);
  
  -- use DISTMEMs
  mem : entity work.QP_DISTMEM
    generic map (
      DATA_WIDTH     => ITEM_WIDTH,
      ITEMS          => ITEMS2*2,
      DISTMEM_TYPE   => GET_DISTMEM_TYPE(ITEMS2*2),
      DEBUG          => false
    ) port map (
      RESET          => RESET,
      DI             => cfg_di,
      WE             => cfg_we,
      WCLK           => CLK,
      ADDRA          => cfg_addr,
      DOA            => cfg_do, 
      ADDRB          => mem0_actual_addr,
      DOB            => mem0_item,
      ADDRC          => mem1_actual_addr,
      DOC            => mem1_item,
      ADDRD          => mem2_actual_addr,
      DOD            => mem2_item
    );
  -- distmem output data connection
  do_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      cfg_do_reg <= cfg_do;
    end if;
  end process;
  CFG0_DO <= cfg_do_reg;
  CFG1_DO <= cfg_do_reg;
  CFG2_DO <= cfg_do_reg;
  -- addresses augmentation
  mem0_actual_addr <= "001"&reg_in0_addr;
  mem1_actual_addr <= "01" &reg_in1_addr;
  mem2_actual_addr <= "1"  &reg_in2_addr;
  -- synchronization of distmem inputs and outputs
  mem0_vld  <= reg_in0_vld;
  mem0_chen <= reg_in0_chen;
  mem0_foundable <= reg_in0_foundable;
  mem0_addr <= reg_in0_addr;
  mem0_key  <= reg_in0_key;
  mem1_vld  <= reg_in1_vld;
  mem1_foundable <= reg_in1_foundable;
  mem1_chen <= reg_in1_chen;
  mem1_addr <= reg_in1_addr;
  mem1_key  <= reg_in1_key;
  mem2_vld  <= reg_in2_vld;
  mem2_foundable <= reg_in2_foundable;
  mem2_chen <= reg_in2_chen;
  mem2_addr <= reg_in2_addr;
  mem2_key  <= reg_in2_key;
  drdy_reg : process(CLK)
  begin
    if CLK'event and CLK='1' then
      if RESET='1' then
        CFG0_DRDY <= '0';
        CFG1_DRDY <= '0';
        CFG2_DRDY <= '0';
      else
        CFG0_DRDY <= CFG0_EN and not CFG0_WE;
        CFG1_DRDY <= CFG1_EN and not CFG1_WE;
        CFG2_DRDY <= CFG2_EN and not CFG2_WE;
      end if;
    end if;
  end process;   
  
  

-- comparator stage ------------------------------------------------------------  
  -- memory item decoder
  cmp0_item_vld    <= reg_mem0_item(0);       
  cmp0_item_isend  <= reg_mem0_item(1);  
  cmp0_item_length <= reg_mem0_item(log2(KEY_WIDTH)+1 downto 2);
  cmp0_item_key    <= reg_mem0_item(ITEM_WIDTH-1 downto ITEM_WIDTH-KEY_WIDTH);
  cmp1_item_vld    <= reg_mem1_item(0);       
  cmp1_item_isend  <= reg_mem1_item(1);  
  cmp1_item_length <= reg_mem1_item(log2(KEY_WIDTH)+1 downto 2);
  cmp1_item_key    <= reg_mem1_item(ITEM_WIDTH-1 downto ITEM_WIDTH-KEY_WIDTH);   
  cmp2_item_vld    <= reg_mem2_item(0);       
  cmp2_item_isend  <= reg_mem2_item(1);  
  cmp2_item_length <= reg_mem2_item(log2(KEY_WIDTH)+1 downto 2);
  cmp2_item_key    <= reg_mem2_item(ITEM_WIDTH-1 downto ITEM_WIDTH-KEY_WIDTH);                   
  
  -- comparator of keys (NOTE: different behaviour on key equality for start and end of range)  
  cmp0 <= '1' when (reg_mem0_key&'0')>=(cmp0_item_key&cmp0_item_isend) else '0';
  cmp1 <= '1' when (reg_mem1_key&'0')>=(cmp1_item_key&cmp1_item_isend) else '0';
  cmp2 <= '1' when (reg_mem2_key&'0')>=(cmp2_item_key&cmp2_item_isend) else '0';
  
  -- comparision result validation
  cmp0_res <= cmp0 and cmp0_item_vld;
  cmp1_res <= cmp1 and cmp1_item_vld;
  cmp2_res <= cmp2 and cmp2_item_vld;
  
  -- results computing
  not_level0_gen : if TREE_LEVEL>0 generate
    reg_out0_addr(TREE_LEVEL downto 1) <= reg_mem0_addr;
  end generate;
  reg_out0_addr(0) <= cmp0_res;
  reg_out0_key     <= reg_mem0_key;
  reg_out0_vld     <= reg_mem0_vld;
  reg_out0_foundable <= reg_mem0_foundable;
  reg_out0_chen    <= reg_mem0_chen;
  reg_out1_addr    <= reg_mem1_addr&cmp1_res;
  reg_out1_key     <= reg_mem1_key;
  reg_out1_vld     <= reg_mem1_vld;
  reg_out1_foundable <= reg_mem1_foundable;
  reg_out1_chen    <= reg_mem1_chen;
  reg_out2_addr    <= reg_mem2_addr&cmp2_res;
  reg_out2_key     <= reg_mem2_key;
  reg_out2_vld     <= reg_mem2_vld;
  reg_out2_foundable <= reg_mem2_foundable;
  reg_out2_chen    <= reg_mem2_chen;

  
  
-- registers -------------------------------------------------------------------
  -- input interface registers
  input_reg_gen : if INPUT0_REG generate
    input_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_in0_addr <= IN_ADDR;
        reg_in0_key  <= IN_KEY;
        reg_in0_vld  <= IN_VLD;
        reg_in0_foundable <= IN_FOUNDABLE;
        reg_in0_chen <= IN_CHANGE_EN;  
      end if;
    end process;
  end generate;  
  input_noreg_gen : if not INPUT0_REG generate
    reg_in0_addr <= IN_ADDR;
    reg_in0_key  <= IN_KEY;
    reg_in0_vld  <= IN_VLD;
    reg_in0_foundable <= IN_FOUNDABLE; 
    reg_in0_chen <= IN_CHANGE_EN;   
  end generate;
  input1_reg_gen : if INPUT1_REG generate
    input_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_in1_addr <= connect01_addr;
        reg_in1_key  <= connect01_key;
        reg_in1_vld  <= connect01_vld;
        reg_in1_foundable <= connect01_foundable;
        reg_in1_chen <= connect01_chen;  
      end if;
    end process;
  end generate;  
  input1_noreg_gen : if not INPUT1_REG generate
    reg_in1_addr <= connect01_addr;
    reg_in1_key  <= connect01_key;
    reg_in1_vld  <= connect01_vld;
    reg_in1_foundable <= connect01_foundable;
    reg_in1_chen <= connect01_chen;
  end generate;
  input2_reg_gen : if INPUT2_REG generate
    input_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_in2_addr <= connect12_addr;
        reg_in2_key  <= connect12_key;
        reg_in2_vld  <= connect12_vld;
        reg_in2_foundable <= connect12_foundable;
        reg_in2_chen <= connect12_chen;
      end if;
    end process;
  end generate;  
  input2_noreg_gen : if not INPUT2_REG generate
    reg_in2_addr <= connect12_addr;
    reg_in2_key  <= connect12_key;
    reg_in2_vld  <= connect12_vld;
    reg_in2_foundable <= connect12_foundable;
    reg_in2_chen <= connect12_chen;
  end generate;

  -- middle registers
  middle0_reg_gen : if MIDDLE0_REG generate
    middle_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_mem0_addr <= mem0_addr;
        reg_mem0_key  <= mem0_key;
        reg_mem0_vld  <= mem0_vld;
        reg_mem0_chen <= mem0_chen;
        reg_mem0_foundable <= mem0_foundable;
        reg_mem0_item <= mem0_item;
      end if;
    end process;
  end generate;
  middle0_noreg_gen : if not MIDDLE0_REG generate
    reg_mem0_addr <= mem0_addr;
    reg_mem0_key  <= mem0_key;
    reg_mem0_vld  <= mem0_vld;
    reg_mem0_foundable <= mem0_foundable;
    reg_mem0_chen <= mem0_chen;
    reg_mem0_item <= mem0_item;  
  end generate;
  middle1_reg_gen : if MIDDLE1_REG generate
    middle_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_mem1_addr <= mem1_addr;
        reg_mem1_key  <= mem1_key;
        reg_mem1_vld  <= mem1_vld;
        reg_mem1_foundable <= mem1_foundable;
        reg_mem1_chen <= mem1_chen;
        reg_mem1_item <= mem1_item;
      end if;
    end process;
  end generate;
  middle1_noreg_gen : if not MIDDLE1_REG generate
    reg_mem1_addr <= mem1_addr;
    reg_mem1_key  <= mem1_key;
    reg_mem1_vld  <= mem1_vld;
    reg_mem1_foundable <= mem1_foundable;
    reg_mem1_chen <= mem1_chen;
    reg_mem1_item <= mem1_item;  
  end generate;
  middle2_reg_gen : if MIDDLE2_REG generate
    middle_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        reg_mem2_addr <= mem2_addr;
        reg_mem2_key  <= mem2_key;
        reg_mem2_vld  <= mem2_vld;
        reg_mem2_foundable <= mem2_foundable;
        reg_mem2_chen <= mem2_chen;
        reg_mem2_item <= mem2_item;
      end if;
    end process;
  end generate;
  middle2_noreg_gen : if not MIDDLE2_REG generate
    reg_mem2_addr <= mem2_addr;
    reg_mem2_key  <= mem2_key;
    reg_mem2_vld  <= mem2_vld;
    reg_mem2_foundable <= mem2_foundable;
    reg_mem2_chen <= mem2_chen;
    reg_mem2_item <= mem2_item;  
  end generate;
  
  -- output interface registers
  output0_reg_gen : if OUTPUT0_REG generate
    output_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        connect01_addr <= reg_out0_addr;
        connect01_key  <= reg_out0_key;
        connect01_vld  <= reg_out0_vld;
        connect01_foundable <= reg_out0_foundable;
        connect01_chen <= reg_out0_chen;  
      end if;
    end process;
  end generate;
  output0_noreg_gen : if not OUTPUT0_REG generate
    connect01_addr <= reg_out0_addr;
    connect01_key  <= reg_out0_key;
    connect01_vld  <= reg_out0_vld;
    connect01_foundable <= reg_out0_foundable;
    connect01_chen <= reg_out0_chen;  
  end generate;
  output1_reg_gen : if OUTPUT1_REG generate
    output_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        connect12_addr <= reg_out1_addr;
        connect12_key  <= reg_out1_key;
        connect12_vld  <= reg_out1_vld;
        connect12_foundable <= reg_out1_foundable;
        connect12_chen <= reg_out1_chen;  
      end if;
    end process;
  end generate;
  output1_noreg_gen : if not OUTPUT1_REG generate
    connect12_addr <= reg_out1_addr;
    connect12_key  <= reg_out1_key;
    connect12_vld  <= reg_out1_vld;
    connect12_foundable <= reg_out1_foundable;
    connect12_chen <= reg_out1_chen;  
  end generate;
  output2_reg_gen : if OUTPUT2_REG generate
    output_reg : process (CLK)
    begin
      if CLK'event and CLK='1' then
        OUT_ADDR      <= reg_out2_addr;
        OUT_KEY       <= reg_out2_key;
        OUT_VLD       <= reg_out2_vld;
        OUT_FOUNDABLE <= reg_out2_foundable;
        OUT_CHANGE_EN <= reg_out2_chen;  
      end if;
    end process;
  end generate;
  output2_noreg_gen : if not OUTPUT2_REG generate
    OUT_ADDR      <= reg_out2_addr;
    OUT_KEY       <= reg_out2_key;
    OUT_VLD       <= reg_out2_vld;
    OUT_FOUNDABLE <= reg_out2_foundable;
    OUT_CHANGE_EN <= reg_out2_chen;   
  end generate;
  
end architecture;