-- filter_core.vhd : Filter application core for 100G NetCOPE platform
-- Copyright (c) 2014 Brno University of Technology 
-- Author: Lukas Kekely <xkekel00@stud.fit.vutbr.cz>
--         Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: application_core.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--


library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;

use work.math_pack.all;


-- ----------------------------------------------------------------------------
--                             Entity declaration
-- ----------------------------------------------------------------------------

entity filter_core is
generic (
   --! \brief Data width of input FL interface for SZE headers
   --! \details In bits, must be power of 2 and greater than 8.
   HDR_WIDTH         : integer := 128;
   --! \brief Number of items in the packet FIFO
   PACKET_FIFO_ITEMS : integer := 1024;
   --! \brief Number of items in the extracted header fields FIFO
   HEADER_FIFO_ITEMS : integer := 1024;
   --! \brief Width of classification rules' result data
   RULES_DATA_WIDTH  : integer := 16
);
port (
   --! \name Clock and reset signals
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   CLK               : in std_logic;
   --! \brief Reset signal
   RESET             : in std_logic;

   --! \name FrameLink Unaligned interface from network module (data payload)
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   RX_DATA           : in  std_logic_vector(511 downto 0);
   --! \brief Position of the start of the payload
   --! \details Valid only if RX_SOP is set to '1'.
   RX_SOP_POS        : in  std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the payload
   --! \details Valid only if RX_EOP is set to '1'.
   RX_EOP_POS        : in  std_logic_vector(  5 downto 0);
   --! \brief Start of the payload
   RX_SOP            : in  std_logic;
   --! \brief End of the payload
   RX_EOP            : in  std_logic;
   --! \brief Source is ready
   RX_SRC_RDY        : in  std_logic;
   --! \brief Destination is ready
   RX_DST_RDY        : out std_logic;

   --! \name FrameLink interface from network module (data header)
   -- -------------------------------------------------------------------------
   --! \brief Header data
   RX_HDATA          : in  std_logic_vector(HDR_WIDTH-1 downto 0);
   --! \brief Start of the header, active in '0'
   RX_HSOP_N         : in  std_logic;
   --! \brief End of the header, active in '0'
   RX_HEOP_N         : in  std_logic;
   --! \brief Position of the end of the header
   --! \details Valid only if RX_HEOP_N is set to '0'.
   RX_HREM           : in  std_logic_vector(log2(HDR_WIDTH/8)-1 downto 0);
   --! \brief Source is ready, active in '0'
   RX_HSRC_RDY_N     : in  std_logic;
   --! \brief Destination is ready, active in '0'
   RX_HDST_RDY_N     : out std_logic;

   --! \name FrameLink Unaligned interface to DMA module (data payload)
   -- -------------------------------------------------------------------------
   --! \brief Payload data
   TX_DATA           : out std_logic_vector(511 downto 0);
   --! \brief Position of the start of the payload
   --! \details Valid only if TX_SOP is set to '1'.
   TX_SOP_POS        : out std_logic_vector(  2 downto 0);
   --! \brief Position of the end of the payload
   --! \details Valid only if TX_EOP is set to '1'.
   TX_EOP_POS        : out std_logic_vector(  5 downto 0);
   --! \brief Start of the payload
   TX_SOP            : out std_logic;
   --! \brief End of the payload
   TX_EOP            : out std_logic;
   --! \brief Source is ready
   TX_SRC_RDY        : out std_logic;
   --! \brief Destination is ready
   TX_DST_RDY        : in  std_logic;

   --! \name MI32 interface
   -- -------------------------------------------------------------------------
   --! \brief Clock signal
   MI_CLK            : in  std_logic;
   --! \brief Reset signal
   MI_RESET          : in  std_logic;
   --! \brief Input Data
   MI_DWR            : in  std_logic_vector(31 downto 0);
   --! \brief Address
   MI_ADDR           : in  std_logic_vector(31 downto 0);
   --! \brief Read Request
   MI_RD             : in  std_logic;
   --! \brief Write Request
   MI_WR             : in  std_logic;
   --! \brief Byte Enable
   MI_BE             : in  std_logic_vector( 3 downto 0);
   --! \brief Output Data
   MI_DRD            : out std_logic_vector(31 downto 0);
   --! \brief Address Ready
   MI_ARDY           : out std_logic;
   --! \brief Data Ready
   MI_DRDY           : out std_logic
);
end entity filter_core;


-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of filter_core is

   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! FLU2FL converter output interface signals
   signal flu2fl_tx_data               : std_logic_vector(511 downto 0);
   signal flu2fl_tx_sof_n              : std_logic; 
   signal flu2fl_tx_sop_n              : std_logic;
   signal flu2fl_tx_eof_n              : std_logic;
   signal flu2fl_tx_eop_n              : std_logic;
   signal flu2fl_tx_drem               : std_logic_vector(5 downto 0); 
   signal flu2fl_tx_src_rdy_n          : std_logic;
   signal flu2fl_tx_dst_rdy_n          : std_logic;

   --! Header inserter output interface signals
   signal hins_tx_data                 : std_logic_vector(511 downto 0);
   signal hins_tx_sof_n                : std_logic; 
   signal hins_tx_sop_n                : std_logic;
   signal hins_tx_eof_n                : std_logic;
   signal hins_tx_eop_n                : std_logic;
   signal hins_tx_drem                 : std_logic_vector(5 downto 0); 
   signal hins_tx_src_rdy_n            : std_logic;
   signal hins_tx_dst_rdy_n            : std_logic;

   --! HFE packet output interface signals
   signal hfe_packet_data              : std_logic_vector(511 downto 0);
   signal hfe_packet_sof_n             : std_logic; 
   signal hfe_packet_sop_n             : std_logic;
   signal hfe_packet_eof_n             : std_logic;
   signal hfe_packet_eop_n             : std_logic;
   signal hfe_packet_drem              : std_logic_vector(5 downto 0); 
   signal hfe_packet_src_rdy_n         : std_logic;
   signal hfe_packet_dst_rdy_n         : std_logic;

   --! HFE extracted header fields interface (Internet layer)
   signal hfe_header_srcip_v4          : std_logic_vector(31 downto 0);
   signal hfe_header_srcip_v4_vld      : std_logic;
   signal hfe_header_dstip_v4          : std_logic_vector(31 downto 0);
   signal hfe_header_dstip_v4_vld      : std_logic;
   signal hfe_header_srcip_v6          : std_logic_vector(127 downto 0);
   signal hfe_header_srcip_v6_vld      : std_logic;
   signal hfe_header_dstip_v6          : std_logic_vector(127 downto 0);
   signal hfe_header_dstip_v6_vld      : std_logic;
   signal hfe_header_proto             : std_logic_vector(7 downto 0);
   signal hfe_header_proto_vld         : std_logic;

   --! HFE extracted header fields interface (Transport layer)
   signal hfe_header_srctcp            : std_logic_vector(15 downto 0);
   signal hfe_header_srctcp_vld        : std_logic;
   signal hfe_header_dsttcp            : std_logic_vector(15 downto 0);
   signal hfe_header_dsttcp_vld        : std_logic;
   signal hfe_header_srcudp            : std_logic_vector(15 downto 0);
   signal hfe_header_srcudp_vld        : std_logic;
   signal hfe_header_dstudp            : std_logic_vector(15 downto 0);
   signal hfe_header_dstudp_vld        : std_logic;

   --! HFE parsing status interface
   signal hfe_header_parsing_done      : std_logic;
   signal hfe_header_parsing_done_reg  : std_logic;
   signal hfe_header_parsing_done_rise : std_logic;

   --! HFE extracted header fields merging
   signal hfe_header_srcip             : std_logic_vector(127 downto 0);
   signal hfe_header_dstip             : std_logic_vector(127 downto 0);
   signal hfe_header_srcport           : std_logic_vector(15 downto 0);
   signal hfe_header_dstport           : std_logic_vector(15 downto 0);
   signal hfe_header_protocol          : std_logic_vector(7 downto 0);

   --! Extracted header fields FIFO interface signals
   signal header_fifo_data_in          : std_logic_vector(295 downto 0);
   signal header_fifo_data_out         : std_logic_vector(295 downto 0);
   signal header_fifo_rd               : std_logic;
   signal header_fifo_empty            : std_logic;
   
   --! Extracted header fields on header FIFO's output
   signal header_fifo_srcip            : std_logic_vector(127 downto 0);
   signal header_fifo_dstip            : std_logic_vector(127 downto 0);
   signal header_fifo_srcport          : std_logic_vector(15 downto 0);
   signal header_fifo_dstport          : std_logic_vector(15 downto 0);
   signal header_fifo_protocol         : std_logic_vector(7 downto 0);

   --! Filtering engine input interface src/dst ready signals
   signal filter_in_ready              : std_logic;
   signal filter_in_next               : std_logic;
   
   --! Cuckoo filter output interface signals
   signal filter_result                : std_logic_vector(2*RULES_DATA_WIDTH-1 downto 0);
   signal filter_found                 : std_logic_vector(1 downto 0);
   signal filter_ready                 : std_logic;
   signal filter_next                  : std_logic;
   signal filter_deny                  : std_logic;

   --! FIFO for packet output of HFE
   signal packet_fifo_data             : std_logic_vector(511 downto 0);
   signal packet_fifo_sof_n            : std_logic; 
   signal packet_fifo_sop_n            : std_logic;
   signal packet_fifo_eof_n            : std_logic;
   signal packet_fifo_eop_n            : std_logic;
   signal packet_fifo_drem             : std_logic_vector(5 downto 0); 
   signal packet_fifo_src_rdy_n        : std_logic;
   signal packet_fifo_dst_rdy_n        : std_logic;
 
   --! Dispatcher output interface signals
   signal dispatcher_data              : std_logic_vector(511 downto 0);
   signal dispatcher_sof_n             : std_logic; 
   signal dispatcher_sop_n             : std_logic;
   signal dispatcher_eof_n             : std_logic;
   signal dispatcher_eop_n             : std_logic;
   signal dispatcher_drem              : std_logic_vector(5 downto 0); 
   signal dispatcher_src_rdy_n         : std_logic;
   signal dispatcher_dst_rdy_n         : std_logic;

   --! MI32 interface for cuckoo filter (clocked at CLK)
   signal mi_async_dwr                 : std_logic_vector(31 downto 0);
   signal mi_async_addr                : std_logic_vector(31 downto 0);
   signal mi_async_be                  : std_logic_vector(3 downto 0);
   signal mi_async_rd                  : std_logic;
   signal mi_async_wr                  : std_logic;
   signal mi_async_drd                 : std_logic_vector(31 downto 0);
   signal mi_async_ardy                : std_logic;
   signal mi_async_drdy                : std_logic;


-- ----------------------------------------------------------------------------
--                             Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                           FLU2FL converter
   -- -------------------------------------------------------------------------

   flu2fl_i : entity work.flu2fl
   generic map (
      DATA_WIDTH      => 512,
      SOP_POS_WIDTH   => 3,
      IN_PIPE_EN      => true,
      IN_PIPE_OUTREG  => true,
      OUT_PIPE_EN     => true,
      OUT_PIPE_OUTREG => true
   )
   port map (
      --! Common interface 
      CLK             => CLK,
      RESET           => RESET,
      
      --! Input interface (FLU)
      RX_DATA         => RX_DATA,
      RX_SOP_POS      => RX_SOP_POS,
      RX_EOP_POS      => RX_EOP_POS,
      RX_SOP          => RX_SOP,
      RX_EOP          => RX_EOP,
      RX_SRC_RDY      => RX_SRC_RDY,
      RX_DST_RDY      => RX_DST_RDY,

      --! Output interface (FL)
      TX_DATA         => flu2fl_tx_data,
      TX_SOF_N        => flu2fl_tx_sof_n,
      TX_SOP_N        => flu2fl_tx_sop_n,
      TX_EOF_N        => flu2fl_tx_eof_n,
      TX_EOP_N        => flu2fl_tx_eop_n,
      TX_DREM         => flu2fl_tx_drem,
      TX_SRC_RDY_N    => flu2fl_tx_src_rdy_n,
      TX_DST_RDY_N    => flu2fl_tx_dst_rdy_n
   );


   -- -------------------------------------------------------------------------
   --                Header insertion into FrameLink stream
   -- -------------------------------------------------------------------------

   fl_hins_i : entity work.fl_hins
   generic map (
      DATA_WIDTH           => 512,
      HDR_WIDTH            => HDR_WIDTH
   )
   port map (
      --! Clock and reset signals
      CLK                  => CLK,
      RESET                => RESET,

      --! Input FrameLink payload interface
      RX_PAYLOAD_DATA      => flu2fl_tx_data,
      RX_PAYLOAD_SOF_N     => flu2fl_tx_sof_n,
      RX_PAYLOAD_SOP_N     => flu2fl_tx_sop_n,
      RX_PAYLOAD_EOF_N     => flu2fl_tx_eof_n,
      RX_PAYLOAD_EOP_N     => flu2fl_tx_eop_n,
      RX_PAYLOAD_REM       => flu2fl_tx_drem,
      RX_PAYLOAD_SRC_RDY_N => flu2fl_tx_src_rdy_n,
      RX_PAYLOAD_DST_RDY_N => flu2fl_tx_dst_rdy_n,

      --! Input FrameLink header interface
      RX_HDR_DATA          => RX_HDATA,
      RX_HDR_SOF_N         => RX_HSOP_N,
      RX_HDR_SOP_N         => RX_HSOP_N,
      RX_HDR_EOF_N         => RX_HEOP_N,
      RX_HDR_EOP_N         => RX_HEOP_N,
      RX_HDR_REM           => RX_HREM,
      RX_HDR_SRC_RDY_N     => RX_HSRC_RDY_N,
      RX_HDR_DST_RDY_N     => RX_HDST_RDY_N,

      --! Output FrameLink interface
      TX_DATA              => hins_tx_data,
      TX_SOF_N             => hins_tx_sof_n,
      TX_SOP_N             => hins_tx_sop_n,
      TX_EOF_N             => hins_tx_eof_n,
      TX_EOP_N             => hins_tx_eop_n,
      TX_REM               => hins_tx_drem,
      TX_SRC_RDY_N         => hins_tx_src_rdy_n,
      TX_DST_RDY_N         => hins_tx_dst_rdy_n
   );


   -- -------------------------------------------------------------------------
   --                        Header field extraction
   -- -------------------------------------------------------------------------

   --! HFE module instantiation
   -- -------------------------------------------------------------------------

   hfe_i : entity work.hfe_m_top
   generic map (
      DATA_WIDTH      => 512,
      IN_PIPE_EN      => true,
      IN_PIPE_OUTREG  => true,
      MID_PIPE_EN     => true,
      MID_PIPE_OUTREG => true,
      OUT_PIPE_EN     => true,
      OUT_PIPE_OUTREG => true,
      EXTRACT_LATENCY => 1
   )
   port map (
      --! Common signals
      CLK             => CLK,
      RESET           => RESET,

      --! FrameLink input interface
      RX_DATA         => hins_tx_data,
      RX_SOF_N        => hins_tx_sof_n,
      RX_SOP_N        => hins_tx_sop_n,
      RX_EOF_N        => hins_tx_eof_n,
      RX_EOP_N        => hins_tx_eop_n,
      RX_DREM         => hins_tx_drem,
      RX_SRC_RDY_N    => hins_tx_src_rdy_n,
      RX_DST_RDY_N    => hins_tx_dst_rdy_n,

      --! FrameLink output interface
      TX_DATA         => hfe_packet_data,
      TX_SOF_N        => hfe_packet_sof_n,
      TX_SOP_N        => hfe_packet_sop_n,
      TX_EOF_N        => hfe_packet_eof_n,
      TX_EOP_N        => hfe_packet_eop_n,
      TX_DREM         => hfe_packet_drem,
      TX_SRC_RDY_N    => hfe_packet_src_rdy_n,
      TX_DST_RDY_N    => hfe_packet_dst_rdy_n,

      --! Extracted fields
      -------------------------------------------------------------------------

      --! Internet layer
      SRCIPV4         => hfe_header_srcip_v4,
      SRCIPV4_VLD     => hfe_header_srcip_v4_vld,
      DSTIPV4         => hfe_header_dstip_v4,
      DSTIPV4_VLD     => hfe_header_dstip_v4_vld,
      SRCIPV6         => hfe_header_srcip_v6,
      SRCIPV6_VLD     => hfe_header_srcip_v6_vld,
      DSTIPV6         => hfe_header_dstip_v6,
      DSTIPV6_VLD     => hfe_header_dstip_v6_vld,
      PROTOCOL        => hfe_header_proto,
      PROTOCOL_VLD    => hfe_header_proto_vld,

      --! Transport layer
      SRCTCP          => hfe_header_srctcp,
      SRCTCP_VLD      => hfe_header_srctcp_vld,
      DSTTCP          => hfe_header_dsttcp,
      DSTTCP_VLD      => hfe_header_dsttcp_vld,
      SRCUDP          => hfe_header_srcudp,
      SRCUDP_VLD      => hfe_header_srcudp_vld,
      DSTUDP          => hfe_header_dstudp,
      DSTUDP_VLD      => hfe_header_dstudp_vld,

      --! Don't expect any new *_VLD to appear after this
      PARSING_DONE    => hfe_header_parsing_done
   );

   --! Capturing rising edge on hfe_header_parsing_done
   -- -------------------------------------------------------------------------

   --! register for hfe_header_parsing_done
   hfe_header_parsing_done_reg_p : process(CLK)
   begin
      if rising_edge(CLK) then
         hfe_header_parsing_done_reg <= hfe_header_parsing_done;
      end if;
   end process hfe_header_parsing_done_reg_p;

   --! rising edge campture
   hfe_header_parsing_done_rise <= '1' when 
                                       hfe_header_parsing_done = '1' AND
                                       hfe_header_parsing_done_reg = '0'
                                   else
                                   '0';

   --! Merging extracted header fields together
   -- -------------------------------------------------------------------------

   --! Source IP
   hfe_header_srcip    <= hfe_header_srcip_v4 & (95 downto 0 => '0') when
                             hfe_header_srcip_v4_vld = '1'
                          else
                          hfe_header_srcip_v6 when
                             hfe_header_srcip_v6_vld = '1'
                          else
                          (others => '0');

   --! Destination IP
   hfe_header_dstip    <= hfe_header_dstip_v4 & (95 downto 0 => '0') when
                             hfe_header_dstip_v4_vld = '1'
                          else
                          hfe_header_dstip_v6 when
                             hfe_header_dstip_v6_vld = '1'
                          else
                          (others => '0');

   --! Source port
   hfe_header_srcport  <= hfe_header_srctcp when hfe_header_srctcp_vld = '1'
                          else
                          hfe_header_srcudp when hfe_header_srcudp_vld = '1'
                          else
                          (others => '0');

   --! Destination port
   hfe_header_dstport  <= hfe_header_dsttcp when hfe_header_dsttcp_vld = '1'
                          else
                          hfe_header_dstudp when hfe_header_dstudp_vld = '1'
                          else
                          (others => '0');

   --! Protocol
   hfe_header_protocol <= hfe_header_proto when hfe_header_proto_vld = '1'
                          else
                          (others => '0');


   -- -------------------------------------------------------------------------
   --                  FIFO for extracted header fields
   --   (ensures their alignment into a single word before classification)
   -- -------------------------------------------------------------------------

   --! Merging extracted header fields into one data word
   header_fifo_data_in <= hfe_header_protocol &
                          hfe_header_dstport  &
                          hfe_header_srcport  &
                          hfe_header_dstip    &
                          hfe_header_srcip;

   --! FIFO instantiation
   header_fifo_i : entity work.FIFO
   generic map (
      DATA_WIDTH   => 296, -- 128 + 128 + 16 + 16 + 8
      DISTMEM_TYPE => 16,
      ITEMS        => HEADER_FIFO_ITEMS,
      BLOCK_SIZE   => 4
   )
   port map (
      CLK          => CLK,
      RESET        => RESET,

      --! Write interface
      DATA_IN      => header_fifo_data_in,
      WRITE_REQ    => hfe_header_parsing_done_rise,
      FULL         => open, -- FIFO is expected to never overflow
      LSTBLK       => open,

      --! Read interface
      DATA_OUT     => header_fifo_data_out,
      READ_REQ     => header_fifo_rd,
      EMPTY        => header_fifo_empty
   );

   --! Decomposition of header FIFO's data word on the output
   header_fifo_srcip    <= header_fifo_data_out(127 downto   0);
   header_fifo_dstip    <= header_fifo_data_out(255 downto 128);
   header_fifo_srcport  <= header_fifo_data_out(271 downto 256);
   header_fifo_dstport  <= header_fifo_data_out(287 downto 272);
   header_fifo_protocol <= header_fifo_data_out(295 downto 288);


   -- -------------------------------------------------------------------------
   --                    Filtering engine (cuckoo filter)
   -- -------------------------------------------------------------------------

   --! src/dst ready interconnection between header FIFO and filtering engine
   filter_in_ready <= NOT header_fifo_empty;
   header_fifo_rd  <= filter_in_next;

   --! filtering engine instantiation
   filtering_engine_i : entity work.AFILTER_CORE_V2
   generic map (
      RULES_DATA_WIDTH => RULES_DATA_WIDTH
   )
   port map (
      CLK              => CLK,
      RESET            => RESET, 
       
      --! Data input interface
      IN_SRCIP         => header_fifo_srcip,
      IN_DSTIP         => header_fifo_dstip,
      IN_SRCPORT       => header_fifo_srcport,
      IN_DSTPORT       => header_fifo_dstport,
      IN_PROTOCOL      => header_fifo_protocol,
      IN_READY         => filter_in_ready,
      IN_NEXT          => filter_in_next,
       
      --! Output of filtering core
      RESULT           => filter_result,
      FOUND            => filter_found,
      RESULT_READY     => filter_ready,
      RESULT_NEXT      => filter_next,

      --! MI32 configuration interface
      MI_DWR           => mi_async_dwr,
      MI_ADDR          => mi_async_addr,
      MI_BE            => mi_async_be,
      MI_RD            => mi_async_rd,
      MI_WR            => mi_async_wr,
      MI_DRD           => mi_async_drd,
      MI_ARDY          => mi_async_ardy,
      MI_DRDY          => mi_async_drdy
   );

   --! filter_found to filter_deny conversion
   filter_deny <= NOT filter_found(1) AND NOT filter_found(0);


   -- -------------------------------------------------------------------------
   --    FIFO for packet latency (while header fields are being extracted)
   -- -------------------------------------------------------------------------

   packet_fifo_i : entity work.FL_FIFO
   generic map (
      DATA_WIDTH   => 512,
      USE_BRAMS    => true,
      ITEMS        => PACKET_FIFO_ITEMS,
      BLOCK_SIZE   => 4,
      STATUS_WIDTH => 1,
      PARTS        => 2
   )
   port map (
      CLK          => CLK,
      RESET        => RESET,

      RX_DATA      => hfe_packet_data,
      RX_SOF_N     => hfe_packet_sof_n,
      RX_SOP_N     => hfe_packet_sop_n,
      RX_EOF_N     => hfe_packet_eof_n,
      RX_EOP_N     => hfe_packet_eop_n,
      RX_REM       => hfe_packet_drem,
      RX_SRC_RDY_N => hfe_packet_src_rdy_n,
      RX_DST_RDY_N => hfe_packet_dst_rdy_n,

      TX_DATA      => packet_fifo_data,
      TX_SOF_N     => packet_fifo_sof_n,
      TX_SOP_N     => packet_fifo_sop_n,
      TX_EOF_N     => packet_fifo_eof_n,
      TX_EOP_N     => packet_fifo_eop_n,
      TX_REM       => packet_fifo_drem,
      TX_SRC_RDY_N => packet_fifo_src_rdy_n,
      TX_DST_RDY_N => packet_fifo_dst_rdy_n,

      LSTBLK       => open,
      FULL         => open,
      EMPTY        => open,
      STATUS       => open,
      FRAME_RDY    => open
   );


   -- -------------------------------------------------------------------------
   --                          Dispatcher unit
   -- -------------------------------------------------------------------------

   dispatcher_i : entity work.dispatcher
   generic map(
      DATA_WIDTH    => 512,
      RESULT_WIDTH  => 2*RULES_DATA_WIDTH
   )
   port map(
      CLK           => CLK,
      RESET         => RESET,

      RESULT        => filter_result,
      DENY          => filter_deny,
      CONTROL_READY => filter_ready,
      CONTROL_NEXT  => filter_next,

      RX_DATA       => packet_fifo_data,
      RX_SOF_N      => packet_fifo_sof_n,
      RX_SOP_N      => packet_fifo_sop_n,
      RX_EOF_N      => packet_fifo_eof_n,
      RX_EOP_N      => packet_fifo_eop_n,
      RX_DREM       => packet_fifo_drem,
      RX_SRC_RDY_N  => packet_fifo_src_rdy_n,
      RX_DST_RDY_N  => packet_fifo_dst_rdy_n,

      TX_DATA       => dispatcher_data,
      TX_SOF_N      => dispatcher_sof_n,
      TX_SOP_N      => dispatcher_sop_n,
      TX_EOF_N      => dispatcher_eof_n,
      TX_EOP_N      => dispatcher_eop_n,
      TX_DREM       => dispatcher_drem,
      TX_SRC_RDY_N  => dispatcher_src_rdy_n,
      TX_DST_RDY_N  => dispatcher_dst_rdy_n
   );


   -- -------------------------------------------------------------------------
   --                           FL2FLU converter
   -- -------------------------------------------------------------------------

   fl2flu_i : entity work.fl2flu
   generic map (
      DATA_WIDTH      => 512,
      SOP_POS_WIDTH   => 3,
      IN_PIPE_EN      => true,
      IN_PIPE_OUTREG  => true,
      OUT_PIPE_EN     => true,
      OUT_PIPE_OUTREG => true
   )
   port map (
      --! Common interface 
      CLK             => CLK,
      RESET           => RESET,
        
      --! Input interface (FL)
      RX_DATA         => dispatcher_data,
      RX_SOF_N        => dispatcher_sof_n,
      RX_SOP_N        => dispatcher_sop_n,
      RX_EOF_N        => dispatcher_eof_n,
      RX_EOP_N        => dispatcher_eop_n,
      RX_REM          => dispatcher_drem,
      RX_SRC_RDY_N    => dispatcher_src_rdy_n,
      RX_DST_RDY_N    => dispatcher_dst_rdy_n,
      
      --! Output interface (FLU)
      TX_DATA         => TX_DATA,
      TX_SOP_POS      => TX_SOP_POS,
      TX_EOP_POS      => TX_EOP_POS,
      TX_SOP          => TX_SOP,
      TX_EOP          => TX_EOP,
      TX_SRC_RDY      => TX_SRC_RDY,
      TX_DST_RDY      => TX_DST_RDY
   );


   -- -------------------------------------------------------------------------
   --         Clock domain crossing for cuckoo filter MI32 interface
   -- -------------------------------------------------------------------------

   mi_async_i: entity work.mi32_async_handshake
   port map (
      --! Write interface
      CLK_M     => MI_CLK,
      RESET_M   => MI_RESET,
      MI_M_DWR  => MI_DWR,
      MI_M_ADDR => MI_ADDR,
      MI_M_BE   => MI_BE,
      MI_M_RD   => MI_RD,
      MI_M_WR   => MI_WR,
      MI_M_DRD  => MI_DRD,
      MI_M_ARDY => MI_ARDY,
      MI_M_DRDY => MI_DRDY,
      
      --! Read interface
      CLK_S     => CLK,
      RESET_S   => RESET,
      MI_S_DWR  => mi_async_dwr,
      MI_S_ADDR => mi_async_addr,
      MI_S_BE   => mi_async_be,
      MI_S_RD   => mi_async_rd,
      MI_S_WR   => mi_async_wr,
      MI_S_DRD  => mi_async_drd,
      MI_S_ARDY => mi_async_ardy,
      MI_S_DRDY => mi_async_drdy
   );

end architecture full;
