-- extract_full.vhd: HFE-M Extractor full architecture, extracting most
-- important fields.
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_extract_full is
   generic (
      --* FrameLink width
      DATA_WIDTH           : integer := 64;
      --* Width of OFFSET_* signals (which are in Bytes).
      --* 7 means that headers should fit in first 128 B of packet
      --* (realistic guess). 14 Eth + 4*4 VLAN/MPLS + min. 40 IPv6 = min. 70
      OFFSET_WIDTH         : integer := 8;
      --* 0 or 1. 1 for higher frequency and lower resources, 
      --* but results one cycle later
      LATENCY              : integer := 1
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink input interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --------------------------------------------------------------------
      -- Analysis results
      --------------------------------------------------------------------
      -- Link layer ------------------------------------------------------
      --* Offset of the link layer (Ethernet) header start from SOF (in Bytes)
      LINK_OFFSET       : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Ethernet header present
      ACTIVE_ETHERNET   : in std_logic;
      --* Link layer analysis still running (above inputs are not valid)
      RUNNING_LINK      : in std_logic;

      -- VLAN and MPLS ---------------------------------------------------
      --* Offset of the first VLAN header start from SOF (in Bytes)
      VLAN0_OFFSET      : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second VLAN header start from SOF (in Bytes)
      VLAN1_OFFSET      : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the first MPLS header start from SOF (in Bytes)
      MPLS0_OFFSET      : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* Offset of the second MPLS header start from SOF (in Bytes)
      MPLS1_OFFSET      : in std_logic_vector(OFFSET_WIDTH-1 downto 0);

      --* First VLAN header present
      ACTIVE_VLAN0      : in std_logic;
      --* Second VLAN header present
      ACTIVE_VLAN1      : in std_logic;
      --* First MPLS header present
      ACTIVE_MPLS0      : in std_logic;
      --* Second MPLS header present
      ACTIVE_MPLS1      : in std_logic;

      --* VLAN & MPLS layer analysis still running (above inputs are not valid)
      RUNNING_VLAN_MPLS : in std_logic;

      -- Internet layer ------------------------------------------------
      --* Offset of the Internet layer (IP) header start from SOF (in Bytes)
      INTERNET_OFFSET   : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* IPv4 header present
      ACTIVE_IPV4       : in std_logic;
      --* IPv6 header present
      ACTIVE_IPV6       : in std_logic;
      --* Internet layer analysis still running (above inputs are not valid)
      RUNNING_INTERNET  : in std_logic;
      --* IPv4 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV4        : in std_logic;
      --* IPv6 header present but has errors (may appear after RUNNING_INTERNET)
      ERROR_IPV6        : in std_logic;

      -- Transport layer -----------------------------------------------
      --* Offset of the Transport layer header start from SOF (in Bytes)
      TRANSPORT_OFFSET  : in std_logic_vector(OFFSET_WIDTH-1 downto 0);
      --* TCP header present
      ACTIVE_TCP        : in std_logic;
      --* UDP header present
      ACTIVE_UDP        : in std_logic;
      --* ICMP header present
      ACTIVE_ICMP       : in std_logic;
      --* IGMP header present
      ACTIVE_IGMP       : in std_logic;
      --* MPLS in IP header present
      ACTIVE_MPLS_TUN   : in std_logic;
      --* SCTP header present
      ACTIVE_SCTP       : in std_logic;
      --* IPv6 in IP header present
      ACTIVE_IPV6_TUN   : in std_logic;
      --* IPv4 in IP header present
      ACTIVE_IPV4_TUN   : in std_logic;
      --* IPv6 No Next Header identifier present
      ACTIVE_IPV6_NO_NEXT:in std_logic;
      --* Last observed next-layer protocol
      PROTOCOL_IN       : in std_logic_vector(7 downto 0);
      --* First packet fragment
      FRAGMENT_FIRST_IN : in std_logic;
      --* Last packet fragment
      FRAGMENT_LAST_IN  : in std_logic;
      --* Transport layer analysis still running (above inputs are not valid)
      RUNNING_TRANSPORT : in std_logic;

      --------------------------------------------------------------------
      -- Extracted fields
      --------------------------------------------------------------------
      -- Internet layer -------------------------------------------------
      --+ Source IPv4 address
      SRCIPV4           : out std_logic_vector(31 downto 0);
      SRCIPV4_VLD       : out std_logic;
      --+ Destination IPv4 address
      DSTIPV4           : out std_logic_vector(31 downto 0);
      DSTIPV4_VLD       : out std_logic;
      --+ Source IPv6 address
      SRCIPV6           : out std_logic_vector(127 downto 0);
      SRCIPV6_VLD       : out std_logic;
      --+ Destination IPv6 address
      DSTIPV6           : out std_logic_vector(127 downto 0);
      DSTIPV6_VLD       : out std_logic;
      --* First packet fragment (valid with PROTOCOL_VLD)
      FRAGMENT_FIRST    : out std_logic;
      --* Last packet fragment (valid with PROTOCOL_VLD)
      FRAGMENT_LAST     : out std_logic;
      --+ Protocol
      PROTOCOL          : out std_logic_vector(7 downto 0);
      PROTOCOL_VLD      : out std_logic;

      -- Transport layer ------------------------------------------------
      --+ Source TCP port number
      SRCTCP            : out std_logic_vector(15 downto 0);
      SRCTCP_VLD        : out std_logic;
      --+ Destination TCP port number
      DSTTCP            : out std_logic_vector(15 downto 0);
      DSTTCP_VLD        : out std_logic;
      --+ Source UDP port number
      SRCUDP            : out std_logic_vector(15 downto 0);
      SRCUDP_VLD        : out std_logic;
      --+ Destination UDP port number
      DSTUDP            : out std_logic_vector(15 downto 0);
      DSTUDP_VLD        : out std_logic;

      --* Don't expect any new *_VLD to appear after this
      PARSING_DONE      : out std_logic
   );
end entity;

architecture full of hfe_m_extract_full is

   -- Must be able to count FL words that contain some header
   -- 128 Bytes should be enough with good reserve
   constant COUNTER_WIDTH  : integer := OFFSET_WIDTH - log2(DATA_WIDTH/8);

   -- Main word counter
   signal word_counter     : std_logic_vector(COUNTER_WIDTH-1 downto 0);

   signal sig_srcipv4_vld     : std_logic_vector(3 downto 0);
   signal sig_dstipv4_vld     : std_logic_vector(3 downto 0);
   signal sig_srcipv4_vld_and : std_logic;
   signal sig_dstipv4_vld_and : std_logic;

   signal sig_srcipv6_vld     : std_logic_vector(15 downto 0);
   signal sig_dstipv6_vld     : std_logic_vector(15 downto 0);
   signal sig_srcipv6_vld_and : std_logic;
   signal sig_dstipv6_vld_and : std_logic;

   signal sig_srcport         : std_logic_vector(15 downto 0);
   signal sig_dstport         : std_logic_vector(15 downto 0);
   signal sig_srcport_vld     : std_logic_vector(1 downto 0);
   signal sig_dstport_vld     : std_logic_vector(1 downto 0);
   signal sig_srcport_vld_and : std_logic;
   signal sig_dstport_vld_and : std_logic;

   signal sig_parsing_done : std_logic;

   signal active_tcp_udp   : std_logic;

   signal reg_active_ipv4     : std_logic;
   signal reg_active_ipv6     : std_logic;
   signal reg_active_tcp      : std_logic;
   signal reg_active_udp      : std_logic;
   signal reg_protocol        : std_logic_vector(7 downto 0);
   signal reg_fragment_first  : std_logic;
   signal reg_fragment_last   : std_logic;
   signal reg_running_link    : std_logic;
   signal reg_running_vlan_mpls:std_logic;
   signal reg_running_internet: std_logic;
   signal reg_running_transport:std_logic;

   signal ones             : std_logic_vector(127 downto 0);

begin

   ones  <= (others => '1');

   --* Count FL words, reset with SOF, saturate
   word_counter_p : process(CLK)
   begin
      if CLK'event and CLK = '1' then
         if RESET = '1' then
            word_counter <= (others => '0');
         else
            if FL_SRC_RDY_N = '0' and FL_DST_RDY_N = '0' then
               if FL_EOF_N = '0' then
                  word_counter <= (others => '0');
               else
                  if word_counter /= ones(COUNTER_WIDTH-1 downto 0) then
                     word_counter <= word_counter + 1;
                  end if;
               end if;
            end if;
         end if;
      end if;
   end process;

   -- Do not delay control signals if latency is zero
   latency_0_gen : if LATENCY = 0 generate
      reg_active_ipv4      <= ACTIVE_IPV4;
      reg_active_ipv6      <= ACTIVE_IPV6;
      reg_active_tcp       <= ACTIVE_TCP;
      reg_active_udp       <= ACTIVE_UDP;
      reg_protocol         <= PROTOCOL_IN;
      reg_fragment_first   <= FRAGMENT_FIRST_IN;
      reg_fragment_last    <= FRAGMENT_LAST_IN;
      reg_running_link     <= RUNNING_LINK;
      reg_running_vlan_mpls<= RUNNING_VLAN_MPLS;
      reg_running_internet <= RUNNING_INTERNET;
      reg_running_transport<= RUNNING_TRANSPORT;
   end generate;

   -- Delay control signals if latency is one
   latency_1_gen : if LATENCY = 1 generate
      reg_latency_p : process(CLK)
      begin
         if CLK'event and CLK = '1' then
            if RESET = '1' then
               reg_active_ipv4      <= '0';
               reg_active_ipv6      <= '0';
               reg_active_tcp       <= '0';
               reg_active_udp       <= '0';
               reg_running_link     <= '0';
               reg_running_vlan_mpls<= '0';
               reg_running_internet <= '0';
               reg_running_transport<= '0';
            else
               reg_active_ipv4      <= ACTIVE_IPV4;
               reg_active_ipv6      <= ACTIVE_IPV6;
               reg_active_tcp       <= ACTIVE_TCP;
               reg_active_udp       <= ACTIVE_UDP;
               reg_protocol         <= PROTOCOL_IN;
               reg_fragment_first   <= FRAGMENT_FIRST_IN;
               reg_fragment_last    <= FRAGMENT_LAST_IN;
               reg_running_link     <= RUNNING_LINK;
               reg_running_vlan_mpls<= RUNNING_VLAN_MPLS;
               reg_running_internet <= RUNNING_INTERNET;
               reg_running_transport<= RUNNING_TRANSPORT;
            end if;
         end if;
      end process;
   end generate;

   -- Internet layer -------------------------------------------------
   get_ipv4_gen : for i in 0 to 3 generate
      --* Extract source IPv4
      get_srcipv4 : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 12+i  -- SRC IP address starts at byte 12
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => INTERNET_OFFSET,
         ACTIVE         => ACTIVE_IPV4,
         BYTE_OUT       => SRCIPV4(31-i*8 downto 24-i*8),
         VALID          => sig_srcipv4_vld(i)
      );

      --* Extract destination IPv4
      get_dstipv4 : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 16+i  -- DST IP address starts at byte 16
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => INTERNET_OFFSET,
         ACTIVE         => ACTIVE_IPV4,
         BYTE_OUT       => DSTIPV4(31-i*8 downto 24-i*8),
         VALID          => sig_dstipv4_vld(i)
      );
   end generate;

   SRCIPV4_VLD <= sig_srcipv4_vld_and and reg_active_ipv4;
   DSTIPV4_VLD <= sig_dstipv4_vld_and and reg_active_ipv4;

   sig_srcipv4_vld_and <= sig_srcipv4_vld(0) and sig_srcipv4_vld(1) and
                  sig_srcipv4_vld(2) and sig_srcipv4_vld(3);

   sig_dstipv4_vld_and <= sig_dstipv4_vld(0) and sig_dstipv4_vld(1) and
                  sig_dstipv4_vld(2) and sig_dstipv4_vld(3);

   get_ipv6_gen : for i in 0 to 15 generate
      --* Extract source IPv6
      get_srcipv6 : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 8+i  -- SRC IP address starts at byte 8
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => INTERNET_OFFSET,
         ACTIVE         => ACTIVE_IPV6,
         BYTE_OUT       => SRCIPV6(127-i*8 downto 120-i*8),
         VALID          => sig_srcipv6_vld(i)
      );

      --* Extract destination IPv6
      get_dstipv6 : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 24+i  -- DST IP address starts at byte 24
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => INTERNET_OFFSET,
         ACTIVE         => ACTIVE_IPV6,
         BYTE_OUT       => DSTIPV6(127-i*8 downto 120-i*8),
         VALID          => sig_dstipv6_vld(i)
      );
   end generate;

   SRCIPV6_VLD <= sig_srcipv6_vld_and and reg_active_ipv6;
   DSTIPV6_VLD <= sig_dstipv6_vld_and and reg_active_ipv6;

   sig_srcipv6_vld_and <= sig_srcipv6_vld(0) and sig_srcipv6_vld(1) and
                  sig_srcipv6_vld(2) and sig_srcipv6_vld(3) and
                  sig_srcipv6_vld(4) and sig_srcipv6_vld(5) and
                  sig_srcipv6_vld(6) and sig_srcipv6_vld(7) and
                  sig_srcipv6_vld(8) and sig_srcipv6_vld(9) and
                  sig_srcipv6_vld(10) and sig_srcipv6_vld(11) and
                  sig_srcipv6_vld(12) and sig_srcipv6_vld(13) and
                  sig_srcipv6_vld(14) and sig_srcipv6_vld(15);

   sig_dstipv6_vld_and <= sig_dstipv6_vld(0) and sig_dstipv6_vld(1) and
                  sig_dstipv6_vld(2) and sig_dstipv6_vld(3) and
                  sig_dstipv6_vld(4) and sig_dstipv6_vld(5) and
                  sig_dstipv6_vld(6) and sig_dstipv6_vld(7) and
                  sig_dstipv6_vld(8) and sig_dstipv6_vld(9) and
                  sig_dstipv6_vld(10) and sig_dstipv6_vld(11) and
                  sig_dstipv6_vld(12) and sig_dstipv6_vld(13) and
                  sig_dstipv6_vld(14) and sig_dstipv6_vld(15);

   --* Protocol extracted in the analysis part
   PROTOCOL <= reg_protocol;
   PROTOCOL_VLD <= not reg_running_transport and (reg_active_ipv4 or reg_active_ipv6);
   FRAGMENT_FIRST <= reg_fragment_first;
   FRAGMENT_LAST  <= reg_fragment_last;

   get_port_gen : for i in 0 to 1 generate
      --* Extract source port
      get_srcport : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 0+i  -- SRC port starts at byte 0
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => TRANSPORT_OFFSET,
         ACTIVE         => active_tcp_udp,
         BYTE_OUT       => sig_srcport(15-i*8 downto 8-i*8),
         VALID          => sig_srcport_vld(i)
      );
      --* Extract destination port
      get_dstport : entity work.hfe_m_getbyte
      generic map (
         DATA_WIDTH     => DATA_WIDTH,
         COUNTER_WIDTH  => COUNTER_WIDTH,
         LATENCY        => LATENCY,
         BYTE_OFFSET    => 2+i  -- DST port starts at byte 2
      )
      port map (
         CLK            => CLK,
         RESET          => RESET,
         FL_DATA        => FL_DATA,
         FL_DREM        => FL_DREM,
         FL_SRC_RDY_N   => FL_SRC_RDY_N,
         FL_DST_RDY_N   => FL_DST_RDY_N,
         FL_SOF_N       => FL_SOF_N,
         FL_EOF_N       => FL_EOF_N,
         FL_SOP_N       => FL_SOP_N,
         FL_EOP_N       => FL_EOP_N,
         WORD_COUNTER   => word_counter,
         HEADER_OFFSET  => TRANSPORT_OFFSET,
         ACTIVE         => active_tcp_udp,
         BYTE_OUT       => sig_dstport(15-i*8 downto 8-i*8),
         VALID          => sig_dstport_vld(i)
      );
   end generate;

   active_tcp_udp <= ACTIVE_TCP or ACTIVE_UDP;

   SRCTCP <= sig_srcport;
   DSTTCP <= sig_dstport;
   SRCUDP <= sig_srcport;
   DSTUDP <= sig_dstport;

   sig_dstport_vld_and <= sig_dstport_vld(0) and sig_dstport_vld(1);
   sig_srcport_vld_and <= sig_srcport_vld(0) and sig_srcport_vld(1);

   SRCTCP_VLD <= sig_srcport_vld_and and reg_active_tcp;
   DSTTCP_VLD <= sig_dstport_vld_and and reg_active_tcp;
   SRCUDP_VLD <= sig_srcport_vld_and and reg_active_udp;
   DSTUDP_VLD <= sig_dstport_vld_and and reg_active_udp;

   -- Don't expect any new results if:
   -- No layer analyzer is running AND
   -- We have DST IPV4 or IPV4 is not active AND
   -- We have DST IPV6 or IPV6 is not active AND
   -- We have DST port or (TCP nor UDP is not active)
   sig_parsing_done <= '1' when
      (reg_running_link = '0' and reg_running_vlan_mpls = '0' and
       reg_running_internet = '0' and reg_running_transport = '0') and
      (reg_active_ipv4 = '0' or sig_dstipv4_vld_and = '1') and
      (reg_active_ipv6 = '0' or sig_dstipv6_vld_and = '1') and
      ((reg_active_tcp = '0' and reg_active_udp = '0') or sig_dstport_vld_and = '1')
      else '0';

   PARSING_DONE <= sig_parsing_done;

end architecture full;
