-- eth_phy.vhd : Ethernet PHY layer (PCS and PMA sublayers) implementation for
--               Fiberblaze FB1CG card
--!
--! \file
--! \brief Ethernet PHY layer implementation for Fiberblaze FB1CG card
--! \author Jiri Matousek <xmatou06@stud.fit.vutbr.cz>
--! \date 2014
--!
--! \section License
--!
--! Copyright (C) 2014 CESNET
--!
--! Redistribution and use in source and binary forms, with or without
--! modification, are permitted provided that the following conditions
--! are met:
--! 1. Redistributions of source code must retain the above copyright
--!    notice, this list of conditions and the following disclaimer.
--! 2. Redistributions in binary form must reproduce the above copyright
--!    notice, this list of conditions and the following disclaimer in
--!    the documentation and/or other materials provided with the
--!    distribution.
--! 3. Neither the name of the Company nor the names of its contributors
--!    may be used to endorse or promote products derived from this
--!    software without specific prior written permission.
--!
--! This software is provided ``as is'', and any express or implied
--! warranties, including, but not limited to, the implied warranties of
--! merchantability and fitness for a particular purpose are disclaimed.
--! In no event shall the company or contributors be liable for any
--! direct, indirect, incidental, special, exemplary, or consequential
--! damages (including, but not limited to, procurement of substitute
--! goods or services; loss of use, data, or profits; or business
--! interruption) however caused and on any theory of liability, whether
--! in contract, strict liability, or tort (including negligence or
--! otherwise) arising in any way out of the use of this software, even
--! if advised of the possibility of such damage.
--!


library ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
use ieee.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                         Architecture declaration
-- ----------------------------------------------------------------------------

architecture full of eth_phy is

   --! Constants declaration
   -- -------------------------------------------------------------------------

   --! MI splitter constants
   constant MI_SPLIT_DATA_WIDTH : integer := 32;
   constant MI_SPLIT_ITEMS      : integer := 2;
 
   --! Signals declaration
   -- -------------------------------------------------------------------------

   --! MI Splitter outputs
   signal mi_split_dwr  : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH-1 downto 0);
   signal mi_split_addr : std_logic_vector(MI_SPLIT_ITEMS*32-1 downto 0);
   signal mi_split_cs   : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_rd   : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_wr   : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_be   : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH/8-1 downto 0);
   signal mi_split_drd  : std_logic_vector(MI_SPLIT_ITEMS*MI_SPLIT_DATA_WIDTH-1 downto 0);
   signal mi_split_ardy : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);
   signal mi_split_drdy : std_logic_vector(MI_SPLIT_ITEMS-1 downto 0);

-- ----------------------------------------------------------------------------
--                            Architecture body
-- ----------------------------------------------------------------------------

begin


   -- -------------------------------------------------------------------------
   --                              MI splitter
   -- -------------------------------------------------------------------------

   mi_split_cs_dec : entity work.dec1fn
      generic map (2)
      port map (MI_ADDR(19 downto 19), mi_split_cs);
   mi_split_rd   <= mi_split_cs and (1 downto 0 => MI_RD);
   mi_split_wr   <= mi_split_cs and (1 downto 0 => MI_WR);
   mi_split_dwr  <= MI_DWR & MI_DWR;
   mi_split_addr <= MI_ADDR & MI_ADDR;
   mi_split_be   <= MI_BE & MI_BE;

   mi_ardy_mux : entity work.GEN_MUX
      generic map (1,2)
      port map (mi_split_ardy,MI_ADDR(19 downto 19),DATA_OUT(0)=>MI_ARDY);
   mi_drdy_or : entity work.GEN_OR
      generic map (2)
      port map (mi_split_drdy,MI_DRDY);
   mi_drd_mux : entity work.GEN_MUX_ONEHOT
      generic map (32,2)
      port map (mi_split_drd,mi_split_drdy,MI_DRD);


   -- -------------------------------------------------------------------------
   --                     Ethernet PHY layer implementation
   -- -------------------------------------------------------------------------
   lr4_eth_gen : if not SR10_MODE generate
      eth_phy_i : entity work.phy_100ge_caui4
      port map (
         RESET      => RESET,
         DRPCLK     => DRPCLK,

         -- CGMII interface
         RXCLK      => CGMII_RXCLK,
         CGMII_RXD  => CGMII_RXD,
         CGMII_RXC  => CGMII_RXC,
         TXCLK      => CGMII_TXCLK,
         CGMII_TXD  => CGMII_TXD,
         CGMII_TXC  => CGMII_TXC,
         CLK_STABLE => CLK_STABLE,

         -- Transceiver reference clocks
         REFCLK_P   => CFP2_GTZ_REFCLK_P,
         REFCLK_N   => CFP2_GTZ_REFCLK_N,

         -- CAUI4 - GTZ serial ports
         RXP        => CFP2_RX_P(9 downto 6),
         RXN        => CFP2_RX_N(9 downto 6),
         TXP        => CFP2_TX_P(9 downto 6),
         TXN        => CFP2_TX_N(9 downto 6),
         RXPOLARITY => "1111", -- The RX serial data polarity is inverted on the FB100G1 card
         TXPOLARITY => "1111", -- The TX serial data polarity is inverted on the FB100G1 card

         -- Signal detect from the transceiver/PMD
         SIGNAL_DET => SIGNAL_DET,

         -- Management (MI32)
         MI_CLK     => MI_CLK,
         MI_DWR     => mi_split_dwr((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
         MI_ADDR    => mi_split_addr((0+1)*32-1 downto 0*32),
         MI_RD      => mi_split_rd(0),
         MI_WR      => mi_split_wr(0),
         MI_BE      => mi_split_be((0+1)*MI_SPLIT_DATA_WIDTH/8-1 downto 0*MI_SPLIT_DATA_WIDTH/8),
         MI_DRD     => mi_split_drd((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
         MI_ARDY    => mi_split_ardy(0),
         MI_DRDY    => mi_split_drdy(0)
      );
   end generate;

   sr10_eth_gen : if SR10_MODE generate
      eth_phy_i : entity work.phy_100ge_caui
      port map (
         RESET      => RESET,
         DRPCLK     => DRPCLK,

         -- CGMII interface
         RXCLK      => CGMII_RXCLK,
         CGMII_RXD  => CGMII_RXD,
         CGMII_RXC  => CGMII_RXC,
         TXCLK      => CGMII_TXCLK,
         CGMII_TXD  => CGMII_TXD,
         CGMII_TXC  => CGMII_TXC,
         CLK_STABLE => CLK_STABLE,

         -- Transceiver reference clocks
         GTZ_REFCLK_P   => CFP2_GTZ_REFCLK_P,
         GTZ_REFCLK_N   => CFP2_GTZ_REFCLK_N,
         GTH_REFCLK_P   => CFP2_GTH_REFCLK_P,
         GTH_REFCLK_N   => CFP2_GTH_REFCLK_N,

         -- CAUI4 - GTZ serial ports
         RXP        => CFP2_RX_P,
         RXN        => CFP2_RX_N,
         TXP        => CFP2_TX_P,
         TXN        => CFP2_TX_N,
         RXPOLARITY => "1111011111",
         TXPOLARITY => "1111011111",

         -- Signal detect from the transceiver/PMD
         SIGNAL_DET => SIGNAL_DET(0),

         -- Management (MI32)
         MI_CLK     => MI_CLK,
         MI_DWR     => mi_split_dwr((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
         MI_ADDR    => mi_split_addr((0+1)*32-1 downto 0*32),
         MI_RD      => mi_split_rd(0),
         MI_WR      => mi_split_wr(0),
         MI_BE      => mi_split_be((0+1)*MI_SPLIT_DATA_WIDTH/8-1 downto 0*MI_SPLIT_DATA_WIDTH/8),
         MI_DRD     => mi_split_drd((0+1)*MI_SPLIT_DATA_WIDTH-1 downto 0*MI_SPLIT_DATA_WIDTH),
         MI_ARDY    => mi_split_ardy(0),
         MI_DRDY    => mi_split_drdy(0)
      );
   end generate;


   -- -------------------------------------------------------------------------
   --                  MDIO controller (for CFP2 transceiver)
   -- -------------------------------------------------------------------------

   cfp2_mdio_inst : entity work.mdio_ctrl_mi32
   generic map (
      DIV_CLK    => X"38"
   )
   port map (
      RESET      => MI_RESET,
      CLK        => MI_CLK,

      -- MDIO interface
      MDC        => CFP2_MDC,
      MDIO_I     => CFP2_MDIO_I,
      MDIO_O     => CFP2_MDIO_O,
      MDIO_OE    => CFP2_MDIO_OE,

      -- MI32 interface
      MI_DWR     => mi_split_dwr((1+1)*MI_SPLIT_DATA_WIDTH-1 downto 1*MI_SPLIT_DATA_WIDTH),
      MI_ADDR    => mi_split_addr((1+1)*32-1 downto 1*32),
      MI_RD      => mi_split_rd(1),
      MI_WR      => mi_split_wr(1),
      MI_BE      => mi_split_be((1+1)*MI_SPLIT_DATA_WIDTH/8-1 downto 1*MI_SPLIT_DATA_WIDTH/8),
      MI_DRD     => mi_split_drd((1+1)*MI_SPLIT_DATA_WIDTH-1 downto 1*MI_SPLIT_DATA_WIDTH),
      MI_ARDY    => mi_split_ardy(1),
      MI_DRDY    => mi_split_drdy(1)
   );

end architecture full;
