-- cgmii_enc_arch.vhd: CGMII_ENC full architecture.
-- Copyright (C) 2012 CESNET
-- Author(s): Vaclav Hummel <xhumme00@stud.fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;

-- ----------------------------------------------------------------------------
--                      Architecture declaration
-- ----------------------------------------------------------------------------
architecture full of cgmii_obuf_enc is

   --Constants --------------------------------------------
   constant NUM_OF_LANES   : integer := 8;

   --! Input data signal
   signal pipe_rx_data     : std_logic_vector(511 downto 0) := (others => '0');
   --! Input SOP_V signal
   signal pipe_rx_sop_v    : std_logic_vector(7 downto 0) := (others => '0');
   --! Input EOP_V signal  
   signal pipe_rx_eop_v    : std_logic_vector(7 downto 0) := (others => '0');
   --! Input EOP_POS signal
   signal pipe_rx_eop_pos  : std_logic_vector(2 downto 0) := (others => '0');

   --Substitution multiplexers ----------------------------
   --Data types declaration
   --! Data lane selection bus
   type LANE_SEL_BUS is array(0 to NUM_OF_LANES-1) of std_logic_vector(3 downto 0);
   --! Data lane bus
   type LANE_DATA_BUS is array(0 to NUM_OF_LANES-1) of std_logic_vector(63 downto 0);

   --! Lane output sel. bus
   signal data_lane_sel    : LANE_SEL_BUS;
   --! Lane data input
   signal data_lane_in     : LANE_DATA_BUS;
   --! Lane data output
   signal data_lane_out    : LANE_DATA_BUS;
   --! CGMII control bus output
   signal cgmii_cmd_out   : std_logic_vector(63 downto 0);
   --! Get next CRC value
   signal next_crc_en      : std_logic;

begin

   --------------------------------------------------------
   --Registers   
   --------------------------------------------------------

   --! \brief Input pipeline register

   input_pipe_regp:process(CLK)
   begin
      if(CLK = '1' and CLK'event)then
         if(RESET = '1')then
            pipe_rx_sop_v     <= (others=>'0'); 
            pipe_rx_eop_v     <= (others=>'0'); 
         else
            pipe_rx_data      <= RX_DATA;
            pipe_rx_sop_v     <= RX_SOP_V;
            pipe_rx_eop_v     <= RX_EOP_V;
            pipe_rx_eop_pos   <= RX_EOP_POS;
         end if;
      end if;
   end process;

   --------------------------------------------------------
   --Output multiplexers for DATA/IDLE/CRC
   --------------------------------------------------------
   INPUT_MUX_MAP_GEN:for i in 0 to NUM_OF_LANES-1 generate
      --! Data lanes map (map from pipelined data)
      data_lane_in(i) <= pipe_rx_data((i+1)*64-1 downto i*64);
   end generate;

   OUTPUT_MUX_GEN:for i in 0 to NUM_OF_LANES-1 generate
      --! \brief Output data selection path.
      --! \details This multiplexer could select between  DATA,PREAMBLE, CRC+IFG and IDLE word.
      data_lane_muxp:process(data_lane_in,data_lane_sel,CRC32_DATA_IN)
      begin
         case data_lane_sel(i) is
            --Data & CRC, later EFD
            when "0000" => data_lane_out(i) <= data_lane_in(i);
            when "0001" => data_lane_out(i) <= CRC32_DATA_IN(7 downto 0)  & data_lane_in(i)(55 downto 0);
            when "0010" => data_lane_out(i) <= CRC32_DATA_IN(15 downto 0) & data_lane_in(i)(47 downto 0);
            when "0011" => data_lane_out(i) <= CRC32_DATA_IN(23 downto 0) & data_lane_in(i)(39 downto 0);
            when "0100" => data_lane_out(i) <= CRC32_DATA_IN(31 downto 0) & data_lane_in(i)(31 downto 0);
            --Data & CRC & EFD + idles
            when "0101" => data_lane_out(i) <= x"FD" & CRC32_DATA_IN(31 downto 0) & data_lane_in(i)(23 downto 0);
            when "0110" => data_lane_out(i) <= x"07FD" & CRC32_DATA_IN(31 downto 0) & data_lane_in(i)(15 downto 0);
            when "0111" => data_lane_out(i) <= x"0707FD" & CRC32_DATA_IN(31 downto 0) & data_lane_in(i)(7 downto 0);
            --CRC & EFD + idle
            when "1000" => data_lane_out(i) <= x"070707FD" & CRC32_DATA_IN(31 downto 0);
            when "1001" => data_lane_out(i) <= x"07070707FD" & CRC32_DATA_IN(31 downto 8);
            when "1010" => data_lane_out(i) <= x"0707070707FD" & CRC32_DATA_IN(31 downto 16);
            when "1011" => data_lane_out(i) <= x"070707070707FD" & CRC32_DATA_IN(31 downto 24);
            --EFD, Idles only
            when "1100" => data_lane_out(i) <= x"07070707070707FD";
            --Idles only
            when "1101" => data_lane_out(i) <= x"0707070707070707";
            --Preamble
            when others => data_lane_out(i) <= x"D5555555555555FB"; 
         end case;
      end process;
   end generate;
 
   --------------------------------------------------------
   --FSM
   --------------------------------------------------------

   FSM_I:entity work.CGMII_ENC_FSM
   port map(
      -----------------------------------------------------
      --! \name Common interface
      -----------------------------------------------------
      --! Clock signal
      CLK         => CLK,
      --! Synchronous reset
      RESET       => RESET,

      -----------------------------------------------------
      --! \name Input Intefrace 
      -----------------------------------------------------
      --! SOP_V from component input (not delayed)
      SOP_V       => RX_SOP_V, 
      --! EOP_V from component input (not delayed)
      EOP_V       => RX_EOP_V,

      --! Delayed SOP_V input
      PIPE_SOP_V     => pipe_rx_sop_v,
      --! Delayed EOP_V input
      PIPE_EOP_V     => pipe_rx_eop_v,
      --! Delayed EOP_POS input
      PIPE_EOP_POS   => pipe_rx_eop_pos,

      -----------------------------------------------------
      --! \name Output interface
      -----------------------------------------------------
      --! Lane 0 selection signal
      LANE0_SEL      => data_lane_sel(0),
      --! Lane 1 selection signal 
      LANE1_SEL      => data_lane_sel(1),
      --! Lane 2 selection signal 
      LANE2_SEL      => data_lane_sel(2),
      --! Lane 3 selection signal
      LANE3_SEL      => data_lane_sel(3),
      --! Lane 4 selection signal
      LANE4_SEL      => data_lane_sel(4),
      --! Lane 5 selection signal 
      LANE5_SEL      => data_lane_sel(5),
      --! Lane 6 selection signal 
      LANE6_SEL      => data_lane_sel(6),
      --! Lane 7 selection signal
      LANE7_SEL      => data_lane_sel(7),

      --! Get next CRC value
      NEXT_CRC       => next_crc_en,

      --! Output XLGMII control bus
      CGMII_CMD     => cgmii_cmd_out
   );

   --------------------------------------------------------
   --Output signal map
   --------------------------------------------------------
   --! CGMII output data lane map
   OUTPUT_MAP_GEN:for i in 0 to NUM_OF_LANES-1 generate
      CGMII_TXD((i+1)*64-1 downto i*64) <= data_lane_out(i);
   end generate;

   --! CGMII output control lane map
   CGMII_TXC  <= cgmii_cmd_out;

   --! Read next CRC value
   CRC32_READ_REQ <= next_crc_en; 

end architecture full;
