-- tree_3stages_distmem_ent.vhd: Interfaces of three stages (levels) of a binary search tree implemented using distmems
-- Copyright (C) 2013 Brno University of Technology
-- Author(s): Lukas Kekely <ikekely@fit.vutbr.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
-- $Id: tree_3stages_distmem_ent.vhd 4531 2013-12-20 16:29:12Z xkekel00 $
--
-- TODO:
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_arith.all;
use IEEE.std_logic_unsigned.all;
use IEEE.numeric_std.all;
use WORK.math_pack.all;
use WORK.tree_func.all;

-- ----------------------------------------------------------------------------
--                        Entity declaration
-- ----------------------------------------------------------------------------
entity tree_3stages_distmem is
  generic(
    -- which is the first implemented level (row) of the tree (root=0)
    TREE_LEVEL     : integer := 0;
    -- width of keys inside tree nodes
    KEY_WIDTH      : integer := 128;
    -- type of memory used in implementation
    -- MEM_TYPE       : MEMORY_TYPE := distmem; -- only distmem supported
    -- use registers on input interface of stage 0?
    INPUT0_REG      : boolean := false;
    -- use registers between memory and comparator of stage 0?
    MIDDLE0_REG     : boolean := false; 
    -- use registers on output interface of stage 0?
    OUTPUT0_REG     : boolean := false;
    -- use registers on input interface of stage 1?
    INPUT1_REG      : boolean := false;
    -- use registers between memory and comparator of stage 1?
    MIDDLE1_REG     : boolean := false; 
    -- use registers on output interface of stage 1?
    OUTPUT1_REG     : boolean := false;
    -- use registers on input interface of stage 2?
    INPUT2_REG      : boolean := false;
    -- use registers between memory and comparator of stage 2?
    MIDDLE2_REG     : boolean := false; 
    -- use registers on output interface of stage 2?
    OUTPUT2_REG     : boolean := false
  );
  port(
    CLK           : in  std_logic;
    RESET         : in  std_logic;
    
    -- input interface
    IN_ADDR       : in  std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
    IN_KEY        : in  std_logic_vector(KEY_WIDTH-1 downto 0);
    IN_VLD        : in  std_logic;
    IN_FOUNDABLE  : in  std_logic;
    IN_CHANGE_EN  : in  std_logic;
    
    -- output interface
    OUT_ADDR      : out std_logic_vector(TREE_LEVEL+2 downto 0);
    OUT_KEY       : out std_logic_vector(KEY_WIDTH-1 downto 0);
    OUT_VLD       : out std_logic;
    OUT_FOUNDABLE : out std_logic;
    OUT_CHANGE_EN : out std_logic;
    
    -- configuration interface 0
    CFG0_DI        : in  std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG0_DO        : out std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG0_DRDY      : out std_logic;
    CFG0_ADDR      : in  std_logic_vector(max(0,TREE_LEVEL-1) downto 0);
    CFG0_WE        : in  std_logic; -- '1' for write, '0' for read
    CFG0_EN        : in  std_logic; -- '1' to enabled operation specified by WE on item ADDR
    
    -- configuration interface 1
    CFG1_DI        : in  std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG1_DO        : out std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG1_DRDY      : out std_logic;
    CFG1_ADDR      : in  std_logic_vector(TREE_LEVEL downto 0);
    CFG1_WE        : in  std_logic; -- '1' for write, '0' for read
    CFG1_EN        : in  std_logic; -- '1' to enabled operation specified by WE on item ADDR
    
    -- configuration interface 2
    CFG2_DI        : in  std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG2_DO        : out std_logic_vector(1+KEY_WIDTH+1+log2(KEY_WIDTH)-1 downto 0);
    CFG2_DRDY      : out std_logic;
    CFG2_ADDR      : in  std_logic_vector(TREE_LEVEL+1 downto 0);
    CFG2_WE        : in  std_logic; -- '1' for write, '0' for read
    CFG2_EN        : in  std_logic  -- '1' to enabled operation specified by WE on item ADDR
    -- NOTE: Only one CFGx_EN can be active in the same time!!!
  );
end entity;