-- ipv4.vhd: HFE-M IPv4 analyzer
-- Copyright (C) 2011 CESNET
-- Author(s): Viktor Puš <pus@cesnet.cz>
--
-- Redistribution and use in source and binary forms, with or without
-- modification, are permitted provided that the following conditions
-- are met:
-- 1. Redistributions of source code must retain the above copyright
--    notice, this list of conditions and the following disclaimer.
-- 2. Redistributions in binary form must reproduce the above copyright
--    notice, this list of conditions and the following disclaimer in
--    the documentation and/or other materials provided with the
--    distribution.
-- 3. Neither the name of the Company nor the names of its contributors
--    may be used to endorse or promote products derived from this
--    software without specific prior written permission.
--
-- This software is provided ``as is'', and any express or implied
-- warranties, including, but not limited to, the implied warranties of
-- merchantability and fitness for a particular purpose are disclaimed.
-- In no event shall the company or contributors be liable for any
-- direct, indirect, incidental, special, exemplary, or consequential
-- damages (including, but not limited to, procurement of substitute
-- goods or services; loss of use, data, or profits; or business
-- interruption) however caused and on any theory of liability, whether
-- in contract, strict liability, or tort (including negligence or
-- otherwise) arising in any way out of the use of this software, even
-- if advised of the possibility of such damage.
--
--
--

library IEEE;
use IEEE.std_logic_1164.all;
use IEEE.std_logic_unsigned.all;
use IEEE.std_logic_arith.all;
use IEEE.numeric_std.all;
use work.math_pack.all;

-- ----------------------------------------------------------------------------
--                      Entity declaration
-- ----------------------------------------------------------------------------

entity hfe_m_ipv4 is
   generic (
      --* FrameLink width
      DATA_WIDTH  : integer := 64;
      --* Width of WORD_COUNTER
      COUNTER_WIDTH:integer := 4
   );
   port (
      CLK         : in std_logic;
      RESET       : in std_logic;

      --+ FrameLink interface (passive port)
      FL_DATA     : in std_logic_vector(DATA_WIDTH-1 downto 0);
      FL_DREM     : in std_logic_vector(log2(DATA_WIDTH/8)-1 downto 0);
      FL_SRC_RDY_N: in std_logic;
      FL_DST_RDY_N: in std_logic;
      FL_SOF_N    : in std_logic;
      FL_EOF_N    : in std_logic;
      FL_SOP_N    : in std_logic;
      FL_EOP_N    : in std_logic;

      --* Counter of FL words (incremented by 1 with every word)
      WORD_COUNTER   : in std_logic_vector(COUNTER_WIDTH-1 downto 0);

      --* Offset of the header start from SOF (in Bytes)
      OFFSET_IN      : in std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --* Previous layer detected the IPv4 header
      ACTIVE_IN      : in std_logic;

      --* Offset of the next header start from SOF (in Bytes)
      OFFSET_OUT     : out std_logic_vector(COUNTER_WIDTH+log2(DATA_WIDTH/8)-1
         downto 0);

      --+ Active for each known next layer protocol
      ACTIVE_TCP           : out std_logic;
      ACTIVE_UDP           : out std_logic;
      ACTIVE_ICMP          : out std_logic;
      ACTIVE_IGMP          : out std_logic;
      ACTIVE_MPLS          : out std_logic;
      ACTIVE_SCTP          : out std_logic;
      ACTIVE_IPV6          : out std_logic;
      ACTIVE_IPV4          : out std_logic;
      --* Last observed next-layer protocol
      PROTOCOL             : out std_logic_vector(7 downto 0);
      --* Is set to 1 when packet is NOT fragmented or has fragment offset = 0
      FRAGMENT_FIRST       : out std_logic;
      --* Is set to 1 when packet is NOT fragmented or is
      --* the last fragment of the fragmented packet
      FRAGMENT_LAST        : out std_logic;

      --* Active for unknown next layer protocol
      ACTIVE_UNKNOWN : out std_logic;

      --* Packet header has errors
      MALFORMED      : out std_logic;

      --* Still parsing, outputs are not valid yet
      RUNNING        : out std_logic
   );
end entity;

architecture behavioral of hfe_m_ipv4 is

   signal version    : std_logic_vector(3 downto 0);
   signal version_bad: std_logic;
   signal length     : std_logic_vector(3 downto 0);
   signal length_b   : std_logic_vector(5 downto 0);
   signal ver_len_vld: std_logic;

   signal nexth   : std_logic_vector(7 downto 0);
   signal proto_vld  : std_logic;

   signal frag_l     : std_logic_vector(7 downto 0);
   signal frag_h     : std_logic_vector(7 downto 0);
   signal frag       : std_logic_vector(12 downto 0);
   signal frag_l_vld : std_logic;
   signal frag_h_vld : std_logic;
   signal frag_vld   : std_logic;

   signal sig_active_tcp           : std_logic;
   signal sig_active_udp           : std_logic;
   signal sig_active_icmp          : std_logic;
   signal sig_active_igmp          : std_logic;
   signal sig_active_mpls          : std_logic;
   signal sig_active_sctp          : std_logic;
   signal sig_active_ipv6          : std_logic;
   signal sig_active_ipv4          : std_logic;
   signal sig_active_unknown       : std_logic;

begin

   --* Get the Version and Header Length fields
   get_version_length_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 0 -- Version and Header Length fields are in byte 0
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER=> WORD_COUNTER,
      HEADER_OFFSET=>OFFSET_IN,
      ACTIVE      => ACTIVE_IN,
      VALID       => ver_len_vld,
      BYTE_OUT(3 downto 0) => length,
      BYTE_OUT(7 downto 4) => version
   );

   -- Convert to length in Bytes
   length_b <= length & "00";

   -- Compute and output the next header offset
   OFFSET_OUT <= OFFSET_IN + length_b;

   -- Detect malformed IPv4 header
   -- TODO: Compute header checksum
   version_bad <= '1' when version /= X"4" and ver_len_vld = '1' else '0';
   MALFORMED <= version_bad;

   --* Get the Fragment Offset field
   get_frag_l_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 7 -- Fragment Offset field in in bytes 6 and 7
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => frag_l_vld,
      BYTE_OUT       => frag_l
   );
   --* Get the Fragment Offset field
   get_frag_h_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 6 -- Fragment Offset field is in bytes 6 and 7
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => frag_h_vld,
      BYTE_OUT       => frag_h
   );

   -- frag_vld ignored, because proto_vld is later in packet
   frag_vld <= frag_h_vld and frag_l_vld; 
   frag <= frag_h(4 downto 0) & frag_l;
   FRAGMENT_FIRST <= '1' when frag = X"0000000000000" else '0';
   FRAGMENT_LAST <= '1' when frag_h(5) = '0' else '0';

   --* Get the Protocol field
   get_nexth_i : entity work.hfe_m_getbyte
   generic map(
      DATA_WIDTH     => DATA_WIDTH,
      COUNTER_WIDTH  => COUNTER_WIDTH,
      BYTE_OFFSET    => 9 -- Protocol field in in byte 9
   )
   port map(
      CLK         => CLK,
      RESET       => RESET,

      FL_DATA     => FL_DATA,
      FL_DREM     => FL_DREM,
      FL_SRC_RDY_N=> FL_SRC_RDY_N,
      FL_DST_RDY_N=> FL_DST_RDY_N,
      FL_SOF_N    => FL_SOF_N,
      FL_EOF_N    => FL_EOF_N,
      FL_SOP_N    => FL_SOP_N,
      FL_EOP_N    => FL_EOP_N,

      WORD_COUNTER   => WORD_COUNTER,
      HEADER_OFFSET  => OFFSET_IN,
      ACTIVE         => ACTIVE_IN,
      VALID          => proto_vld,
      BYTE_OUT       => nexth
   );

   sig_active_tcp <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"06"
                     and frag = X"0000000000000" else '0';
   sig_active_udp  <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"11"
                     and frag = X"0000000000000" else '0';
   sig_active_icmp <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and (nexth = X"01" or nexth = X"3A")
                     and frag = X"0000000000000" else '0';
   sig_active_igmp <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"02"
                     and frag = X"0000000000000" else '0';
   sig_active_mpls <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"89"
                     and frag = X"0000000000000" else '0';
   sig_active_sctp <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"84"
                     and frag = X"0000000000000" else '0';
   sig_active_ipv6 <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"29"
                     and frag = X"0000000000000" else '0';
   sig_active_ipv4 <= '1' when ACTIVE_IN = '1' and proto_vld = '1'
                     and nexth = X"04"
                     and frag = X"0000000000000" else '0';

   sig_active_unknown <= '1' when ACTIVE_IN = '1' and proto_vld = '1' and not
   (sig_active_tcp  = '1' or sig_active_udp  = '1' or
    sig_active_icmp = '1' or sig_active_igmp = '1' or
    sig_active_mpls = '1' or sig_active_sctp = '1' or
    sig_active_ipv6 = '1' or sig_active_ipv4 = '1')
    else '0';

   ACTIVE_TCP    <= sig_active_tcp;
   ACTIVE_UDP    <= sig_active_udp;
   ACTIVE_ICMP   <= sig_active_icmp;
   ACTIVE_IGMP   <= sig_active_igmp;
   ACTIVE_MPLS   <= sig_active_mpls;
   ACTIVE_SCTP   <= sig_active_sctp;
   ACTIVE_IPV6   <= sig_active_ipv6;
   ACTIVE_IPV4   <= sig_active_ipv4;
   ACTIVE_UNKNOWN<= sig_active_unknown;
   PROTOCOL      <= nexth;

   RUNNING <= ACTIVE_IN and not proto_vld;

end architecture behavioral;
